 /*
  * 
 * <LIC_AMD_STD>
 * Copyright (c) 2004 Advanced Micro Devices, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * The full GNU General Public License is included in this distribution in the
 * file called COPYING
 * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * File Contents: This file contains the  the Geode frame buffer
  *                device driver specific ioctl functions definitions.
  *
  * Notes:         All the geode frame buffer driver ioctl functions 
  *                use the durango library to access the hardware.
  *
  * Project:       Geode Frame buffer device driver
  * </DOC_AMD_STD>
  * 
  */

#include <linux/module.h>
#ifdef CONFIG_FB_NSC_GAL
#include "nsc_common_fb.h"

#include "nsc_galproto.h"
#include "nsc_ioctl.h"

#define GEODEFBDEBUG 0

#if GEODEFBDEBUG
#define DPRINTK(fmt, args...) printk(KERN_INFO "%s: " fmt, __FUNCTION__ , ## args)
#else
#define DPRINTK(fmt, args...)
#endif

/*
 * Look for our signature and if subfunction is
 * within range
 */

#define IO_PACKET_ISVALID(x)\
        ((x.dwSubfunction < GALFN_LASTFUNCTION_SUPPORTED) &&\
        (x.dwSignature == FBGAL_SIGNATURE))

static char bDispatchTableInitialized = 0;
static int MaxPacketSize;
static char *IoctlBuf;

static IOCTL_TABLE IoctlTable[] = {
/* General Adapter level functions */
   {Nscgfxfn_getadapterinfo, sizeof(GAL_ADAPTERINFO), sizeof(GAL_ADAPTERINFO)}
   ,
   {Nscgfxfn_setsoftvgastate, sizeof(GAL_SOFTVGASTATE), 0}
   ,
   {Nscgfxfn_getsoftvgastate, sizeof(GAL_SOFTVGASTATE),
    sizeof(GAL_SOFTVGASTATE)}
   ,
   {Nscgfxfn_waituntilidle, sizeof(GAL_WAITUNTILIDLE), 0}
   ,
   {Nscgfxfn_waitverticalblank, sizeof(GAL_WAITVERTICALBLANK), 0}
   ,
   {Nscgfxfn_setcrtenable, sizeof(GAL_CRTENABLE), 0}
   ,
   {Nscgfxfn_writereg, sizeof(GAL_HWACCESS), 0}
   ,
   {Nscgfxfn_readreg, sizeof(GAL_HWACCESS), sizeof(GAL_HWACCESS)}
   ,

/* Change/Get Display hardware state */
   {Nscgfxfn_ismodesupported, sizeof(GAL_DISPLAYMODE), sizeof(GAL_DISPLAYMODE)}
   ,
   {Nscgfxfn_setdisplaymode, sizeof(GAL_DISPLAYMODE), 0}
   ,
   {Nscgfxfn_getdisplaymode, sizeof(GAL_DISPLAYMODE), sizeof(GAL_DISPLAYMODE)}
   ,
   {Nscgfxfn_setbpp, sizeof(GAL_DISPLAYPARAMS), 0}
   ,
   {Nscgfxfn_setdisplaybpp, sizeof(GAL_DISPLAYPARAMS), 0}
   ,
   {Nscgfxfn_getdisplaybpp, sizeof(GAL_DISPLAYPARAMS),
    sizeof(GAL_DISPLAYPARAMS)}
   ,
   {Nscgfxfn_setdisplaypitch, sizeof(GAL_DISPLAYPARAMS), 0}
   ,
   {Nscgfxfn_getdisplaypitch, sizeof(GAL_DISPLAYPARAMS),
    sizeof(GAL_DISPLAYPARAMS)}
   ,
   {Nscgfxfn_setdisplayoffset, sizeof(GAL_DISPLAYPARAMS), 0}
   ,
   {Nscgfxfn_getdisplayoffset, sizeof(GAL_DISPLAYPARAMS),
    sizeof(GAL_DISPLAYPARAMS)}
   ,
   {Nscgfxfn_getrefreshfromdotclock, sizeof(GAL_DOTCLKTOREFRESH),
    sizeof(GAL_DOTCLKTOREFRESH)}
   ,
   {Nscgfxfn_getdisplaytiming, sizeof(GAL_DISPLAYTIMING),
    sizeof(GAL_DISPLAYTIMING)}
   ,
   {Nscgfxfn_setdisplaytiming, sizeof(GAL_DISPLAYTIMING), 0}
   ,
   {Nscgfxfn_setdisplaypalette, sizeof(GAL_PALETTE), 0}
   ,
   {Nscgfxfn_getdisplaypalette, sizeof(GAL_PALETTE), sizeof(GAL_PALETTE)}
   ,
   {Nscgfxfn_setdisplaypaletteentry, sizeof(GAL_PALETTE_ENTRY), 0}
   ,
   {Nscgfxfn_getdisplaypaletteentry, sizeof(GAL_PALETTE_ENTRY),
    sizeof(GAL_PALETTE_ENTRY)}
   ,
   {Nscgfxfn_setfixedtimings, sizeof(GAL_DISPLAYTIMING), 0}
   ,

/*Hardware cursor funtions*/
   {Nscgfxfn_setcursorenable, sizeof(GAL_CURSORENABLE), 0}
   ,
   {Nscgfxfn_getcursorenable, sizeof(GAL_CURSORENABLE),
    sizeof(GAL_CURSORENABLE)}
   ,
   {Nscgfxfn_setcursorposition, sizeof(GAL_CURSORPOSITION), 0}
   ,
   {Nscgfxfn_getcursorposition, sizeof(GAL_CURSORPOSITION),
    sizeof(GAL_CURSORPOSITION)}
   ,
   {Nscgfxfn_setcursorcolors, sizeof(GAL_CURSORCOLORS), 0}
   ,
   {Nscgfxfn_getcursorcolors, sizeof(GAL_CURSORCOLORS),
    sizeof(GAL_CURSORCOLORS)}
   ,
   {Nscgfxfn_setcursorshape32, sizeof(GAL_SETCURSORSHAPE), 0}
   ,
   {Nscgfxfn_setcursorshape64, sizeof(GAL_SETCURSORSHAPE), 0}
   ,

/*grafix rendering funtions*/
   {Nscgfxfn_setsolidpattern, sizeof(GAL_SETSOLIDPATTERN), 0}
   ,
   {Nscgfxfn_setrasteroperation, sizeof(GAL_RASTEROPERATION), 0}
   ,
   {Nscgfxfn_setsolidsource, sizeof(GAL_SETSOLIDSOURCE), 0}
   ,
   {Nscgfxfn_patternfill, sizeof(GAL_PATTERNFILL), 0}
   ,
   {Nscgfxfn_setmonosource, sizeof(GAL_SETMONOSOURCE), 0}
   ,
   {Nscgfxfn_setmonopattern, sizeof(GAL_SETMONOPATTERN), 0}
   ,
   {Nscgfxfn_screentoscreenblt, sizeof(GAL_SCREENTOSCREENBLT), 0}
   ,
   {Nscgfxfn_screentoscreenxblt, sizeof(GAL_SCREENTOSCREENXBLT), 0}
   ,
   {Nscgfxfn_bresenhamline, sizeof(GAL_BRESENHAMLINE), 0}
   ,
   {Nscgfxfn_color_pattern_fill, sizeof(GAL_COLOR_PATTERNFILL), 0}
   ,
   {Nscgfxfn_color_bitmap_to_screen_blt,
    sizeof(GAL_COLOR_BITMAP_TO_SCREEN_BLT), 0}
   ,
   {Nscgfxfn_color_bitmap_to_screen_xblt,
    sizeof(GAL_COLOR_BITMAP_TO_SCREEN_XBLT), 0}
   ,
   {Nscgfxfn_mono_bitmap_to_screen_blt, sizeof(GAL_MONO_BITMAP_TO_SCREEN_BLT),
    0}
   ,
   {Nscgfxfn_text_blt, sizeof(GAL_TEXT_BLT), 0}
   ,

/*VGA Support functions*/
   {Nscgfxfn_vgamodeswitch, sizeof(GAL_VGAREGS), 0}
   ,
   {Nscgfxfn_vgaclearextended, sizeof(GAL_VGAREGS), 0}
   ,
   {Nscgfxfn_vgapitch, sizeof(GAL_VGAREGS), sizeof(GAL_VGAREGS)}
   ,
   {Nscgfxfn_vgarestore, sizeof(GAL_VGAREGS), sizeof(GAL_VGAREGS)}
   ,
   {Nscgfxfn_vgasave, sizeof(GAL_VGAREGS), sizeof(GAL_VGAREGS)}
   ,
   {Nscgfxfn_vgamode, sizeof(GAL_VGAREGS), sizeof(GAL_VGAREGS)}
   ,

/*Compression functions*/
   {Nscgfxfn_setcompressionstate, sizeof(GAL_COMPRESSIONSTATE), 0}
   ,
   {Nscgfxfn_getcompressionstate, sizeof(GAL_COMPRESSIONSTATE),
    sizeof(GAL_COMPRESSIONSTATE)}
   ,
   {Nscgfxfn_setcompressionparams, sizeof(GAL_COMPRESSIONPARAMS), 0}
   ,
   {Nscgfxfn_getcompressionparams, sizeof(GAL_COMPRESSIONPARAMS),
    sizeof(GAL_COMPRESSIONPARAMS)}
   ,

/* Panel Support functions */
#ifdef CONFIG_FB_NSC_FP
   {Nscpnlfn_setpanelparams, sizeof(GAL_PNLPARAMS), 0}
   ,
   {Nscpnlfn_getpanelparams, sizeof(GAL_PNLPARAMS), sizeof(GAL_PNLPARAMS)}
   ,
   {Nscpnlfn_initpanel, sizeof(GAL_PNLPARAMS), 0}
   ,
   {Nscpnlfn_savepanel, sizeof(GAL_PNLPARAMS), 0}
   ,
   {Nscpnlfn_restorepanel, sizeof(GAL_PNLPARAMS), 0}
   ,
   {Nscpnlfn_powerup, sizeof(GAL_PNLPARAMS), 0}
   ,
   {Nscpnlfn_powerdown, sizeof(GAL_PNLPARAMS), 0}
   ,
   {Nscpnlfn_enabledinbios, sizeof(GAL_PNLBIOS), sizeof(GAL_PNLBIOS)}
   ,
   {Nscpnlfn_infofrombios, sizeof(GAL_PNLBIOS), sizeof(GAL_PNLBIOS)}
   ,
   {Nscenable_panning, sizeof(GAL_ENABLEPANNING), 0}
   ,
#else
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
#endif

#ifdef CONFIG_FB_NSC_TV
/* TV Support functions */
   {Nscgfxfn_settvparams, sizeof(GAL_TVPARAMS), 0}
   ,
   {Nscgfxfn_gettvparams, sizeof(GAL_TVPARAMS), sizeof(GAL_TVPARAMS)}
   ,
   {Nscgfxfn_settvtiming, sizeof(GAL_TVTIMING), 0}
   ,
   {Nscgfxfn_gettvtiming, sizeof(GAL_TVTIMING), sizeof(GAL_TVTIMING)}
   ,
   {Nscgfxfn_settvenable, sizeof(GAL_TVPARAMS), 0}
   ,
   {Nscgfxfn_gettvenable, sizeof(GAL_TVPARAMS), sizeof(GAL_TVPARAMS)}
   ,
   {Nscgfxfn_istvmodesupported, sizeof(GAL_TVPARAMS), sizeof(GAL_TVPARAMS)}
   ,
#else
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,
#endif

/* Video Support functions */
   {Nscgfxfn_setvideoenable, sizeof(GAL_VIDEOENABLE), 0}
   ,
   {Nscgfxfn_setvideoformat, sizeof(GAL_VIDEOFORMAT), 0}
   ,
   {Nscgfxfn_setvideosize, sizeof(GAL_VIDEOSIZE), 0}
   ,
   {Nscgfxfn_setvideooffset, sizeof(GAL_VIDEOOFFSET), 0}
   ,
   {Nscgfxfn_setvideowindow, sizeof(GAL_VIDEOWINDOW), 0}
   ,
   {Nscgfxfn_setvideoscale, sizeof(GAL_VIDEOSCALE), 0}
   ,
   {Nscgfxfn_setvideofilter, sizeof(GAL_VIDEOFILTER), 0}
   ,
   {Nscgfxfn_setvideocolorkey, sizeof(GAL_VIDEOCOLORKEY), 0}
   ,
   {Nscgfxfn_setvideodownscale_enable, sizeof(GAL_VIDEODOWNSCALEENABLE), 0}
   ,
   {Nscgfxfn_setvideodownscale_config, sizeof(GAL_VIDEODOWNSCALECONFIG), 0}
   ,
   {Nscgfxfn_setvideodownscale_coefficients, sizeof(GAL_VIDEODOWNSCALECOEFF),
    0}
   ,
   {Nscgfxfn_setvideosource, sizeof(GAL_VIDEOSOURCE), 0}
   ,
   {Nscgfxfn_setvideointerlaced, sizeof(GAL_SETVIDEOINTERLACED), 0}
   ,
   {Nscgfxfn_setvideocursor, sizeof(GAL_VIDEOCURSOR), 0}
   ,
   {Nscgfxfn_setvideorequest, sizeof(GAL_VIDEOREQUEST), 0}
   ,
   {Nscgfxfn_setalphaenable, sizeof(GAL_ALPHAENABLE), 0}
   ,
   {Nscgfxfn_setalphawindow, sizeof(GAL_ALPHAWINDOW), 0}
   ,
   {Nscgfxfn_setalphavalue, sizeof(GAL_ALPHAVALUE), 0}
   ,
   {Nscgfxfn_setalphapriority, sizeof(GAL_ALPHAPRIORITY), 0}
   ,
   {Nscgfxfn_setalphacolor, sizeof(GAL_ALPHACOLOR), 0}
   ,
   {Nscgfxfn_setalpharegion, sizeof(GAL_ALPHAREGION), 0}
   ,
   {Nscgfxfn_setvideooutsidealpha, sizeof(GAL_VIDEOOUTSIDEALPHA), 0}
   ,
   {Nscgfxfn_setvideopalette, sizeof(GAL_VIDEOPALETTE), 0}
   ,
   {Nscgfxfn_getvideoinfo, sizeof(GAL_VIDEOINFO), sizeof(GAL_VIDEOINFO)}
   ,
   {Nscgfxfn_Unsupported, sizeof(GAL_BASE), 0}
   ,

/* VIP Supported functions */
   {Nscgfxfn_setvipenable, sizeof(GAL_VIPENABLE), 0}
   ,
   {Nscgfxfn_setvipcapturerunmode, sizeof(GAL_VIPCAPTURERUNMODE), 0}
   ,
   {Nscgfxfn_setvipbase, sizeof(GAL_VIPBASE), 0}
   ,
   {Nscgfxfn_setvippitch, sizeof(GAL_VIPPITCH), 0}
   ,
   {Nscgfxfn_setvipmode, sizeof(GAL_VIPMODE), 0}
   ,
   {Nscgfxfn_setvipbusrth, sizeof(GAL_VIPBUS_RTH), 0}
   ,
   {Nscgfxfn_setviplastline, sizeof(GAL_VIPLASTLINE), 0}
   ,
   {Nscgfxfn_testvipoddfield, sizeof(GAL_TESTVIPODDFIELD),
    sizeof(GAL_TESTVIPODDFIELD)}
   ,
   {Nscgfxfn_testvipbasesupdated, sizeof(GAL_TESTVIPBASESUPDATED),
    sizeof(GAL_TESTVIPBASESUPDATED)}
   ,
   {Nscgfxfn_setvbienable, sizeof(GAL_VBIENABLE), 0}
   ,
   {Nscgfxfn_setvbimode, sizeof(GAL_VBIMODE), 0}
   ,
   {Nscgfxfn_setvbibase, sizeof(GAL_VBIBASE), 0}
   ,
   {Nscgfxfn_setvbipitch, sizeof(GAL_VBIPITCH), 0}
   ,
   {Nscgfxfn_setvbidirect, sizeof(GAL_SETVBIDIRECT), 0}
   ,
   {Nscgfxfn_setvbiinterrupt, sizeof(GAL_VBIINTERRUPT), 0}
   ,
   {Nscgfxfn_setgenlockenable, sizeof(GAL_GENLOCKENABLE), 0}
   ,
   {Nscgfxfn_settoplineinodd, sizeof(GAL_TOPLINEINODD), 0}
   ,
   {Nscgfxfn_setgenlockdelay, sizeof(GAL_GENLOCKDELAY), 0}
   ,
   {Nscgfxfn_setmacrovisionenable, sizeof(GAL_MACROVISIONENABLE), 0}
   ,

   {Nscgfxfn_getvipenable, sizeof(GAL_VIPENABLE), sizeof(GAL_VIPENABLE)}
   ,
   {Nscgfxfn_getvipbase, sizeof(GAL_VIPBASE), sizeof(GAL_VIPBASE)}
   ,
   {Nscgfxfn_getvippitch, sizeof(GAL_VIPPITCH), sizeof(GAL_VIPPITCH)}
   ,
   {Nscgfxfn_getvipmode, sizeof(GAL_VIPMODE), sizeof(GAL_VIPMODE)}
   ,
   {Nscgfxfn_getvipbusrth, sizeof(GAL_VIPBUS_RTH), sizeof(GAL_VIPBUS_RTH)}
   ,
   {Nscgfxfn_getvipline, sizeof(GAL_VIPLINE), sizeof(GAL_VIPLINE)}
   ,
   {Nscgfxfn_getvbienable, sizeof(GAL_VBIENABLE), sizeof(GAL_VBIENABLE)}
   ,
   {Nscgfxfn_getvbibase, sizeof(GAL_VBIBASE), sizeof(GAL_VBIBASE)}
   ,
   {Nscgfxfn_getvbipitch, sizeof(GAL_VBIPITCH), sizeof(GAL_VBIPITCH)}
   ,
   {Nscgfxfn_getvbimode, sizeof(GAL_VBIMODE), sizeof(GAL_VBIMODE)}
   ,
   {Nscgfxfn_getvbidirect, sizeof(GAL_GETVBIDIRECT), sizeof(GAL_GETVBIDIRECT)}
   ,
   {Nscgfxfn_getvbiinterrupt, sizeof(GAL_VBIINTERRUPT),
    sizeof(GAL_VBIINTERRUPT)}
   ,
   {Nscgfxfn_testvipfifooverflow, sizeof(GAL_TESTVIPOVERFLOW),
    sizeof(GAL_TESTVIPOVERFLOW)}
   ,

/* Second generation rendering routines */
   {Nscgfxfn_seticonenable, sizeof(GAL_ICONENABLE), 0}
   ,
   {Nscgfxfn_seticoncolors, sizeof(GAL_ICONCOLORS), 0}
   ,
   {Nscgfxfn_seticonposition, sizeof(GAL_ICONPOSITION), 0}
   ,
   {Nscgfxfn_seticonshape64, sizeof(GAL_ICONSHAPE64), 0}
   ,

   {Nscgfxfn_gfx2setsourcestride, sizeof(GAL_STRIDE), 0}
   ,
   {Nscgfxfn_setdestinationstride, sizeof(GAL_STRIDE), 0}
   ,
   {Nscgfxfn_setsourcetransparency, sizeof(GAL_SOURCETRANSPARENCY), 0}
   ,
   {Nscgfxfn_setpatternorigin, sizeof(GAL_PATTERNORIGIN), 0}
   ,
   {Nscgfxfn_gfx2setalphamode, sizeof(GAL_GFX2ALPHAMODE), 0}
   ,
   {Nscgfxfn_gfx2setalphavalue, sizeof(GAL_GFX2ALPHAVALUE), 0}
   ,
   {Nscgfxfn_gfx2patternfill, sizeof(GAL_GFX2PATTERNFILL), 0}
   ,
   {Nscgfxfn_gfx2colorpatternfill, sizeof(GAL_GFX2COLORPATTERNFILL), 0}
   ,
   {Nscgfxfn_gfx2screentoscreenblt, sizeof(GAL_GFX2SCREENTOSCREENBLT), 0}
   ,
   {Nscgfxfn_gfx2monoexpandblt, sizeof(GAL_GFX2MONOEXPANDBLT), 0}
   ,
   {Nscgfxfn_gfx2colorbmptoscrblt, sizeof(GAL_GFX2COLORBMPTOSCRBLT), 0}
   ,
   {Nscgfxfn_gfx2monobmptoscrblt, sizeof(GAL_GFX2MONOBMPTOSCRBLT), 0}
   ,
   {Nscgfxfn_gfx2textblt, sizeof(GAL_GFX2TEXTBLT), 0}
   ,
   {Nscgfxfn_gfx2bresenhamline, sizeof(GAL_GFX2BRESENHAMLINE), 0}
   ,
   {Nscgfxfn_gfx2synctovblank, sizeof(GAL_GFX2SYNCTOVBLANK), 0}
   ,

/* Change/Get Video routines */
   {Nscgfxfn_setcolorspaceYUV, sizeof(GAL_COLORSPACEYUV), 0}
   ,
   {Nscgfxfn_setvideoyuvpitch, sizeof(GAL_VIDEOYUVPITCH), 0}
   ,
   {Nscgfxfn_setvideoyuvoffsets, sizeof(GAL_VIDEOYUVOFFSETS), 0}
   ,
   {Nscgfxfn_setvideoleftcrop, sizeof(GAL_VIDEOLEFTCROP),
    sizeof(GAL_VIDEOLEFTCROP)}
   ,
   {Nscgfxfn_setvideoverticaldownscale, sizeof(GAL_VIDEOVERTICALDOWNSCALE),
    sizeof(GAL_VIDEOVERTICALDOWNSCALE)}
   ,
   {Nscgfxfn_setvbisource, sizeof(GAL_VBISOURCE), sizeof(GAL_VBISOURCE)}
   ,
   {Nscgfxfn_setvbilines, sizeof(GAL_VBILINES), sizeof(GAL_VBILINES)}
   ,
   {Nscgfxfn_setvbitotal, sizeof(GAL_VBITOTAL), sizeof(GAL_VBITOTAL)}
   ,
   {Nscgfxfn_setverticalscaleroffset, sizeof(GAL_VSCALEROFFSET),
    sizeof(GAL_VSCALEROFFSET)}
   ,

   {Nscgfxfn_getvbisource, sizeof(GAL_VBISOURCE), sizeof(GAL_VBISOURCE)}
   ,
   {Nscgfxfn_getvbilines, sizeof(GAL_VBILINES), sizeof(GAL_VBILINES)}
   ,
   {Nscgfxfn_getvbitotal, sizeof(GAL_VBITOTAL), sizeof(GAL_VBITOTAL)}
   ,
   {Nscgfxfn_getverticalscaleroffset, sizeof(GAL_VSCALEROFFSET),
    sizeof(GAL_VSCALEROFFSET)}
   ,
   {Nscgfxfn_getvideointerlaced, sizeof(GAL_GETVIDEOINTERLACED),
    sizeof(GAL_GETVIDEOINTERLACED)}
   ,
   {Nscgfxfn_getcolorspaceYUV, sizeof(GAL_COLORSPACEYUV),
    sizeof(GAL_COLORSPACEYUV)}
   ,
   {Nscgfxfn_getgenlockenable, sizeof(GAL_GENLOCKENABLE),
    sizeof(GAL_GENLOCKENABLE)}
   ,
   {Nscgfxfn_getgenlockdelay, sizeof(GAL_GENLOCKDELAY),
    sizeof(GAL_GENLOCKDELAY)}
   ,
   {Nscgfxfn_getvideocursor, sizeof(GAL_VIDEOCURSOR), sizeof(GAL_VIDEOCURSOR)}
   ,
   {Nscgfxfn_readcrc, sizeof(GAL_READCRC), sizeof(GAL_READCRC)}
   ,
   {Nscgfxfn_readwindowcrc, sizeof(GAL_READWINDOWCRC),
    sizeof(GAL_READWINDOWCRC)}
   ,
   {Nscgfxfn_getmacrovisionenable, sizeof(GAL_MACROVISIONENABLE),
    sizeof(GAL_MACROVISIONENABLE)}
   ,
   {Nscgfxfn_getalphaenable, sizeof(GAL_ALPHAENABLE), sizeof(GAL_ALPHAENABLE)}
   ,
   {Nscgfxfn_getalphasize, sizeof(GAL_ALPHASIZE), sizeof(GAL_ALPHASIZE)}
   ,
   {Nscgfxfn_getalphavalue, sizeof(GAL_ALPHAVALUE), sizeof(GAL_ALPHAVALUE)}
   ,
   {Nscgfxfn_getalphapriority, sizeof(GAL_ALPHAPRIORITY),
    sizeof(GAL_ALPHAPRIORITY)}
   ,
   {Nscgfxfn_getalphacolor, sizeof(GAL_ALPHACOLOR), sizeof(GAL_ALPHACOLOR)}
   ,
   {Nscgfxfn_getvideoyuvpitch, sizeof(GAL_VIDEOYUVPITCH),
    sizeof(GAL_VIDEOYUVPITCH)}
   ,
   {Nscgfxfn_getvideoyuvoffsets, sizeof(GAL_VIDEOYUVOFFSETS),
    sizeof(GAL_VIDEOYUVOFFSETS)}
   ,

/* Additional VGA Support functions */
   {Nscgfxfn_vgatestpci, sizeof(GAL_VGATESTPCI), sizeof(GAL_VGATESTPCI)}
   ,
   {Nscgfxfn_vgagetpcicommand, sizeof(GAL_VGAGETPCICOMMAND),
    sizeof(GAL_VGAGETPCICOMMAND)}
   ,
   {Nscgfxfn_vgaseqreset, sizeof(GAL_VGASEQRESET), sizeof(GAL_VGASEQRESET)}
   ,
   {Nscgfxfn_vgasetgraphicsbits, sizeof(GAL_VGASETGRAPHICSBITS),
    sizeof(GAL_VGASETGRAPHICSBITS)}
   ,
};

/*-----------------------------------------------------------------
 * Nscgfxfn_Unsupported 
 *
 * Description: This function will be called when ioctl function 
 *                              is not supported by geode frame buffer driver. 
 * parameters :
 *              pvarg : This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: It returns negative error value.
 *------------------------------------------------------------------*/
int
Nscgfxfn_Unsupported(PGAL_BASE arg)
{
   return -EBADRQC;
}

/*--------------------------------------------------------------------
 * Nscgfxfn_getadapterinfo 
 * 
 * Description: This function gets the galdevice adapter information
 *                              to the user. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *              passed  to the ioctl by the application.
 *      return: 0 was return on getting the gal adapter information
 *                              other wise -ve value will be returned.
 *--------------------------------------------------------------------*/
int
Nscgfxfn_getadapterinfo(PGAL_BASE arg)
{
   PGAL_ADAPTERINFO sAdapterInfo = (PGAL_ADAPTERINFO) arg;

   sAdapterInfo->dwCPUVersion = gfx_detect_cpu();
   sAdapterInfo->dwVideoVersion = gfx_detect_video();
   sAdapterInfo->dwFrameBufferBase = gfx_get_frame_buffer_base();
   sAdapterInfo->dwFrameBufferSize = GXFBMemSize;
   sAdapterInfo->dwGfxRegisterBase = gfx_get_cpu_register_base();
   sAdapterInfo->dwVidRegisterBase = gfx_get_vid_register_base();
   sAdapterInfo->dwVipRegisterBase = gfx_get_vip_register_base();
   {
      Q_WORD msrValue;

      /* GX2 : Can have CRT or TFT only */
      gfx_msr_read(RC_ID_DF, MBD_MSR_CONFIG, &msrValue);
      sAdapterInfo->dwCPUType =
            ((msrValue.low & RCDF_CONFIG_FMT_MASK) ==
             RCDF_CONFIG_FMT_FP) ? 1 : 0;
      printk("CPU Type = %d\n", sAdapterInfo->dwCPUType);
   }
   sAdapterInfo->dwGpsRegisterBase = gfx_get_graphics_register_base();
   sAdapterInfo->dwGpsRegisterBase = sAdapterInfo->dwGfxRegisterBase;
   sAdapterInfo->dwMaxSupportedPixelClock =
         gfx_get_max_supported_pixel_clock();
   return 0;
}

/*----------------------------------------------------------------------
 * Nscgfxfn_setsoftvgastate
 * 
 * Description: This function sets the softvga state
 *      parameters:
 *           pvarg:     This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: 0 was retured on success otherwise error value will be 
 *                              returned.
 *--------------------------------------------------------------------*/
int
Nscgfxfn_setsoftvgastate(PGAL_BASE arg)
{
   PGAL_SOFTVGASTATE sSoftVgaState = (PGAL_SOFTVGASTATE) arg;

   if (sSoftVgaState->bSoftVgaEnable)
      gfx_enable_softvga();
   else
      gfx_disable_softvga();
   return 0;
}

/*----------------------------------------------------------------------
 * Nscgfxfn_getsoftvgastate
 * 
 * Description: This function gets the softvga state
 *      parameters:
 *           pvarg:     This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: 0 was retured on success otherwise error value will be 
 *                              returned.
 *--------------------------------------------------------------------*/
int
Nscgfxfn_getsoftvgastate(PGAL_BASE arg)
{
   PGAL_SOFTVGASTATE sSoftVgaState = (PGAL_SOFTVGASTATE) arg;

   sSoftVgaState->bSoftVgaEnable = gfx_get_softvga_active();

   return 0;
}

int
Nscgfxfn_vgatestpci(PGAL_BASE arg)
{
   PGAL_VGATESTPCI sVgatestpci = (PGAL_VGATESTPCI) arg;

   sVgatestpci->softvga = gfx_vga_test_pci();
   return 0;
}

int
Nscgfxfn_vgagetpcicommand(PGAL_BASE arg)
{
   PGAL_VGAGETPCICOMMAND sVgagetpcicommand = (PGAL_VGAGETPCICOMMAND) arg;

   sVgagetpcicommand->value = gfx_vga_get_pci_command();
   return 0;
}

int
Nscgfxfn_vgaseqreset(PGAL_BASE arg)
{
   PGAL_VGASEQRESET sVgaseqreset = (PGAL_VGASEQRESET) arg;

   sVgaseqreset->statusok = gfx_vga_seq_reset(sVgaseqreset->reset);
   return 0;
}

int
Nscgfxfn_vgasetgraphicsbits(PGAL_BASE arg)
{
   PGAL_VGASETGRAPHICSBITS sVgasetgraphicsbits =
         (PGAL_VGASETGRAPHICSBITS) arg;
   sVgasetgraphicsbits->statusok = gfx_vga_set_graphics_bits();
   return 0;
}

/*-------------------------------------------------------------------
 * Nscgfxfn_setcrtenable
 * 
 * Description: This function enable the gal crt state based on 
 *                              user information.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: 0 was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setcrtenable(PGAL_BASE arg)
{
   PGAL_CRTENABLE sCrtEnable = (PGAL_CRTENABLE) arg;

   gfx_set_crt_enable(sCrtEnable->wCrtEnable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_waituntilidle
 * 
 * Description: This routine waits until the graphics engine is idle.
 *                              It must be used before a display driver allows direct
 *                              access to frame buffer data.    
 *  parameters:
 *               pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: 0 was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_waituntilidle(PGAL_BASE arg)
{
   gfx_wait_until_idle();

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_waitverticalblank
 *
 * Description: This function wait until start of vertical blank.
 *
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_waitverticalblank(PGAL_BASE arg)
{
   gfx_wait_vertical_blank();

   return 0;
}

/*--------------------------------------------------------------------
 * Nscgfxfn_writereg
 *
 * Description: This function writes the user information to the
 *                              gal hardware.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: 0 was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_writereg(PGAL_BASE arg)
{
   PGAL_HWACCESS hwAccess = (PGAL_HWACCESS) arg;

   if (hwAccess->dwType == GAL_REG) {
      if (hwAccess->dwByteCount == 4)
         WRITE_REG32(hwAccess->dwOffset, hwAccess->dwValue);
      else if (hwAccess->dwByteCount == 2)
         WRITE_REG16(hwAccess->dwOffset, hwAccess->dwValue);
      else if (hwAccess->dwByteCount == 1) {
         WRITE_REG8(hwAccess->dwOffset, hwAccess->dwValue);
      }
   } else if (hwAccess->dwType == GAL_VID) {
      if (hwAccess->dwByteCount == 4) {
         WRITE_VID32(hwAccess->dwOffset, hwAccess->dwValue);
      }
   } else if (hwAccess->dwType == GAL_VIP) {
      if (hwAccess->dwByteCount == 4) {
         WRITE_VIP32(hwAccess->dwOffset, hwAccess->dwValue);
      }
   }

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_readreg
 *
 * Description: This function reads the user information from the
 *                              gal hardware.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
                        passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_readreg(PGAL_BASE arg)
{
   PGAL_HWACCESS hwAccess = (PGAL_HWACCESS) arg;

   if (hwAccess->dwType == GAL_REG) {
      if (hwAccess->dwByteCount == 4) {
         hwAccess->dwValue = READ_REG32(hwAccess->dwOffset);
      } else if (hwAccess->dwByteCount == 2) {
         hwAccess->dwValue = READ_REG16(hwAccess->dwOffset);
      }
   } else if (hwAccess->dwType == GAL_VID) {
      if (hwAccess->dwByteCount == 4) {
         hwAccess->dwValue = READ_VID32(hwAccess->dwOffset);
      }
   } else if (hwAccess->dwType == GAL_VIP) {
      if (hwAccess->dwByteCount == 4) {
         hwAccess->dwValue = READ_VIP32(hwAccess->dwOffset);
      }
   }
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_ismodesupported
 *
 * Description: This function verifies that a given display mode is
 *                              supported or not.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_ismodesupported(PGAL_BASE arg)
{
   PGAL_DISPLAYMODE sDisplayMode = (PGAL_DISPLAYMODE) arg;

   sDisplayMode->dwSupported =
         gfx_is_display_mode_supported(sDisplayMode->wXres,
                                       sDisplayMode->wYres,
                                       sDisplayMode->wBpp,
                                       sDisplayMode->wRefresh);
   sDisplayMode->dwReturnValue = 1;
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setdisplaymode
 *
 * Description  This function programs the display controller 
 *                              for the specified display mode. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setdisplaymode(PGAL_BASE arg)
{
   PGAL_DISPLAYMODE sDisplayMode = (PGAL_DISPLAYMODE) arg;

   gfx_set_display_mode(sDisplayMode->wXres, sDisplayMode->wYres,
                        sDisplayMode->wBpp, sDisplayMode->wRefresh);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getdisplaymode
 *
 * Description: This function reads the current display parameters.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getdisplaymode(PGAL_BASE arg)
{
   PGAL_DISPLAYMODE sDisplayMode = (PGAL_DISPLAYMODE) arg;
   int xres, yres, bpp, hz;

   sDisplayMode->dwReturnValue =
         gfx_get_display_mode(&xres, &yres, &bpp, &hz);
   sDisplayMode->wXres = xres;
   sDisplayMode->wYres = yres;
   sDisplayMode->wBpp = bpp;
   sDisplayMode->wRefresh = hz;
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setbpp
 *
 * Description: This function specifies the bpp of the current 
 *                              mode so that the graphics engine can be programmed 
 *                              accordingly.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which 
 *                              was passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setbpp(PGAL_BASE arg)
{
   PGAL_DISPLAYPARAMS sDisplayParams = (PGAL_DISPLAYPARAMS) arg;

   gfx_set_bpp(sDisplayParams->wBpp);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setdisplaybpp
 *
 * Description: This function specifies the bpp of the current 
 *                              mode so that the graphics engine can be programmed 
 *                              accordingly.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which 
 *                              was passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setdisplaybpp(PGAL_BASE arg)
{
   PGAL_DISPLAYPARAMS sDisplayParams = (PGAL_DISPLAYPARAMS) arg;

   gfx_set_display_bpp(sDisplayParams->wBpp);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getdisplaybpp
 *
 * Description: This function reads the bpp of the current mode 
 *                              that the graphics engine can be programmed.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getdisplaybpp(PGAL_BASE arg)
{
   PGAL_DISPLAYPARAMS sDisplayParams = (PGAL_DISPLAYPARAMS) arg;

   sDisplayParams->wBpp = gfx_get_display_bpp();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setdisplaypitch
 *
 * Description: This function sets the pitch of the current
 *                              display mode, which is the number of bytes between
 *                              scanlines.
 *  parameters:
 *               pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setdisplaypitch(PGAL_BASE arg)
{
   PGAL_DISPLAYPARAMS sDisplayParams = (PGAL_DISPLAYPARAMS) arg;

   gfx_set_display_pitch(sDisplayParams->wPitch);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getdisplaypitch
 *
 * Description: This function returns the pitch of the current
 *                              display mode, which is the number of bytes between
 *                              scanlines.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getdisplaypitch(PGAL_BASE arg)
{
   PGAL_DISPLAYPARAMS sDisplayParams = (PGAL_DISPLAYPARAMS) arg;

   sDisplayParams->wPitch = gfx_get_display_pitch();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setdisplayoffset
 *
 * Description: This function sets the frame buffer start address.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setdisplayoffset(PGAL_BASE arg)
{
   PGAL_DISPLAYPARAMS sDisplayParams = (PGAL_DISPLAYPARAMS) arg;

   gfx_set_display_offset(sDisplayParams->dwOffset);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getdisplayoffset
 *
 * Description: This function returns the offset to the start of
 *                              frame buffer in graphics memory.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getdisplayoffset(PGAL_BASE arg)
{
   PGAL_DISPLAYPARAMS sDisplayParams = (PGAL_DISPLAYPARAMS) arg;

   sDisplayParams->dwOffset = gfx_get_display_offset();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getrefreshfromdotclock
 *
 * Description: This function takes display mode parameters,
 *                              and returns the refresh rate of the closest display
 *                              mode.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getrefreshfromdotclock(PGAL_BASE arg)
{
   PGAL_DOTCLKTOREFRESH sDclk2refresh = (PGAL_DOTCLKTOREFRESH) arg;
   int hz;

   sDclk2refresh->dwReturnValue =
         gfx_get_refreshrate_from_frequency(sDclk2refresh->wXres,
                                            sDclk2refresh->wYres,
                                            sDclk2refresh->wBpp, &hz,
                                            sDclk2refresh->dwDotClock);
   sDclk2refresh->wRefreshRate = hz;
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getdisplaytiming
 *
 * Description: This function gets display timings.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
                        passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getdisplaytiming(PGAL_BASE arg)
{
   PGAL_DISPLAYTIMING sDisplayTiming = (PGAL_DISPLAYTIMING) arg;

   sDisplayTiming->dwDotClock = gfx_get_clock_frequency();
   sDisplayTiming->wPitch = gfx_get_display_pitch();
   sDisplayTiming->wBpp = gfx_get_display_bpp();
   sDisplayTiming->wHTotal = gfx_get_htotal();
   sDisplayTiming->wHActive = gfx_get_hactive();
   sDisplayTiming->wHSyncStart = gfx_get_hsync_start();
   sDisplayTiming->wHSyncEnd = gfx_get_hsync_end();
   sDisplayTiming->wHBlankStart = gfx_get_hblank_start();
   sDisplayTiming->wHBlankEnd = gfx_get_hblank_end();
   sDisplayTiming->wVTotal = gfx_get_vtotal();
   sDisplayTiming->wVActive = gfx_get_vactive();
   sDisplayTiming->wVSyncStart = gfx_get_vsync_start();
   sDisplayTiming->wVSyncEnd = gfx_get_vsync_end();
   sDisplayTiming->wVBlankStart = gfx_get_vblank_start();
   sDisplayTiming->wVBlankEnd = gfx_get_vblank_end();
   sDisplayTiming->wPolarity = gfx_get_sync_polarities();

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setdisplaytiming
 *
 * Description: This function sets  display timings.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setdisplaytiming(PGAL_BASE arg)
{
   PGAL_DISPLAYTIMING sDisplayTiming = (PGAL_DISPLAYTIMING) arg;

   gfx_set_display_timings(sDisplayTiming->wBpp,
                           sDisplayTiming->wPolarity,
                           sDisplayTiming->wHActive,
                           sDisplayTiming->wHBlankStart,
                           sDisplayTiming->wHSyncStart,
                           sDisplayTiming->wHSyncEnd,
                           sDisplayTiming->wHBlankEnd,
                           sDisplayTiming->wHTotal,
                           sDisplayTiming->wVActive,
                           sDisplayTiming->wVBlankStart,
                           sDisplayTiming->wVSyncStart,
                           sDisplayTiming->wVSyncEnd,
                           sDisplayTiming->wVBlankEnd,
                           sDisplayTiming->wVTotal,
                           sDisplayTiming->dwDotClock);
   gfx_set_display_pitch(sDisplayTiming->wPitch);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setfixedtimings
 *
 * Description: This function enables a flat panel and fixed timings.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setfixedtimings(PGAL_BASE arg)
{
   PGAL_DISPLAYTIMING sDisplayTiming = (PGAL_DISPLAYTIMING) arg;

   gfx_set_fixed_timings(sDisplayTiming->wHActive,      /* panel Xres */
                         sDisplayTiming->wVActive,      /* panel Yres */
                         sDisplayTiming->wHTotal,       /* Total Xres */
                         sDisplayTiming->wVTotal,       /* Total Yres */
                         sDisplayTiming->wBpp);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setdisplaypaletteentry
 *
 * Description: This function sets the entire palette used for 8 bpp
 *                              graphics data.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setdisplaypaletteentry(PGAL_BASE arg)
{
   PGAL_PALETTE_ENTRY sPalette = (PGAL_PALETTE_ENTRY) arg;

   gfx_set_display_palette_entry(sPalette->dwIndex, sPalette->dwPalette);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getdisplaypaletteentry
 *
 * Description: This function returns a single 32-bit 8:8:8
 *                              RGB entry of the 8 bpp graphics palette.
 *  parameters:
 *               pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getdisplaypaletteentry(PGAL_BASE arg)
{
   PGAL_PALETTE_ENTRY sPalette = (PGAL_PALETTE_ENTRY) arg;
   unsigned long palette;

   gfx_get_display_palette_entry(sPalette->dwIndex, &palette);
   sPalette->dwPalette = palette;
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setdisplaypalette
 *
 * Description: This function sets the entire palette used
 *                              for 8bpp graphics data.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setdisplaypalette(PGAL_BASE arg)
{
   PGAL_PALETTE sPalette = (PGAL_PALETTE) arg;
   unsigned long *colors;

   colors = (unsigned long *)sPalette->dwColors;
   gfx_set_display_palette(colors);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getdisplaypalette
 *
 * Description: This function returns the current palette used
 *                              for 8bpp graphics data.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getdisplaypalette(PGAL_BASE arg)
{
   PGAL_PALETTE sPalette = (PGAL_PALETTE) arg;
   unsigned long *colors;

   colors = (unsigned long *)sPalette->dwColors;
   gfx_get_display_palette(colors);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setcompressionstate
 *
 * Description: This function enables or disables display compression.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setcompressionstate(PGAL_BASE arg)
{
   PGAL_COMPRESSIONSTATE sCompState = (PGAL_COMPRESSIONSTATE) arg;

   gfx_set_compression_enable(sCompState->bCompressionState);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getcompressionstate
 *
 * Description: This function gets the display compression.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getcompressionstate(PGAL_BASE arg)
{
   PGAL_COMPRESSIONSTATE sCompState = (PGAL_COMPRESSIONSTATE) arg;

   sCompState->bCompressionState = gfx_get_compression_enable();

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setcompressionparams
 *
 * Description: This function sets the compression parameters of the
 *                              frame buffer device.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setcompressionparams(PGAL_BASE arg)
{
   PGAL_COMPRESSIONPARAMS sCompParams = (PGAL_COMPRESSIONPARAMS) arg;

   if (sCompParams->dwFlags & GAL_COMPRESSION_OFFSET)
      gfx_set_compression_offset(sCompParams->dwCompOffset);
   if (sCompParams->dwFlags & GAL_COMPRESSION_PITCH)
      gfx_set_compression_pitch(sCompParams->dwCompPitch);
   if (sCompParams->dwFlags & GAL_COMPRESSION_SIZE)
      gfx_set_compression_size(sCompParams->dwCompSize);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getcompressionparams
 *
 * Description: This function gets the compression parameters of the
 *                              frame buffer device.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getcompressionparams(PGAL_BASE arg)
{
   PGAL_COMPRESSIONPARAMS sCompParams = (PGAL_COMPRESSIONPARAMS) arg;

   if (sCompParams->dwFlags & GAL_COMPRESSION_OFFSET)
      sCompParams->dwCompOffset = gfx_get_compression_offset();

   if (sCompParams->dwFlags & GAL_COMPRESSION_PITCH)
      sCompParams->dwCompPitch = gfx_get_compression_pitch();

   if (sCompParams->dwFlags & GAL_COMPRESSION_SIZE)
      sCompParams->dwCompSize = gfx_get_compression_size();

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setcursorenable
 *
 * Description: This function enable or disable the hardware cursor.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setcursorenable(PGAL_BASE arg)
{
   PGAL_CURSORENABLE sCursorEnable = (PGAL_CURSORENABLE) arg;

   gfx_set_cursor_enable(sCursorEnable->bCursorEnable);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getcursorenable
 *
 * Description: This function gets the curent cursor state.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getcursorenable(PGAL_BASE arg)
{
   PGAL_CURSORENABLE sCursorEnable = (PGAL_CURSORENABLE) arg;

   sCursorEnable->bCursorEnable = gfx_get_cursor_enable();

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setcursorcolors
 *
 * Description: This function sets the colors of the hardware cursor.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setcursorcolors(PGAL_BASE arg)
{
   PGAL_CURSORCOLORS sCursorCol = (PGAL_CURSORCOLORS) arg;

   gfx_set_cursor_colors(sCursorCol->dwBgColor, sCursorCol->dwFgColor);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getcursorcolors
 *
 * Description: This function sets the colors of the hardware cursor.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getcursorcolors(PGAL_BASE arg)
{
   PGAL_CURSORCOLORS sCursorCol = (PGAL_CURSORCOLORS) arg;

   sCursorCol->dwBgColor = gfx_get_cursor_color(0);
   sCursorCol->dwFgColor = gfx_get_cursor_color(1);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setcursorposition
 *
 * Description: This function sets the position of the hardware cursor.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setcursorposition(PGAL_BASE arg)
{
   PGAL_CURSORPOSITION sCursorPos = (PGAL_CURSORPOSITION) arg;

   gfx_set_cursor_position(sCursorPos->dwMemOffset,
                           sCursorPos->wXPos,
                           sCursorPos->wYPos,
                           sCursorPos->wXHot, sCursorPos->wYHot);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getcursorposition
 *
 * Description: This function gets the position of the hardware cursor.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getcursorposition(PGAL_BASE arg)
{
   PGAL_CURSORPOSITION sCursorPos = (PGAL_CURSORPOSITION) arg;

   sCursorPos->dwMemOffset = gfx_get_cursor_offset();
   sCursorPos->wXPos = 0;
   sCursorPos->wYPos = 0;
   sCursorPos->wXHot = 0;
   sCursorPos->wYHot = 0;

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setcursorshape32
 *
 * Description: This function loads 32x32 cursor pattern. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setcursorshape32(PGAL_BASE arg)
{
   PGAL_SETCURSORSHAPE sCursorShape = (PGAL_SETCURSORSHAPE) arg;

   gfx_set_cursor_shape32(sCursorShape->dwMemOffset,
                          (unsigned long *)sCursorShape->dwAndMask,
                          (unsigned long *)sCursorShape->dwXorMask);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setcursorshape64
 *
 * Description: This function loads 64x64 cursor pattern. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/

int
Nscgfxfn_setcursorshape64(PGAL_BASE arg)
{
   PGAL_SETCURSORSHAPE sCursorShape = (PGAL_SETCURSORSHAPE) arg;

   gfx_set_cursor_shape64(sCursorShape->dwMemOffset,
                          (unsigned long *)sCursorShape->dwAndMask,
                          (unsigned long *)sCursorShape->dwXorMask);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setsolidpattern
 *
 * Description: This function used to specify a solid pattern color
 *                              for future rendering. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setsolidpattern(PGAL_BASE arg)
{
   PGAL_SETSOLIDPATTERN sSetSolidPat = (PGAL_SETSOLIDPATTERN) arg;

   gfx_set_solid_pattern(sSetSolidPat->dwColor);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setsolidsource
 *
 * Description: This function specifies a constant source data value
 *                              for raster operations that use both pattern and 
 *                              source data. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setsolidsource(PGAL_BASE arg)
{
   PGAL_SETSOLIDSOURCE sSetSolidSrc = (PGAL_SETSOLIDSOURCE) arg;

   gfx_set_solid_source(sSetSolidSrc->dwColor);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setmonosource
 *
 * Description: This function specifies an 8x8 monochrome source 
 *                              used in future rendering operations.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: '0' was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setmonosource(PGAL_BASE arg)
{
   PGAL_SETMONOSOURCE sSetMonoSrc = (PGAL_SETMONOSOURCE) arg;

   gfx_set_mono_source(sSetMonoSrc->dwBgColor,
                       sSetMonoSrc->dwFgColor, sSetMonoSrc->cTransparency);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setmonopattern
 *
 * Description: This function specifies an 8x8 monochrome pattern 
 *                              used in future rendering operations.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: '0' was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setmonopattern(PGAL_BASE arg)
{
   PGAL_SETMONOPATTERN sSetMonoPat = (PGAL_SETMONOPATTERN) arg;

   gfx_set_mono_pattern(sSetMonoPat->dwBgColor,
                        sSetMonoPat->dwFgColor,
                        sSetMonoPat->dwData0,
                        sSetMonoPat->dwData1, sSetMonoPat->cTransparency);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setrasteroperation
 *
 * Description: This function specifies the raster operation for 
 *                              future rendering.  
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setrasteroperation(PGAL_BASE arg)
{
   PGAL_RASTEROPERATION sSetRop = (PGAL_RASTEROPERATION) arg;

   gfx_set_raster_operation(sSetRop->cRop);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_patternfill
 *
 * Description: This function renders pattern data to a rectangular 
 *                              region. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero  was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_patternfill(PGAL_BASE arg)
{
   PGAL_PATTERNFILL sPatternFill = (PGAL_PATTERNFILL) arg;

   gfx_pattern_fill(sPatternFill->wXPos,
                    sPatternFill->wYPos,
                    sPatternFill->wWidth, sPatternFill->wHeight);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_screentoscreenblt
 *
 * Description: This function is used to perform a screen to screen 
 *                              BLT operation. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_screentoscreenblt(PGAL_BASE arg)
{
   PGAL_SCREENTOSCREENBLT sScreen2ScreenBlt = (PGAL_SCREENTOSCREENBLT) arg;

   gfx_screen_to_screen_blt(sScreen2ScreenBlt->wXStart,
                            sScreen2ScreenBlt->wYStart,
                            sScreen2ScreenBlt->wXEnd,
                            sScreen2ScreenBlt->wYEnd,
                            sScreen2ScreenBlt->wWidth,
                            sScreen2ScreenBlt->wHeight);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_screentoscreenxblt
 *
 * Description: This function is used to perform a screen to screen 
 *                              BLT operation using a transparency color. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_screentoscreenxblt(PGAL_BASE arg)
{
   PGAL_SCREENTOSCREENXBLT sScreen2ScreenXBlt = (PGAL_SCREENTOSCREENXBLT) arg;

   gfx_screen_to_screen_xblt(sScreen2ScreenXBlt->wXStart,
                             sScreen2ScreenXBlt->wYStart,
                             sScreen2ScreenXBlt->wXEnd,
                             sScreen2ScreenXBlt->wYEnd,
                             sScreen2ScreenXBlt->wWidth,
                             sScreen2ScreenXBlt->wHeight,
                             sScreen2ScreenXBlt->dwColor);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_bresenhamline
 *
 * Description: This function is used to draw a single pixel line
 *                              using the specified Bresenham parameters. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_bresenhamline(PGAL_BASE arg)
{
   PGAL_BRESENHAMLINE sBresenhamLine = (PGAL_BRESENHAMLINE) arg;

   gfx_bresenham_line(sBresenhamLine->wX1,
                      sBresenhamLine->wY1,
                      sBresenhamLine->wLength,
                      sBresenhamLine->wErr,
                      sBresenhamLine->wE1,
                      sBresenhamLine->wE2, sBresenhamLine->wFlags);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_color_pattern_fill
 *
 * Description: . 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_color_pattern_fill(PGAL_BASE arg)
{
   PGAL_COLOR_PATTERNFILL sColorPat = (PGAL_COLOR_PATTERNFILL) arg;

   gfx_color_pattern_fill(sColorPat->wDstx,
                          sColorPat->wDsty,
                          sColorPat->wWidth,
                          sColorPat->wHeight,
                          (unsigned long *)sColorPat->dwPattern);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_color_bitmap_to_screen_blt
 *
 * Description: . 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_color_bitmap_to_screen_blt(PGAL_BASE arg)
{
   PGAL_COLOR_BITMAP_TO_SCREEN_BLT sBmp2Scr =
         (PGAL_COLOR_BITMAP_TO_SCREEN_BLT) arg;

   gfx_color_bitmap_to_screen_blt(sBmp2Scr->wSrcx,
                                  sBmp2Scr->wSrcy,
                                  sBmp2Scr->wDstx,
                                  sBmp2Scr->wDsty,
                                  sBmp2Scr->wWidth,
                                  sBmp2Scr->wHeight,
                                  (unsigned char *)sBmp2Scr->dwData,
                                  sBmp2Scr->wPitch);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_color_bitmap_to_screen_xblt
 *
 * Description: . 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_color_bitmap_to_screen_xblt(PGAL_BASE arg)
{
   PGAL_COLOR_BITMAP_TO_SCREEN_XBLT sBmp2Scr =
         (PGAL_COLOR_BITMAP_TO_SCREEN_XBLT) arg;

   gfx_color_bitmap_to_screen_xblt(sBmp2Scr->wSrcx,
                                   sBmp2Scr->wSrcy,
                                   sBmp2Scr->wDstx,
                                   sBmp2Scr->wDsty,
                                   sBmp2Scr->wWidth,
                                   sBmp2Scr->wHeight,
                                   (unsigned char *)sBmp2Scr->dwData,
                                   sBmp2Scr->wPitch, sBmp2Scr->dwColor);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_mono_bitmap_to_screen_blt
 *
 * Description: . 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_mono_bitmap_to_screen_blt(PGAL_BASE arg)
{
   PGAL_MONO_BITMAP_TO_SCREEN_BLT sBmp2Scr =
         (PGAL_MONO_BITMAP_TO_SCREEN_BLT) arg;

   gfx_mono_bitmap_to_screen_blt(sBmp2Scr->wSrcx,
                                 sBmp2Scr->wSrcy,
                                 sBmp2Scr->wDstx,
                                 sBmp2Scr->wDsty,
                                 sBmp2Scr->wWidth,
                                 sBmp2Scr->wHeight,
                                 (unsigned char *)sBmp2Scr->dwData,
                                 sBmp2Scr->wPitch);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_text_blt
 *
 * Description: . 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_text_blt(PGAL_BASE arg)
{
   PGAL_TEXT_BLT sTextBlt = (PGAL_TEXT_BLT) arg;

   gfx_text_blt(sTextBlt->wDstx,
                sTextBlt->wDsty,
                sTextBlt->wWidth,
                sTextBlt->wHeight, (unsigned char *)sTextBlt->dwData);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_vgamodeswitch
 *
 * Description: This function signals the beginning or end of a 
 *                              mode switch.  
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_vgamodeswitch(PGAL_BASE arg)
{
   PGAL_VGAMODEDATA sVgaData = (PGAL_VGAMODEDATA) arg;

   gfx_vga_mode_switch(sVgaData->dwFlags);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_vgaclearextended
 *
 * Description: This will clear the Svga data.
 *
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_vgaclearextended(PGAL_BASE arg)
{
   gfx_vga_clear_extended();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_vgapitch
 *
 * Description: This function sets VGA register values in VGA 
 *                              structure for specified pitch. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_vgapitch(PGAL_BASE arg)
{
   PGAL_VGAMODEDATA sVgaData = (PGAL_VGAMODEDATA) arg;

   gfx_vga_pitch((gfx_vga_struct *) & sVgaData->sVgaRegs, sVgaData->dwFlags);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_vgarestore
 *
 * Description: This function sets the VGA hardware to values in the
 *                              VGA structure.  
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_vgarestore(PGAL_BASE arg)
{
   PGAL_VGAMODEDATA sVgaData = (PGAL_VGAMODEDATA) arg;

   gfx_vga_restore((gfx_vga_struct *) & sVgaData->sVgaRegs,
                   sVgaData->dwFlags);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_vgasave
 *
 * Description: This function saves the current VGA state in the
 *                              VGA structure.  
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_vgasave(PGAL_BASE arg)
{
   PGAL_VGAMODEDATA sVgaData = (PGAL_VGAMODEDATA) arg;

   gfx_vga_save((gfx_vga_struct *) & sVgaData->sVgaRegs, sVgaData->dwFlags);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_vgamode
 *
 * Description:This function sets VGA register values in VGA 
 *                              structure for specified mode. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_vgamode(PGAL_BASE arg)
{
   PGAL_VGAMODEDATA sVgaData = (PGAL_VGAMODEDATA) arg;

   gfx_vga_mode((gfx_vga_struct *) & sVgaData->sVgaRegs,
                sVgaData->wXres, sVgaData->wYres, sVgaData->wBpp,
                sVgaData->wRefresh);
   return 0;
}

#ifdef CONFIG_FB_NSC_FP
/*-----------------------------------------------------------------
 * Nscpnlfn_enabledinbios
 *
 * Description: This function gets the panel state from Bios. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_enabledinbios(PGAL_BASE arg)
{
   PGAL_PNLBIOS sPnlBios = (PGAL_PNLBIOS) arg;

   sPnlBios->state = FPBP;
   return 0;
}

/*-----------------------------------------------------------------
 * Nscpnlfn_infofrombios
 *
 * Description: This function gets the panel state from Bios. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_infofrombios(PGAL_BASE arg)
{
   PGAL_PNLBIOS sPnlBios = (PGAL_PNLBIOS) arg;

   sPnlBios->XRes = FPBX;
   sPnlBios->YRes = FPBY;
   sPnlBios->Bpp  = FPBD;
   sPnlBios->Freq = FPBF;
   return 0;
}

/*-----------------------------------------------------------------
 * Nscpnlfn_setpanelparams
 *
 * Description: This function sets the panel parameters. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_setpanelparams(PGAL_BASE arg)
{
   PGAL_PNLPARAMS sPnlParams = (PGAL_PNLPARAMS) arg;

   Pnl_SetPanelParam(&(sPnlParams->PanelParams));
   return 0;
}

/*-----------------------------------------------------------------
 * Nscpnlfn_getpanelparams
 *
 * Description: This function gets the panel parameters. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_getpanelparams(PGAL_BASE arg)
{
   PGAL_PNLPARAMS sPnlParams = (PGAL_PNLPARAMS) arg;

   Pnl_GetPanelParam(&(sPnlParams->PanelParams));
   return 0;
}

/*-----------------------------------------------------------------
 *  Nscpnlfn_initpanel
 *
 * Description: This function intializes the panel. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_initpanel(PGAL_BASE arg)
{
   PGAL_PNLPARAMS sPnlParams = (PGAL_PNLPARAMS) arg;

   Pnl_InitPanel((&(sPnlParams->PanelParams)));
   return 0;
}

/*-----------------------------------------------------------------
 *  Nscpnlfn_savepanel
 *
 * Description: This function saves the current panel state. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_savepanel(PGAL_BASE arg)
{
   Pnl_SavePanelState();

   return 0;
}

/*-----------------------------------------------------------------
 *  Nscpnlfn_restorepanel
 *
 * Description: This function restores the current panel state. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_restorepanel(PGAL_BASE arg)
{
   Pnl_RestorePanelState();

   return 0;
}

/*-----------------------------------------------------------------
 *  Nscpnlfn_powerup
 *
 * Description: This function Powers up the 9211 for panel. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_powerup(PGAL_BASE arg)
{
   Pnl_PowerUp();

   return 0;
}

/*-----------------------------------------------------------------
 *  Nscpnlfn_powerdown
 *
 * Description: This function Powers down the 9211 for panel. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *              return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscpnlfn_powerdown(PGAL_BASE arg)
{
   Pnl_PowerDown();
   return 0;
}

/*-----------------------------------------------------------------
 *  Nscenable_panning
 *
 * Description: This routine  enables the panning when the Mode 
 *              is bigger than the panel size.
 *
 *  parameters:
 *           x: x-positon of the screen.
 *           y: y-positon of the screen.
 *                              
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscenable_panning(PGAL_BASE arg)
{
   PGAL_ENABLEPANNING sEnablePanning = (PGAL_ENABLEPANNING) arg;

   gfx_enable_panning(sEnablePanning->x, sEnablePanning->y);
   return 0;
}

/*Panel**********************************************************/
#endif

#ifdef CONFIG_FB_NSC_TV
/*TV**********************************************************/

/*-----------------------------------------------------------------
 * Nscgfxfn_settvparams
 *
 * Description: This function sets the tv parameters of
 *                              tvparameters structure. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_settvparams(PGAL_BASE arg)
{
   PGAL_TVPARAMS sTvParams = (PGAL_TVPARAMS) arg;

   /* sequence might be important */
   if (sTvParams->dwFlags & GAL_TVRESOLUTION) {
      gfx_set_tv_display(sTvParams->wWidth, sTvParams->wHeight);
   }
   if (sTvParams->dwFlags & GAL_TVFORMAT) {
      gfx_set_tv_format(sTvParams->wStandard, sTvParams->wType);
   }
   if (sTvParams->dwFlags & GAL_TVOUTPUT) {
      gfx_set_tv_output(sTvParams->wOutput);
   }
   if (sTvParams->dwFlags & GAL_TVSTATE) {
      gfx_set_tv_enable(sTvParams->bState);
   }
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gettvparams
 *
 * Description: This function gets the tv parameters of the
 *                              tvparameters structure to the user. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gettvparams(PGAL_BASE arg)
{
   PGAL_TVPARAMS sTvParams = (PGAL_TVPARAMS) arg;
   unsigned int state = 0;

   if (sTvParams->dwFlags & GAL_TVSTATE) {
      sTvParams->bState = gfx_get_tv_enable(&state);
   }
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_settvtiming
 *
 * Description: This function sets the tv timing registers
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_settvtiming(PGAL_BASE arg)
{
   PGAL_TVTIMING sTvTiming = (PGAL_TVTIMING) arg;
   unsigned long VideoType = gfx_detect_video();

   if (VideoType == GFX_CPU_SC1200) {
      /* TV Format */
      WRITE_VID32(SC1200_TVOUT_HORZ_TIM, sTvTiming->HorzTim);
      WRITE_VID32(SC1200_TVOUT_HORZ_SYNC, sTvTiming->HorzSync);
      WRITE_VID32(SC1200_TVOUT_VERT_SYNC, sTvTiming->VertSync);
      WRITE_VID32(SC1200_TVOUT_LINE_END, sTvTiming->LineEnd);
      WRITE_VID32(SC1200_TVOUT_VERT_DOWNSCALE, sTvTiming->VertDownscale);
      WRITE_VID32(SC1200_TVOUT_HORZ_SCALING, sTvTiming->HorzScaling);
      WRITE_VID32(SC1200_TVENC_TIM_CTRL_1, sTvTiming->TimCtrl1);
      WRITE_VID32(SC1200_TVENC_TIM_CTRL_2, sTvTiming->TimCtrl2);
      WRITE_VID32(SC1200_TVENC_SUB_FREQ, sTvTiming->Subfreq);
      WRITE_VID32(SC1200_TVENC_DISP_POS, sTvTiming->DispPos);
      WRITE_VID32(SC1200_TVENC_DISP_SIZE, sTvTiming->DispSize);
      /* TV Output */
      WRITE_VID32(SC1200_TVENC_TIM_CTRL_2, sTvTiming->TimCtrl2);
      WRITE_VID32(SC1200_TVOUT_DEBUG, sTvTiming->Debug);
      /* TV Enable */
      WRITE_VID32(SC1200_TVENC_DAC_CONTROL, sTvTiming->DacCtrl);
   }
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gettvtiming
 *
 * Description: This function gets the tv timing registers
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gettvtiming(PGAL_BASE arg)
{
   PGAL_TVTIMING sTvTiming = (PGAL_TVTIMING) arg;
   unsigned long VideoType = gfx_detect_video();

   if (VideoType == GFX_CPU_SC1200) {
      /* TV Format */
      sTvTiming->HorzTim = READ_VID32(SC1200_TVOUT_HORZ_TIM);
      sTvTiming->HorzSync = READ_VID32(SC1200_TVOUT_HORZ_SYNC);
      sTvTiming->VertSync = READ_VID32(SC1200_TVOUT_VERT_SYNC);
      sTvTiming->LineEnd = READ_VID32(SC1200_TVOUT_LINE_END);
      sTvTiming->VertDownscale = READ_VID32(SC1200_TVOUT_VERT_DOWNSCALE);
      sTvTiming->HorzScaling = READ_VID32(SC1200_TVOUT_HORZ_SCALING);
      sTvTiming->TimCtrl1 = READ_VID32(SC1200_TVENC_TIM_CTRL_1);
      sTvTiming->TimCtrl2 = READ_VID32(SC1200_TVENC_TIM_CTRL_2);
      sTvTiming->Subfreq = READ_VID32(SC1200_TVENC_SUB_FREQ);
      sTvTiming->DispPos = READ_VID32(SC1200_TVENC_DISP_POS);
      sTvTiming->DispSize = READ_VID32(SC1200_TVENC_DISP_SIZE);
      /* TV Output */
      sTvTiming->TimCtrl2 = READ_VID32(SC1200_TVENC_TIM_CTRL_2);
      sTvTiming->Debug = READ_VID32(SC1200_TVOUT_DEBUG);
      /* TV Enable */
      sTvTiming->DacCtrl = READ_VID32(SC1200_TVENC_DAC_CONTROL);
   }

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_settvenable
 *
 * Description: This function sets the tv enable
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_settvenable(PGAL_BASE arg)
{
   PGAL_TVPARAMS sTvParams = (PGAL_TVPARAMS) arg;

   gfx_set_tv_enable(sTvParams->bState);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gettvenable
 *
 * Description: This function gets the tv enable to the user. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gettvenable(PGAL_BASE arg)
{
   PGAL_TVPARAMS sTvParams = (PGAL_TVPARAMS) arg;
   unsigned int state = 0;

   sTvParams->bState = 0;
   gfx_get_tv_enable(&state);
   sTvParams->bState = state;
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_istvmodesupported
 *
 * Description: This function verifies that a given tv mode is
 *                              supported or not.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_istvmodesupported(PGAL_BASE arg)
{
   PGAL_TVPARAMS sTvParams = (PGAL_TVPARAMS) arg;

   sTvParams->bState = gfx_is_tv_display_mode_supported(sTvParams->wWidth,
                                                        sTvParams->wHeight,
                                                        sTvParams->wStandard);
   return 0;
}

/*TV**********************************************************/
#endif
/*Video**********************************************************/
/*-----------------------------------------------------------------
 * Nscgfxfn_setvideoenable
 *
 * Description: This function enable or disables video overlay.
 *
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/

int
Nscgfxfn_setvideoenable(PGAL_BASE arg)
{
   PGAL_VIDEOENABLE sSetVideo = (PGAL_VIDEOENABLE) arg;

   gfx_set_video_enable(sSetVideo->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideoformat
 *
 * Description: This function sets the format of the video 
 *                              overlay data.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: 
 *                              zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideoformat(PGAL_BASE arg)
{
   PGAL_VIDEOFORMAT sSetVideo = (PGAL_VIDEOFORMAT) arg;

   gfx_set_video_format(sSetVideo->format);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideosize
 *
 * Description: This function sets the size of the source buffer. 
 *                              It is used to determine how much data is 
 *                              transferred between the display controller and the 
 *                              video hardware. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideosize(PGAL_BASE arg)
{
   PGAL_VIDEOSIZE sSetVideo = (PGAL_VIDEOSIZE) arg;

   gfx_set_video_size(sSetVideo->width, sSetVideo->height);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideooffset
 *
 * Description: This function sets the base offset of the video 
 *                              buffer in graphics memory. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideooffset(PGAL_BASE arg)
{
   PGAL_VIDEOOFFSET sSetVideo = (PGAL_VIDEOOFFSET) arg;

   gfx_set_video_offset(sSetVideo->offset);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideoyuvoffsets
 *
 * Description: This function sets the base offset of the YUV video 
 *                              buffer in graphics memory. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideoyuvoffsets(PGAL_BASE arg)
{
   PGAL_VIDEOYUVOFFSETS sSetVideo = (PGAL_VIDEOYUVOFFSETS) arg;

   gfx_set_video_yuv_offsets(sSetVideo->dwYoffset,
                             sSetVideo->dwUoffset, sSetVideo->dwVoffset);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideoyuvpitch
 *
 * Description: This function sets the pitch of the YUV video 
 *                              buffer in graphics memory. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideoyuvpitch(PGAL_BASE arg)
{
   PGAL_VIDEOYUVPITCH sSetVideo = (PGAL_VIDEOYUVPITCH) arg;

   gfx_set_video_yuv_pitch(sSetVideo->y_pitch, sSetVideo->uv_pitch);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvideoyuvpitch
 *
 * Description: This function gets the Video YUV pitch.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvideoyuvpitch(PGAL_BASE arg)
{
   PGAL_VIDEOYUVPITCH sGetVideoyuvpitch = (PGAL_VIDEOYUVPITCH) arg;

   gfx_get_video_yuv_pitch(&(sGetVideoyuvpitch->y_pitch),
                           &(sGetVideoyuvpitch->uv_pitch));

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideowindow
 *
 * Description: This function sets the location of the video 
 *                              overlay window.  
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideowindow(PGAL_BASE arg)
{
   PGAL_VIDEOWINDOW sSetVideo = (PGAL_VIDEOWINDOW) arg;

   gfx_set_video_window(sSetVideo->x, sSetVideo->y, sSetVideo->w,
                        sSetVideo->h);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideoscale
 *
 * Description: This function sets the scale factors of the video 
 *                              overlay window. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideoscale(PGAL_BASE arg)
{
   PGAL_VIDEOSCALE sSetVideo = (PGAL_VIDEOSCALE) arg;

   gfx_set_video_scale(sSetVideo->srcw, sSetVideo->srch,
                       sSetVideo->dstw, sSetVideo->dsth);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideofilter
 *
 * Description: This function sets the hardware filtering for the 
 *                              video overlay. 
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideofilter(PGAL_BASE arg)
{
   PGAL_VIDEOFILTER sSetVideo = (PGAL_VIDEOFILTER) arg;

   gfx_set_video_filter(sSetVideo->xfilter, sSetVideo->yfilter);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideocolorkey
 *
 * Description: This function sets the color key value used by the 
 *                              video overlay hardware.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero  was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideocolorkey(PGAL_BASE arg)
{
   PGAL_VIDEOCOLORKEY sSetVideo = (PGAL_VIDEOCOLORKEY) arg;

   gfx_set_video_color_key(sSetVideo->key, sSetVideo->mask,
                           sSetVideo->bluescreen);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideodownscale_enable
 *
 * Description: This function enables or disables horizontal
 *                              downscaling.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideodownscale_enable(PGAL_BASE arg)
{
   PGAL_VIDEODOWNSCALEENABLE sSetVideo = (PGAL_VIDEODOWNSCALEENABLE) arg;

   gfx_set_video_downscale_enable(sSetVideo->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideodownscale_config
 *
 * Description: This function programs the horizontal video 
 *                              downscale factor.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideodownscale_config(PGAL_BASE arg)
{
   PGAL_VIDEODOWNSCALECONFIG sSetVideo = (PGAL_VIDEODOWNSCALECONFIG) arg;

   gfx_set_video_downscale_config(sSetVideo->type, sSetVideo->m);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideodownscale_coefficients
 *
 * Description: This function programs the filter coefficients for 
 *                              horizontal video downscaling.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideodownscale_coefficients(PGAL_BASE arg)
{
   PGAL_VIDEODOWNSCALECOEFF sSetVideo = (PGAL_VIDEODOWNSCALECOEFF) arg;

   gfx_set_video_downscale_coefficients(sSetVideo->coef1,
                                        sSetVideo->coef2, sSetVideo->coef3,
                                        sSetVideo->coef4);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideosource
 *
 * Description: This function selects the souce for incoming video
 *                              data to the video processor.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideosource(PGAL_BASE arg)
{
   PGAL_VIDEOSOURCE sSetVideo = (PGAL_VIDEOSOURCE) arg;
   VideoSourceType source = VIDEO_SOURCE_MEMORY;

   switch (sSetVideo->source) {

   case GAL_VIDEO_SOURCE_MEMORY:

      source = VIDEO_SOURCE_MEMORY;
      break;

   case GAL_VIDEO_SOURCE_DVIP:

      source = VIDEO_SOURCE_DVIP;
      break;
   }
   gfx_set_video_source(source);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideointerlaced
 *
 * Description: This function programs the video overlay mode for
 *                              interlaced or progressive scanning.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideointerlaced(PGAL_BASE arg)
{
   PGAL_SETVIDEOINTERLACED sSetVideo = (PGAL_SETVIDEOINTERLACED) arg;

   gfx_set_video_interlaced(sSetVideo->enable);
   return 0;
}

int
Nscgfxfn_getvideointerlaced(PGAL_BASE arg)
{
   PGAL_GETVIDEOINTERLACED sGetVideointerlaced =
         (PGAL_GETVIDEOINTERLACED) arg;

   sGetVideointerlaced->interlaced = gfx_get_video_interlaced();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setcolorspaceYUV
 *
 * Description: This function programs the color space for 
 *                              graphics/video blending.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setcolorspaceYUV(PGAL_BASE arg)
{
   PGAL_COLORSPACEYUV sSetVideo = (PGAL_COLORSPACEYUV) arg;

   gfx_set_color_space_YUV(sSetVideo->colorspace);
   return 0;
}

int
Nscgfxfn_getcolorspaceYUV(PGAL_BASE arg)
{
   PGAL_COLORSPACEYUV sSetVideo = (PGAL_COLORSPACEYUV) arg;

   sSetVideo->colorspace = gfx_get_color_space_YUV();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideocursor
 *
 * Description: This function programs the video cursor color key
 *                              and color mask.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideocursor(PGAL_BASE arg)
{
   PGAL_VIDEOCURSOR sSetVideo = (PGAL_VIDEOCURSOR) arg;

   gfx_set_video_cursor(sSetVideo->key, sSetVideo->mask,
                        sSetVideo->select_color2, sSetVideo->color1,
                        sSetVideo->color2);
   return 0;
}

int
Nscgfxfn_getvideocursor(PGAL_BASE arg)
{
   PGAL_VIDEOCURSOR sSetVideo = (PGAL_VIDEOCURSOR) arg;

   gfx_get_video_cursor(&sSetVideo->key, &sSetVideo->mask,
                        &sSetVideo->select_color2, &sSetVideo->color1,
                        &sSetVideo->color2);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideorequest
 *
 * Description: This function sets the horizontal(pixel)
 *                              and vertical(line) video request values.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideorequest(PGAL_BASE arg)
{
   PGAL_VIDEOREQUEST sSetVideo = (PGAL_VIDEOREQUEST) arg;

   gfx_set_video_request(sSetVideo->x, sSetVideo->y);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setalphaenable
 *
 * Description: This function enables or disables the currently 
 *                              selected alpha region.
 *  parameters:
 *              pvarg:  This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setalphaenable(PGAL_BASE arg)
{
   PGAL_ALPHAENABLE sSetVideo = (PGAL_ALPHAENABLE) arg;

   gfx_set_alpha_enable(sSetVideo->enable);
   return 0;
}

int
Nscgfxfn_getalphaenable(PGAL_BASE arg)
{
   PGAL_ALPHAENABLE sGetalphaenable = (PGAL_ALPHAENABLE) arg;

   gfx_get_alpha_enable(&sGetalphaenable->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setalphawindow
 *
 * Description:This function sets the position and size of the 
 *                         currently selected alpha region.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setalphawindow(PGAL_BASE arg)
{
   PGAL_ALPHAWINDOW sSetVideo = (PGAL_ALPHAWINDOW) arg;

   gfx_set_alpha_window(sSetVideo->x, sSetVideo->y, sSetVideo->width,
                        sSetVideo->height);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setalphavalue
 *
 * Description: This function the alpha of the currently selected 
 *                              alpha region.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setalphavalue(PGAL_BASE arg)
{
   PGAL_ALPHAVALUE sSetVideo = (PGAL_ALPHAVALUE) arg;

   gfx_set_alpha_value(sSetVideo->alpha, sSetVideo->delta);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setalphapriority
 *
 * Description: This function sets the priority of the currently 
 *                              selected alpha region.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setalphapriority(PGAL_BASE arg)
{
   PGAL_ALPHAPRIORITY sSetVideo = (PGAL_ALPHAPRIORITY) arg;

   gfx_set_alpha_priority(sSetVideo->priority);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setalphacolor
 *
 * Description: This function sets the color of the currently 
 *                              selected alpha region.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setalphacolor(PGAL_BASE arg)
{
   PGAL_ALPHACOLOR sSetVideo = (PGAL_ALPHACOLOR) arg;

   gfx_set_alpha_color(sSetVideo->color);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setalpharegion
 *
 * Description: This function selects the alpha region for successive
 *                              calls to set the parameters.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setalpharegion(PGAL_BASE arg)
{
   PGAL_ALPHAREGION sSetVideo = (PGAL_ALPHAREGION) arg;

   gfx_select_alpha_region(sSetVideo->region);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideooutsidealpha
 *
 * Description: This function programs what is displayed inside the 
 *                              video overlay area.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideooutsidealpha(PGAL_BASE arg)
{
   PGAL_VIDEOOUTSIDEALPHA sSetVideo = (PGAL_VIDEOOUTSIDEALPHA) arg;

   gfx_set_no_ck_outside_alpha(sSetVideo->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideopalette
 *
 * Description: This function sets the palette in the video overlay 
 *                              hardware.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: 
 *                              zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideopalette(PGAL_BASE arg)
{
   PGAL_VIDEOPALETTE sSetVideo = (PGAL_VIDEOPALETTE) arg;

   if (sSetVideo->identity) {
      gfx_set_video_palette(NULL);
   } else {
      gfx_set_video_palette(sSetVideo->palette);
   }
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvideoinfo
 *
 * Description: This function reads the video information 
 *                              of the hardware to the user.
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvideoinfo(PGAL_BASE arg)
{
   PGAL_VIDEOINFO sGetVideo = (PGAL_VIDEOINFO) arg;

   sGetVideo->enable = gfx_get_video_enable();
   sGetVideo->format = gfx_get_video_format();
   sGetVideo->filter = gfx_get_video_filter();
   sGetVideo->src_size = gfx_get_video_src_size();
   sGetVideo->dst_size = gfx_get_video_dst_size();
   sGetVideo->line_size = gfx_get_video_line_size();
   sGetVideo->xclip = gfx_get_video_xclip();
   sGetVideo->offset = gfx_get_video_offset();
   sGetVideo->scale = gfx_get_video_scale();
   sGetVideo->position = gfx_get_video_position();
   sGetVideo->color_key_src = gfx_get_video_color_key_src();
   sGetVideo->color_key = gfx_get_video_color_key();
   sGetVideo->color_key_mask = gfx_get_video_color_key_mask();
   gfx_get_video_downscale_enable(&sGetVideo->downscale_enable);
   gfx_get_video_downscale_config(&sGetVideo->downscale_type,
                                  &sGetVideo->downscale_mask);
   gfx_get_video_downscale_coefficients(&sGetVideo->downscale_coef1,
                                        &sGetVideo->downscale_coef2,
                                        &sGetVideo->downscale_coef3,
                                        &sGetVideo->downscale_coef4);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvipenable
 *
 * Description: This function enables the Video input port.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/

int
Nscgfxfn_setvipenable(PGAL_BASE arg)
{
   PGAL_VIPENABLE sSetVipenable = (PGAL_VIPENABLE) arg;

   gfx_set_vip_enable(sSetVipenable->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvipenable
 *
 * Description: This function get the state(enable/disable) of the Video 
 *                              input port.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvipenable(PGAL_BASE arg)
{
   PGAL_VIPENABLE sGetVipenable = (PGAL_VIPENABLE) arg;

   sGetVipenable->enable = gfx_get_vip_enable();
   return 0;
}

int
Nscgfxfn_setdestinationstride(PGAL_BASE arg)
{
   PGAL_STRIDE sStride = (PGAL_STRIDE) arg;

   gfx2_set_destination_stride(sStride->stride);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvipcapturerunmode
 *
 * Description: This function enables the VIP capture run mode.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/

int
Nscgfxfn_setvipcapturerunmode(PGAL_BASE arg)
{
   PGAL_VIPCAPTURERUNMODE sSetVipcapturerunmode =
         (PGAL_VIPCAPTURERUNMODE) arg;

   gfx_set_vip_capture_run_mode(sSetVipcapturerunmode->mode);
   return 0;
}

int
Nscgfxfn_setsourcetransparency(PGAL_BASE arg)
{
   PGAL_SOURCETRANSPARENCY sTrans = (PGAL_SOURCETRANSPARENCY) arg;

   gfx2_set_source_transparency(sTrans->color, sTrans->mask);
   return 0;
}

int
Nscgfxfn_setpatternorigin(PGAL_BASE arg)
{
   PGAL_PATTERNORIGIN sPat = (PGAL_PATTERNORIGIN) arg;

   gfx2_set_pattern_origin(sPat->x, sPat->y);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvippitch
 *
 * Description: This function sets the Video input port pitch.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvippitch(PGAL_BASE arg)
{
   PGAL_VIPPITCH sSetVipPitch = (PGAL_VIPPITCH) arg;

   gfx_set_vip_pitch(sSetVipPitch->pitch);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvippitch
 *
 * Description: This function gets the Video input port pitch.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvippitch(PGAL_BASE arg)
{
   PGAL_VIPPITCH sGetVipPitch = (PGAL_VIPPITCH) arg;

   sGetVipPitch->pitch = gfx_get_vip_pitch();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvipmode
 *
 * Description: This function sets the Video input port mode.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvipmode(PGAL_BASE arg)
{
   PGAL_VIPMODE sSetVipMode = (PGAL_VIPMODE) arg;

   gfx_set_vip_mode(sSetVipMode->mode);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvipmode
 *
 * Description: This function gets the Video input port mode.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvipmode(PGAL_BASE arg)
{
   PGAL_VIPMODE sGetVipMode = (PGAL_VIPMODE) arg;

   sGetVipMode->mode = gfx_get_vip_mode();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvipbase
 *
 * Description: This function sets the Video input port base.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvipbase(PGAL_BASE arg)
{
   PGAL_VIPBASE sSetVipBase = (PGAL_VIPBASE) arg;

   gfx_set_vip_base(sSetVipBase->even, sSetVipBase->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvipbusrth
 *
 * Description: This function sets the Video input port Bus request 
 *                              threshold high.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvipbusrth(PGAL_BASE arg)
{
   PGAL_VIPBUS_RTH sSetVipBRTH = (PGAL_VIPBUS_RTH) arg;

   gfx_set_vip_bus_request_threshold_high(sSetVipBRTH->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvipbusrth
 *
 * Description: This function gets the Video input port bus threshold 
 *                              request high.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvipbusrth(PGAL_BASE arg)
{
   PGAL_VIPBUS_RTH sGetVipBRTH = (PGAL_VIPBUS_RTH) arg;

   sGetVipBRTH->enable = gfx_get_vip_bus_request_threshold_high();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setviplastline
 *
 * Description: This function sets the Video input port last line.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setviplastline(PGAL_BASE arg)
{
   PGAL_VIPLASTLINE sSetViplastline = (PGAL_VIPLASTLINE) arg;

   gfx_set_vip_last_line(sSetViplastline->last_line);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvipbase
 *
 * Description: This function gets the Video input port base.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvipbase(PGAL_BASE arg)
{
   PGAL_VIPBASE sGetVipBase = (PGAL_VIPBASE) arg;

   sGetVipBase->address = gfx_get_vip_base(sGetVipBase->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvipline
 *
 * Description: This function gets the Video input port line.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvipline(PGAL_BASE arg)
{
   PGAL_VIPLINE sGetVipline = (PGAL_VIPLINE) arg;

   sGetVipline->status = gfx_get_vip_line();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_testvipoddfield
 *
 * Description: This function tests the Video input port odd field.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_testvipoddfield(PGAL_BASE arg)
{
   PGAL_TESTVIPODDFIELD sTestVipoddfield = (PGAL_TESTVIPODDFIELD) arg;

   sTestVipoddfield->status = gfx_test_vip_odd_field();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_testvipbasesupdated
 *
 * Description: This function test the Video input port bases updated.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_testvipbasesupdated(PGAL_BASE arg)
{
   PGAL_TESTVIPBASESUPDATED sTestVipbasesupdated =
         (PGAL_TESTVIPBASESUPDATED) arg;

   sTestVipbasesupdated->status = gfx_test_vip_bases_updated();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_testvipfifooverflow
 *
 * Description: This function test the Video input port FIFO overflow.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_testvipfifooverflow(PGAL_BASE arg)
{
   PGAL_TESTVIPOVERFLOW sTestVipoverflow = (PGAL_TESTVIPOVERFLOW) arg;

   sTestVipoverflow->status = gfx_test_vip_fifo_overflow();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbienable
 *
 * Description: This function enables the VBI. 
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbienable(PGAL_BASE arg)
{
   PGAL_VBIENABLE sSetVbienable = (PGAL_VBIENABLE) arg;

   gfx_set_vbi_enable(sSetVbienable->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbienable
 *
 * Description: This function gets the VBI enable state.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbienable(PGAL_BASE arg)
{
   PGAL_VBIENABLE sGetVbienable = (PGAL_VBIENABLE) arg;

   sGetVbienable->enable = gfx_get_vbi_enable();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbibase
 *
 * Description: This function sets the VBI base.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbibase(PGAL_BASE arg)
{
   PGAL_VBIBASE sSetVbiBase = (PGAL_VBIBASE) arg;

   gfx_set_vbi_base(sSetVbiBase->even, sSetVbiBase->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbibase
 *
 * Description: This function gets the VBI base.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbibase(PGAL_BASE arg)
{
   PGAL_VBIBASE sGetVbiBase = (PGAL_VBIBASE) arg;

   sGetVbiBase->address = gfx_get_vbi_base(sGetVbiBase->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbipitch 
 *
 * Description: This function sets the VBI pitch.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbipitch(PGAL_BASE arg)
{
   PGAL_VBIPITCH sSetVbiPitch = (PGAL_VBIPITCH) arg;

   gfx_set_vbi_pitch(sSetVbiPitch->pitch);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbipitch
 *
 * Description: This function gets the VBI pitch.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbipitch(PGAL_BASE arg)
{
   PGAL_VBIPITCH sGetVbiPitch = (PGAL_VBIPITCH) arg;

   sGetVbiPitch->pitch = gfx_get_vbi_pitch();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbimode
 *
 * Description: This function sets the VBI mode.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbimode(PGAL_BASE arg)
{
   PGAL_VBIMODE sSetVbiMode = (PGAL_VBIMODE) arg;

   gfx_set_vbi_mode(sSetVbiMode->mode);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbimode
 *
 * Description: This function gets the VBI mode.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbimode(PGAL_BASE arg)
{
   PGAL_VBIMODE sGetVbiMode = (PGAL_VBIMODE) arg;

   sGetVbiMode->mode = gfx_get_vbi_mode();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbidirect
 *
 * Description: This function gets the VBI direct lines.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbidirect(PGAL_BASE arg)
{
   PGAL_GETVBIDIRECT sGetVbidirect = (PGAL_GETVBIDIRECT) arg;

   sGetVbidirect->direct_lines = gfx_get_vbi_direct(sGetVbidirect->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbidirect
 *
 * Description: This function sets the VBI direct lines.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbidirect(PGAL_BASE arg)
{
   PGAL_SETVBIDIRECT sSetVbidirect = (PGAL_SETVBIDIRECT) arg;

   gfx_set_vbi_direct(sSetVbidirect->even_lines, sSetVbidirect->odd_lines);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbiinterrupt
 *
 * Description: This function sets the VBI interrupt.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbiinterrupt(PGAL_BASE arg)
{
   PGAL_VBIINTERRUPT sSetVbiinterrupt = (PGAL_VBIINTERRUPT) arg;

   gfx_set_vbi_interrupt(sSetVbiinterrupt->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbiinterrupt
 *
 * Description: This function gets the VBI interrupt.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbiinterrupt(PGAL_BASE arg)
{
   PGAL_VBIINTERRUPT sGetVbiinterrupt = (PGAL_VBIINTERRUPT) arg;

   sGetVbiinterrupt->enable = gfx_get_vbi_interrupt();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_seticonenable
 *
 * Description: This function enables the video icon. 
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_seticonenable(PGAL_BASE arg)
{
   PGAL_ICONENABLE sSetIconenable = (PGAL_ICONENABLE) arg;

   gfx_set_icon_enable(sSetIconenable->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_seticoncolors
 *
 * Description: This function sets the video icon colors. 
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_seticoncolors(PGAL_BASE arg)
{
   PGAL_ICONCOLORS sSetIconcolors = (PGAL_ICONCOLORS) arg;

   gfx_set_icon_colors(sSetIconcolors->color0, sSetIconcolors->color1,
                       sSetIconcolors->color2);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_seticonposition
 *
 * Description: This function sets the icon position. 
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_seticonposition(PGAL_BASE arg)
{
   PGAL_ICONPOSITION sSetIconposi = (PGAL_ICONPOSITION) arg;

   gfx_set_icon_position(sSetIconposi->memoffset, sSetIconposi->xpos);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_seticonshape64
 *
 * Description: This function intializes the icon buffer according to 
 *                              the current mode. 
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_seticonshape64(PGAL_BASE arg)
{
   PGAL_ICONSHAPE64 sSetIconshape64 = (PGAL_ICONSHAPE64) arg;

   gfx_set_icon_shape64(sSetIconshape64->memoffset,
                        (unsigned long *)sSetIconshape64->andmask,
                        (unsigned long *)sSetIconshape64->xormask,
                        sSetIconshape64->lines);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2setsourcestride
 *
 * Description: This function sets source stride.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2setsourcestride(PGAL_BASE arg)
{
   PGAL_STRIDE sSetsourcestride = (PGAL_STRIDE) arg;

   gfx2_set_source_stride(sSetsourcestride->stride);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2setsourcetransparency
 *
 * Description: This function sets source transparency.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2setsourcetransparency(PGAL_BASE arg)
{
   PGAL_SOURCETRANSPARENCY sSetsourcetransparency =
         (PGAL_SOURCETRANSPARENCY) arg;

   gfx2_set_source_transparency(sSetsourcetransparency->color,
                                sSetsourcetransparency->mask);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2setalphamode
 *
 * Description: This function sets the alphamode.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2setalphamode(PGAL_BASE arg)
{
   PGAL_GFX2ALPHAMODE sAlpha = (PGAL_GFX2ALPHAMODE) arg;

   gfx2_set_alpha_mode(sAlpha->mode);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2setalphavalue
 *
 * Description: This function sets the alpha value for GX2.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2setalphavalue(PGAL_BASE arg)
{
   PGAL_GFX2ALPHAVALUE sAlpha = (PGAL_GFX2ALPHAVALUE) arg;

   gfx2_set_alpha_value(sAlpha->value);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2patternfill
 *
 * Description: This function fills the pattern for GX2.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2patternfill(PGAL_BASE arg)
{
   PGAL_GFX2PATTERNFILL sPat = (PGAL_GFX2PATTERNFILL) arg;

   gfx2_pattern_fill(sPat->dstoffset, sPat->width, sPat->height);
   return 0;
}

int
Nscgfxfn_gfx2colorpatternfill(PGAL_BASE arg)
{
   PGAL_GFX2COLORPATTERNFILL sPat = (PGAL_GFX2COLORPATTERNFILL) arg;

   gfx2_color_pattern_fill(sPat->dstoffset, sPat->width, sPat->height,
                           (unsigned long *)sPat->pattern);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2screentoscreenblt
 *
 * Description: This function do screen to screen blt of GX2.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2screentoscreenblt(PGAL_BASE arg)
{
   PGAL_GFX2SCREENTOSCREENBLT sBlt = (PGAL_GFX2SCREENTOSCREENBLT) arg;

   gfx2_screen_to_screen_blt(sBlt->srcoffset, sBlt->dstoffset, sBlt->width,
                             sBlt->height, sBlt->flags);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2monoexpandblt
 *
 * Description: This function do mono expand blt of GX2.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2monoexpandblt(PGAL_BASE arg)
{
   PGAL_GFX2MONOEXPANDBLT sBlt = (PGAL_GFX2MONOEXPANDBLT) arg;

   gfx2_mono_expand_blt(sBlt->srcbase, sBlt->srcx, sBlt->srcy,
                        sBlt->dstoffset, sBlt->width, sBlt->height,
                        sBlt->byte_packed);
   return 0;
}

int
Nscgfxfn_gfx2colorbmptoscrblt(PGAL_BASE arg)
{
   PGAL_GFX2COLORBMPTOSCRBLT sBlt = (PGAL_GFX2COLORBMPTOSCRBLT) arg;

   gfx2_color_bitmap_to_screen_blt(sBlt->srcx, sBlt->srcy,
                                   sBlt->dstoffset, sBlt->width, sBlt->height,
                                   (unsigned char *)sBlt->data, sBlt->pitch);
   return 0;
}

int
Nscgfxfn_gfx2monobmptoscrblt(PGAL_BASE arg)
{
   PGAL_GFX2MONOBMPTOSCRBLT sBlt = (PGAL_GFX2MONOBMPTOSCRBLT) arg;

   gfx2_mono_bitmap_to_screen_blt(sBlt->srcx, sBlt->srcy,
                                  sBlt->dstoffset, sBlt->width, sBlt->height,
                                  (unsigned char *)sBlt->data, sBlt->pitch);
   return 0;
}

int
Nscgfxfn_gfx2textblt(PGAL_BASE arg)
{
   PGAL_GFX2TEXTBLT sBlt = (PGAL_GFX2TEXTBLT) arg;

   gfx2_text_blt(sBlt->dstoffset, sBlt->width, sBlt->height,
                 (unsigned char *)sBlt->data);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2bresenhamline
 *
 * Description: This function used to draw bresenhamline for GX2.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2bresenhamline(PGAL_BASE arg)
{
   PGAL_GFX2BRESENHAMLINE sLine = (PGAL_GFX2BRESENHAMLINE) arg;

   gfx2_bresenham_line(sLine->dstoffset,
                       sLine->length, sLine->initerr,
                       sLine->axialerr, sLine->diagerr, sLine->flags);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_gfx2synctovblank
 *
 * Description: This function used to sync to vertical blank for GX2.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_gfx2synctovblank(PGAL_BASE arg)
{
   gfx2_sync_to_vblank();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvideoyuvoffsets
 *
 * Description: This function gets the Video yuv offsets.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvideoyuvoffsets(PGAL_BASE arg)
{
   PGAL_VIDEOYUVOFFSETS sGetVideoyuvoffsets = (PGAL_VIDEOYUVOFFSETS) arg;

   gfx_get_video_yuv_offsets(&sGetVideoyuvoffsets->dwYoffset,
                             &sGetVideoyuvoffsets->dwUoffset,
                             &sGetVideoyuvoffsets->dwVoffset);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideoleftcrop
 *
 * Description: This function sets the video left crop.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideoleftcrop(PGAL_BASE arg)
{
   PGAL_VIDEOLEFTCROP sSetVideoleftcrop = (PGAL_VIDEOLEFTCROP) arg;

   sSetVideoleftcrop->status = gfx_set_video_left_crop(sSetVideoleftcrop->x);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvideoverticaldownscale
 *
 * Description: This function sets the Video vertical down scale.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvideoverticaldownscale(PGAL_BASE arg)
{
   PGAL_VIDEOVERTICALDOWNSCALE sSetVideovdownscale =
         (PGAL_VIDEOVERTICALDOWNSCALE) arg;

   sSetVideovdownscale->status =
         gfx_set_video_vertical_downscale(sSetVideovdownscale->srch,
                                          sSetVideovdownscale->dsth);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbisource
 *
 * Description: This function sets the VBI source.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbisource(PGAL_BASE arg)
{
   PGAL_VBISOURCE sSetVbisource = (PGAL_VBISOURCE) arg;

   sSetVbisource->status = gfx_set_vbi_source(sSetVbisource->source);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbisource
 *
 * Description: This function gets the VBI source.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbisource(PGAL_BASE arg)
{
   PGAL_VBISOURCE sGetVbisource = (PGAL_VBISOURCE) arg;

   sGetVbisource->status = gfx_get_vbi_source(&sGetVbisource->source);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbilines
 *
 * Description: This function sets VBI lines.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbilines(PGAL_BASE arg)
{
   PGAL_VBILINES sSetVbilines = (PGAL_VBILINES) arg;

   sSetVbilines->status =
         gfx_set_vbi_lines(sSetVbilines->even, sSetVbilines->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbilines
 *
 * Description: This function gets the VBI lines.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbilines(PGAL_BASE arg)
{
   PGAL_VBILINES sGetVbilines = (PGAL_VBILINES) arg;

   sGetVbilines->lines = gfx_get_vbi_lines(sGetVbilines->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setvbitotal
 *
 * Description: This function sets the vbi total.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setvbitotal(PGAL_BASE arg)
{
   PGAL_VBITOTAL sSetVbitotal = (PGAL_VBITOTAL) arg;

   sSetVbitotal->status =
         gfx_set_vbi_total(sSetVbitotal->even, sSetVbitotal->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getvbitotal
 *
 * Description: This function gets the VBI total.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getvbitotal(PGAL_BASE arg)
{
   PGAL_VBITOTAL sGetVbitotal = (PGAL_VBITOTAL) arg;

   sGetVbitotal->total = gfx_get_vbi_lines(sGetVbitotal->odd);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setverticalscaleroffset
 *
 * Description: This function sets the verticalscaleroffset.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setverticalscaleroffset(PGAL_BASE arg)
{
   PGAL_VSCALEROFFSET sSetVscaleroffset = (PGAL_VSCALEROFFSET) arg;

   sSetVscaleroffset->status =
         gfx_set_vertical_scaler_offset(sSetVscaleroffset->offset);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getverticalscaleroffset
 *
 * Description: This function gets the vertical scaler offset.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getverticalscaleroffset(PGAL_BASE arg)
{
   PGAL_VSCALEROFFSET sGetVscaleroffset = (PGAL_VSCALEROFFSET) arg;

   sGetVscaleroffset->status =
         gfx_get_vertical_scaler_offset(&sGetVscaleroffset->offset);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getgenlockenable
 *
 * Description: This function gets the genlockenable state.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getgenlockenable(PGAL_BASE arg)
{
   PGAL_GENLOCKENABLE sGetgenlockenable = (PGAL_GENLOCKENABLE) arg;

   sGetgenlockenable->enable = gfx_get_genlock_enable();

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setgenlockenable
 *
 * Description: This function sets the genlock.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setgenlockenable(PGAL_BASE arg)
{
   PGAL_GENLOCKENABLE sSetgenlockenable = (PGAL_GENLOCKENABLE) arg;

   gfx_set_genlock_enable(sSetgenlockenable->enable);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getgenlockdelay
 *
 * Description: This function gets the genlockdelay.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getgenlockdelay(PGAL_BASE arg)
{
   PGAL_GENLOCKDELAY sGetgenlockdelay = (PGAL_GENLOCKDELAY) arg;

   sGetgenlockdelay->delay = gfx_get_genlock_delay();

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setgenlockdelay
 *
 * Description: This function sets the genlock delay.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setgenlockdelay(PGAL_BASE arg)
{
   PGAL_GENLOCKDELAY sSetgenlockdelay = (PGAL_GENLOCKDELAY) arg;

   gfx_set_genlock_delay(sSetgenlockdelay->delay);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_settoplineinodd
 *
 * Description: This function sets the topline.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_settoplineinodd(PGAL_BASE arg)
{
   PGAL_TOPLINEINODD sSettoplineinodd = (PGAL_TOPLINEINODD) arg;

   gfx_set_top_line_in_odd(sSettoplineinodd->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_readcrc
 *
 * Description: This function reads the crc.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_readcrc(PGAL_BASE arg)
{
   PGAL_READCRC sReadcrc = (PGAL_READCRC) arg;

   sReadcrc->crc = gfx_read_crc();

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_readwindowcrc
 *
 * Description: This function reads the window crc.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_readwindowcrc(PGAL_BASE arg)
{
   PGAL_READWINDOWCRC sReadwindowcrc = (PGAL_READWINDOWCRC) arg;

   sReadwindowcrc->crc = gfx_read_window_crc(sReadwindowcrc->source,
                                             sReadwindowcrc->x,
                                             sReadwindowcrc->y,
                                             sReadwindowcrc->width,
                                             sReadwindowcrc->height,
                                             sReadwindowcrc->crc32);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getmacrovisionenable
 *
 * Description: This function gets the macrovision enable state.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getmacrovisionenable(PGAL_BASE arg)
{
   PGAL_MACROVISIONENABLE sGetmacrovisionenable =
         (PGAL_MACROVISIONENABLE) arg;

   sGetmacrovisionenable->enable = gfx_get_macrovision_enable();
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_setmacrovisionenable
 *
 * Description: This function enables the macrovision.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_setmacrovisionenable(PGAL_BASE arg)
{
   PGAL_MACROVISIONENABLE sSetmacrovisionenable =
         (PGAL_MACROVISIONENABLE) arg;

   gfx_set_macrovision_enable(sSetmacrovisionenable->enable);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getalphasize
 *
 * Description: This function gets the alpha size.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getalphasize(PGAL_BASE arg)
{
   PGAL_ALPHASIZE sGetalphasize = (PGAL_ALPHASIZE) arg;

   gfx_get_alpha_size(sGetalphasize->x, sGetalphasize->y,
                      sGetalphasize->width, sGetalphasize->height);

   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getalphavalue
 *
 * Description: This function gets the alpha value.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getalphavalue(PGAL_BASE arg)
{
   PGAL_ALPHAVALUE sGetalphavalue = (PGAL_ALPHAVALUE) arg;

   gfx_get_alpha_value(&sGetalphavalue->alpha, &sGetalphavalue->delta);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getalphapriority
 *
 * Description: This function gets the alpha priority.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getalphapriority(PGAL_BASE arg)
{
   PGAL_ALPHAPRIORITY sGetalphapriority = (PGAL_ALPHAPRIORITY) arg;

   gfx_get_alpha_priority(&sGetalphapriority->priority);
   return 0;
}

/*-----------------------------------------------------------------
 * Nscgfxfn_getalphacolor
 *
 * Description: This function gets the alpha color.
 *                              
 *  parameters:
 *       pvarg: This is void pointer  points to the data which was 
 *                              passed to the ioctl by the application.
 *      return: zero was returned on success other wise error value
 *                              will be returned.
 *------------------------------------------------------------------*/
int
Nscgfxfn_getalphacolor(PGAL_BASE arg)
{
   PGAL_ALPHACOLOR sGetalphacolor = (PGAL_ALPHACOLOR) arg;

   gfx_get_alpha_color(&sGetalphacolor->color);
   return 0;
}

/*-----------------------------------------------------------------
 * Init_Dispatch_Table
 *
 * Description: This function intializes the all the graphic ioctl 
 *                              to the respective ioctl function pointer.
 *  parameters: none
 *      return: none
 *------------------------------------------------------------------*/
void
Init_Dispatch_Table()
{
   int iFnIndex = 0, Total = sizeof(IoctlTable) / sizeof(IOCTL_TABLE);

   MaxPacketSize = 0;

   while (iFnIndex < Total) {
      if (MaxPacketSize < IoctlTable[iFnIndex].inbytes)
         MaxPacketSize = IoctlTable[iFnIndex].inbytes;
      iFnIndex++;
   }

   IoctlBuf = kmalloc(MaxPacketSize, GFP_KERNEL);

   bDispatchTableInitialized = 1;
}

/*-----------------------------------------------------------------
 * geodefb_ioctl_ext
 *
 * Description: This function provide an interface to the Geode frame
 *                              buffer specific input output control commands.
 *  parameters:
 *       inode: Pointer to "inode" structure which contains the device
 *                              reference.
 *        file: Pointer to a "file" structure. 
 *         cmd: The IOCTL code which was issued by the application.
 *         arg: Pointer to the data which was passed to the IOCTL by
 *                              the application.
 *         con: The console from which the ioctl was issued.
 *        info: This is a pointer to the frame buffer information 
 *                              structure.  
 *      return: Returns 0 if the ioctl was handled successfully.
 *----------------------------------------------------------------*/
int
geodefb_ioctl_ext(struct inode *inode, struct file *file, u_int cmd,
                  u_long arg, int con, struct fb_info *info)
{
   int retval = -EINVAL;
   GAL_BASE sNscgfxbase;

   if (!bDispatchTableInitialized) {
      Init_Dispatch_Table();
   }
   if (!copy_from_user(&sNscgfxbase, (void *)arg, sizeof(sNscgfxbase))) {
      if (IO_PACKET_ISVALID(sNscgfxbase)) {
         DPRINTK("%d %d %d\n", sNscgfxbase.dwSubfunction,
                 IoctlTable[sNscgfxbase.dwSubfunction].inbytes,
                 IoctlTable[sNscgfxbase.dwSubfunction].outbytes);
         if ((IoctlTable[sNscgfxbase.dwSubfunction].inbytes) &&
             copy_from_user(IoctlBuf, (void *)arg,
                            IoctlTable[sNscgfxbase.dwSubfunction].inbytes)) {
            DPRINTK("Error reading from user\n");
            return -EFAULT;
         }

         retval = IoctlTable[sNscgfxbase.dwSubfunction].func(IoctlBuf);
         ((GAL_BASE *) IoctlBuf)->dwReturnValue = 0;

         if ((IoctlTable[sNscgfxbase.dwSubfunction].outbytes) &&
             copy_to_user((void *)arg, IoctlBuf,
                          IoctlTable[sNscgfxbase.dwSubfunction].outbytes)) {
            DPRINTK("Error writing to user\n");
            return -EFAULT;
         }
      }
   }
   return 0;
}
#endif
