 /*
  * 
 * <LIC_AMD_STD>
 * Copyright (c) 2004 Advanced Micro Devices, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * The full GNU General Public License is included in this distribution in the
 * file called COPYING
 * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * File Contents: This file contains the main functions of the Geode 
  *                frame buffer device driver, which includes functions 
  *                init, setup and ioctl etc.
  * [Notes:]           All the geode frame buffer driver functions can be 
  *                invoked through fbmem.c file functions
  *
  * Project:       Geode Frame buffer device driver
  * </DOC_AMD_STD>
  * 
  */


/* Enable/Disable PowerManagement by toggling the value 0/1 */
#define PM 1

#include "nsc_common_fb.h"
#include "nsc_galproto.h"

#if PM
/* powermanagement related functions */
#include <linux/pm.h>
#include "nsc_savreg.h"
static int gx1_geode_pm_callback(struct pm_dev *dev, pm_request_t rqst, void *data);
#endif

#define GX1_GEODE_DEFMODE     5
#define GX1_NUM_TV_MODES      4

#ifdef MODULE
void gx1_cleanup_module(void);
#endif

static char con2fb_map[MAX_NR_CONSOLES];
typedef struct _MemOffset
{
   unsigned short xres;
   unsigned short yres;
   unsigned short bpp;
   unsigned long CBOffset;
   unsigned short CBPitch;
   unsigned short CBSize;
   unsigned long CurOffset;
   unsigned long OffScreenOffset;

}
MemOffset_t;

/*
 * predefined memory address for compression and cursor offsets
 * if COMPRESSION enabled.
 */

static MemOffset_t GeodeMemOffset[] = {
   /* CRT and TFT modes */

   {640, 480, 8, 640, 1024, 272, 0x78000, 0x78100},
   {640, 480, 16, 1280, 2048, 272, 0x610, 0xF0000},
   {640, 480, 32, 2560, 4096, 272, 0xB10, 0x1E0000},
   {800, 600, 8, 800, 1024, 224, 0x96000, 0x96100},
   {800, 600, 16, 1600, 2048, 272, 0x12C000, 0x12C100},
   {800, 600, 32, 3200, 4096, 272, 0xD90, 0x258000},
   {1024, 768, 8, 0xC0000, 272, 272, 0xF3000, 0xF3100},
   {1024, 768, 16, 0x180000, 272, 272, 0x1B3000, 0x1B3100},
   {1024, 768, 32, 0x300000, 272, 272, 0x303000, 0x303100},
   {1152, 864, 8, 1152, 2048, 272, 0x590, 0x1B0000},
   {1152, 864, 16, 2304, 4096, 272, 0xA10, 0x360000},
   {1280, 1024, 8, 1280, 2048, 272, 0x610, 0x200000},
   {1280, 1024, 16, 2560, 4096, 272, 0xB10, 0x400000},

#ifdef CONFIG_FB_NSC_TV
   /* PAL TV modes */

   {704, 576, 16, 1408, 2048, 272, 0x690, 0x120000},
   {720, 576, 16, 1440, 2048, 272, 0x6B0, 0x120000},
   {768, 576, 16, 1536, 2048, 256, 0x700, 0x120000},

   /* NTSC TV modes */

   {704, 480, 16, 1408, 2048, 272, 0x690, 0xF0000},
   {720, 480, 16, 1440, 2048, 272, 0x6B0, 0xF0000}
#endif
};

#define NUM_TOTAL_OFFSET (arraysize(GeodeMemOffset))

/* Predefined Video Modes */

static struct {

   const char *name;
   unsigned int vesa;
   unsigned int Display_Type;
   struct fb_var_screeninfo var;

} gx1_geodefb_predefined[] = {

	{ "640x480-8", 0x101, Display_CRT | Display_FP,

		{ 	640, 480, 640, 480, 0, 0, 8, 0,{0, 8, 0}, 
			{0, 8, 0}, {0, 8, 0}, {0, 0, 0}, 0,
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	}, 

	{ "800x600-8", 0x103, Display_CRT | Display_FP,

		{ 	800, 600, 800, 600, 0, 0, 8, 0, {0, 8, 0}, 
			{0, 8, 0}, {0, 8, 0}, {0, 0, 0}, 0, 
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
		}

	}, 

	{ "1024x768-8", 0x105, Display_CRT | Display_FP, 

     
		{ 	1024, 768, 1024, 768, 0, 0, 8, 0, {0, 8, 0},
			{0, 8, 0}, {0, 8, 0}, {0, 0, 0}, 0, 
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	},

	{ "1152x864-8", 0x190, Display_CRT | Display_FP, 
     
		{	1152, 864, 1152, 864, 0, 0, 8, 0, {0, 8, 0}, 
			{0, 8, 0}, {0, 8, 0}, {0, 0, 0}, 0,
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	}, 

	{ "1280x1024-8", 0x107, Display_CRT | Display_FP, 
     
		{	1280, 1024, 1280, 1024, 0, 0, 8, 0, {0, 8, 0}, 
			{0, 8, 0}, {0, 8, 0}, {0, 0, 0}, 0,
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	}, 

	
	{ "640x480-16", 0x111, Display_CRT | Display_FP
#ifdef CONFIG_FB_NSC_TV
            | Display_TV | Display_TV_NTSC | Display_TV_Square_Pixels
#endif
		,
      
		{	640, 480, 640, 480, 0, 0, 16, 0, {11, 5, 0},
			{5, 6, 0}, {0, 5, 0}, {0, 0, 0}, 0,
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	 },

	{ "800x600-16", 0x114, Display_CRT | Display_FP, 

		{	800, 600, 800, 600, 0, 0, 16, 0,{11, 5, 0}, 
			{5, 6, 0}, {0, 5, 0}, {0, 0, 0}, 0, 
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	},

	{ "1024x768-16", 0x117, Display_CRT | Display_FP, 
   
		{	1024, 768, 1024, 768, 0, 0, 16, 0, {11, 5, 0}, 
			{5, 6, 0}, {0, 5, 0}, {0, 0, 0}, 0, 
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	}, 

    { "1152x864-16", 0x192, Display_CRT | Display_FP, 

		{	1152, 864, 1152, 864, 0, 0, 16, 0,{11, 5, 0},
			{5, 6, 0}, {0, 5, 0}, {0, 0, 0}, 0, 
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	}, 

    { "1280x1024-16", 0x11A, Display_CRT | Display_FP, 

		{	1280, 1024, 1280, 1024, 0, 0, 16, 0,{11, 5, 0},
			{5, 6, 0}, {0, 5, 0}, {0, 0, 0}, 0, 
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 0, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	}, 

#ifdef CONFIG_FB_NSC_TV
         /* TV Modes to the end of the list */

   {                                    /* 720 x 576 (PAL) */

	  "PAL-720x576", 0, Display_TV | Display_TV_PAL |
           Display_TV_No_Scaling,

		{	720, 576, 720, 576, 0, 0, 16, 0,{11, 5, 0},
			{5, 6, 0}, {0, 5, 0}, {0, 0, 0}, 0, 
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 50, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	}, 


	{	/* 768 x 576 (PAL) */

      "PAL-768x576", 0, Display_TV | Display_TV_PAL |
          Display_TV_Square_Pixels,

		{	768, 576, 768, 576, 0, 0, 16, 0, {11, 5, 0},
			{5, 6, 0}, {0, 5, 0}, {0, 0, 0}, 0, 
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 50, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
   }
	},
	{	/* 720 x 480 (NTSC) */
 
	  "NTSC-720x480", 0, Display_TV | Display_TV_NTSC | 
	  Display_TV_No_Scaling, 

		{	720, 480, 720, 480, 0, 0, 16, 0,{11, 5, 0}, 
			{5, 6, 0}, {0, 5, 0}, {0, 0, 0}, 0,
			FB_ACTIVATE_NOW, -1, -1, FB_ACCELF_TEXT, 60, 
               0, 0, 0, 0, 0, 0,
               FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
			FB_VMODE_NONINTERLACED
		}
   }
#endif
};

#define GX1_NUM_TOTAL_MODES    (arraysize(gx1_geodefb_predefined))

extern unsigned int GetVideoMemSize(void);
static int gx1_geodefb_set_cmap(struct fb_cmap *cmap, int kspc, int con,
                                struct fb_info *info);



static int gx1_geodefb_ioctl(struct inode *inode,struct file *file, u_int cmd,
				u_long arg,int con,struct fb_info *info);
/* Interface used by the world */

/* The geodefb_setup function set the display mode */
int gx1_geodefb_setup(char *);

/* Interface to the low level console driver */

/*
 * The gx1_geodefb_init function intialiazes the Geode frame buffer
 * driver
 */
int gx1_geodefb_init(void);

/* Text console acceleration */

#ifdef FBCON_HAS_CFB8
static struct display_switch gx1_fbcon_geode8;
#endif

#ifdef FBCON_HAS_CFB16
static struct display_switch gx1_fbcon_geode16;
#endif
/* Hardware Specific Routines */
static void gx1_Geode_detect(void);

static int gx1_Geode_encode_fix(struct fb_fix_screeninfo *fix,
                                const void *par, struct fb_info_gen *info);

static int gx1_Geode_get_fix(struct fb_fix_screeninfo *fix, int con,
                             struct fb_info *info);

static int gx1_do_fb_set_var(struct fb_var_screeninfo *var, int isactive,
                             struct fb_info_gen *info);

static int gx1_Geode_set_var(struct fb_var_screeninfo *var, int con,
                             struct fb_info *info);

static int gx1_Geode_decode_var(const struct fb_var_screeninfo *var,
                                void *par, struct fb_info_gen *info);

static int gx1_Geode_get_var(struct fb_var_screeninfo *var, int con,
                             struct fb_info *info);

static int gx1_Geode_encode_var(struct fb_var_screeninfo *var,
                                const void *par, struct fb_info_gen *info);

static void gx1_Geode_get_par(void *par, struct fb_info_gen *info);

static void gx1_Geode_set_par(const void *par, struct fb_info_gen *info);

static int gx1_Geode_getcolreg(unsigned regno, unsigned *red, unsigned *green,
                               unsigned *blue, unsigned *transp,
                               struct fb_info *info);

static int gx1_Geode_setcolreg(unsigned regno, unsigned red, unsigned green,
                               unsigned blue, unsigned transp,
                               struct fb_info *info);

static int gx1_Geode_blank(int blank_mode, struct fb_info_gen *info);

static void gx1_Geode_set_disp(const void *par, struct display *disp,
                               struct fb_info_gen *info);
#ifdef CONFIG_FB_NSC_DDC
static void gx1_get_ddc_mode(void);
#endif

#ifdef CONFIG_FB_NSC_TV
void gx1_set_tv_pixclock(struct fb_var_screeninfo *var);
#endif
/* Internal routines */
int gx1_get_video_mode(const char *name);
void gx1_set_mode(int xres, int yres, int bpp, int hz, int DType);
void gx1_rectfill(int sx, int sy, int width, int height, unsigned long bg);
void gx1_check_video_memory(void);

/*------------- Hardware specific routines --------------------*/

/*-----------------------------------------------------------------
 * gx1_get_stride 
 *
 * Description: This function gets the stride (line lenth) of the screen.
 *       
 *  parameters:
 *        xres: X-resolution of the display screen.
 *        yres: Y-resolution of the display screen.
 *         bpp: This gives the bits-per-pixel of the diplay screen. 
 *      return: It returns the screen line lenth.
 *---------------------------------------------------------------*/
int
gx1_get_stride(int xres, int yres, int bpp)
{
   int line_delta = xres * (bpp >> 3);

   if (line_delta > 2048)
      line_delta = 4096;
   else if (line_delta > 1024)
      line_delta = 2048;
   else
      line_delta = 1024;
   return (line_delta);
}

#ifdef CONFIG_FB_NSC_TV
/*-----------------------------------------------------------------
 * is_tv_supported 
 *
 * Description: This function checks whether tv is enabled or not.
 *  parameters: none
 *      return: If tv is enabled it returns 0 else it returns 1.
 *---------------------------------------------------------------*/
inline int
is_tv_supported(void)
{
   if (is_tv_set == -1) {
      is_tv_set = (gfx_detect_video() == GFX_VID_SC1200) ? 1 : 0;
   }
   return (is_tv_set);
}

void
gx1_set_tv_pixclock(struct fb_var_screeninfo *var)
{
   int pll;

   TVStandardType tvformat = TV_STANDARD_NTSC;

   if (var->reserved[0] & Display_TV_PAL)
      tvformat = TV_STANDARD_PAL;

   gfx_get_tv_display_mode_frequency(var->xres, var->yres, tvformat, &pll);

   DPRINTK("%d, %x\n", tvformat, pll);

   var->pixclock = 1E6 / fixedformat2frequency(pll);
   /* strip off all CRT msaks */
   var->reserved[0] &= Display_TVMASK;
}
#endif

#ifdef CONFIG_FB_NSC_TV
int
gx1_set_if_tv_mode(struct fb_var_screeninfo *var)
{
   int i, ret = -1;

   for (i = 0; i < GX1_NUM_TOTAL_MODES; i++) {
      if ((gx1_geodefb_predefined[i].Display_Type & Display_TV) &&
          (var->xres == gx1_geodefb_predefined[i].var.xres) &&
          (var->yres == gx1_geodefb_predefined[i].var.yres) &&
          (var->bits_per_pixel ==
           gx1_geodefb_predefined[i].var.bits_per_pixel)) {
         DPRINTK("Found %d\n", i);

         var->reserved[0] = gx1_geodefb_predefined[i].Display_Type;
         if (var->reserved[0] & Display_TVMASK) {
            /* check if user wants to set a different vfreq */
            if ((var->reserved[0] & Display_TV_NTSC)) {
               vfreq = 60;
            }
            if ((var->reserved[0] & Display_TV_PAL)) {
               vfreq = 50;
            }
            gx1_set_tv_pixclock(var);
            ret = i;
         } else {
            /* knock off TV flags */
            var->reserved[0] &= ~Display_TVMASK;
         }
         break;
      }
   }
   return ret;
}

/**
 *      nsc_set_all_vcs - set all virtual consoles to match
 *      @fbidx: frame buffer index (e.g. fb0, fb1, ...)
 *      @fb: frame buffer ops structure
 *      @var: frame buffer screen structure to set
 *      @info: frame buffer info structure
 *
 *      Set all virtual consoles to match screen info set in @var
 *      for device @info.
 *
 *      Returns negative errno on error, or zero on success.
 *
 */

static int
nsc_set_all_vcs(int fbidx, struct fb_ops *fb,
                struct fb_var_screeninfo *var, struct fb_info *info)
{
   int unit, err;

   var->activate |= FB_ACTIVATE_TEST;
   err = fb->fb_set_var(var, NSC_PROC_CONSOLE(info), info);
   var->activate &= ~FB_ACTIVATE_TEST;
   if (err)
      return err;
   for (unit = 0; unit < MAX_NR_CONSOLES; unit++)
      if (fb_display[unit].conp && con2fb_map[unit] == fbidx)
         fb->fb_set_var(var, unit, info);
   return 0;
}

#endif
/*-----------------------------------------------------------------
 * gx1_Geode_detect
 *
 * Description: This routine will search the display mode , which 
 *                              can be supported or not. If the video mode
 *                              was specified on the command line, it will override
 *                              the default mode. This is the first function called
 *                              in a module to setup the default mode.
 *      parameters:     none
 *              return: none
 *-----------------------------------------------------------------*/
static void
gx1_Geode_detect(void)
{
   int i;
   int xres, yres, bpp, hz, ret;

   DPRINTK("geodefb_default_valid : %d vfreq : %d\n", geodefb_default_valid,
           vfreq);
   if((vfreq == 56 && ((geodefb_default.xres != 800) || 
                       (geodefb_default.yres != 600))) ||
       (vfreq == 70 && ((geodefb_default.xres != 1024) || 
                        (geodefb_default.yres != 768)))) {
     vfreq = 60;
     geodefb_default.xres = 640; 
     geodefb_default.yres = 480; 
     geodefb_default.bits_per_pixel = 16; 
   }
   if (!geodefb_default_valid) {
      int i, found = -1;

      gfx_get_display_mode(&xres, &yres, &bpp, &hz);

      DPRINTK("gfx_get_display_mode %d %d %d %d\n", xres, yres, bpp, hz);

      /*
       * search if we can support this mode, else set default 
       */

      for (i = 0; i < GX1_NUM_TOTAL_MODES; i++) {
         if ((xres == gx1_geodefb_predefined[i].var.xres) &&
             (yres == gx1_geodefb_predefined[i].var.yres) &&
             (bpp == gx1_geodefb_predefined[i].var.bits_per_pixel)) {
#ifdef CONFIG_FB_NSC_TV
            if (is_tv_supported()) {
               if (gx1_geodefb_predefined[i].Display_Type & Display_TV) {
                  found = i;
                  break;
               }
            } else
#endif
            {
               found = i;
               break;
            }
         }
      }
      if (found >= 0) {
         geodefb_default = gx1_geodefb_predefined[found].var;
         geodefb_default.reserved[0] =
               gx1_geodefb_predefined[found].Display_Type;
         if ((geodefb_default.xres == 640)
             && (geodefb_default.bits_per_pixel == 16) && ((vfreq == 72)
                                                           || (vfreq == 75)
                                                           || (vfreq ==
                                                               85))) {
            geodefb_default.reserved[0] = Display_CRT | Display_FP;
         }
#ifdef CONFIG_FB_NSC_TV
         /* is this a TV resolution */
         if (geodefb_default.reserved[0] & Display_TV) {
            gx1_set_tv_pixclock(&geodefb_default);
         } else                         /* just crt mode */
#endif
         {
            set_crt_pixclock(&geodefb_default);
         }
      } else {
         int ret = -1;

         /* Default mode */
         DPRINTK("%s", "Default mode\n");
         geodefb_default = gx1_geodefb_predefined[GX1_GEODE_DEFMODE].var;
         geodefb_default.reserved[0] =
               gx1_geodefb_predefined[GX1_GEODE_DEFMODE].Display_Type;
         if ((geodefb_default.xres == 640)
             && (geodefb_default.bits_per_pixel == 16) && ((vfreq == 72)
                                                           || (vfreq == 75)
                                                           || (vfreq ==
                                                               85))) {
            geodefb_default.reserved[0] = Display_CRT | Display_FP;
         }
#ifdef CONFIG_FB_NSC_TV
         if (is_tv_supported() && (geodefb_default.reserved[0] & Display_TV)) {
            /* check if TV mode */
            ret = gx1_set_if_tv_mode(&geodefb_default);
         }
#endif
         /* if not a TV mode then set to CRT */
         if (ret == -1)
            set_crt_pixclock(&geodefb_default);

         DPRINTK("frequency %dx%d %d %x %x\n", geodefb_default.xres,
                 geodefb_default.yres,
                 geodefb_default.bits_per_pixel,
                 vfreq, geodefb_default.pixclock);

      }
   } else {                             /* do a sanity check on the mode accepted */

      xres = geodefb_default.xres;
      yres = geodefb_default.yres;
      bpp = geodefb_default.bits_per_pixel;
      hz = vfreq;

      /* check if the requested mode can be supported */
      ret = gfx_is_display_mode_supported(xres, yres, bpp, hz);

      DPRINTK("gfx_is_display_mode_supported %d\n", ret);
     
      if (ret >= 0) {                   /* CRT mode */
         ret = -1;
         if ((geodefb_default.xres == 640) &&
             (geodefb_default.bits_per_pixel == 16) &&
             ((vfreq == 72) || (vfreq == 75) || (vfreq == 85))) {
            geodefb_default.reserved[0] = Display_CRT | Display_FP;
         }
#ifdef CONFIG_FB_NSC_TV
         /* Check if a TV mode, if platform supports it */
         if (is_tv_supported() && (geodefb_default.reserved[0] & Display_TV)) {
            ret = gx1_set_if_tv_mode(&geodefb_default);
         }
#endif
         /* if not a TV mode then set to CRT */
         if (ret == -1) {
            set_crt_pixclock(&geodefb_default);
         }
#ifdef CONFIG_FB_NSC_FP
         /* Enable FP if CRT selected */
         if ((geodefb_default.reserved[0] == Display_CRT))
            geodefb_default.reserved[0] = Display_FP;
#endif
      }
#ifdef CONFIG_FB_NSC_TV
      else if (is_tv_supported()) {
         /* if Not a CRT mode try if its TV mode, 
          * check if running on SC1200 with TV enabled/support
          * based system 
          */
         ret = gx1_set_if_tv_mode(&geodefb_default);
         printk(" return  value after tv_mode test = %d", ret);

         if (ret < 0) {

            printk("Cannot support requested TV mode %dx%d %d @%dHz\n",
                   xres, yres, bpp, hz);
            /* default crt mode */

            geodefb_default = gx1_geodefb_predefined[GX1_GEODE_DEFMODE].var;
            geodefb_default.reserved[0] =
                  gx1_geodefb_predefined[GX1_GEODE_DEFMODE].Display_Type;
            ret = gx1_set_if_tv_mode(&geodefb_default);
            /* if not a TV mode then set to CRT */
            if (ret == -1) {
               set_crt_pixclock(&geodefb_default);
            }

            DPRINTK("%d %d %d, %X\n",
                    geodefb_default.xres, geodefb_default.yres,
                    geodefb_default.bits_per_pixel,
                    geodefb_default.reserved[0]);

         }
      }
#endif
      else {                            /* wrong mode, fall back */
         /* default crt mode */
         vfreq = 60; 
         geodefb_default = gx1_geodefb_predefined[GX1_GEODE_DEFMODE].var;
         geodefb_default.reserved[0] =
               gx1_geodefb_predefined[GX1_GEODE_DEFMODE].Display_Type;
         DPRINTK("%d %d %d, %X\n",
                 geodefb_default.xres, geodefb_default.yres,
                 geodefb_default.bits_per_pixel, geodefb_default.reserved[0]);
         set_crt_pixclock(&geodefb_default);
      }
   }

   for (i = 0; i < 256; i++) {
      Geode_palette[i].red = i;
      Geode_palette[i].green = i;
      Geode_palette[i].blue = i;
   }

   memset((char *)GeodeMem_virt, 0, GXFBMemSize);
   DPRINTK("%d %d %d, %d %X\n",
           geodefb_default.xres, geodefb_default.yres,
           geodefb_default.bits_per_pixel,
           geodefb_default.pixclock, geodefb_default.reserved[0]);

#ifdef CONFIG_FB_NSC_TV
   geodefb_default.reserved[0] &= ~Display_TVO;
   /* If TV Overscan setup by user */
   if (defaultTVO.TV_Overscan_On) {
      if (((signed int)defaultTVO.TVOx >= 0) &&
          ((signed int)defaultTVO.TVOy >= 0) &&
          ((signed int)defaultTVO.TVOw > 0) &&
          ((signed int)defaultTVO.TVOh > 0)) {
         if (((defaultTVO.TVOx + defaultTVO.TVOw) <= geodefb_default.xres) &&
             ((defaultTVO.TVOy + defaultTVO.TVOh) <= geodefb_default.yres)) {
            geodefb_default.reserved[0] |= Display_TVO;
            geodefb_default.reserved[1] = defaultTVO.TVOx;
            geodefb_default.reserved[2] = defaultTVO.TVOw;
            geodefb_default.reserved[3] = defaultTVO.TVOy;
            geodefb_default.reserved[4] = defaultTVO.TVOh;
         }
      }
   }
#endif
}

/*-----------------------------------------------------------------
 * gx1_Geode_encode_fix
 *
 * Description: This function should fill in the fix structure
 *                              based on the values in the `par' structure.
 *  parameters:
 *                 fix: It is a pointer to the frame buffer fixed screen 
 *                              information.    
 *      fb_par: It is a void pointer to the frame buffer parameters. 
 *        info: This info pointer holds the generic frame buffer 
 *                              device information.
 *      return: It returns 0 on success. 
 *              On failure it returns erro value.  
 *------------------------------------------------------------------*/
static int
gx1_Geode_encode_fix(struct fb_fix_screeninfo *fix, const void *fb_par,
                     struct fb_info_gen *info)
{
   /* parameters are used to get the display mode */
   struct geodefb_par *par = (struct geodefb_par *)fb_par;
   int stride = gx1_get_stride(par->xres_virtual,
                               par->yres_virtual, par->bpp);

   DPRINTK("%dx%d %d\n", par->xres_virtual, par->yres_virtual, par->bpp);
   assert(par != NULL);
   assert(fix != NULL);
   assert(info != NULL);

   if ((info == NULL) || (fix == NULL) || (par == NULL))
      return (-EINVAL);

   memset(fix, 0, sizeof(struct fb_fix_screeninfo));
   strcpy(fix->id, geodefb_name);

#ifdef CONFIG_FB_NSC_TV
   if ((par->flags & Display_TVO) && (is_tv_supported())) {
      fix->smem_start = (GeodeMem_phys +
                         (stride * par->TVOy) +
                         (par->TVOx << ((par->bpp >> 3) - 1)));
      fix->smem_len = GXFBMemSize -
            ((stride * par->TVOy) + (par->TVOx << ((par->bpp >> 3) - 1)));
   } else
#endif
   {
      fix->smem_start = GeodeMem_phys;
      fix->smem_len = GXFBMemSize;
   }
   fix->mmio_start = GeodeRegs_phys;
   fix->mmio_len = GeodeRegs_phys_size;

   fix->type = FB_TYPE_PACKED_PIXELS;
   fix->type_aux = 0;
   if (par->bpp == 8)
      fix->visual = FB_VISUAL_PSEUDOCOLOR;
   else
      fix->visual = FB_VISUAL_TRUECOLOR;

   fix->xpanstep = 0;
   fix->ypanstep = 0;
   fix->ywrapstep = 0;
   fix->line_length = stride;
   fix->accel = FB_ACCEL_NSC_GEODE;

   return (0);
}

/*-----------------------------------------------------------------
 * gx1_Geode_get_var
 *
 * Description: This function get the user defined part of the display.
 *              This function will call the geode  specific functions 
 *              Geode_get_par, Geode_encode_var to get the user defined
 *              part of the display.  
 *
 *  parameters:
 *         var: It is a pointer to the frame buffer variable screen information.     
 *         con: It specifies the virtual console number.
 *        info: It is a pointer to the frame buffer information structure.
 *
 *      return: On success it returns the 0 value. 
 *
 *------------------------------------------------------------------*/

static int
gx1_Geode_get_var(struct fb_var_screeninfo *var, int con,
                  struct fb_info *info)
{
   struct fb_info_gen *info2 = (struct fb_info_gen *)info;
   struct geodefb_par par;

   if (con == -1) {
      gx1_Geode_get_par(&par, info2);
      gx1_Geode_encode_var(var, &par, info2);
   } else
      *var = fb_display[con].var;
   return 0;
}

/*-----------------------------------------------------------------
 * gx1_Geode_get_fix
 *
 * Description:   This function gets the fixed part of the display.
 *                This function calls the Geode_get_par, Geode_decode_var and 
 *                Geode_encode_fix to get the fixed screen information.
 *
 *  parameters:
 *         fix: Pointer to fb_fix_screeninfo which has to be updated
 *              with the data for console indicated by con.
 *         con: The console whose data has to be returned in fix structure.
 *        Info: Pointer to fb_info.
 *
 *      return: It returns 0 on success. 
 *              On failure it returns erro value.  
 *------------------------------------------------------------------*/
static int
gx1_Geode_get_fix(struct fb_fix_screeninfo *fix, int con,
                  struct fb_info *info)
{
   struct fb_info_gen *info2 = (struct fb_info_gen *)info;
   struct geodefb_par par;

   if (con == -1) {
      gx1_Geode_get_par(&par, info2);
   } else {
      int err;

      if ((err = gx1_Geode_decode_var(&fb_display[con].var, &par, info2)))
         return err;
   }
   memset(fix, 0, sizeof(struct fb_fix_screeninfo));
   return gx1_Geode_encode_fix(fix, &par, info2);
}

/*-----------------------------------------------------------------
 * gx1_do_fb_set_var
 *
 * Description: This function change the video mode of the frame buffer device. 
 *              This function make use of the  function Geode_decode_var, 
 *              Geode_set_par and Geode_encode_var.
 *
 *  parameters:
 *         var: Pointer to fb_var_screeninfo which has the parameters
 *              to set the hardware. 
 *    isactive: It specifies the current console number.
 *        info: This pointer points to the generic frame buffer
 *              device information structure.  
 *
 *      return: On success it returns the 0 value. 
 *              On failure it returns the error value.
 *
 *------------------------------------------------------------------*/

static int
gx1_do_fb_set_var(struct fb_var_screeninfo *var, int isactive,
                  struct fb_info_gen *info)
{
   int err, activate;
   struct geodefb_par par;

   DPRINTK("%d\n", isactive);

   if ((err = gx1_Geode_decode_var(var, &par, info))) {
      return err;
   }
   activate = var->activate;
   if (((var->activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_NOW) && isactive)
      gx1_Geode_set_par(&par, info);

   gx1_Geode_encode_var(var, &par, info);
   var->activate = activate;
   return 0;
}

/*-----------------------------------------------------------------
 * gx1_Geode_set_var
 *
 * Description: This function set the user defined part of the display. 
 *              This function change the video mode using fbgen_do_set_var
 *              function.It calls the function fbgen_set_disp to set 
 *              the frame buffer display switch. It allocates and installs
 *              the frame buffer color map using  the functions 
 *              fb_alloc_cmap, and fbgen_install_cmap functions.
 *
 *  parameters:
 *         var: It is a pointer to the frame buffer variable screen information. 
 *         con: It specifies the virtual console number.
 *        info: It is a pointer to the frame buffer information structure.
 *
 *      return: On success it returns the 0 value.
 *              On failure it returns the error value.
 *------------------------------------------------------------------*/
static int
gx1_Geode_set_var(struct fb_var_screeninfo *var, int con,
                  struct fb_info *info)
{
   struct fb_info_gen *info2 = (struct fb_info_gen *)info;
   int err, oldxres, oldyres, oldbpp, oldxres_virtual,
         oldyres_virtual, oldyoffset, oldpixclock;

   DPRINTK("%d\n", con);

   if ((err = gx1_do_fb_set_var(var, con == currcon, info2)))
      return err;
   if ((var->activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_NOW) {
      DPRINTK("%s", "- activate\n");
      oldxres = fb_display[con].var.xres;
      oldyres = fb_display[con].var.yres;
      oldxres_virtual = fb_display[con].var.xres_virtual;
      oldyres_virtual = fb_display[con].var.yres_virtual;
      oldbpp = fb_display[con].var.bits_per_pixel;
      oldyoffset = fb_display[con].var.yoffset;
      oldpixclock = fb_display[con].var.pixclock;
      fb_display[con].var = *var;
      if ((oldxres != var->xres || oldyres != var->yres
           || oldxres_virtual != var->xres_virtual
           || oldyres_virtual != var->yres_virtual
           || oldbpp != var->bits_per_pixel
           || oldpixclock != var->pixclock || oldyoffset != var->yoffset)
#ifdef CONFIG_FB_NSC_TV
          || (var->reserved[0] & Display_TVO)   /* if TVO, force mode change */
#endif
            ) {
         fbgen_set_disp(con, info2);
         if (info->changevar)
            (*info->changevar) (con);
         fb_alloc_cmap(&fb_display[con].cmap, 0, 0);
         fbgen_install_cmap(con, info2);
      }
   }
   var->activate = 0;
   return 0;
}

#ifdef CONFIG_FB_NSC_TV
static int
gx1_Geode_decode_tv_var(const struct fb_var_screeninfo *var,
                        struct geodefb_par *par, struct fb_info_gen *info)
{
   /* check if this is a tv mode, if platform supports it */

   int i, found = -1;
   int xres = 0, yres = 0, bpp = 0;
   int tvo = 0;

   DPRINTK("1 %dx%d  %dx%d %d (%d, %d)\n",
           var->xres,
           var->yres,
           var->xres_virtual,
           var->yres_virtual,
           var->bits_per_pixel, var->pixclock, par->refresh);
   DPRINTK("1 %dx%d %d %d %dx%d %d\n",
           var->xres_virtual,
           var->yres_virtual,
           var->left_margin,
           var->upper_margin,
           var->right_margin, var->lower_margin, var->hsync_len);
   DPRINTK("1 %X %dx%d  %dx%d\n",
           var->reserved[0],
           var->reserved[1],
           var->reserved[3], var->reserved[2], var->reserved[4]);

   tvo = (var->reserved[0] & Display_TVO);
   for (i = 0; i < GX1_NUM_TOTAL_MODES; i++) {
      if ((gx1_geodefb_predefined[i].Display_Type & Display_TVMASK) &&
          (var->xres_virtual == gx1_geodefb_predefined[i].var.xres) &&
          (var->yres_virtual == gx1_geodefb_predefined[i].var.yres) &&
          (var->bits_per_pixel ==
           gx1_geodefb_predefined[i].var.bits_per_pixel)) {
         found = i;
         xres = gx1_geodefb_predefined[found].var.xres;
         yres = gx1_geodefb_predefined[found].var.yres;
         bpp = gx1_geodefb_predefined[found].var.bits_per_pixel;
         break;
      }
   }                                    /* for end */
   DPRINTK("Found = %d\n", found);
   if (found >= 0) {
      int pll;
      TVStandardType tvformat = TV_STANDARD_NTSC;

      ((struct fb_var_screeninfo *)var)->reserved[0] =
            gx1_geodefb_predefined[found].Display_Type;
      /* knock off non TV flags */
      ((struct fb_var_screeninfo *)var)->reserved[0] &= Display_TVMASK;
      DPRINTK("TVO %d %d %d %d\n",
              var->reserved[1],
              var->reserved[3], var->reserved[2], var->reserved[4]);

      if ((tvo) &&
          (var->reserved[1] >= 0) &&
          (var->reserved[3] >= 0) &&
          (var->reserved[2] > 0) &&
          (var->reserved[4] > 0) &&
          ((var->reserved[1] + var->reserved[2]) <=
           var->xres_virtual) &&
          ((var->reserved[3] + var->reserved[4]) <= var->yres_virtual)) {
         ((struct fb_var_screeninfo *)var)->reserved[0] |= Display_TVO;
      }

      par->refresh = 60;

      if (var->reserved[0] & Display_TV_PAL) {
         tvformat = TV_STANDARD_PAL;
         par->refresh = 50;
      }

      gfx_get_tv_display_mode_frequency(xres, yres, tvformat, &pll);

      ((struct fb_var_screeninfo *)var)->pixclock =
            1E6 / fixedformat2frequency(pll);
   }
   DPRINTK("2 %dx%d  %dx%d %d (%d, %d)\n",
           var->xres,
           var->yres,
           var->xres_virtual,
           var->yres_virtual,
           var->bits_per_pixel, var->pixclock, par->refresh);
   DPRINTK("2 %dx%d %d %d %dx%d %d\n",
           var->xres_virtual,
           var->yres_virtual,
           var->left_margin,
           var->upper_margin,
           var->right_margin, var->lower_margin, var->hsync_len);
   DPRINTK("2 %X %dx%d  %dx%d\n",
           var->reserved[0],
           var->reserved[1],
           var->reserved[2], var->reserved[3], var->reserved[4]);

   return found;
}
#endif

/*-----------------------------------------------------------------
 * gx1_Geode_decode_var
 *
 * Description: This routine get the video parameters out
 *                              of `var'. If a value doesn't fit, round it up, 
 *                              if it's too big, return -EINVAL.
 *  parameters:
 *         var: Pointer to fb_var_screeninfo which has the 
 *                              parameters to set the hardware. 
 *      fb_par: This is void pointer which points to the frame 
 *                              buffer parameters.  
 *        info: This pointer points to the generic frame buffer
 *                              device structure.  
 *      return: On decoding the variable screen information 
 *                              it returns 0.  
 *-----------------------------------------------------------------*/
static int
gx1_Geode_decode_var(const struct fb_var_screeninfo *var,
                     void *fb_par, struct fb_info_gen *info)
{
   int i, ret = -1, hz;
   int stride;
   unsigned long Required_FBsize;
   struct geodefb_par *par = (struct geodefb_par *)fb_par;
   int pll16_16;

   assert(par != NULL);
   assert(var != NULL);
   assert(info != NULL);

   if ((info == NULL) || (var == NULL) || (par == NULL))
      return (-EINVAL);

   DPRINTK("1 %dx%d  %dx%d %d (%d, %d, %d)\n",
           var->xres,
           var->yres,
           var->xres_virtual,
           var->yres_virtual,
           var->bits_per_pixel,
           var->pixclock, vfreq, info->info.var.pixclock);
   DPRINTK("1 %dx%d %d %d %dx%d %d\n",
           var->xres_virtual,
           var->yres_virtual,
           var->left_margin,
           var->upper_margin,
           var->right_margin, var->lower_margin, var->hsync_len);
   DPRINTK("1 %X %dx%d  %dx%d\n",
           var->reserved[0],
           var->reserved[1],
           var->reserved[3], var->reserved[2], var->reserved[4]);
   /* find the mode in the predefined list */
#ifdef CONFIG_FB_NSC_TV
   if ((var->reserved[0] & Display_TVO) && is_tv_supported()) {
      for (i = 0; i < GX1_NUM_TOTAL_MODES; i++) {
         if ((var->xres_virtual == gx1_geodefb_predefined[i].var.xres) &&
             (var->yres_virtual == gx1_geodefb_predefined[i].var.yres) &&
             (var->bits_per_pixel ==
              gx1_geodefb_predefined[i].var.bits_per_pixel)) {
            ret = i;
            break;
         }
      }                                 /* for end */
      DPRINTK("22 ret = %d\n", ret);
   } else
#endif
   {
      for (i = 0; i < GX1_NUM_TOTAL_MODES; i++) {
         if ((var->xres == gx1_geodefb_predefined[i].var.xres) &&
             (var->yres == gx1_geodefb_predefined[i].var.yres) &&
             (var->bits_per_pixel ==
              gx1_geodefb_predefined[i].var.bits_per_pixel)) {
            ret = i;
            break;
         }
      }                                 /* for end */
      DPRINTK("23 ret = %d\n", ret);
   }
   if (ret >= 0) {
      stride = gx1_get_stride(var->xres, var->yres, var->bits_per_pixel);
      Required_FBsize = (unsigned long)(stride * var->yres);

      DPRINTK("dtype %X\n", gx1_geodefb_predefined[ret].Display_Type);
      DPRINTK("\ndecode_var stride = %d\n, Required_FBsize = %d\n, \
                                GXFBMemSize = %d compression = %d \n", stride, (unsigned int)Required_FBsize / 1024, (unsigned int)GXFBMemSize / 1024, compression);

      if (Required_FBsize > GXFBMemSize) {
         printk("Cannot support this mode on this platform\n");
         printk("Required FB size beyond the available FB memory\n");
         return (-EINVAL);
      } else {
         if (Required_FBsize == GXFBMemSize) {
            compression = 0;
         }
      }
      DPRINTK("25 ret = %d\n", ret);
   }
   if (ret >= 0) {
      DPRINTK("26 ret = %d\n", ret);
#ifdef CONFIG_FB_NSC_TV
      if ((gx1_geodefb_predefined[ret].Display_Type & Display_TVMASK) &&
          is_tv_supported()) {
         DPRINTK("Can Support TV\n");
         DPRINTK("27 ret = %d\n", ret);

         /* fbset doesn't sent the reserved[0], tvoset is taken care */
         if (!var->reserved[0]) {
            ((struct fb_var_screeninfo *)var)->reserved[0] =
                  gx1_geodefb_predefined[ret].Display_Type;
         }

         /* check if this mode suports CRT also */
         if (var->reserved[0] & ~Display_TVMASK) {
            DPRINTK("Has CRT Support\n");
            DPRINTK("28 ret = %d\n", ret);
            if (does_pixclock_match(var, par, &hz, &pll16_16) <
                calc_err(var->pixclock)) {
               ((struct fb_var_screeninfo *)var)->pixclock =
                     1E6 / fixedformat2frequency(pll16_16);
               par->refresh = hz;
               DPRINTK("pixclock %X hz %d, %X\n",
                       var->pixclock, par->refresh, var->reserved[0]);
               /* the issue is only with NTSC which runs @ 60Hz.
                * Check the freq for non 60 and knock of TV support.
                */
               if (par->refresh != 60) {
                  /* strip off all TV msaks */
                  ((struct fb_var_screeninfo *)var)->reserved[0] &=
                        ~Display_TVMASK;
               }
               ret = 1;                 /* set it, since we got a valid resolution */

               DPRINTK("29 ret = %d\n", ret);
            } else
               ret = -1;
         } else {
            DPRINTK("No CRT Support\n");
            ret = gx1_Geode_decode_tv_var(var, par, info);
            DPRINTK("30 ret = %d\n", ret);
         }
      } else                            /* CRT mode */
#endif

      if (ret >= 0) {
         DPRINTK("31 ret = %d\n", ret);
         if (var->pixclock == 0) {
            hz = vfreq;                 /* default */
            gfx_get_frequency_from_refreshrate(var->xres,
                                               var->yres,
                                               var->bits_per_pixel,
                                               hz, &pll16_16);
            DPRINTK("32 ret = %d\n", ret);
            if (pll16_16)               /* Zero if not found */
               ((struct fb_var_screeninfo *)var)->pixclock =
                     (1E6 / fixedformat2frequency(pll16_16)) + .5;
            else {
               DPRINTK("33 ret = %d\n", ret);
               return (-EINVAL);
            }
         } else {
            int diff16_16;

            diff16_16 = does_pixclock_match(var, par, &hz, &pll16_16);
            DPRINTK("34 ret = %d\n", ret);

            if (diff16_16 > calc_err(var->pixclock)) {  /* error term */
               /* check if the resolution change is requested, 
                * * if not requested then the pll needs to be recalculated 
                */
               int oxres, oyres, obpp, ohz;

               gfx_get_display_mode(&oxres, &oyres, &obpp, &ohz);

               if ((var->xres == Sxres) &&
                   (var->yres == Syres) && (var->bits_per_pixel == Sbpp)) {
                  printk("%dx%d @%d Bpp Mode with %X pixclock not supported\n", var->xres, var->yres, var->bits_per_pixel, var->pixclock);
                  return (-EINVAL);
               }
               ((struct fb_var_screeninfo *)var)->pixclock =
                     1E6 / fixedformat2frequency(pll16_16);
               DPRINTK("35 ret = %d\n", ret);
            }
         }
         par->refresh = hz;
         DPRINTK
               ("var->xres = %d\n, var->yxres = %d var->bits_per_pixel =  %d  , par->refresh = %d",
                var->xres, var->yres, var->bits_per_pixel, par->refresh);

         /* check if the requested mode can be supported */
         ret = gfx_is_display_mode_supported(var->xres, var->yres,
                                             var->bits_per_pixel,
                                             par->refresh);

         DPRINTK("gfx_is_display_mode_supported %d\n", ret);
         if (ret >= 0) {                /* CRT mode */

            DPRINTK("36 ret = %d\n", ret);

            /* Should be a CRT mode */
            ((struct fb_var_screeninfo *)var)->reserved[0] = Display_CRT;

#ifdef CONFIG_FB_NSC_FP
            DPRINTK("flatpanel : %d\n", flatpanel);

            /* Enable FP if CRT selected */
            if ((var->reserved[0] == Display_CRT) && flatpanel) {
               ((struct fb_var_screeninfo *)var)->reserved[0] = Display_FP;
            }
#endif

            DPRINTK("Display_Type: %d\n", var->reserved[0]);
         } else {
            printk("Cannot support this mode on this platform\n");
            DPRINTK("37 ret = %d\n", ret);

            return (-EINVAL);
         }
      }
   } else {
      printk("Cannot support this mode on this platform\n");
      DPRINTK("38 ret = %d\n", ret);
      return (-EINVAL);
   }
   par->pixclock = var->pixclock;

   par->flags = var->reserved[0];
   par->TVOx = var->reserved[1];
   par->TVOw = var->reserved[2];
   par->TVOy = var->reserved[3];
   par->TVOh = var->reserved[4];

#ifdef CONFIG_FB_NSC_TV
   if ((par->flags & Display_TVO) && is_tv_supported()) {
      par->xres = par->TVOw;
      par->yres = par->TVOh;
   } else
#endif
   {
      par->xres = var->xres;
      par->yres = var->yres;
   }
   par->xres_virtual = var->xres_virtual;
   par->yres_virtual = var->yres_virtual;
   par->bpp = var->bits_per_pixel;

   if (var->accel_flags & FB_ACCELF_TEXT)
      par->accel = FB_ACCELF_TEXT;
   else
      par->accel = 0;

   DPRINTK("2 var %dx%d  %dx%d %d (%d, %d)\n",
           var->xres,
           var->yres,
           var->xres_virtual,
           var->yres_virtual,
           var->bits_per_pixel, var->pixclock, par->refresh);
   DPRINTK("2 var %dx%d %d %d %dx%d %d\n",
           var->xres_virtual,
           var->yres_virtual,
           var->left_margin,
           var->upper_margin,
           var->right_margin, var->lower_margin, var->hsync_len);

   DPRINTK("2 %X %dx%d  %dx%d\n",
           var->reserved[0],
           var->reserved[1],
           var->reserved[2], var->reserved[3], var->reserved[4]);

   DPRINTK("1 par %dx%d  %dx%d %d %d %d %d %d %d %d %d %X\n",
           par->xres,
           par->yres,
           par->xres_virtual,
           par->yres_virtual,
           par->bpp,
           par->refresh,
           par->pixclock,
           par->accel,
           par->TVOx, par->TVOw, par->TVOy, par->TVOh, par->flags);

   return (0);
}

/*-----------------------------------------------------------------
 * gx1_Geode_encode_var
 * 
 * Description: Fill the `var' structure based on the values in `par' 
 *                              and other values read out of the hardware.
 *  parameters:
 *         var: Pointer to fb_var_screeninfo which has the parameters
 *                              to set the hardware. 
 *       b_par: This is void pointer which points to the frame buffer
 *                              parameters.  
 *        info: This pointer points to the generic frame buffer
 *                              device structure.  
 *      return: On encoding the variable screen information 
 *                              it returns 0.  
 *-------------------------------------------------------------------*/
static int
gx1_Geode_encode_var(struct fb_var_screeninfo *var,
                     const void *fb_par, struct fb_info_gen *info)
{
   struct geodefb_par *par = (struct geodefb_par *)fb_par;

   assert(par != NULL);
   assert(var != NULL);
   assert(info != NULL);

   if ((info == NULL) || (var == NULL) || (par == NULL))
      return (-EINVAL);

   DPRINTK("1 par %dx%d  %dx%d %d %d %d %d %d %d %d %d %X, %d\n",
           par->xres,
           par->yres,
           par->xres_virtual,
           par->yres_virtual,
           par->bpp,
           par->refresh,
           par->pixclock,
           par->accel,
           par->TVOx, par->TVOw, par->TVOy, par->TVOh, par->flags, vfreq);

   memset(var, 0, sizeof(struct fb_var_screeninfo));
   var->xres = par->xres;
   var->yres = par->yres;
#ifdef CONFIG_FB_NSC_TV
   if ((par->flags & Display_TVO) && is_tv_supported()) {
      var->xres_virtual = par->xres_virtual;
      var->yres_virtual = par->yres_virtual;
   } else
#endif
   {
      var->xres_virtual = par->xres;
      var->yres_virtual = par->yres;
   }
   var->reserved[0] = par->flags;
   var->reserved[1] = par->TVOx;
   var->reserved[2] = par->TVOw;
   var->reserved[3] = par->TVOy;
   var->reserved[4] = par->TVOh;

   var->hsync_len = gfx_get_hsync_end() - gfx_get_hsync_start();
   var->vsync_len = gfx_get_vsync_end() - gfx_get_vsync_start();
   var->right_margin = var->left_margin =
         (gfx_get_htotal() - gfx_get_hactive() - var->hsync_len) / 2;
   var->upper_margin = var->lower_margin =
         (gfx_get_vtotal() - gfx_get_vactive() - var->vsync_len) / 2;

   var->xoffset = 0;
   var->yoffset = 0;
   var->bits_per_pixel = par->bpp;
   var->grayscale = 0;

   switch (var->bits_per_pixel) {
#ifdef FBCON_HAS_CFB8

   case 8:
      /* CLUT */
      var->red.offset = 0;
      var->red.length = 6;
      var->red.msb_right = 0;
      var->blue = var->green = var->red;
      break;
#endif

#ifdef FBCON_HAS_CFB16

   case 16:
      /* RGB 565 */
      var->red.offset = 11;
      var->red.length = 5;
      var->green.offset = 5;
      var->green.length = 6;
      var->blue.offset = 0;
      var->blue.length = 5;
      var->transp.offset = 0;
      var->transp.length = 0;
      break;
#endif

   }

   var->red.msb_right = 0;
   var->green.msb_right = 0;
   var->blue.msb_right = 0;
   var->transp.msb_right = 0;
   var->nonstd = 0;
   var->activate = 0;
   var->height = -1;
   var->width = -1;
   var->accel_flags = (par->accel && ((par->bpp == 8) ||
                                      (par->bpp == 16))) ? FB_ACCELF_TEXT : 0;

   var->vmode = FB_VMODE_NONINTERLACED;

   /* Clock is a 16:16 fixed point number */
   DPRINTK("%X %d\n", par->pixclock, par->refresh);
   var->pixclock = par->pixclock;
   var->sync = 0;
   DPRINTK("res%dx%d, virt%dx%d, offset%dx%d,wh%dx%d\n",
           var->xres,
           var->yres,
           var->xres_virtual,
           var->yres_virtual,
           var->xoffset, var->yoffset, var->width, var->height);

   DPRINTK("sync=%dx%d, margin=%dx%d, bpp=%d, accel=%x, sync=%x,gray=%x\n",
           var->hsync_len,
           var->vsync_len,
           var->right_margin,
           var->upper_margin,
           var->bits_per_pixel, var->accel_flags, var->sync, var->grayscale);

   DPRINTK("RGB %d %d,%d %d, %d %d, %d %d, msb%d %d %d %d, nostd%d,activ%d\n",
           var->red.offset,
           var->red.length,
           var->green.offset,
           var->green.length,
           var->blue.offset,
           var->blue.length,
           var->transp.offset,
           var->transp.length,
           var->red.msb_right,
           var->green.msb_right,
           var->blue.msb_right,
           var->transp.msb_right, var->nonstd, var->activate);

   DPRINTK("reserved %X %dx%d  %dx%d\n",
           var->reserved[0],
           var->reserved[1],
           var->reserved[2], var->reserved[3], var->reserved[4]);

   return (0);
}

/*-----------------------------------------------------------------
 * gx1_Geode_get_par
 * 
 * Description: This function fill the Geode hardware's 'par' 
 *                              structure.
 * parameters:
 *     fb_par:  This is void pointer which points to the frame buffer
 *                              parameters.  
 *       info:  This pointer points to the generic frame buffer
 *                              device structure.  
 *     return:  none.
 *------------------------------------------------------------------*/
static void
gx1_Geode_get_par(void *fb_par, struct fb_info_gen *info)
{
   struct geodefb_par *par = (struct geodefb_par *)fb_par;

   assert(par != NULL);
   assert(info != NULL);

   if ((info == NULL) || (par == NULL))
      return;

   DPRINTK("%s", "\n");

   if (current_par_valid)
      *par = current_par;
   else
      gx1_Geode_decode_var(&geodefb_default, par, info);
}

/*-----------------------------------------------------------------
 * gx1_Geode_set_par
 *
 * Description: This function set the Geode frame buffer device
 *              according to par structure and  
 *              display mode based on display type ( TV or CRT or PANEL).
 * parameters:
 *     fb_par:  This is void pointer which points to the frame
 *                              buffer parameters.  
 *       info:  This pointer points to the generic frame buffer
 *                              device structure.  
 *     return:  none.
 *----------------------------------------------------------------*/
static void
gx1_Geode_set_par(const void *fb_par, struct fb_info_gen *info)
{
   struct geodefb_par *par = (struct geodefb_par *)fb_par;

   assert(par != NULL);
   assert(info != NULL);

   if ((info == NULL) || (par == NULL))
      return;

   DPRINTK("1 par %dx%d  %dx%d %d %d %d %d %d %d %d, %X\n",
           par->xres,
           par->yres,
           par->xres_virtual,
           par->yres_virtual,
           par->bpp,
           par->refresh,
           par->accel,
           par->TVOx, par->TVOw, par->TVOy, par->TVOh, par->flags);

   current_par = *par;
   current_par_valid = 1;

   /* Save the current mode, for pwrmgmt */
#ifdef CONFIG_FB_NSC_TV
   if ((par->flags & Display_TVO) && is_tv_supported()) {
      Sxres = par->xres_virtual;
      Syres = par->yres_virtual;
   } else
#endif
   {
      Sxres = par->xres;
      Syres = par->yres;
   }

   Sbpp = par->bpp;
   Shz = par->refresh;
   SDisplayType = par->flags;

   gx1_set_mode(Sxres, Syres, Sbpp, Shz, SDisplayType);
}

/*-----------------------------------------------------------------
 * gx1_set_mode 
 *
 * Description: This function sets the display mode.
 *       
 *  parameters:
 *        xres: X-resolution of the display screen.
 *        yres: Y-resolution of the display screen.
 *         bpp: Bits-per-pixel of the diplay screen. 
 *          hz: Vertical frequency of the display screen.
 *      return: none.
 *---------------------------------------------------------------*/
void
gx1_set_mode(int xres, int yres, int bpp, int hz, int DType)
{
   int i;
   int pxres, pyres, pbpp, phz, pstride, pheight;
   static int first_prev_mode = 0;      /* to avoid VESA previous mode */

   DPRINTK("%d %d %d %d, %X\n", xres, yres, bpp, hz, DType);

   /* Get the current operating display mode */
   gfx_get_display_mode(&pxres, &pyres, &pbpp, &phz);

   DPRINTK("prev mode %d %d %d %d\n", pxres, pyres, pbpp, phz);

   /* get the stride in bytes for this resolution */
   pstride = gx1_get_stride(pxres, pyres, pbpp);
   pheight = GXFBMemSize / pstride;

   /* get the stride in pixels for this resolution */
   pstride >>= (pbpp >> 3) - 1;

   /* disable compression, since we clear the compression area also */
   if (compression)
      gfx_set_compression_enable(0);

   /* clean the frame buffer */

   if (first_prev_mode) {               /* to avoid VESA previous mode */
      gx1_rectfill(0, 0, pstride, pheight, 0x0);
   } else {
      first_prev_mode = 1;
   }
   DPRINTK(" first_prev_mode = %d \n", first_prev_mode);
   /* Set the display depth */
   gfx_set_display_bpp(bpp);

#ifdef CONFIG_FB_NSC_TV
   if (DType & Display_TV) {
      gfx_set_tv_display(xres, yres);

      /* TV stuff goes here */
      if (DType & Display_TV_NTSC) {
         if (DType & Display_TV_Square_Pixels)
            gfx_set_tv_format(TV_STANDARD_NTSC, GFX_ON_TV_SQUARE_PIXELS);
         else
            gfx_set_tv_format(TV_STANDARD_NTSC, GFX_ON_TV_NO_SCALING);

      } else if (DType & Display_TV_PAL) {
         if (DType & Display_TV_Square_Pixels)
            gfx_set_tv_format(TV_STANDARD_PAL, GFX_ON_TV_SQUARE_PIXELS);
         else
            gfx_set_tv_format(TV_STANDARD_PAL, GFX_ON_TV_NO_SCALING);

      }
      gfx_set_tv_output(tv_output);
      gfx_set_tv_enable(1);
   } else
#endif
   {
#ifdef CONFIG_FB_NSC_TV
      /* disable TV if TV is being supported. CRT modes
       * don't go well with TV.
       */
      if (is_tv_supported())
         gfx_set_tv_enable(0);
#endif
      if (DType & Display_CRT) {
         gfx_set_display_mode(xres, yres, bpp, hz);
      }
#ifdef CONFIG_FB_NSC_FP
      else if (DType & Display_FP) {
         gfx_set_fixed_timings(FPBX, FPBY, xres, yres, bpp);
      }
#endif
   }

   gfx_set_crt_enable(CRT_ENABLE);

   if (compression) {
      int match = -1;

      /* find the index to our operating mode the offsets are located */

      for (i = 0; i < NUM_TOTAL_OFFSET; i++) {
         if ((xres == GeodeMemOffset[i].xres) &&
             (yres == GeodeMemOffset[i].yres) &&
             (bpp == GeodeMemOffset[i].bpp)) {
            match = i;
            break;
         }
      }
      DPRINTK("Found the Match @ %d\n", match);
      if (match >= 0) {
         /* set the compression parameters */
         gfx_set_compression_offset(GeodeMemOffset[i].CBOffset);
         gfx_set_compression_pitch(GeodeMemOffset[i].CBPitch);
         gfx_set_compression_size((GeodeMemOffset[i].CBSize - 16));

         /* enable compression buffer, all parameters already set */
         gfx_set_compression_enable(1);
      }
   }

}

/*-----------------------------------------------------------------
 * gx1_Geode_setcolreg
 *
 * Description: This function set a single color register. 
 *                              The values supplied have a 16 bit magnitude.
 *                              This function make the first 16 colors of the 
 *                              palette available to frame buffer console.
 * parameters:
 *      regno:  It is an integer which represents color register number.
 *        red:    
 *      green:
 *       blue:
 *     transp:  These red, green, blue, and transp are the inter values
 *                              which represents the color attributes for geode palette.
 *       info:  This is a pointer to the frame buffer information 
 *                              structure.  
 *     return:  Return != 0 for invalid regno.
 *----------------------------------------------------------------*/
static int
gx1_Geode_setcolreg(u_int regno, u_int red, u_int green, u_int blue,
                    u_int transp, struct fb_info *info)
{
   assert(info != NULL);
   if (((current_par.bpp == 8) && (regno > 255)) ||
       ((current_par.bpp != 8) && (regno > 15)))
      return (1);

   if (((current_par.bpp == 8) && (regno < 256)) ||
       ((current_par.bpp == 16) && (regno < 16))) {
      Geode_palette[regno].red = red >> 8;
      Geode_palette[regno].green = green >> 8;
      Geode_palette[regno].blue = blue >> 8;
   }

   switch (current_par.bpp) {
#ifdef FBCON_HAS_CFB8
   case 8:
      gfx_set_display_palette_entry(regno,
                                    (Geode_palette[regno].red << 16) |
                                    (Geode_palette[regno].green << 8) |
                                    Geode_palette[regno].blue);
      break;
#endif

#ifdef FBCON_HAS_CFB16
   case 16:
      fbcon_cmap.cfb16[regno] = ((red & 0xf800) |
                                 ((green & 0xfc00) >> 5) |
                                 ((blue & 0xf800) >> 11));
      break;
#endif

   }
   return (0);

}

/*-----------------------------------------------------------------
 * gx1_Geode_getcolreg
 *
 * Description: This function read  a single color register. 
 *                              and split it into colors/transparent.The return 
 *                              values must have 16 bit magnitude.
 *  parameters:
 *       regno: It is an integer which represents color register number.
 *         red:    
 *       green:
 *        blue:
 *      transp: These red, green, blue, and transp are the inter values
 *                              which represents the color attributes for geode palette.
 *        info: This is a pointer to the frame buffer information 
 *                              structure.  
 *      return: Return != 0 for invalid regno.
 *---------------------------------------------------------------*/
static int
gx1_Geode_getcolreg(u_int regno, u_int * red, u_int * green,
                    u_int * blue, u_int * transp, struct fb_info *info)
{
   int t;

   assert(red != NULL);
   assert(green != NULL);
   assert(blue != NULL);
   assert(transp != NULL);
   assert(info != NULL);
   if (regno > 255)
      return (1);
   if (((current_par.bpp == 8) && (regno < 256)) ||
       ((current_par.bpp == 16) && (regno < 16))) {
      t = Geode_palette[regno].red;
      *red = (t << 8) | t;
      t = Geode_palette[regno].green;
      *green = (t << 8) | t;
      t = Geode_palette[regno].blue;
      *blue = (t << 8) | t;
   }

   *transp = 0;
   return (0);
}

/*----------- Interfaces to hardware functions -------------- */

static struct fbgen_hwswitch gx1_Geode_hwswitch = {
   detect:gx1_Geode_detect,
   encode_fix:gx1_Geode_encode_fix,
   decode_var:gx1_Geode_decode_var,
   encode_var:gx1_Geode_encode_var,
   get_par:gx1_Geode_get_par,
   set_par:gx1_Geode_set_par,
   getcolreg:gx1_Geode_getcolreg,
   setcolreg:gx1_Geode_setcolreg,
   blank:gx1_Geode_blank,
   set_disp:gx1_Geode_set_disp
};

/*--------------- Generic routines ----------------------------*/

/*-----------------------------------------------------------------
 * gx1_Geode_set_disp
 *
 * Description: This function fill in a pointer to appropriate
 *                              low level text console operations for the video 
 *                              mode 'par' of the video hardware. If it don't 
 *                              have any appropriate operations, it will fill in a 
 *                              pointer to dummy operations, and there will be no 
 *                              text output.
 *  parameters:
 *      fb_par: This is a void pointer, it points to frame
 *                              buffer parameters.  
 *        disp: This is a pointer to the Geode frame buffer display 
 *                              structure.
 *        info: This is a pointer to the generic frame buffer 
 *                              information structure.  
 *      return: none.
 *----------------------------------------------------------------*/
static void
gx1_Geode_set_disp(const void *fb_par, struct display *disp,
                   struct fb_info_gen *info)
{
   assert(fb_par != NULL);
   assert(disp != NULL);
   assert(info != NULL);

   DPRINTK("%X %dx%d  %dx%d\n",
           disp->var.reserved[0],
           disp->var.reserved[1],
           disp->var.reserved[2],
           disp->var.reserved[3], disp->var.reserved[4]);

   if ((info == NULL) || (disp == NULL) || (fb_par == NULL))
      return;
#ifdef CONFIG_FB_NSC_TV
   if ((disp->var.reserved[0] & Display_TVO) && (is_tv_supported())) {
      int stride = gx1_get_stride(disp->var.xres_virtual,
                                  disp->var.yres_virtual,
                                  disp->var.bits_per_pixel);

      disp->screen_base = (char *)(GeodeMem_virt +
                                   (stride * disp->var.reserved[3]) +
                                   (disp->var.
                                    reserved[1] <<
                                    ((disp->var.bits_per_pixel >> 3) - 1)));
   } else
#endif
   {
      disp->screen_base = (char *)GeodeMem_virt;
   }

   DPRINTK("0x%8X\n", (int)disp->screen_base);

   switch (disp->var.bits_per_pixel) {
#ifdef FBCON_HAS_CFB8
   case 8:
      if (disp->var.accel_flags & FB_ACCELF_TEXT) {
         disp->dispsw = &gx1_fbcon_geode8;
      } else
         disp->dispsw = &fbcon_cfb8;
      break;
#endif

#ifdef FBCON_HAS_CFB16

   case 16:
      if (disp->var.accel_flags & FB_ACCELF_TEXT) {
         disp->dispsw = &gx1_fbcon_geode16;
      } else
         disp->dispsw = &fbcon_cfb16;
      disp->dispsw_data = &fbcon_cmap.cfb16;
      break;
#endif

   default:
      disp->dispsw = &fbcon_dummy;
      break;
   }
   disp->scrollmode = SCROLL_YREDRAW;
}

/*-------------------------------------------------------------
 * geodefb_set_cmap
 *
 * Description: This function will set the color map to the
 *                              console frame buffer device.
 *  parameters:
 *        cmap: This is a fb_cmap pointer, which has the colormap
 *                              to be set for the console indicated by 'con'.
 *        kspc: It is an integer for kernel/user mode switch. 
 *         con: The console for which the colormap has to be set.
 *        info: It is a pointer to the frame buffer information
 *                              structure.
 *      return: It returns 0 on success otherwise it returns error value. 
 *------------------------------------------------------------*/
static int
gx1_geodefb_set_cmap(struct fb_cmap *cmap, int kspc, int con,
                     struct fb_info *info)
{
   int err;
   int cmap_len = 0;

   assert(cmap != NULL);
   assert(info != NULL);
   if (info->var.bits_per_pixel == 8)
      cmap_len = 256;
   else
      cmap_len = 16;

   if (!fb_display[con].cmap.len) {
      /* no colormap allocated? */
      if ((err = fb_alloc_cmap(&fb_display[con].cmap, cmap_len, 0)))
         return (err);
   }

   /* current console? */

   if (con == currcon)
      return (fb_set_cmap(cmap, kspc, gx1_Geode_setcolreg, info));
   else
      fb_copy_cmap(cmap, &fb_display[con].cmap, kspc ? 0 : 1);
   return (0);
}

/*-----------------------------------------------------------------
 * gx1_geodefb_ioctl
 *
 * Description: This function provide an interface to the Geode frame
 *                              buffer specific input output control commands.
 *  parameters:
 *               inode: Pointer to "inode" structure which contains the device
 *                              reference.
 *        file: Pointer to a "file" structure. 
 *                 cmd: The IOCTL code which was issued by the application.
 *         arg: Pointer to the data which was passed to the IOCTL by
 *                              the application.
 *         con: The console from which the ioctl was issued.
 *        info: This is a pointer to the frame buffer information 
 *                              structure.  
 *      return: Returns 0 if the ioctl was handled successfully.
 *----------------------------------------------------------------*/
static int
gx1_geodefb_ioctl(struct inode *inode, struct file *file,
                  u_int cmd, u_long arg, int con, struct fb_info *info)
{
   struct fb_ops *fb = info->fbops;
   struct fb_var_screeninfo var;
   int i;

   switch (cmd) {
   case FB_DEBUG:
      {
         int option = -1, i;

         printk("FB_DEBUG IOCLT\n");
         i = copy_from_user(&option, (void *)arg, sizeof(int));
         printk("FB_DEBUG IOCLT %d %d\n", option, i);
         if (i)
            return -EFAULT;
         switch (option) {
         case 0:
            putcs_dbg ^= 1;
            return copy_to_user((void *)arg, &putcs_dbg,
                                sizeof(int)) ? -EFAULT : 0;
         }
      }
   case FB_SUPPORT:
      {
	unsigned int FBSupport=0;
	int i;
         i = copy_from_user(&FBSupport, (void *)arg, sizeof(unsigned int));
         if (i)
            return -EFAULT;
	 FBSupport = 0;
#ifdef CONFIG_FB_NSC_GAL
	 FBSupport |= NSC_GAL_SUPPORT;
#endif
#ifdef CONFIG_FB_NSC_TV
	 FBSupport |= NSC_TV_SUPPORT;
#endif
#ifdef CONFIG_FB_NSC_DDC
	 FBSupport |= NSC_DDC_SUPPORT;
#endif
#ifdef CONFIG_FB_NSC_FP
	 FBSupport |= NSC_FP_SUPPORT;
#endif
        i=copy_to_user((void *)arg, &FBSupport, sizeof(unsigned int));
        if (i)
           return -EFAULT;
        else
           return 0;
      }
   case FBIOGET_VSCREENINFO:
      if ((i = fb->fb_get_var(&var, NSC_PROC_CONSOLE(info), info)))
         return i;
      if (var.reserved[0] & Display_TVO) {
         var.xres = var.xres_virtual;
         var.yres = var.yres_virtual;
      }

      return copy_to_user((void *)arg, &var, sizeof(var)) ? -EFAULT : 0;
#ifdef CONFIG_FB_NSC_GAL
   case FBIOGAL_API:
      return geodefb_ioctl_ext(inode, file, cmd, arg, con, info);
#endif
#ifdef CONFIG_FB_NSC_TV
   case PUT_TVO:
      {
         int fbidx = GET_FB_IDX(inode->i_rdev);
         if (!is_tv_supported()) {
            printk("This is not a TV supported Platform\n");
            return (-EINVAL);
         }

         if (copy_from_user(&var, (void *)arg, sizeof(var)))
            return -EFAULT;
         if (var.activate & FB_ACTIVATE_ALL) {
            DPRINTK("PUT_TVO %d\n", fbidx);
            i = nsc_set_all_vcs(fbidx, fb, &var, info);
         } else {
            DPRINTK("%dx%d  %dx%d %d %d\n",
                    var.xres,
                    var.yres,
                    var.xres_virtual,
                    var.yres_virtual, var.bits_per_pixel, var.pixclock);
            DPRINTK("%dx%d %d %d %dx%d %d\n",
                    var.xres_virtual,
                    var.yres_virtual,
                    var.left_margin,
                    var.upper_margin,
                    var.right_margin, var.lower_margin, var.hsync_len);

            DPRINTK("%X %dx%d  %dx%d\n",
                    var.reserved[0],
                    var.reserved[1],
                    var.reserved[3], var.reserved[2], var.reserved[4]);

            /* if requesting for TVO */
            if (var.reserved[0] & Display_TVO) {
               DPRINTK("Display TVO\n");
               /* check if parameters valid */
               if (((signed int)var.reserved[1] >= 0) &&
                   ((signed int)var.reserved[3] >= 0) &&
                   ((signed int)var.reserved[2] > 0) &&
                   ((signed int)var.reserved[4] > 0) &&
                   ((var.reserved[1] + var.reserved[2]) <=
                    var.xres_virtual) &&
                   ((var.reserved[3] + var.reserved[4]) <=
                    var.yres_virtual)) {
                  struct fb_var_screeninfo cur_var;

                  i = fb->fb_get_var(&cur_var, con, info);

                  DPRINTK("get:PUT_TVO %dx%d  %dx%d %d (%d, %d)\n",
                          cur_var.xres,
                          cur_var.yres,
                          cur_var.xres_virtual,
                          cur_var.yres_virtual,
                          cur_var.bits_per_pixel, cur_var.pixclock, i);
                  DPRINTK("get:PUT_TVO %d %d %d %dx%d\n",
                          cur_var.reserved[0],
                          cur_var.reserved[1],
                          cur_var.reserved[3],
                          cur_var.reserved[2], cur_var.reserved[4]);

                  /* If currently TVO already active */
                  if (cur_var.reserved[0] & Display_TVO) {
                     var.xres = cur_var.xres_virtual;
                     var.yres = cur_var.yres_virtual;
                     var.reserved[0] |= Display_TVO;
                     /* knock off any non-tv flags */
                  } else {              /* TVO applied for first time on this console */

                     var.xres_virtual = var.xres;
                     var.yres_virtual = var.yres;
                     var.reserved[0] |= Display_TVO;
                  }
                  /* knock off any non-tv flags */
                  var.reserved[0] &= Display_TVMASK;
               } else {                 /* return error */

                  return -EFAULT;
               }
            } else {                    /* disabling TVO */

               DPRINTK("!Display_TVO\n");

               var.xres = var.xres_virtual;
               var.yres = var.yres_virtual;
               var.reserved[0] &= ~Display_TVO;
            }
         }
         i = gx1_Geode_set_var(&var, NSC_PROC_CONSOLE(info), info);
         if (i)
            return i;
         return 0;
      }
#endif
   }
   return (-EINVAL);
}

/* Geode frame buffer driver file operations */

static struct fb_ops gx1_geodefb_ops = {
   owner:THIS_MODULE,
   fb_get_fix:gx1_Geode_get_fix,
   fb_get_var:gx1_Geode_get_var,
   fb_set_var:gx1_Geode_set_var,
   fb_get_cmap:fbgen_get_cmap,
   fb_set_cmap:gx1_geodefb_set_cmap,
   fb_ioctl:gx1_geodefb_ioctl,
};

#ifdef CONFIG_FB_NSC_DDC
/*----------------------------------------------------------------
 * gx1_get_ddc_mode     
 *
 * Description: This function gets the display mode supported by 
 *              ddc. If no mode is supported by ddc the default mode
 *              set to  640x480-16 with ddc parameters.Default vfreq
 *              set to 60 hz.
 *  parameters: none. 
 *      return: none  
 *--------------------------------------------------------------*/

static void
gx1_get_ddc_mode(void)
{
   int i, mode_res, found = -1;

   unsigned long ddc_xres, ddc_yres, ddc_hz;
   int stride;
   unsigned long Required_FBsize;

   printk(KERN_INFO "Quering monitor for Display modes supported :\n");

   /*intialization */

   ddc_init();

   /* Find the best ddc available and use the Geode's mode
    * * values to program the timings for the chip
    */
   while (found == -1) {
      /* Get the next resolution supported by Monitor */

      nextFromList(&ddc_xres, &ddc_yres, &ddc_hz);

      printk("Do we Support %ld %ld %ld %d",
             ddc_xres, ddc_yres, ddc_hz, ddc_depth);

      if (ddc_xres == 0)                /* if end of the list */
         break;
      /* check if this mode can be supported by our hardware */
      mode_res = gfx_is_display_mode_supported(ddc_xres, ddc_yres,
                                               ddc_depth, ddc_hz);

      /* if not supported test next supported resolution */

      if (mode_res < 0) {
         printk(" - NO \n");
         continue;
      }
      stride = gx1_get_stride(ddc_xres, ddc_yres, ddc_depth);
      Required_FBsize = (stride * ddc_yres);
      if(Required_FBsize > GXFBMemSize) {
          printk("In sufficient memory for the supported mode\n");
          continue;
      }
      printk(" - YES \n");

      /*
       * search if we can support this mode from the predefined table, 
       * else set default 
       */

      for (i = 0; i < GX1_NUM_TOTAL_MODES; i++) {
         if ((ddc_xres == gx1_geodefb_predefined[i].var.xres) &&
             (ddc_yres == gx1_geodefb_predefined[i].var.yres) &&
             (ddc_depth == gx1_geodefb_predefined[i].var.bits_per_pixel) &&
             (gx1_geodefb_predefined[i].Display_Type & Display_CRT)) {
            /* set geodefb default mode */
            geodefb_default = gx1_geodefb_predefined[i].var;
            geodefb_default.reserved[0] =
                  gx1_geodefb_predefined[i].Display_Type;
            geodefb_default_valid = 1;
            vfreq = ddc_hz;
            found = i;
            break;
         }
      }
   }

   /*cleanup */

   ddc_cleanup();
}
#endif

/*----------------------------------------------------------------
 * gx1_get_video_mode
 *
 * Description: This function get the video mode specified by
 *                              the pointer name if it supports otherwise it set
 *                              the default video mode.
 *  parameters:
 *        name: It is a character pointer holds requested 
 *                              video mode by the user.       
 *      return: Return >= 0 if it succeeds the requested mode else 
 *                              it return 0 with default mode.
 *--------------------------------------------------------------*/
int
gx1_get_video_mode(const char *name)
{
   int i;

   DPRINTK("%s\n", name);
   for (i = 0; i < GX1_NUM_TOTAL_MODES; i++) {
         DPRINTK("%s == %s\n", name, gx1_geodefb_predefined[i].name);
      if (!strcmp(name, gx1_geodefb_predefined[i].name)) {
         DPRINTK("Found mode %s @ %d\n", name, i);
         geodefb_default = gx1_geodefb_predefined[i].var;
         geodefb_default.reserved[0] = gx1_geodefb_predefined[i].Display_Type;
         if(geodefb_default.reserved[0] & Display_TV) {
            if(!(((vfreq == 72) || (vfreq == 75) || (vfreq == 85)) && 
                 ((geodefb_default.xres == 640) && 
                 (geodefb_default.yres == 480))))
             vfreq = 60;
         }
         return (i);
      }
   }
   DPRINTK("Mode %s Not supported, Defaulting\n", name);
   /* set geodefb default mode */
   vfreq = 60; 
   geodefb_default = gx1_geodefb_predefined[GX1_GEODE_DEFMODE].var;
   geodefb_default.reserved[0] =
         gx1_geodefb_predefined[GX1_GEODE_DEFMODE].Display_Type;
   return (0);
}

void
gx1_check_video_memory(void)
{
   int stride;
   unsigned long Required_FBsize;

   stride = gx1_get_stride(geodefb_default.xres, geodefb_default.yres,
                           geodefb_default.bits_per_pixel);
   Required_FBsize = (stride * geodefb_default.yres);

   DPRINTK("\ndecode_var stride = %d\n, Required_FBsize = %d\n, \
                                GXFBMemSize = %d compression = %d \n", stride, (unsigned int)Required_FBsize / 1024, (unsigned int)GXFBMemSize / 1024, compression);

   if (Required_FBsize > GXFBMemSize) {
      printk("In sufficient memory for the requested mode\n");
      printk("Display mode defaulted to 640x480@16\n");
      /* set geodefb default mode */
      vfreq = 60;
      geodefb_default = gx1_geodefb_predefined[GX1_GEODE_DEFMODE].var;
      geodefb_default.reserved[0] =
            gx1_geodefb_predefined[GX1_GEODE_DEFMODE].Display_Type;
   } else {
      if (Required_FBsize == GXFBMemSize) {
         compression = 0;
      }
   }
}

static int
gx1_get_vesa_mode(void)
{
   /* fix VESA number */
   if (vesa != ~0) {
      int i;

      vesa &= 0x1DFF;
      /* mask out clearscreen, acceleration and so on */
      /* static settings */
      for (i = 0; i < GX1_NUM_TOTAL_MODES; i++) {
         if (vesa == gx1_geodefb_predefined[i].vesa) {
            geodefb_default = gx1_geodefb_predefined[i].var;
            geodefb_default.reserved[0] =
                  gx1_geodefb_predefined[i].Display_Type;
            return (i);
         }
      }
      DPRINTK("Mode %d Not supported, Defaulting\n", vesa);
      /* set geodefb default mode */
      vfreq = 60; 
      geodefb_default = gx1_geodefb_predefined[GX1_GEODE_DEFMODE].var;
      geodefb_default.reserved[0] =
         gx1_geodefb_predefined[GX1_GEODE_DEFMODE].Display_Type;
   }
   return (0);
}

#ifdef CONFIG_FB_NSC_TV
/*----------------------------------------------------------------
 * gx1_get_tv_overscan_geom
 *
 * Description: This function parse the tv-overscan parameters 
 *              X-offset, Y-offset, width, and height of the screen.
 *
 *  parameters:
 *           X: Pointer to the X-offset of the display screen.
 *           Y: Pointer to the Y-offset of the display screen.
 *           W: Pointer to the width of the display screen.
 *           H: Pointer to the height of the display screen.
 *
 *      return: none. 
 *--------------------------------------------------------------*/

int
gx1_get_tv_overscan_geom(char *opt, int *X, int *Y, int *W, int *H)
{
   char *tv_opt;
   int ok=1;
   if(opt == NULL)
      tv_opt = strtok(NULL, delimits);
   else
      tv_opt = opt;
   if(tv_opt) {
      *X = simple_strtol(tv_opt, NULL, 0);
   } else
      ok &= 0;
   tv_opt = strtok(NULL, delimits);
   if(tv_opt) {
      *Y = simple_strtol(tv_opt, NULL, 0);
   } else
      ok &= 0;
   tv_opt = strtok(NULL, delimits);
   if(tv_opt) {
      *W = simple_strtol(tv_opt, NULL, 0);
   } else
      ok &= 0;
   tv_opt = strtok(NULL, delimits);
   if(tv_opt) {
      *H = simple_strtol(tv_opt, NULL, 0);
   } else
      ok &= 0;
   return(ok);
}
#endif

/*---------------------------------------------------------------------
 * gx1_geodefb_setup
 *
 * Description: This function setup the display mode based 
 *              on the option specified by the user if it 
 *              supports, other wise it will set the default 
 *              display mode.
 *  parameters:
 *         options:     This is character pointer holds the display mode 
 *                              specified from user.
 *      return: Returns 0 if the display mode was set successfully. 
 *---------------------------------------------------------------------*/
int
gx1_geodefb_setup(char *options)
{

   char *this_opt;

   assert(options != NULL);
   if (!options || !*options)
      return (0);

   DPRINTK("%s\n", options);

   for (this_opt = strtok(options, delimits); this_opt;
        this_opt = strtok(NULL, delimits)) {
      DPRINTK("%s\n", this_opt);
      if (!strcmp(this_opt, "inverse")) {
         Geodefb_inverse = 1;
         fb_invert_cmaps();
      } else if (!strcmp(this_opt, "font")) {
         this_opt = strtok(NULL, delimits);
         strncpy(default_fontname, this_opt, sizeof(default_fontname));
      } else if (!strcmp(this_opt, "vesa")) {
         this_opt = strtok(NULL, delimits);
         vesa = simple_strtoul(this_opt, NULL, 0);
         DPRINTK("vesa: %x\n", vesa);
      } else if (!strcmp(this_opt, "vfreq")) {
         this_opt = strtok(NULL, delimits);
         vfreq = simple_strtoul(this_opt, NULL, 0);
         DPRINTK("vfreq: %x\n", vfreq);
         if((vfreq != 56) && (vfreq != 60) && 
            (vfreq != 70) && (vfreq != 72) && 
            (vfreq != 75) && (vfreq != 85)) {
            vfreq = 60;	
         }
      } else if (!strcmp(this_opt, "vmode")) {
         int i;
         this_opt = strtok(NULL, delimits);
         i = strlen(this_opt) + 1;
         DPRINTK("this_opt: %s %d\n", this_opt, i);
         if(i < 20) { /* can fit in the memory */
            strcpy(kvmode, this_opt);
            vmode = kvmode;
            DPRINTK("vmode: %s\n", vmode);
         }
      }
#ifdef CONFIG_FB_NSC_FP
      else if (!strcmp(this_opt, "flatpanel")) {
         this_opt = strtok(NULL, delimits);
         flatpanel = simple_strtoul(this_opt, NULL, 0);
      }
#endif
#ifdef CONFIG_FB_NSC_DDC
      else if (!strcmp(this_opt, "ddc")) {
         this_opt = strtok(NULL, delimits);
         ddc = simple_strtoul(this_opt, NULL, 0);
      } else if (!strcmp(this_opt, "ddc_depth")) {
         this_opt = strtok(NULL, delimits);
         ddc_depth = simple_strtoul(this_opt, NULL, 0);
         DPRINTK("ddc_depth: %d\n", ddc_depth);
      }
#endif
      else if (!strcmp(this_opt, "compression")) {
         this_opt = strtok(NULL, delimits);
         compression = simple_strtoul(this_opt, NULL, 0);
      }
#ifdef CONFIG_FB_NSC_TV
      else if (!strcmp(this_opt, "tv_output")) {
         this_opt = strtok(NULL, delimits);
         tv_output = simple_strtoul(this_opt, NULL, 0);
         if (tv_output < 1 || tv_output > 4)
            tv_output = 2;              /* default to S_VIDEO */
      } else if (!strcmp(this_opt, "TVO")) {
         defaultTVO.TV_Overscan_On = gx1_get_tv_overscan_geom(NULL, 
                                  &defaultTVO.TVOx, &defaultTVO.TVOy,
                                  &defaultTVO.TVOw, &defaultTVO.TVOh);
         DPRINTK("XYWH: %d %d %d %d, %d\n",
                 defaultTVO.TVOx,
                 defaultTVO.TVOy, 
                 defaultTVO.TVOw, 
                 defaultTVO.TVOh,
                 defaultTVO.TV_Overscan_On);
      }
#endif
   }

   DPRINTK("Geode Options: %x %d", vesa, vfreq);
   DPRINTK("default mode: xres=%d, yres=%d, bpp=%d, DType=%X\n",
           geodefb_default.xres, geodefb_default.yres,
           geodefb_default.bits_per_pixel, geodefb_default.reserved[0]);
   return (0);
}

/*----------------------------------------------------------------
 * gx1_geodefb_init
 *
 * Description: This is the entry point to the driver. This 
 *              function detects a National display controller 
 *              (CS5530, CS1200) and initializes the durango 
 *              interface by memory mapping the physical address 
 *              space required for the controller.
 *              A software device /dev/nscgal is created to receive 
 *              IOCTLs from GAL.Then it registers the  geode frame 
 *              buffer device.
 *  Parameters: none
 *              return: 0 returned if the geode frame buffer device 
 *                              is created < 0 returned on error.
 *---------------------------------------------------------------*/
int
gx1_geodefb_init(void)
{
   unsigned int address, address_size;
   DPRINTK("%X", cpu_version);

   if (!cpu_version) {
      /* NO GEODE PROCESSOR DETECTED */
      return (-ENXIO);
   }

   printk("National Geode FrameBuffer Driver Version %d.%d.%d\n",
          NSCFB_MAJOR, NSCFB_MINOR, NSCFB_SUB);

   if (vmode) {
      gx1_get_video_mode((const char *)vmode);
      geodefb_default_valid = 1;
   }
   if (vesa != ~0) {
      gx1_get_vesa_mode();
      geodefb_default_valid = 1;
   }
#ifdef MODULE
#ifdef CONFIG_FB_NSC_TV
   if (TVO) {
     char *opt = strtok(TVO, delimits);
   DPRINTK("%s", TVO);
     defaultTVO.TV_Overscan_On = gx1_get_tv_overscan_geom(opt, 
                             &defaultTVO.TVOx, &defaultTVO.TVOy,
                             &defaultTVO.TVOw, &defaultTVO.TVOh);
     DPRINTK("XYWH: %d %d %d %d, %d\n",
             defaultTVO.TVOx,
             defaultTVO.TVOy, 
             defaultTVO.TVOw, 
             defaultTVO.TVOh,
             defaultTVO.TV_Overscan_On);
   }
#endif
   if (font) {
      strncpy(default_fontname, font, sizeof(default_fontname));
   }
#endif /*  MODULE */

   if (compression != 0) {
      compression = 1;
   }
#ifdef CONFIG_FB_NSC_FP
   if (flatpanel != 1) {
      flatpanel = 0;
   }
#ifdef CONFIG_FB_NSC_TV
   else {
      is_tv_set = 0;
   }
#endif
#endif

#ifdef CONFIG_FB_NSC_DDC
   if ((ddc_depth != 8) && (ddc_depth != 16)) {
      ddc_depth = 8;
   }
#endif

#if PM
   {
      struct pm_dev *pmdev;

      pmdev = pm_register(PM_SYS_DEV, PM_SYS_VGA, gx1_geode_pm_callback);
   }
#endif
   /* DISPLAY DETECTION MESSAGE */
   printk("Geode Processor Detected (Type = %d,Version = %d.%d)\n",
          cpu_version & 0xFF, (cpu_version >> 8) & 0xFF,
          (cpu_version >> 16) & 0xFF);

#if defined(CONFIG_FB_NSC_DDC) && defined(CONFIG_FB_NSC_FP)
   if ((flatpanel == 1) && (ddc == 1))
      ddc = 0;
#endif

#ifdef CONFIG_FB_NSC_FP
   /* Panel is selected from bios, check if user turned on panel */
   DPRINTK("flatpanel : %d ", flatpanel);

   /* Check if bios supports panel. 
    * To be checked before SoftVGA put to sleep 
    */
   DPRINTK(" Before Quering From Bios: flatpanel : %d ", flatpanel);

   FPBP = Pnl_IsPanelEnabledInBIOS();

   DPRINTK(" Quered From Bios: flatpanel : %d ", FPBP);

   /* if OK, get the panel details from the Softvga */
   if (FPBP) {
      FPBX = FPBY = FPBD = FPBF = 0;
      Pnl_GetPanelInfoFromBIOS(&FPBX, &FPBY, &FPBD, &FPBF);
      DPRINTK("FPBX:%d FPBY:%d FPBD:%d FPBF:%d \n", FPBX, FPBY, FPBD, FPBF);
   }
   if (flatpanel) {
      flatpanel = FPBP;
   }
   DPRINTK("flatpanel %d\n", flatpanel);
   if (flatpanel == 1) {
      if ((geodefb_default.xres == 1152) &&  (geodefb_default.yres == 864)) {
          vfreq = 75;
      }
      else {
          vfreq = 60;
      }   
   }
#endif

   /* Initialize durango API here  SET DURANGO REGISTER 
    * POINTERS. The method of mapping from a physical address
    * to a linear address is operating system independent.
    * Set variables to linear address 
    */
   GeodeRegs_phys = address = gfx_get_cpu_register_base();
   GeodeRegs_phys_size = address_size = 0x10000;
   gfx_virt_regptr = (unsigned char *)ioremap(address, address_size);
   gfx_virt_spptr = gfx_virt_regptr;    /* scratch buffer */

   DPRINTK("Geodefb: Gfx Registers at 0x%x,mapped to 0x%p, size %dk\n",
           address, gfx_virt_regptr, address_size / 1024);

   address = gfx_get_vid_register_base();
   address_size = 0x1000;

   gfx_virt_vidptr = (unsigned char *)ioremap(address, address_size);
   DPRINTK("Geodefb: 5530 registers at 0x%x, mapped to 0x%p, size %dk\n",
           address, (unsigned int *)gfx_virt_vidptr, address_size / 1024);

   GeodeMem_phys = address = gfx_get_frame_buffer_base();
   GXFBMemSize = address_size = GetVideoMemSize();
   if (geodefb_default_valid) {
      gx1_check_video_memory();
   }

   DPRINTK("Video Memory Size at %dK,\n", GXFBMemSize / 1024);
   gfx_virt_fbptr = (unsigned char *)ioremap(address, address_size);

   GeodeMem_virt = (unsigned int)gfx_virt_fbptr;

   DPRINTK("framebuffer at 0x%x, mapped to 0x%p, size %dk\n",
           address, (unsigned char *)gfx_virt_fbptr, address_size / 1024);

   /* CHECK IF REGISTERS WERE MAPPED SUCCESSFULLY */
   if (!((unsigned)gfx_virt_regptr & (unsigned)gfx_virt_vidptr &
         (unsigned)gfx_virt_fbptr)) {

      DPRINTK("%s", "Could not map hardware registers.\n");

      return (-EBUSY);
   }
#ifdef CONFIG_FB_NSC_DDC
   /* try ddc only if only selected */
   if (ddc == 1)
      gx1_get_ddc_mode();
#endif

   /* disable softvga and never enable it again, to take complete
    * control 
    */
   gfx_disable_softvga();

   /* Tell softvga we are changing values and never call this with 0 */
   gfx_vga_mode_switch(1);

   strcpy(gen.info.modename, geodefb_name);
   gen.info.node = -1;
   gen.info.flags = FBINFO_FLAG_DEFAULT;
   gen.info.fbops = &gx1_geodefb_ops;
   gen.info.disp = &disp;
   gen.info.changevar = NULL;
   gen.info.switch_con = &fbgen_switch;
   gen.info.updatevar = &fbgen_update_var;
   gen.info.blank = &fbgen_blank;
   strcpy(gen.info.fontname, default_fontname);
   gen.parsize = sizeof(struct fb_info_gen);
   gen.fbhw = &gx1_Geode_hwswitch;

   gen.fbhw->detect();

   /* This should give a reasonable default video mode */

   gx1_Geode_get_var(&disp.var, -1, &gen.info);
   disp.var.activate = FB_ACTIVATE_NOW;
   gx1_do_fb_set_var(&disp.var, 1, &gen);

   fbgen_set_disp(-1, &gen);
   fbgen_install_cmap(0, &gen);
   if (register_framebuffer(&gen.info) < 0) {
      DPRINTK("%s", "register_framebuffer failed\n");
      return (-EINVAL);
   }

   DPRINTK("fb%d: %s frame buffer device, using %dK of video memory\n",
           GET_FB_IDX(gen.info.node), gen.info.modename, GXFBMemSize >> 10);
   return (0);
}

#if PM
static SAV_REG_GX1_GP SaveRegGp;
static SAV_REG_GX1_DP SaveRegDp;
static SAV_REG_5530 SaveReg5530;
#endif
/*----------------------------------------------------------------
 * gx1_Geode_blank
 *
 * Description: This function blank the screen if
 *                              blank_mode != 0, else unblank the screen.
 *  parameters:
 *       blank: This is an integer which specify the blank_mode.
 *      return: Return 0 if blanking succeeded, != 0 if
 *                              unblanking failed.
 *--------------------------------------------------------------*/
static int
gx1_Geode_blank(int blank, struct fb_info_gen *info)
{
   DPRINTK("%d\n", blank);

   /*
    * (Un)Blank the screen
    * 0 unblank, 1 blank, 2 no vsync, 3 no hsync, 4 off 
    */
   switch (blank) {
   case VESA_POWERDOWN:
   case VESA_VSYNC_SUSPEND:
      gfx_set_crt_enable(CRT_DISABLE);
#ifdef CONFIG_FB_NSC_FP
      if (flatpanel)
         Pnl_PowerDown();
#endif
#ifdef CONFIG_FB_NSC_TV
      if (is_tv_supported())
         gfx_set_tv_enable(0);
#endif
      break;

   case VESA_HSYNC_SUSPEND:
      gfx_set_crt_enable(CRT_STANDBY);
#ifdef CONFIG_FB_NSC_FP
      if (flatpanel)
         Pnl_PowerDown();
#endif
#ifdef CONFIG_FB_NSC_TV
      if (is_tv_supported()) {
         gfx_set_tv_enable(0);
      }
#endif
      break;

   case VESA_NO_BLANKING:
      DPRINTK(" SDis type: %X, TVMask: %X\n", SDisplayType, Display_TVMASK);
      gfx_set_crt_enable(CRT_ENABLE);
#ifdef CONFIG_FB_NSC_FP
      if (flatpanel)
         Pnl_PowerUp();
#endif
#ifdef CONFIG_FB_NSC_TV
      {
         int test_display;

         test_display = (SDisplayType & Display_TVMASK);
         DPRINTK("test_display : %d ", test_display);
      }
      if (is_tv_supported() && (SDisplayType & Display_TVMASK)) {
         gfx_set_tv_enable(1);
      }
#endif
      break;
   }
   return (0);
}

/*---------------------------------------------------------------
 * gx1_geode_pm_callback
 *
 * Description: Geode Power management request callback.
 *              The device request callback will be called before the
 *              device/system enters a suspend state (ACPI D1-D3) or
 *              or after the device/system resumes from suspend (ACPI D0).
 *              For PM_SUSPEND, the ACPI D-state being entered is passed
 *              as the "data" argument to the callback.  The device
 *              driver should save (PM_SUSPEND) or restore (PM_RESUME)
 *              device context when the request callback is called.
 *              Once a driver returns 0 (success) from a suspend
 *              request, it should not process any further requests or
 *              access the device hardware until a call to "pm_access" is 
 *              made.
 *
 *  Parameters:
 *         dev: M device previously returned from pm_register.
 *        rqst: request type.
 *        data: data, if any, associated with the request.
 *
 *     Returns: 0 if the request is successful
 *              EINVAL if the request is not supported
 *              EBUSY if the device is now busy and can not handle the
 *              request.
 *              ENOMEM if the device was unable to handle the request due
 *              to memory.
 *          
 *------------------------------------------------------------------------- */
#if PM
static int
gx1_geode_pm_callback(struct pm_dev *dev, pm_request_t rqst, void *data)
{
   switch (rqst) {
   case PM_SUSPEND:
      DPRINTK("%s", "suspend\n");
      gfx_set_crt_enable(CRT_DISABLE);
      GX1SaveRestoreGPReg(REG_SAVE, &SaveRegGp);
      GX1SaveRestoreDPReg(REG_SAVE, &SaveRegDp);
      GX1SaveRestore5530Reg(REG_SAVE, &SaveReg5530);
#ifdef CONFIG_FB_NSC_FP
      if (flatpanel)
         Pnl_PowerDown();
#endif
#ifdef CONFIG_FB_NSC_TV
      if (is_tv_supported())
         gfx_set_tv_enable(0);
#endif
      break;
   case PM_RESUME:
      DPRINTK("%s", "resume\n");
      GX1SaveRestoreDPReg(REG_RESTORE, &SaveRegDp);
      GX1SaveRestore5530Reg(REG_RESTORE, &SaveReg5530);
      GX1SaveRestoreGPReg(REG_RESTORE, &SaveRegGp);

      gx1_set_mode(Sxres, Syres, Sbpp, Shz, SDisplayType);

#ifdef CONFIG_FB_NSC_FP
      if (flatpanel)
         Pnl_PowerUp();
#endif
#ifdef CONFIG_FB_NSC_TV
      if (is_tv_supported())
         gfx_set_tv_enable(1);
#endif
      break;
   }
   return 0;
}
#endif

/*----------------------------------------------------------------
 * gx1_geodefb_bmove 
 *
 * Description: This function provide text console acceleration.
 *                              It accelerates screen to screen blts.
 * parameters:
 *          p:  It is a pointer to the display structure.
 *         sy:  It holds the integer value of font height.
 *         sx:  It holds the integer value of font width.
 *         dy:  It holds the integer value of font height.
 *         dx:  It holds the integer value of font width.
 *     height:  height of blt, in pixels.
 *      width:  width of blt, in pixels.
 *     return:  none
 *--------------------------------------------------------------*/
static void
gx1_geodefb_bmove(struct display *p, int sy, int sx, int dy,
                  int dx, int height, int width)
{
   assert(p != NULL);

   if (fontwidthlog(p)) {
      sx <<= fontwidthlog(p);
      dx <<= fontwidthlog(p);
      width <<= fontwidthlog(p);
   } else {
      sx *= fontwidth(p);
      dx *= fontwidth(p);
      width *= fontwidth(p);
   }

   sy *= fontheight(p);
   dy *= fontheight(p);
   height *= fontheight(p);
   gfx_set_solid_pattern(0);
   gfx_set_raster_operation(0xCC);

#ifdef CONFIG_FB_NSC_TV
   if ((p->var.reserved[0] & Display_TVO) && (is_tv_supported())) {
      gfx_screen_to_screen_blt(sx + p->var.reserved[1],
                               sy + p->var.reserved[3],
                               dx + p->var.reserved[1],
                               dy + p->var.reserved[3], width, height);
   } else
#endif
   {
      gfx_screen_to_screen_blt(sx, sy, dx, dy, width, height);
   }

   blit_maybe_busy = 1;

}

/*----------------------------------------------------------------
 * gx1_rectfill 
 *
 * Description: This function accelerates fill rectangles of the
 *                              display screen.
 * parameters:
 *         sx: screen X location
 *         sy: screen Y location
 *     height: height of rectangle, in pixels.
 *      width: width of rectangle, in pixels.
 *         bg: It gives the back ground color of the pattern.
 *     return: none.
 *-----------------------------------------------------------------*/

void
gx1_rectfill(int sx, int sy, int width, int height, unsigned long bg)
{
   gfx_set_solid_pattern(bg);
   gfx_set_raster_operation(0xF0);
   gfx_pattern_fill(sx, sy, width, height);
}

/*----------------------------------------------------------------
 * gx1_geodefb_clear 
 *
 * Description: This function accelerates fill rectangles of the
 *                              display screen.
 * parameters:
 *       conp:  It is a pointer to  virtual console data strucure.
 *          p:  It is a pointer to the display structure.
 *         sy:  It holds the integer value of font height.
 *         sx:  It holds the integer value of font width.
 *         dy:  It holds the integer value of font height.
 *         dx:  It holds the integer value of font width.
 *     height:  height of rectangle, in pixels.
 *      width:  width of rectangle, in pixels.
 *     return:  none
 *-----------------------------------------------------------------*/

static void
gx1_geodefb_clear(struct vc_data *conp, struct display *p,
                  int sy, int sx, int height, int width)
{
   unsigned long bg = 0;

   assert(p != NULL);
   assert(conp != NULL);
   if (fontwidthlog(p)) {
      sx <<= fontwidthlog(p);
      width <<= fontwidthlog(p);
   } else {
      sx *= fontwidth(p);
      width *= fontwidth(p);
   }

   sy *= fontheight(p);
   height *= fontheight(p);
   if (p->var.bits_per_pixel == 8) {
      bg = attr_bgcol_ec(p, conp);
   } else if (p->var.bits_per_pixel == 16) {
      bg = ((u16 *) p->dispsw_data)[attr_bgcol_ec(p, conp)];
   } else {
      bg = ((u32 *) p->dispsw_data)[attr_bgcol_ec(p, conp)];
   }

#ifdef CONFIG_FB_NSC_TV
   if ((p->var.reserved[0] & Display_TVO) && (is_tv_supported())) {
      gx1_rectfill(p->var.reserved[1] + sx,
                   p->var.reserved[3] + sy, width, height, bg);
   } else
#endif
   {
      gx1_rectfill(sx, sy, width, height, bg);
   }

   blit_maybe_busy = 1;
}

/*----------------------------------------------------------------
 * gx1_geodefb_putc
 *
 * Description: This functions calls the console frame buffer driver
 *                              putc function.          
 *
 * parameters:
 *       conp:  It is a pointer to the virtual console data structure. 
 *          p:  It is a pointer to the display data structure. 
 *          c:  It is an integer used as mask to get character source
 *                              pixel data.         
 *         yy:  It gives destination Y position 
 *                              (offset into frame buffer)
 *         xx:  It gives destination X position 
 *     return:  none
 *-----------------------------------------------------------------*/
static void
gx1_geodefb_putc(struct vc_data *conp, struct display *p,
                 int c, int yy, int xx)
{
   u32 fg, bg;
   int fw = fontwidth(p);
   int fh = fontheight(p);

//      int fbw=(fw+7)>>3;
   u8 *cdata = p->fontdata + (c & p->charmask) * fh;

   assert(p != NULL);
   assert(conp != NULL);

   if (blit_maybe_busy)
      Geode_WaitBusy();

   if (p->var.bits_per_pixel == 8) {
      fg = attr_fgcol(p, c);
      bg = attr_bgcol(p, c);
   } else if (p->var.bits_per_pixel == 16) {
      fg = ((u16 *) p->dispsw_data)[attr_fgcol(p, c)];
      bg = ((u16 *) p->dispsw_data)[attr_bgcol(p, c)];
   } else {
      fg = ((u32 *) p->dispsw_data)[attr_fgcol(p, c)];
      bg = ((u32 *) p->dispsw_data)[attr_bgcol(p, c)];
   }

   xx *= fw;
   yy *= fh;

#ifdef CONFIG_FB_NSC_TV
   if ((p->var.reserved[0] & Display_TVO) && (is_tv_supported())) {
      xx += p->var.reserved[1];
      yy += p->var.reserved[3];
   }
#endif

   gfx_set_solid_pattern(0);
   gfx_set_mono_source(bg, fg, 0);
   gfx_set_raster_operation(0xcc);
   gfx_text_blt(xx, yy, fw, fh, cdata);
//      gfx_mono_bitmap_to_screen_blt(0, 0, xx, yy, fw, fh, cdata, fbw);
   blit_maybe_busy = 1;
}

/*----------------------------------------------------------------
 * gx1_geodefb_puts
 *
 * Description: This functions calls the console frame buffer driver
 *                              putcs function.          
 * parameters:
 *       conp:  It is a pointer to the virtual console data structure. 
 *          p:  It is a pointer to the display data structure. 
 *          s:  It is  a pointer used as mask to get string of source
 *                              pixel data.         
 *      count:
 *         yy:  It gives destination Y position 
 *                              (offset into frame buffer)
 *         xx:  It gives destination X position 
 *     return:  none
 *-----------------------------------------------------------------*/
static void
gx1_geodefb_putcs(struct vc_data *conp, struct display *p,
                  const unsigned short *s, int count, int yy, int xx)
{
   u32 fg, bg;
   int fw = fontwidth(p);
   int fh = fontheight(p);

//      int fbw=(fw+7)>>3;
   u8 *cdata;

   assert(p != NULL);
   assert(conp != NULL);

   if (p->var.bits_per_pixel == 8) {
      fg = attr_fgcol(p, *s);
      bg = attr_bgcol(p, *s);
   } else if (p->var.bits_per_pixel == 16) {
      fg = ((u16 *) p->dispsw_data)[attr_fgcol(p, scr_readw(s))];
      bg = ((u16 *) p->dispsw_data)[attr_bgcol(p, scr_readw(s))];
   } else {
      fg = ((u32 *) p->dispsw_data)[attr_fgcol(p, scr_readw(s))];
      bg = ((u32 *) p->dispsw_data)[attr_bgcol(p, scr_readw(s))];
   }

   if (blit_maybe_busy)
      Geode_WaitBusy();

   xx *= fw;
   yy *= fh;

#ifdef CONFIG_FB_NSC_TV
   if ((p->var.reserved[0] & Display_TVO) && (is_tv_supported())) {
      xx += p->var.reserved[1];
      yy += p->var.reserved[3];
   }
#endif

#if GEODEFBDEBUG
   if (putcs_dbg) {
      DPRINTK("%d %d, %d\n", xx, yy, count);
   }
#endif

   gfx_set_solid_pattern(0);
   gfx_set_mono_source(bg, fg, 0);
   gfx_set_raster_operation(0xcc);
   while (count--) {
      cdata = p->fontdata + (scr_readw(s++) & p->charmask) * fh;
      gfx_text_blt(xx, yy, fw, fh, cdata);
//              gfx_mono_bitmap_to_screen_blt(0, 0, xx, yy, fw, fh, cdata, fbw);
      xx += fw;
   }
   blit_maybe_busy = 1;
}

/*----------------------------------------------------------------
 * gx1_geodefb_clear_margins
 *
 *  Description: This function clears the margins of the screen.
 *                                
 *   parameters:
 *         conp: It is a pointer to the virtual console data structure. 
 *            p: It is a pointer to the display data structure.
 *  bottom_only: It gives the value of the bottom only flag.
 *       return: none.
 *-----------------------------------------------------------------*/
static void
gx1_geodefb_clear_margins(struct vc_data *conp,
                          struct display *p, int bottom_only)
{
   unsigned long bg;
   unsigned int sx = conp->vc_cols * fontwidth(p);
   unsigned int sy = conp->vc_rows * fontheight(p);
   unsigned int width, height;

   assert(p != NULL);
   assert(conp != NULL);

   if (blit_maybe_busy)
      Geode_WaitBusy();

   if (p->var.bits_per_pixel == 8) {
      bg = attr_bgcol_ec(p, conp);
   } else if (p->var.bits_per_pixel == 16) {
      bg = ((u16 *) p->dispsw_data)[attr_bgcol_ec(p, conp)];
   } else {
      bg = ((u32 *) p->dispsw_data)[attr_bgcol_ec(p, conp)];
   }

   if (!bottom_only && (width = p->var.xres - sx)) {
#ifdef CONFIG_FB_NSC_TV
      if ((p->var.reserved[0] & Display_TVO) && (is_tv_supported())) {
         gx1_rectfill(sx + p->var.reserved[1], p->var.reserved[3],
                      width, p->var.yres_virtual, bg);
      } else
#endif
      {
         gx1_rectfill(sx, 0, width, p->var.yres_virtual, bg);
      }
   }

   if ((height = p->var.yres - sy)) {
#ifdef CONFIG_FB_NSC_TV
      if ((p->var.reserved[0] & Display_TVO) && (is_tv_supported())) {
         gx1_rectfill(p->var.reserved[1],
                      (p->var.yoffset + sy) + p->var.reserved[3],
                      sx, height, bg);
      } else
#endif
      {
         gx1_rectfill(0, (p->var.yoffset + sy), sx, height, bg);
      }
   }
   blit_maybe_busy = 1;
}

/*----------------------------------------------------------------
 * gx1_geodefb_revc
 *
 *  Description: This function reverse the character color.
 *                                
 *   parameters:
 *            p: It is a pointer to the display data structure.
 *           xx: It is the X-offset of the character.
 *           yy: It is the Y-offset of the character.
 *       return: none.
 *-----------------------------------------------------------------*/
static void
gx1_geodefb_revc(struct display *p, int xx, int yy)
{
   int fw = fontwidth(p);
   int fh = fontheight(p);

   assert(p != NULL);

   if (blit_maybe_busy)
      Geode_WaitBusy();
   xx *= fw;
   yy *= fh;

   gfx_set_solid_pattern(0);
   gfx_set_raster_operation(0x55);      /* XOR */

#ifdef CONFIG_FB_NSC_TV
   if ((p->var.reserved[0] & Display_TVO) && (is_tv_supported())) {
/*              DPRINTK("TVO revc %d %d %dx%d, %dx%d\n", 
                        xx, yy, p->var.reserved[1], p->var.reserved[3], fw, fh);
*/
      gfx_screen_to_screen_blt(p->var.reserved[1] + xx,
                               p->var.reserved[3] + yy,
                               p->var.reserved[1] + xx,
                               p->var.reserved[3] + yy, fw, fh);
   } else
#endif
   {
/*              DPRINTK("revc %d %d, %dx%d\n", xx, yy, fw, fh); */
      gfx_screen_to_screen_blt(xx, yy, xx, yy, fw, fh);
   }
   blit_maybe_busy = 1;
}

/*
 * Switch for low level operations 
 */
#ifdef FBCON_HAS_CFB8
static struct display_switch gx1_fbcon_geode8 = {
   setup:fbcon_cfb8_setup,
   bmove:gx1_geodefb_bmove,
   clear:gx1_geodefb_clear,
   putc:gx1_geodefb_putc,
   putcs:gx1_geodefb_putcs,
   revc:fbcon_cfb8_revc,                /* our func not working for 8bpp - Why not known */
   clear_margins:gx1_geodefb_clear_margins,
   fontwidthmask:FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(12) | FONTWIDTH(16)
};
#endif /* endif for FBCON_HAS_CFB8 */
#ifdef FBCON_HAS_CFB16
static struct display_switch gx1_fbcon_geode16 = {
   setup:fbcon_cfb16_setup,
   bmove:gx1_geodefb_bmove,
   clear:gx1_geodefb_clear,
   putc:gx1_geodefb_putc,
   putcs:gx1_geodefb_putcs,
   revc:gx1_geodefb_revc,
   clear_margins:gx1_geodefb_clear_margins,
   fontwidthmask:FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(12) | FONTWIDTH(16)
};
#endif /* endif  for FBCON_HAS_CFB16 */

#ifdef MODULE
/*----------------------------------------------------------------
 * gx1_cleanup_module 
 *
 * Description: This function unregister the geode frame buffer
 *                              driver.
 *  parameters: none. 
 *      return: none  
 *--------------------------------------------------------------*/
void
gx1_cleanup_module(void)
{
   /* unregister all usages of geode_pm_callback */
#if PM
   pm_unregister_all(gx1_geode_pm_callback);
#endif
   unregister_framebuffer((struct fb_info *)&gen.info);
}
#endif /* MODULE */
