 /*
  * 
 * <LIC_AMD_STD>
 * Copyright (c) 2004 Advanced Micro Devices, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * The full GNU General Public License is included in this distribution in the
 * file called COPYING
 * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * File Contents: This file contains the main functions of the Geode 
  *                frame buffer device driver, which includes functions 
  *                init, setup and ioctl etc.
  * [Notes:]           All the geode frame buffer driver functions can be 
  *                invoked through fbmem.c file functions
  *
  * Project:       Geode Frame buffer device driver
  * </DOC_AMD_STD>
  * 
  */

#define EXTERN_DEFINED
#include "nsc_common_fb.h"

/* Interface used by the world */

/* The geodefb_setup function set the display mode */
int nscfb_setup(char *);

/* Interface to the low level console driver */

/*
 * The geodefb_init function intialiazes the Geode frame buffer
 * driver
 */
int nscfb_init(void);

#ifdef MODULE
extern void gx1_cleanup_module(void);
extern void gx2_cleanup_module(void);
#endif
extern int gx1_geodefb_init(void);
extern int gx2_geodefb_init(void);
extern int gx1_geodefb_setup(char *options);
extern int gx2_geodefb_setup(char *options);
extern void strip_support_for_REDCLOUD(void);

/*------------- common routines --------------------*/

/*-----------------------------------------------------------------
 * Geode_WaitBusy
 *
 * Description: This function sychronize the Geode graphics engine.
 *              While accessing the geode frame buffer it makes 
 *              the graphics engine idle.
 *  parameters: none
 *      return: none
 *---------------------------------------------------------------*/
void
Geode_WaitBusy(void)
{
   gfx_wait_until_idle();
   blit_maybe_busy = 0;
}

unsigned long
frequency2fixedformat(double frequency)
{
   int clock = frequency;
   unsigned long pll;

   frequency *= 1000;
   pll = (clock << 16) | ((int)
                          ((double)(frequency - (clock * 1000)) *
                           (double)65.536));
#if GEODEFBDEBUG
#if 0
   printk("fixedformat_frequency: %X - %X %X %08x\n",
          frequency,
          clock,
          ((int)((double)(frequency - (clock * 1000)) * (double)65.536)),
          pll);
#endif
#endif

   return (pll);
}

double
fixedformat2frequency(unsigned long frequency)
{
   int dec = frequency >> 16;
   int frac = frequency & 0xFFFF;
   double freq = dec + (double)frac / (double)(65536);

#if GEODEFBDEBUG
#if 0
   printk("fixedformat_frequency: %X - %X %X %f = %3.4f\n",
          frequency, dec, frac, (double)frac / (double)(65536), freq);
#endif
#endif
   return freq;
}

void
set_crt_pixclock(struct fb_var_screeninfo *var)
{
   int pll;

   gfx_get_frequency_from_refreshrate(var->xres,
                                      var->yres, var->bits_per_pixel, vfreq,
                                      &pll);
   DPRINTK("%d, %x\n", vfreq, pll);
   var->pixclock = (1E6 / fixedformat2frequency(pll)) + .5;
   /* strip off all TV msaks */
   var->reserved[0] &= ~Display_TVMASK;
}

unsigned int
does_pixclock_match(const struct fb_var_screeninfo *var,
                    struct geodefb_par *par, int *hz, int *pixclock)
{
   int pll16_16, diff16_16;

   pll16_16 = frequency2fixedformat(1E6 / (double)var->pixclock);
   DPRINTK("%d, %d %x\n", vfreq, par->refresh, pll16_16);

   /* get the vhz from pixel clock */
   gfx_get_refreshrate_from_mode(var->xres,
                                 var->yres,
                                 var->bits_per_pixel, hz, pll16_16);

   /* get the pixel clock from vhz */
   gfx_get_frequency_from_refreshrate(var->xres,
                                      var->yres,
                                      var->bits_per_pixel, *hz, pixclock);

   /* compare to see if there is a difference and vhz match */
   diff16_16 = *pixclock - pll16_16;
   diff16_16 = (diff16_16 < 0) ? -diff16_16 : diff16_16;

   DPRINTK("%X - %X = %X\n", *pixclock, pll16_16, diff16_16);

   return diff16_16;
}

unsigned int
calc_err(unsigned long pixclock)
{
   unsigned int err = ((double)PIXCLOCK_ERR * (double)pixclock);

   DPRINTK("%X\n", err);

   return err;
}

/*--------------- Generic routines ----------------------------*/

/*-----------------------------------------------------------------
 * geodefb_pan_display
 *
 * Description: This function Pan or Wrap the Display
 *                              and it looks only at xoffset, yoffset and
 *                              the FB_VMODE_YWRAP flag.
 *      parameters:
 *                 var: Pointer to fb_var_screeninfo which has the parameters
 *                              to set the hardware. 
 *        con:  The console which has to be updated with parameters 
 *                              in 'var' structure.
 *       info:  This is a pointer to the frame buffer information 
 *                              structure.  
 *     return:  error value 
 *----------------------------------------------------------------*/
int
geodefb_pan_display(struct fb_var_screeninfo *var,
                    int con, struct fb_info *info)
{
   assert(var != NULL);
   return (-EINVAL);
}

/**
 *      PROC_CONSOLE - find the attached tty or visible console
 *      @info: frame buffer info structure
 *
 *      Finds the tty attached to the process or visible console if
 *      the process is not directly attached to a tty (e.g. remote
 *      user) for device @info.
 *
 *      Returns -1 errno on error, or tty/visible console number
 *      on success.
 *
 */

int
NSC_PROC_CONSOLE(const struct fb_info *info)
{
   int fgc;

   if (info->display_fg != NULL)
      fgc = info->display_fg->vc_num;
   else
      return -1;

   if (!current->tty)
      return fgc;

   if (current->tty->driver.type != TTY_DRIVER_TYPE_CONSOLE)
      /* XXX Should report error here? */
      return fgc;

   if (MINOR(current->tty->device) < 1)
      return fgc;

   return MINOR(current->tty->device) - 1;
}

/*---------------------------------------------------------------------
 * nscfb_setup
 *
 * Description: This function setup the display mode based 
 *              on the option specified by the user if it 
 *              supports, other wise it will set the default 
 *              display mode.
 *  parameters:
 *         options:     This is character pointer holds the display mode 
 *                              specified from user.
 *      return: Returns 0 if the display mode was set successfully. 
 *---------------------------------------------------------------------*/
int __init
nscfb_setup(char *options)
{
   cpu_version = gfx_detect_cpu();

   if (!cpu_version) {
      /* NO GEODE PROCESSOR DETECTED */
      return (-ENXIO);
   }

   detect_hw_platform = 1;
   if ((cpu_version & 0xFF) == GFX_CPU_REDCLOUD) {
      rc = RC_CRT; /* default */
   }
   if (rc) {
      return (gx2_geodefb_setup(options));
   } else {
      return (gx1_geodefb_setup(options));
   }
}

/*----------------------------------------------------------------
 * nscfb_init
 *
 * Description: This is the entry point to the driver. This 
 *              function detects a National display controller 
 *              (CS5530, CS1200) and initializes the durango 
 *              interface by memory mapping the physical address 
 *              space required for the controller.
 *              Then it registers the  geode frame buffer device.
 *              A software device /dev/fb0 can be used to receive 
 *              IOCTLs from GAL. 
 *  Parameters: none
 *              return: 0 returned if the geode frame buffer device 
 *                              is created < 0 returned on error.
 *---------------------------------------------------------------*/
int __init
nscfb_init(void)
{
   DPRINTK("%s", "\n");

   DebugPort(16);
   if (detect_hw_platform == 0) {
      cpu_version = gfx_detect_cpu();
   }

   if (!cpu_version) {
      /* NO GEODE PROCESSOR DETECTED */
      return (-ENXIO);
   }

   DebugPort(17);
   if ((cpu_version & 0xFF) == GFX_CPU_REDCLOUD) {
      rc = RC_CRT;                      /* default */
      strip_support_for_REDCLOUD();
   }
   if (rc) {
      return (gx2_geodefb_init());
   } else {
      DebugPort(18);
      return (gx1_geodefb_init());
   }
}

#ifdef MODULE

MODULE_AUTHOR("(c) 1999,2001 National Semiconductor Corporation");
MODULE_DESCRIPTION("Accelerated FBDev driver for GX1/SCx200/GX2");
MODULE_PARM(vesa, "i");
MODULE_PARM_DESC(vesa, "Startup videomode (0x000-0x1FF) (default=0x101)");
MODULE_PARM(vfreq, "i");
MODULE_PARM_DESC(vfreq,
                 "Startup vertical frequency (60-85 MHz) (default=60)");
MODULE_PARM(vmode, "s");
MODULE_PARM_DESC(vmode,
                 "Name of the selected resolution in text (default=640x480-16)");
MODULE_PARM(font, "s");
MODULE_PARM_DESC(font, "Name of the font");

#ifdef CONFIG_FB_NSC_FP
MODULE_PARM(flatpanel, "i");
MODULE_PARM_DESC(flatpanel,
                 "Flatpanel support (0=disabled or 1=enabled) (default=0)");
#endif
MODULE_PARM(compression, "i");
MODULE_PARM_DESC(compression,
                 "Compression support (0=disabled or 1=enabled) (default=1)");

#ifdef CONFIG_FB_NSC_TV
MODULE_PARM(tv_output, "i");
MODULE_PARM_DESC(tv_output, "TV output (default=1-S_VIDEO)");
MODULE_PARM(TVO, "s");
MODULE_PARM_DESC(TVO, "TV Overscan option TVO=xoffset:yoffset:width:height");
#endif

#ifdef CONFIG_FB_NSC_DDC
MODULE_PARM(ddc, "i");
MODULE_PARM_DESC(ddc, "DDC support (0=disabled or 1=enabled) (default=0)");
MODULE_PARM(ddc_depth, "i");
MODULE_PARM_DESC(ddc_depth,
                 "DDC Depth when supported  8 or 16 Bpp (default=8)");
#endif

/*------------------------------------------------------------------------
 * init_module 
 *
 * Description: This function will call nscfb_init to creat 
 *                              geode frame buffer driver as a module.
 *  parameters: none. 
 *      return: 0 returned if the geode frame buffer device is created.
 *                              < 0 returned on error.
 *----------------------------------------------------------------------*/
int
init_module(void)
{
   if((vfreq != 56) && (vfreq != 60) && 
        (vfreq != 70) && (vfreq != 72) && 
        (vfreq != 75) && (vfreq != 85)) {
        vfreq = 60;	
   }
   return (nscfb_init());
}

/*----------------------------------------------------------------
 * cleanup_module 
 *
 * Description: This function unregister the geode frame buffer
 *                              driver.
 *  parameters: none. 
 *      return: none  
 *--------------------------------------------------------------*/
void
cleanup_module(void)
{
   if (rc) {
      gx2_cleanup_module();
   } else {
      gx1_cleanup_module();
   }
}
#endif /* MODULE */
