 /*
  * 
 * <LIC_AMD_STD>
 * Copyright (c) 2004 Advanced Micro Devices, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * The full GNU General Public License is included in this distribution in the
 * file called COPYING
 * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * File Contents: This file contains the function definitions to access the  
  *                hardware x200   I/O compaion.
  *
  * SubModule:     Display Data Channel support.
  * </DOC_AMD_STD>
  * 
  */


#include "ddc_x200.h"

/* Glen Han 01/25/2001, modified code to support DORADO board
 *DDC DORADO implementation with GPIO pins
 *SCL is gpio 7, SDA is gpio 18, CSDDC is gpio 9
 */
#define SCx2xx_GPIO_BASE 0x6400
#define SCx2xx_GPDO0 0x0000
#define SCx2xx_GPDI0 0x0004
#define SCx2xx_GPPIN_SEL 0x0020
#define SCx2xx_GPPIN_CFG 0x0024
#define GPOUT (SCx2xx_GPIO_BASE + SCx2xx_GPDO0) 
#define GPIN (SCx2xx_GPIO_BASE + SCx2xx_GPDI0) 
#define GPSEL (SCx2xx_GPIO_BASE + SCx2xx_GPPIN_SEL) 
#define GPCFG (SCx2xx_GPIO_BASE + SCx2xx_GPPIN_CFG)
#define OUTPUT_ENABLE 0x00000003 
#define INPUT_ENABLE 0x00000000 
#define CLOCK_PIN_MASK 0x00000080
#define DATA_PIN_MASK 0x00040000
#define CSDDC_PIN_MASK 0x00000200
#define CLOCK_PIN_SEL 0x00000007
#define DATA_PIN_SEL 0x00000012
#define CSDDC_PIN_SEL 0x00000009

int Get_9211_Details()
{
	unsigned long pmr;
	
#define SC_CB_ADDR      0x9000
#define SC_PMR          0x30
#define SC_PMR_TFTIDE   0x01000000
#define SC_PMR_TFTPP    0x00800000
#define SC_PMR_ETFT     0x00000400

	pmr = gfx_ind(SC_CB_ADDR + SC_PMR);
	RETAILMSG( 1,(TEXT("Get_9211_Details %X\n"),pmr));

	if (!(pmr & SC_PMR_TFTIDE)) {
		// The SC_PMR_TFTIDE bit is off, go check if the TFT is 
		// multiplexed with the parallel port.
		if (pmr & SC_PMR_TFTPP) {
			pmr |= SC_PMR_ETFT;     // Make sure this is on.
			gfx_outd(SC_CB_ADDR + SC_PMR, pmr);
		} else {
			// Failed.  The hardware configuration precludes a TFT.
			return(0); 
		}
	}
	return(1); 
}

/*----------------------------------------------------------------------------
 * init_x200.
 *
 * Description	  : This function is called to configure the  Serial Data
 *					line to probe the monitor.
 *
 * Parameters     : None
 *
 * Returns		  : None
 *
 * Comments       : This function  should be invoked before readbyte or write
 *                   byte routines.
*----------------------------------------------------------------------------
*/
void init_x200() 
{
	RETAILMSG( 0,(TEXT("SCx200 setup\n")));
	requestData_x200();
	ReadByte_x200(0);
	StopBit_x200();
}
/*----------------------------------------------------------------------------
 * ReadByte_x200.
 *
 * Description	  : This function reads the data bit by bit and setting ack
 *                    after the one byte read.
 *
 * Parameters       
 *       ack      : Gets the one byte data.
 * Returns		  : Returning one byte information after read bit.
 *
 * Comments       : This function  should be invoked before readbyte or write
 *                : byte routines.
*----------------------------------------------------------------------------
*/

unsigned char ReadByte_x200(int ack)
{
	unsigned char val = 0;
	int i;
	RETAILMSG( 0,(TEXT("getByte\n")));
	ConfigSDAin_x200();
	for (i = 0; i < 8; i++) {
		val <<= 1;
		val |= ReadBit();
	}

	Ack_x200(ack);
	return val;
}
/*---------------------------------------------------------------------------
 * StartBit_x200.
 *
 * Description	  : This function sending out the start bit of serial data
 *
 *
 * Parameters     : None
 *
 * Returns		  : None
 *
 * Comments       : This function indicates the begining ddc seiral data.
 *
*----------------------------------------------------------------------------
*/
void StartBit_x200()
{
	RETAILMSG( 0,(TEXT("startBit\n")));
	wait(DEFAULTDELAY);
	ClearSCL_x200();
	wait(DEFAULTDELAY);
	SetSDA_x200();
	ConfigSDAout_x200();
	wait(DEFAULTDELAY);
	WaitSCLHigh();
	wait(DEFAULTDELAY);
	ClearSDA_x200();
	wait(DEFAULTDELAY);
}

/*---------------------------------------------------------------------------
 * StopBit_x200.
 *
 * Description	  : This function sending out the stop bit of serial data
 *
 *
 * Parameters     : None
 *
 * Returns		  : None
 *
 * Comments       : This function indicates the end ddc seiral data.
 *
*----------------------------------------------------------------------------
*/
void StopBit_x200() 
{
	RETAILMSG( 0,(TEXT("stopBit\n")));	
	/* always follow master ack, already in SDA out mode */
	wait(DEFAULTDELAY);
	ClearSCL_x200();
	wait(DEFAULTDELAY);
	ClearSDA_x200();
	wait(DEFAULTDELAY);
	WaitSCLHigh();
	wait(DEFAULTDELAY);
	SetSDA_x200();
	wait(2);
}

/*---------------------------------------------------------------------------
 * Ack_x200.
 *
 * Description	  : This sends out the ack data
 *
 * Parameters       
 *        flag    : Value read in ReaByte routine is passed
 * Returns		  : None
 *
 * Comments       : None
*----------------------------------------------------------------------------
*/
void Ack_x200(int flag)
{
	RETAILMSG( 0,(TEXT("ACK\n")));
	wait(DEFAULTDELAY);
	ClearSCL_x200();
	wait(DEFAULTDELAY);
	if (flag) {
		ClearSDA_x200();
	} else {
		SetSDA_x200();
	}
	ConfigSDAout_x200();
	WaitSCLHigh();
	wait(10);		/* let transmitter has time to sense it */
	ClearSCL_x200();
	wait(DEFAULTDELAY);
	SetSDA_x200();	       /* is it real necessary? */
	RETAILMSG( 0,(TEXT("ACK Exit\n")));
}

/*---------------------------------------------------------------------------
 * waitForAck:
 *
 * Description	  : This routine waits the ack SDL data.
 *
 * Parameters     : none
 *
 * Returns		  : None
 *
 * Comments       : None
*----------------------------------------------------------------------------
*/
int waitForAck_x200()
{
	unsigned long target;
	int	i = 0;

	RETAILMSG( 0,(TEXT("waitForACK\n")));
	wait(DEFAULTDELAY);
	ClearSCL_x200();
	wait(DEFAULTDELAY);
	SetSDA_x200();
	ConfigSDAin_x200();
	WaitSCLHigh();
	wait(5);

#ifndef _WIN32_WCE
	/* Try for two times (2 uSec) */
	for (i=0; i<2; i++) {
		wait(DEFAULTDELAY);
		if (GetSDA_x200() == 0) {
		return 1;
		} 
	}
	/* time out */
	return 0;

#else
	target = GetTickCount() + 2;
	for (;;) {
		wait(DEFAULTDELAY);
		if (GetSDA_x200() == 0) {
			return 1;
		} else if(GetTickCount() > target) {
			return 0;
		}
	}
#endif
}

/*---------------------------------------------------------------------------
 * requestData.
 *
 * Description	  : This routine request the data and configure SDA to Rx mode.
 *
 * Parameters     : none
 *
 * Returns		  : None
 *
 * Comments       : None
*----------------------------------------------------------------------------
*/

void requestData_x200() 
{
	RETAILMSG( 0,(TEXT("requestData\n")));
	StartBit_x200();
	SendByte(0xa0);
	ConfigSDAout_x200();
	SendByte(0);

	StartBit_x200();
	SendByte(0xa1);
}


/*---------------------------------------------------------------------------
 * startXfer_x200.
 *
 * Description	  : This routine is called to sets SDA and SCL outto high state
	                when the transfer starts.
 * To start DDC transfer
 * 1. Enable DDC access by making chip control pin active low
 * 2. Configuer SCL and SDA as output pins 
 * 3. Issue an stop condition to clean up any possible slave device mess  
 *   caused by other application.
 *. Toggle clock several times to let slave device finish 
 *  internal housekeeping job
* Parameters     : None
 * Returns		  : None
 *
 * Comments       : SCL should be high when toggling.
*----------------------------------------------------------------------------
*/

void startXfer_x200() 
{
	int i;
	RETAILMSG( 0,(TEXT("startXfer\n")));
	ClearCSDDC_x200();
	wait(10);
	ConfigSCLout_x200();
	ConfigSDAout_x200();
	StopBit_x200();
	wait(10);
	for (i = 0; i < 10; i++)
	{
		ClearSCL_x200();
		wait(DEFAULTDELAY);
		SetSCL_x200();
		wait(DEFAULTDELAY);
	}
}

/*---------------------------------------------------------------------------
 * endXfer_x200.
 *
 * Description		: This routine is called to sets SDA to low state when the
 *                    transfer ends.
 *
 * Parameters		: None
 * Returns			: None
 *
 * Comments         : SCL should be high when toggling.
*----------------------------------------------------------------------------
*/
void endXfer_x200() 
{
	int i;
	wait(DEFAULTDELAY);
	for (i = 0; i < 10; i++)
	{
		ClearSCL_x200();
		wait(DEFAULTDELAY);
		SetSCL_x200();
		wait(DEFAULTDELAY);
	}
	ConfigSDAin_x200();				/* tristate pin */
	ConfigSCLin_x200();				/* tristate pin */
	RETAILMSG( 0,(TEXT("endXfer\n")));	
	wait(10);
}


/***************************************************************
*  H/W DEPENDENT LOW LAYER ROUTINES
****************************************************************/

/*---------------------------------------------------------------------------
 * SetCSDDC_x200.
 *
 * Description	  : This routine disable DDC acesss and enable cs9211 serial
	                access.
 *Set CSDDC chip select control pin high to disable DDC access 
 * and enable cs9211 serial access
 * Parameters     : None
 * Returns		  : None
 *
 * Comments       : None.
*----------------------------------------------------------------------------
*/
void SetCSDDC_x200()
{
	unsigned long data = gfx_ind(GPIN);
	data |= CSDDC_PIN_MASK;
	RETAILMSG( 0,(TEXT("Set CSDDC to disable DDC access \n")));
	gfx_outd(GPOUT, data);
	wait(10); /*10 */
}

/*---------------------------------------------------------------------------
 * ClearCSDDC_x200.
 *
 * Description	  : This routine enable DDC acesss and disble cs9211 serial
	                access.
 * Set CSDDC chip select control pin low to enable DDC access 
 *and disable cs9211 serial access
 * Parameters     : None
 * Returns		  : None
 *
 * Comments       : None.
*----------------------------------------------------------------------------
*/
void ClearCSDDC_x200()
{
	unsigned long data = gfx_ind(GPIN);
	data &= ~CSDDC_PIN_MASK;
	RETAILMSG( 0,(TEXT("Clear CSDDC to enable DDC access %X\n"),data));
	gfx_outd(GPOUT, data);
	wait(10); /* 10 */
}

/*---------------------------------------------------------------------------
 * SetSCL_x200.
 *
 * Description	  : This routine sets the serial clock line to high.
 *
 *
 * Parameters     : None
 * Returns		  : None
 *
 * Comments       : none
*----------------------------------------------------------------------------
*/
void SetSCL_x200() 
{
	unsigned long data = gfx_ind(GPIN);
	data |= CLOCK_PIN_MASK;
	RETAILMSG( 0,(TEXT("SetSCL\n")));
	gfx_outd(GPOUT, data);
}


/*---------------------------------------------------------------------------
 * ClearSCL_x200.
 *
 * Description	  : This routine sets the serial clock line to low.
 *
 *
 * Parameters     : None
 * Returns		  : None
 *
 * Comments       : none
*----------------------------------------------------------------------------
*/
void ClearSCL_x200() 
{
	unsigned long data = gfx_ind(GPIN);
	data &= ~CLOCK_PIN_MASK;
	RETAILMSG( 0,(TEXT("DDCclockLo %X\n"),data));
	gfx_outd(GPOUT, data);
}
/*---------------------------------------------------------------------------
 * GetSCL_x200.
 *
 * Description	  : This routine gets the of SCL data .
 *
 *
 * Parameters     : None
 * Returns		  : returns SCL data.
 *
 * Comments       : none
*----------------------------------------------------------------------------
*/
int GetSCL_x200() 
{
	unsigned long data = gfx_ind(GPIN);
	RETAILMSG( 0,(TEXT("GetSCL %X\n"), data));
	data &= CLOCK_PIN_MASK;
	return (data >> CLOCK_PIN_SEL);
}


/*---------------------------------------------------------------------------
 * SetSDA_x200.
 *
 * Description	  : This routine sets the serial data line to high.
 *
 *
 * Parameters     : None
 * Returns		  : None
 *
 * Comments       : none
*----------------------------------------------------------------------------
*/
void SetSDA_x200() 
{
	unsigned long data = gfx_ind(GPIN);
	data |= DATA_PIN_MASK;
	RETAILMSG( 0,(TEXT("SetSDA %X\n"), data));
	gfx_outd(GPOUT, data);
}

/*---------------------------------------------------------------------------
 * ClearSDA_x200.
 *
 * Description	  : This routine sets the serial data line to low.
 *
 *
 * Parameters     : None
 * Returns		  : None
 *
 * Comments       : none
*----------------------------------------------------------------------------
*/
void ClearSDA_x200() 
{
	unsigned long data = gfx_ind(GPIN);
	data &= ~DATA_PIN_MASK;
	RETAILMSG( 0,(TEXT("ClearSDA %X\n"), data));
	gfx_outd(GPOUT, data);
}

/*---------------------------------------------------------------------------
 * GetSDA_x200.
 *
 * Description	  : This routine gets the of SDL data .
 *
 *
 * Parameters     : None
 * Returns		  : returns SCL data.
 *
 * Comments       : None
*----------------------------------------------------------------------------
*/
int GetSDA_x200() 
{
	unsigned long data = gfx_ind(GPIN);
	RETAILMSG( 0,(TEXT("GetSDA 0x%8X\n"), data));
	data &= DATA_PIN_MASK;
	return (data >> DATA_PIN_SEL);
}

/*---------------------------------------------------------------------------
 * ConfigSDAout_x200.
 *
 * Description	  : This routiine selects GPIO base and sets SDA pin for out.
 *
 *
 * Parameters     : None
 * Returns		  : None.
 *
 * Comments       : None
*----------------------------------------------------------------------------
*/

void ConfigSDAout_x200() 
{
 	gfx_outd(GPSEL, DATA_PIN_SEL);
 	gfx_outd(GPCFG, OUTPUT_ENABLE);
	wait(5); //10
}
/*---------------------------------------------------------------------------
 * ConfigSDAin_x200.
 *
 * Description	  : This routiine selects GPIO base and sets SDA pin for in.
 *
 *
 * Parameters     : None
 * Returns		  : None.
 *
 * Comments       : None
*----------------------------------------------------------------------------
*/
void ConfigSDAin_x200() 
{
 	gfx_outd(GPSEL, DATA_PIN_SEL);
 	gfx_outd(GPCFG, INPUT_ENABLE);
	wait(5); /* 10 */
}
/*---------------------------------------------------------------------------
 * ConfigSCLout_x200.
 *
 * Description	  : This routiine selects GPIO base and sets SCL pin for in.
 *
 *
 * Parameters     : None
 * Returns		  : None.
 *
 * Comments       : None
*----------------------------------------------------------------------------
*/



void ConfigSCLout_x200() 
{
	ClearSCL_x200();	/* don't disturb any device */
 	gfx_outd(GPSEL, CLOCK_PIN_SEL);
 	gfx_outd(GPCFG, OUTPUT_ENABLE);
	wait(5); //10
}

/*---------------------------------------------------------------------------
 * ConfigSCLin_x200.
 *
 * Description	  : This routiine selects GPIO base and sets SCL pin for in.
 *
 *
 * Parameters     : None
 * Returns		  : None.
 *
 * Comments       : None
*----------------------------------------------------------------------------
*/
void ConfigSCLin_x200() 
{
 	gfx_outd(GPSEL, CLOCK_PIN_SEL);
 	gfx_outd(GPCFG, INPUT_ENABLE);
	wait(5); /* 10 */
}
