 /*
  * 
 * <LIC_AMD_STD>
 * Copyright (c) 2004 Advanced Micro Devices, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * The full GNU General Public License is included in this distribution in the
 * file called COPYING
 * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * File Contents: This file contains the DDC initialization function 
  *                definitions. 
  *  
  * SubModule:     Display Data Channel support.
  * </DOC_AMD_STD>
  * 
  */

#include "ddc.h"
#include "ddc_defs.h"

#define DisplayDataLength 128
#define MAX_MODES 50

/*Forward prototype declarations */
static unsigned long displaySupports;
static unsigned char displayData[DisplayDataLength];
static void WaitSCLHigh(void);
static void getBlock(unsigned char *, int);
static void wait(int uSec);
static int  ReadBit(void);
static void SendBit(int);
static void SendByte(unsigned char);
static void standardTiming(unsigned char *data); 
static void predefinedSettings(unsigned char Tim, Predefined_DDC *array,
								 int start, int count);
static int checkHeader(void); 
static void add2List(unsigned long xres,unsigned long yres,unsigned long hz);
static void sortList(void);
static void dumpList(void);

static unsigned long Current;
static unsigned long SupportedModes[MAX_MODES];
static unsigned long TempModes[MAX_MODES];
static unsigned long temp;
static Platform_DDC_ops *ddc_ops=0x0;

/* functions to support the ddc protocols are included in this file struct.*/
#ifdef PLATFORM_5530
#include "ddc_5530.c"

Platform_DDC_ops ddc_5530={
	init_5530,
	ReadByte_5530,
	StartBit_5530,
	StopBit_5530,
	Ack_5530,
	waitForAck_5530,
	requestData_5530,
	startXfer_5530,
	endXfer_5530,
	SetSCL_5530,
	ClearSCL_5530,
	ClearSDA_5530,
	SetSDA_5530,
	GetSDA_5530,
	GetSCL_5530
};
#endif

#ifdef PLATFORM_x200
#include "ddc_x200.c"
Platform_DDC_ops ddc_x200={
	init_x200,
	ReadByte_x200,
	StartBit_x200,
	StopBit_x200,
	Ack_x200,
	waitForAck_x200,
	requestData_x200,
	startXfer_x200,
	endXfer_x200,
	SetSCL_x200,
	ClearSCL_x200,
	ClearSDA_x200,
	SetSDA_x200,
	GetSDA_x200,
	GetSCL_x200
};
#endif

/*----------------------------------------------------------------------------
 * SortList.
 *
 * Description	  : This function used for sorting out modes in order
 *
 * Parameters     : None
 *
 * Returns		  : None
 *
 * Comments       : This is keep the supported modes by the hw in order.
*----------------------------------------------------------------------------
*/
void sortList()
{
	int i,j;
	int length;
	length = MAX_MODES - 1;	

	for(i=0;i<MAX_MODES;i++)
	{
		for (j=0;j<length;j++)
		{
			if(SupportedModes[j] > SupportedModes[j+1])
			{
				temp = SupportedModes[j+1];
				SupportedModes[j+1] = SupportedModes[j];
				SupportedModes[j] = temp;
			}
		}
		length--;
	}
      /*Now put it it descending order */
	for(i=0;i<MAX_MODES;i++)
	{
		TempModes[i] = SupportedModes[MAX_MODES - 1 - i];
	}
      /* Now put the TempModes in same order back into SupportedModes */
	for(i=0;i<MAX_MODES;i++)
	{
		SupportedModes[i] = TempModes[i];
	}
}

/*----------------------------------------------------------------------------
 * add2List.
 *
 * Description		: This fuction will add the mode data into the prepared list.
 *                    next at the end. 	
 * Parameters       :
 *
 *     xres         : Specifies the horz resolution.
 *     yres         : Specifies the  vert resolution.
 *      hz          : Specfies the mode timing.
 *
 * Returns			: None.
 *
 * Comments         : This is used to navigate the list and sets the desird
 *                    mode.
 *----------------------------------------------------------------------------
*/

void add2List(unsigned long xres, unsigned long yres, unsigned long hz)
{
	RETAILMSG( 0,(_T("%d %d %d = "), xres, yres, hz));
	SupportedModes[Current] = (xres << 20) | (yres << 8) | (hz & 0xff);
	RETAILMSG( 0,(_T("%x\n"), SupportedModes[Current]));
	Current ++;
}

/*----------------------------------------------------------------------------
 * nextFromList.
 *
 * Description		: This fuction will check read  the mode list and points to
 *                    next in the list. 	
 * Parameters       :
 *	 xres			: Specifies the horz resolution.
 *     yres         : Specifies the  vert resolution.
 *      hz          : Specfies the mode timing.
 *
 * Returns			: None.
 *
 * Comments         : This is used to navigate the list and sets the desired
 *                    mode.
 *----------------------------------------------------------------------------
*/
void nextFromList(unsigned long *xres, unsigned long *yres, unsigned long *hz)
{
	*hz = SupportedModes[Current]  & 0xff;
	*yres = (SupportedModes[Current]  >> 8) & 0xfff;
	*xres = (SupportedModes[Current]  >> 20) & 0xfff;
	Current ++;
}

/*----------------------------------------------------------------------------
 * dumpList.
 *
 * Description	  : This function to prepare supported mode list.
	
 * Parameters     : None
 *
 * Returns		  : None.
 *
 * Comments       : This may be called to check the list of modes supported.
 *
 *----------------------------------------------------------------------------
*/

void dumpList() 
{
	int unsigned long i;
	for(i=0; i< Current; i++) {
		RETAILMSG( 0,(_T("%d %d %d\n"),
			(SupportedModes[i]  >> 20) & 0xfff,
			(SupportedModes[i]  >> 8) & 0xfff,
			SupportedModes[i]  & 0xff));
	}
}


/*----------------------------------------------------------------------------
 * PredefiedSettings.
 *
 * Description		: This function is to get the VESA estimated manual timing
                      based on manfacturer's list.
 * Parameters       :
 *     Tim          : Specifies the timing data from header read from hw.
 *     array        : Specifies the data of std mfgtiming.
 *     Start        : Specfies the index.* not used here?.
 *     count        : Specifies the no of mode count.
 * Returns			: None
 *
 * Comments         : List is prapared based on VESA timings.
 *
 *----------------------------------------------------------------------------
*/
void predefinedSettings(unsigned char Tim, Predefined_DDC *array,
								 int start, int count) 
{ 
        int i;
	for (i = 0; i < count; i++) {
		if(Tim & ((unsigned char)(0x1 << i)))
			add2List(array[i].xres, array[i].yres, array[i].hz);
	}
}
/*----------------------------------------------------------------------------
 * standardTiming.
 *
 * Description		: This function is to make standard timing list with VESA
					  standard timings.
 * Parameters       :
 *       data       : Pointer has mode data.
 * Returns			: None.
 *
 * Comments         : None.
 *
 *----------------------------------------------------------------------------
*/
void standardTiming(unsigned char *data) 
{
	unsigned long width, height;
	unsigned long freq, ratio, pos;
	int i;
	for (i = 0; i < 8; i++) {
		/* value = (HorizTiming /8) - 31 */
		pos = VESA1_EST_STAND_TIM + (i <<1); /* Each timing uses 2 Bytes as info. */
		if((data[pos] <= 0x1 ) && (data[pos + 1] <= 0x1))
			continue;
		width = (data[pos] + 31) << 3;
		freq = 60+(data[pos + 1] & 0x3F);
		ratio = (data[pos + 1] & 0xC0) >> 6;
		height = (width / asp[ratio].xRatio) * asp[ratio].yRatio;

		add2List(width, height, freq);
		
	RETAILMSG( 0,(_T("%d(%d %d) = %3dX%3d %dHz\n"),i, data[pos], data[pos+1], width, height, freq));
	}
}

/*----------------------------------------------------------------------------
 * CheckHeader.
 *
 * Description		:	This function checks the read ddc header with std EDID
 *						header.
 * Parameters		:	None
 *
 * Returns			:	returns TRUE if both headers are equal.
 *
 * Comments         :	This is to check the read header data.
 *
 *----------------------------------------------------------------------------
*/

int checkHeader()
{
	int i;
	for (i = 0; i < 8; i++) {
		if(displayData[i] != EDID_Header[i])
			return 0;
	}
	return 1;
}

/*----------------------------------------------------------------------------
 * DumpData.
 *
 * Description		: This function looks like redundant
 *
 * Parameters		: None
 *
 * Returns			:
 *
 * Comments			: None
 *
 *----------------------------------------------------------------------------
*/
static void dumpData(unsigned char *data, int length)
{
	int i;
	for (i = 0; i < length; i++) {
		RETAILMSG( 0,(_T("%d = %2X %3d\n"),i, data[i],data[i]));
	}
}

/*----------------------------------------------------------------------------
 * ddc_init.
 *
 * Description	  : This function detects the type of video HW and gets ddc data
 *
 *
 * Parameters     : None
 *
 * Returns		  : None
 *
 * Comments       : This readsthe header and sort list of modes.
 *
 *----------------------------------------------------------------------------
*/

void ddc_init() 
{
	unsigned long video_hw = gfx_detect_video();

#ifdef PLATFORM_5530
	if (video_hw == GFX_VID_CS5530)
		ddc_ops = &ddc_5530;
#endif

#ifdef PLATFORM_x200
	if ((video_hw == GFX_VID_SC1200) && Get_9211_Details())
		ddc_ops = &ddc_x200;
#endif

	displaySupports = 0;
	memset(displayData, 0x0, DisplayDataLength);
	memset(SupportedModes, 0x0, sizeof(unsigned long) * MAX_MODES);
	Current = 0;

	if(ddc_ops == 0x0) /* none mapped, ERROR */
		return;

	ddc_ops->startXfer();
	ddc_ops->init();
	getBlock(displayData, DisplayDataLength);
	ddc_ops->endXfer();
	dumpData(displayData, DisplayDataLength);
    if(checkHeader()) 
	{
		standardTiming(displayData);
		predefinedSettings(displayData[VESA1_EST_MANU_TIM], MfgTim2, 7, 1);
		predefinedSettings(displayData[VESA1_EST_TIM2], EstTim2, 0, 8);
		predefinedSettings(displayData[VESA1_EST_TIM1], EstTim1, 0, 8);
		dumpList();
		sortList();
		dumpList();
	Current = 0;
	}
}

/*----------------------------------------------------------------------------
 * ddc_Cleanup.
 *
 * Description	  : This function will bw called at the end of ddc.
 *
 *
 * Parameters     : None
 *
 * Returns		  : None
 *
 * Comments       : it may be enhanced.
 *
 *----------------------------------------------------------------------------
*/

void ddc_cleanup()
{
	RETAILMSG( 0,(TEXT("DDC1_2B::~DDC1_2B\n")));
}


/* Wait for microSeconds */
void wait(int uSec) 
{
	gfx_delay_milliseconds(uSec);
}

/*----------------------------------------------------------------------------
 * getBlock.
 *
 * Description		: This function reads the ddc data
 *
 *
 * Parameters       :
 *      block       : data will be stored in this ponter
 *      length      : Specifies the length of data to be read.
 * Returns			: None
 *
 * Comments         : Block of data will be read by readbyte,readbit routines.
 *
 *----------------------------------------------------------------------------
*/
void getBlock(unsigned char *block, int length) 
{

	int cksum = 0;
	int i, end = length - 1;

	RETAILMSG( 0,(TEXT("getBlock\n")));
	ddc_ops->requestData();

	for (i = 0; i < length; i++) {
		block[i] = ddc_ops->ReadByte((i != end));
		cksum += (int)block[i];
	}
	ddc_ops->StopBit();

	RETAILMSG( 0,(TEXT("getBlock %d\n"),cksum));

	if ((cksum & 0xff) != 0) {
		RETAILMSG( 1,(TEXT("Checksum Error\n")));
	}
}

/*----------------------------------------------------------------------------
 * ReadBit.
 *
 * Description	  : This function read the DDC data by one bit
 *
 *
 * Parameters     : None
 * Returns		  : One bit information by a flag.
 *
 * Comments       : Date will be read by  a bit.
 *
*----------------------------------------------------------------------------
*/

int ReadBit()
{
	RETAILMSG( 0,(TEXT("getBit\n")));
	wait(DEFAULTDELAY);
	ddc_ops->ClearSCL();
	wait(DEFAULTDELAY);
	WaitSCLHigh();
	wait(DEFAULTDELAY);
	return ddc_ops->GetSDA();
}

/*----------------------------------------------------------------------------
 * WaitSCLHigh.
 *
 * Description		: This function provides the conditional delay till SCL clock
 *                    goes high.
 *
 * Parameters       :
 *       data       : None.
 * Returns			: None
 *
 * Comments         : Date will be send by bit by bit.
 *
*----------------------------------------------------------------------------
*/
void WaitSCLHigh()
{
	unsigned long target=0;

	RETAILMSG( 0,(TEXT("WaitSCLHigh\n")));
	ddc_ops->SetSCL();
	/* slow down trasfer  */
#ifndef _WIN32_WCE
	/* introduce delay for SetSCL to work. This is not required in the
	* else part, since GetTickCount() function call adds the delay 
	*/
	wait(DEFAULTDELAY); 

	for (;;) {
		if (ddc_ops->GetSCL() == 1) {
			break;
		} 
          /* else if (GetTickCount() > target) {
			RETAILMSG( 1,(TEXT("timeout")));
		}*/
		RETAILMSG( 0,(TEXT("WaitSCLHigh - Delay 2 u Sec\n")));
	}
#else
	target = GetTickCount() + 2;

	for (;;) {
		if (ddc_ops->GetSCL() == 1) {
			break;
		} else if (GetTickCount() > target) {
			RETAILMSG( 1,(TEXT("timeout")));
			return;
		}
		RETAILMSG( 0,(TEXT("WaitSCLHigh - Delay 2 u Sec\n")));
	}
#endif
}

/*----------------------------------------------------------------------------
 * SendByte.
 *
 * Description		: This function is send one byte of data
 *
 *
 * Parameters       :
 *       data       : Specifies the data to be send.
 * Returns			: None
 *
 * Comments         : Date will be send by bit by bit.
 *
*----------------------------------------------------------------------------
*/
void SendByte(unsigned char data) 
{
	int i;
	RETAILMSG( 0,(TEXT("sendByte Enter\n")));
	for (i = 7; i >= 0; i--) {
		SendBit((data >> i) & 1);
	}

	if (! ddc_ops->waitForAck()) {
		RETAILMSG( 1,(TEXT("NACK received\n")));
	}
	ddc_ops->ClearSCL();
	RETAILMSG( 0,(TEXT("sendByte Exit\n")));
}


/*----------------------------------------------------------------------------
 * SendBit.
 *
 * Description		: This function is send one bit
 *
 *
 * Parameters       :
 *       flag       : Specifies masked data bit by a flag High or Low.
 * Returns			: None
 *
 * Comments         : None.
 *
*----------------------------------------------------------------------------
*/
void SendBit(int flag) 
{
	RETAILMSG( 0,(TEXT("sendBit\n")));
	wait(DEFAULTDELAY);
	ddc_ops->ClearSCL();
	wait(DEFAULTDELAY);
	if (flag) {
		ddc_ops->SetSDA();
	} else {
		ddc_ops->ClearSDA();
	}
	wait(DEFAULTDELAY);
	WaitSCLHigh();
}

