% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bb.R
\name{bb}
\alias{bb}
\title{Bounding box generator}
\usage{
bb(
  x = NA,
  ext = NULL,
  cx = NULL,
  cy = NULL,
  width = NULL,
  height = NULL,
  xlim = NULL,
  ylim = NULL,
  relative = FALSE,
  asp.target = NULL,
  asp.limit = NULL,
  current.projection = NULL,
  projection = NULL,
  output = c("bbox", "matrix", "extent")
)
}
\arguments{
\item{x}{One of the following:
\itemize{
\item A shape from class \code{\link[sf:sf]{sf}}, \code{\link[stars:st_as_stars]{stars}}, \code{sp}, or \code{raster}.
\item A bounding box (\code{\link[sf:st_bbox]{st_bbox}}, \code{Extent} (\code{raster} package, which will no longer be supported in the future versions), numeric vector of 4 (default order: xmin, ymin, xmax, ymax), or a 2x2 matrix).
\item Open Street Map search query. The bounding is automatically generated by querying \code{x} from Open Street Map Nominatim. See \code{\link{geocode_OSM}}}.}

\item{ext}{Extension factor of the bounding box. If 1, the bounding box is unchanged. Values smaller than 1 reduces the bounding box, and values larger than 1 enlarges the bounding box. This argument is a shortcut for both \code{width} and \code{height} with \code{relative=TRUE}. If a negative value is specified, then the shortest side of the bounding box (so width or height) is extended with \code{ext}, and the longest side is extended with the same absolute value. This is especially useful for bounding boxes with very low or high aspect ratios.}

\item{cx}{center x coordinate}

\item{cy}{center y coordinate}

\item{width}{width of the bounding box. These are either absolute or relative (depending on the argument \code{relative}).}

\item{height}{height of the bounding box. These are either absolute or relative (depending on the argument \code{relative}).}

\item{xlim}{limits of the x-axis. These are either absolute or relative (depending on the argument \code{relative}).}

\item{ylim}{limits of the y-axis. See \code{xlim}.}

\item{relative}{boolean that determines whether relative values are used for \code{width}, \code{height}, \code{xlim} and \code{ylim} or absolute. If \code{x} is unspecified, \code{relative} is set to \code{"FALSE"}.}

\item{asp.target}{target aspect ratio, which is width/height, of the returned bounding box.}

\item{asp.limit}{maximum aspect ratio, which is width/height. Number greater than or equal to 1. For landscape bounding boxes, \code{1/asp.limit} will be used. The returned bounding box will have an aspect ratio between \code{1/asp.limit} and \code{asp.limit}.}

\item{current.projection}{projection that corresponds to the bounding box specified by \code{x}.}

\item{projection}{projection to transform the bounding box to.}

\item{output}{output format of the bounding box, one of:
\itemize{
\item \code{"bbox"} a \code{sf::bbox} object, which is a numeric vector of 4: xmin, ymin, xmax, ymax. This representation used by the \code{\link[sf:sf]{sf}} package.
\item \code{"matrix"} a 2 by 2 numeric matrix, where the rows correspond to x and y, and the columns to min and max. This representation used by the \code{sp} package.
\item \code{"extent"} an \code{raster::extent} object, which is a numeric vector of 4: xmin, xmax, ymin, ymax. This representation used by the \code{raster} package.
}}
}
\value{
bounding box (see argument \code{output})
}
\description{
Swiss army knife for bounding boxes. Modify an existing bounding box or create a new bounding box from scratch. See details.
}
\details{
An existing bounding box (defined by \code{x}) can be modified as follows:
\itemize{
\item Using the extension factor \code{ext}.
\item Changing the width and height with \code{width} and \code{height}. The argument \code{relavitve} determines whether relative or absolute values are used.
\item Setting the x and y limits. The argument \code{relavitve} determines whether relative or absolute values are used.}

A new bounding box can be created from scratch as follows:
\itemize{
\item Using the extension factor \code{ext}.
\item Setting the center coorinates \code{cx} and \code{cy}, together with the \code{width} and \code{height}.
\item Setting the x and y limits \code{xlim} and \code{ylim}
}
}
\examples{
if (require(tmap)) {

    ## load shapes
    data(NLD_muni)
    data(World)

    ## get bounding box (similar to sp's function bbox)
    bb(NLD_muni)

    ## extent it by factor 1.10
    bb(NLD_muni, ext=1.10)

    ## double the width
    bb(NLD_muni, width=2, relative = TRUE)

    ## crop both dimensions from 0.25 to 0.75
    bb(NLD_muni, xlim=c(.25, .75), ylim=c(.25, .75), relative = TRUE)

    ## extent it such that aspect ratio is 1
    bb(NLD_muni, asp.target = 1)

    ## convert to longlat (EPSG 4326)
    bb(NLD_muni, projection=4326)
}

\dontrun{
if (require(tmap)) {
    bb("Limburg", projection = 28992)
    bb_italy <- bb("Italy", projection = "+proj=eck4")

    tm_shape(World, bbox=bb_italy) + tm_polygons()
    # shorter alternative: tm_shape(World, bbox="Italy") + tm_polygons()
}}
}
\seealso{
\code{\link{geocode_OSM}}
}
