#' @title Inverse Gaussian Distribution Functions
#' @name inverse_gaussian_distribution
#' @description Functions to compute the probability density function, cumulative distribution function, and quantile function for the Inverse Gaussian distribution.
#' @param x quantile
#' @param mu mean parameter (mu > 0; default is 1)
#' @param lambda scale parameter (lambda > 0; default is 1)
#' @param p probability (0 <= p <= 1)
#' @return A single numeric value with the computed probability density, log-probability density, cumulative distribution, log-cumulative distribution, or quantile depending on the function called.
#' @seealso [Boost Documentation](https://www.boost.org/doc/libs/latest/libs/math/doc/html/math_toolkit/dist_ref/dists/inverse_gaussian_dist.html) for more details on the mathematical background.
#' @examples
#' # Inverse Gaussian distribution with mu = 3, lambda = 4
#' dist <- inverse_gaussian_distribution(3, 4)
#' # Apply generic functions
#' cdf(dist, 0.5)
#' logcdf(dist, 0.5)
#' pdf(dist, 0.5)
#' logpdf(dist, 0.5)
#' hazard(dist, 0.5)
#' chf(dist, 0.5)
#' mean(dist)
#' median(dist)
#' mode(dist)
#' range(dist)
#' quantile(dist, 0.2)
#' standard_deviation(dist)
#' support(dist)
#' variance(dist)
#' skewness(dist)
#' kurtosis(dist)
#' kurtosis_excess(dist)
#'
#' # Convenience functions
#' inverse_gaussian_pdf(2, 3, 4)
#' inverse_gaussian_lpdf(2, 3, 4)
#' inverse_gaussian_cdf(2, 3, 4)
#' inverse_gaussian_lcdf(2, 3, 4)
#' inverse_gaussian_quantile(0.5, 3, 4)
NULL

#' @rdname inverse_gaussian_distribution
#' @export
inverse_gaussian_distribution <- function(mu = 1, lambda = 1) {
  structure(
    list(
      extptr = .Call(`inverse_gaussian_init_`, mu, lambda),
      mean = mu,
      scale = lambda,
      shape = lambda / mu
    ),
    class = c("inverse_gaussian_distribution", "boost_distribution")
  )
}

#' @rdname inverse_gaussian_distribution
#' @export
inverse_gaussian_pdf <- function(x, mu = 1, lambda = 1) {
  pdf(inverse_gaussian_distribution(mu, lambda), x)
}

#' @rdname inverse_gaussian_distribution
#' @export
inverse_gaussian_lpdf <- function(x, mu = 1, lambda = 1) {
  logpdf(inverse_gaussian_distribution(mu, lambda), x)
}

#' @rdname inverse_gaussian_distribution
#' @export
inverse_gaussian_cdf <- function(x, mu = 1, lambda = 1) {
  cdf(inverse_gaussian_distribution(mu, lambda), x)
}

#' @rdname inverse_gaussian_distribution
#' @export
inverse_gaussian_lcdf <- function(x, mu = 1, lambda = 1) {
  logcdf(inverse_gaussian_distribution(mu, lambda), x)
}

#' @rdname inverse_gaussian_distribution
#' @export
inverse_gaussian_quantile <- function(p, mu = 1, lambda = 1) {
  quantile(inverse_gaussian_distribution(mu, lambda), p)
}
