% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/expAv.R
\name{expAv}
\alias{expAv}
\title{Matrix exponential of sparse matrix multiplied by a vector.}
\usage{
expAv(
  A,
  v,
  transpose = FALSE,
  uniformization = TRUE,
  tol = 1e-08,
  ...,
  cache = A
)
}
\arguments{
\item{A}{Sparse matrix (usually a generator)}

\item{v}{Vector (or matrix)}

\item{transpose}{Calculate \code{expm(t(A)) \%*\% v} ? (faster due to the way sparse matrices are stored)}

\item{uniformization}{Use uniformization method?}

\item{tol}{Accuracy if A is a generator matrix and v a probability vector.}

\item{...}{Extra configuration parameters}

\item{cache}{Re-use internal AD calculations by setting an attribute on this object (\code{A} by default - use NULL to disable caching).}
}
\value{
Vector (or matrix)
}
\description{
Calculates \code{expm(A) \%*\% v} using plain series summation. The number of terms is determined adaptively when \code{uniformization=TRUE}.
The uniformization method essentially pushes the spectrum of the operator inside a zero centered disc, within which a tight uniform error bound is available. This effectively reduces the number of terms needed to calculate the series to a given accuracy.
If \code{A} is a generator matrix (i.e. \code{expm(A)} is a probability matrix) and if \code{v} is a probability vector, then the relative error of the result is bounded by \code{tol}.
However, note that series summation may be unstable depending on the spectral radius of A. If you get \code{NaN} values, consider setting \code{rescale_freq=1} for better stability (see details).
}
\details{
Additional supported arguments via \code{...} currently include:
\itemize{
\item \code{Nmax} Integer (\code{2e9} by default). Use no more than this number of terms even if the specified accuracy cannot be met. When using \code{expAv} as part of likelihood optimization, you can set a lower value to avoid long unnecessary computation when the optimizer tries extreme parameters. For example, if the spectral radius of \code{A} cannot realistically exceed some known value \code{rhomax} one can set \code{Nmax=qpois(tol, rhomax, lower.tail = FALSE)}.
\item \code{warn} Logical (\code{TRUE} by default). Give warning if number of terms is truncated by \code{Nmax} (autodiff code only).
\item \code{trace} Logical (\code{FALSE} by default). Trace the number of terms when it adaptively changes (autodiff code only).
\item \code{rescale_freq} Integer (\code{50} by default) controlling how often to rescale for numerical stability. Set to a lower value for more frequent rescaling at the cost of longer computation time. The default value should suffice for a generator matrix of spectral radius up to at least \code{1e6} (\code{.Machine$double.xmax^(1/50)}).
\item \code{rescale} Logical; Set to \code{FALSE} to disable rescaling for higher speed. All other values are ignored.
}
}
\examples{
set.seed(1); A <- Matrix::rsparsematrix(5, 5, .5)
expAv(A, 1:5) ## Matrix::expm(A) \%*\% 1:5
F <- MakeTape(function(x) expAv(A*x, 1:5, trace=TRUE), 1)
F(1)
F(2) ## More terms needed => trigger retaping
}
\references{
Grassmann, W. K. (1977). Transient solutions in Markovian queueing systems. \emph{Computers & Operations Research}, 4(1), 47--53.

Sherlock, C. (2021). Direct statistical inference for finite Markov jump processes via the matrix exponential. \emph{Computational Statistics}, 36(4), 2863--2887.
}
