% Generated by roxygen2: do not edit by hand
% Please edit documentation in
%   R/gemIntertemporal_EndogenousEquilibriumInterestRate.R
\name{gemIntertemporal_EndogenousEquilibriumInterestRate}
\alias{gemIntertemporal_EndogenousEquilibriumInterestRate}
\title{Some Examples Illustrating Endogenous Equilibrium Interest Rates in (Timeline) Transitional Equilibrium Paths}
\usage{
gemIntertemporal_EndogenousEquilibriumInterestRate(...)
}
\arguments{
\item{...}{arguments to be passed to the function sdm2.}
}
\description{
These examples illustrate (endogenous) equilibrium primitive interest rates in a transitional equilibrium path,
which is an intertemporal path distinct from a steady-state equilibrium.
Assume that the velocity of money is equal to one, that is, money circulates once per period.

The interest rate calculated here is adjusted from the nominal interest rate based on the growth rate of the money supply,
which is equal to the nominal interest rate when the money stock remains unchanged.
We refer to this kind of interest rate as the primitive interest rate,
which usually differs from the real interest rate obtained by adjusting the nominal rate based on the inflation rate.
}
\examples{
\donttest{
#### There are two types of economic agents in this example: firms and a consumer.
# Suppose the consumer needs to use money to buy products,
# and firms need to use money to buy labor.
set.seed(123)
eis <- 1 # the elasticity of intertemporal substitution
np <- 3 # the number of economic periods
alpha <- runif(np, 1, 3)
beta <- runif(np, 0.9, 1) |>
  cumprod() |>
  proportions()

f <- function(ir = rep(0.1, np), return.ge = FALSE) {
  ir[np] <- 1e6

  n <- 3 * np # the number of commodity kinds
  m <- np + 1 # the number of agent kinds

  names.commodity <- c(
    paste0("prod", 1:np),
    paste0("lab", 1:np),
    paste0("money", 1:np)
  )

  names.agent <- c(
    paste0("firm", 1:np),
    "consumer"
  )

  # the exogenous supply matrix.
  S0Exg <- matrix(NA, n, m, dimnames = list(names.commodity, names.agent))
  S0Exg[paste0("lab", 1:np), "consumer"] <- 100
  S0Exg[paste0("money", 1:np), "consumer"] <- 1

  # the output coefficient matrix.
  B <- matrix(0, n, m, dimnames = list(names.commodity, names.agent))
  for (k in 1:np) {
    B[paste0("prod", k), paste0("firm", k)] <- 1
  }

  dstl.firm <- list()
  for (k in 1:np) {
    dstl.firm[[k]] <- node_new(
      "prod",
      type = "FIN", rate = c(1, ir[k]),
      "cc1", paste0("money", k)
    )
    node_set(dstl.firm[[k]], "cc1",
             type = "Leontief", a = 1 / alpha[k],
             paste0("lab", k)
    )
  }

  dst.consumer <- node_new(
    "util",
    type = "SCES", es = eis,
    alpha = 1, beta = beta,
    paste0("cc", 1:np)
  )
  for (k in 1:np) {
    node_set(dst.consumer, paste0("cc", k),
             type = "FIN", rate = c(1, ir[k]),
             paste0("prod", k), paste0("money", k)
    )
  }

  ge <- sdm2(
    A = c(
      dstl.firm, dst.consumer
    ),
    B = B,
    S0Exg = S0Exg,
    names.commodity = names.commodity,
    names.agent = names.agent,
    numeraire = "prod1",
  )

  tmp <- rowSums(ge$SV)
  ts.trading.value <- tmp[paste0("prod", 1:np)] + tmp[paste0("lab", 1:np)] +
    tmp[paste0("money", 1:np)]
  ir.new <- ts.trading.value[1:(np - 1)] / ts.trading.value[2:np] - 1
  ir.new <- pmax(1e-6, ir.new)

  ir.new[np] <- 1e6
  ir <- ir.new

  cat("ir: ", ir, "\n")

  if (return.ge) {
    ge$ts.trading.value <- unname(ts.trading.value)
    return(ge)
  } else {
    return(ir)
  }
}

mat.ir <- iterate(rep(0.1, np), f, tol = 1e-3)

# When eis equals 1 and np equals 3, compute the
# interest rates using the closed-form formulas.
compute_ir <- function(beta) {
  b1 <- beta[1]
  b2 <- beta[2]
  b3 <- beta[3]

  A <- sqrt(b2^2 + 4 * b2 * b3)
  B <- sqrt(b1^2 + 2 * b1 * (b2 + A))

  r1 <- (b1 - b2 - A + B) / (b2 + A)
  r2 <- (b2 - 2 * b3 + A) / (2 * b3)

  c(r1 = r1, r2 = r2)
}
compute_ir(beta)

ge <- f(tail(mat.ir, 1), return.ge = TRUE)
ge$p

#### There are three types of economic agents in this example: firms, a laborer, and a money owner.
# Suppose the laborer and the money owner need to use money to buy products,
# and firms need to use money to buy products and labor.
# Formally, the money owner borrows money from himself and pays interest to himself.
eis <- 0.8 # the elasticity of intertemporal substitution
Gamma.beta <- 0.8 # the subjective discount factor
gr <- 0 # the steady-state growth rate
np <- 20 # the number of economic periods

f <- function(ir = rep(0.25, np - 1), return.ge = FALSE,
              y1 = 20, # the product supply in the first period
              alpha.firm = rep(2, np - 1) # the efficiency parameters of firms
) {
  n <- 2 * np # the number of commodity kinds
  m <- np + 1 # the number of agent kinds

  names.commodity <- c(
    paste0("prod", 1:np),
    paste0("lab", 1:(np - 1)),
    "money"
  )
  names.agent <- c(
    paste0("firm", 1:(np - 1)),
    "laborer", "moneyOwner"
  )

  # the exogenous supply matrix.
  S0Exg <- matrix(NA, n, m, dimnames = list(names.commodity, names.agent))
  S0Exg[paste0("lab", 1:(np - 1)), "laborer"] <- 100 * (1 + gr)^(0:(np - 2))
  S0Exg["money", "moneyOwner"] <- 100
  S0Exg["prod1", "laborer"] <- y1

  # the output coefficient matrix.
  B <- matrix(0, n, m, dimnames = list(names.commodity, names.agent))
  for (k in 1:(np - 1)) {
    B[paste0("prod", k + 1), paste0("firm", k)] <- 1
  }

  dstl.firm <- list()
  for (k in 1:(np - 1)) {
    dstl.firm[[k]] <- node_new(
      "prod",
      type = "FIN", rate = c(1, ir[k]),
      "cc1", "money"
    )
    node_set(dstl.firm[[k]], "cc1",
             type = "CD", alpha = alpha.firm[k], beta = c(0.5, 0.5),
             paste0("prod", k), paste0("lab", k)
    )
  }

  dst.laborer <- node_new(
    "util",
    type = "CES", es = eis,
    alpha = 1, beta = prop.table(Gamma.beta^(1:np)),
    paste0("cc", 1:(np - 1)), paste0("prod", np)
  )

  for (k in 1:(np - 1)) {
    node_set(dst.laborer, paste0("cc", k),
             type = "FIN", rate = c(1, ir[k]),
             paste0("prod", k), "money"
    )
  }

  dst.moneyOwner <- node_new(
    "util",
    type = "CES", es = eis,
    alpha = 1, beta = prop.table(Gamma.beta^(1:(np - 1))),
    paste0("cc", 1:(np - 1))
  )
  for (k in 1:(np - 1)) {
    node_set(dst.moneyOwner, paste0("cc", k),
             type = "FIN", rate = c(1, ir[k]),
             paste0("prod", k), "money"
    )
  }

  ge <- sdm2(
    A = c(dstl.firm, dst.laborer, dst.moneyOwner),
    B = B,
    S0Exg = S0Exg,
    names.commodity = names.commodity,
    names.agent = names.agent,
    numeraire = "prod1",
    policy = makePolicyHeadTailAdjustment(gr = gr, np = np, type = c("tail"))
  )

  tmp <- rowSums(ge$SV)
  ts.trading.value <- (tmp[paste0("prod", 1:(np - 1))] + tmp[paste0("lab", 1:(np - 1))]) * (1 + ir)
  ir.new <- ts.trading.value[1:(np - 2)] / ts.trading.value[2:(np - 1)] - 1
  ir.new <- pmax(1e-6, ir.new)
  ir.new[np - 1] <- ir.new[np - 2]

  ir <- ir.new
  cat("ir: ", ir, "\n")

  if (return.ge) {
    ge$ts.trading.value <- ts.trading.value
    return(ge)
  } else {
    return(ir)
  }
}

## Calculate equilibrium interest rates.
## Warning: Running the program below may take several minutes.
# mat.ir <- iterate(rep(0.1, np - 1), f, tol = 1e-4)
# sserr(eis, Gamma.beta, gr, prepaid = TRUE)

## Below are the calculated equilibrium interest rates.
ir <- c(0.4297, 0.3449, 0.3014, 0.2782, 0.2656, 0.2587, 0.2548, 0.2527,
        0.2515, 0.2508, 0.2505, 0.2503, 0.2501, 0.2501, 0.2500, 0.2500,
        0.2500, 0.2500, 0.2500)

ge <- f(ir, TRUE)

plot(ge$z[1:(np - 1)], type = "o")
ge$ts.trading.value[1:(np - 2)] / ge$ts.trading.value[2:(np - 1)] - 1
ir
}
}
