package progressbar;

import java.util.Comparator;
import java.util.Date;
import java.util.TreeSet;

/**
 * This class is a simple implementation of the Java 1.3 java.util.Timer API
 */

public class Timer {
    // This sorted set stores the tasks that this Timer is responsible for.
    // It uses a comparator (defined below) to sort the task by execution time.
    TreeSet tasks = new TreeSet(new TimerTaskComparator());

    // This is the thread the timer uses to execute the tasks
    TimerThread timer;

    /** This constructor creates a Timer that does not use a daemon thread */
    public Timer() { this(false); }

    /** The main constructor: the internal thread is a daemon if specified */
    public Timer(boolean isDaemon) {
	timer = new TimerThread(isDaemon);
	timer.start();
    }

    /** Stop the timer thread, and discard all scheduled tasks */
    public void cancel() {
	synchronized(tasks) {
	    timer.pleaseStop();	// set a flag asking the thread to stop
	    tasks.clear();	// Discard all tasks
	    tasks.notify();	// Wake up the thread if it is in wait()
	}
    }
			      
    /** Schedule a single execution after delay milliseconds */
    public void schedule(TimerTask task, long delay) {
	task.schedule(System.currentTimeMillis() + delay, 0, false);
	schedule(task);
    }

    /** Schedule a single execution at the specified time */
    public void schedule(TimerTask task, Date time) {
	task.schedule(time.getTime(), 0, false);
	schedule(task);
    }

    /** Schedule a periodic execution starting at the specified time */
    public void schedule(TimerTask task, Date firstTime, long period) {
	task.schedule(firstTime.getTime(), period, false);
	schedule(task);
    }

    /** Schedule a periodic execution starting after the specified delay */
    public void schedule(TimerTask task, long delay, long period) {
	task.schedule(System.currentTimeMillis() + delay, period, false);
	schedule(task);
    }

    /** 
     * Schedule a periodic execution starting after the specified delay.
     * Schedule fixed-rate executions period ms after the start of the last.
     * Instead of fixed-interval executions measured from the end of the las.
     */
    public void scheduleAtFixedRate(TimerTask task, long delay, long period) {
	task.schedule(System.currentTimeMillis() + delay, period, true);
	schedule(task);
    }

    /** Schedule a periodic execution starting after the specified time */
    public void scheduleAtFixedRate(TimerTask task, Date firstTime, 
				    long period) {
	task.schedule(firstTime.getTime(), period, true);
	schedule(task);
    }

    // This internal method adds a task to the sorted set of tasks
    void schedule(TimerTask task) {
	synchronized(tasks) {
	    tasks.add(task);
	    tasks.notify();	// wake up the thread if it is waiting
	}
    }

    /**
     * This inner class is used to sort tasks by next execution time.
     */
    static class TimerTaskComparator implements Comparator {
	public int compare(Object a, Object b) {
	    TimerTask t1 = (TimerTask)a;
	    TimerTask t2 = (TimerTask)b;
	    long diff = t1.nextTime - t2.nextTime;
	    if (diff < 0)
		return -1;
	    else if (diff > 0)
		return 1;
	    else
		return 0;
	}

	public boolean equals(Object o) {
	    return this == o;
	}
    }

    /**
     * This inner class defines the thread that runs each of the tasks at
     * their scheduled times
     */
    class TimerThread extends Thread {
	// This flag will be set to true to tell the thread to stop 
	// running. Note that it is declared as volatile, which means
	// that it may be changed asynchronously by another thread, so
	// threads must always read its true value, and not use a cached
	// version.
	volatile boolean stopped = false;

	// the constructor
	public TimerThread(boolean isDaemon) {setDaemon(isDaemon);}

	// Ask the thread to stop by setting the flag
	public void pleaseStop() { stopped = true; }

	// This is the body of the thread
	public void run() {
	    TimerTask readyToRun = null; // Is there a task to run right now?
	    
	    // The thread loops until the stopped flag is set to true
	    while (!stopped) {
		// If there is a task that is ready to run, then run it
		if (readyToRun != null) {
		    if (readyToRun.cancelled) {
			readyToRun = null;
			continue;
		    }
		    // Run the task.
		    readyToRun.run();
		    // Ask it to reschedule itself, and if it wants to run
		    // again, then insert it back into the set of tasks.
		    if (readyToRun.reschedule())
			schedule(readyToRun);
		    // We've run it, so there is nothing to run now
		    readyToRun = null;
		    // Go back to the top of the loop to see if we've
		    // been stopped
		    continue;
		}

		// Now acquire a lock on the set of tasks
		synchronized(tasks) {
		    long timeout; // how many ms 'till the next execution?
		    
		    if (tasks.isEmpty()) {
			timeout = 0;
		    }
		    else {
			// If there are scheduled tasks, then get the first one
			// Since the set is sorted, this is the next one.
			TimerTask t = (TimerTask)tasks.first();
			timeout = t.nextTime - System.currentTimeMillis();
			// Check whether it needs to run now
			if (timeout <= 0) {
			    readyToRun = t;
			    tasks.remove(t);
			    continue;
			}
		    }

		    // If we get here, there is nothing ready to run now
		    // so wait for time to run out, or wait 'till notify is
		    // called when somethind new is added to the set of 
		    // tasks.
		    try {
			tasks.wait(timeout);
		    }
		    catch (InterruptedException e) {}
		}
	    }
	}
    }
}
