/* vim: set sw=8: */

/*
 * dependent.c: 
 *
 * Copyright (C) 2000 Jody Goldberg (jgoldberg@home.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */
#include "config.h"
#include "dependent.h"
#include "workbook.h"
#include "sheet.h"
#include "cell.h"
#include "eval.h"
#include "parse-util.h"

/**
 * dependent_changed:
 * @cell : the dependent that changed
 * @queue_recalc: also queue a recalc for the dependent.
 *
 * Registers the expression with the sheet and optionally queues a recalc.
 */
void
dependent_changed (Dependent *dep, CellPos const *pos, gboolean queue_recalc)
{
	dependent_link (dep, pos);
	if (queue_recalc)
		dependent_queue_recalc (dep);
}

/**
 * dependent_link:
 * @dep : the dependent that changed
 * @pos: The optionally NULL position of the dependent.
 *
 * Adds the dependent to the workbook wide list of dependents.
 */
void
dependent_link (Dependent *dep, const CellPos *pos)
{
	Workbook *wb;

	g_return_if_fail (dep != NULL);
	g_return_if_fail (dep->expression != NULL);
	g_return_if_fail (dep->sheet != NULL);
	g_return_if_fail (!(dep->flags & DEPENDENT_IN_EXPR_LIST));

	wb = dep->sheet->workbook;

#if 0
	if (g_list_find (wb->dependents, dep)) {
		/* Anything that shows here is a bug.  */
		dependent_debug_name (dep, stderr);
		g_warning ("Doubly linked dependent");
		return;
	}
#endif

	wb->dependents = g_list_prepend (wb->dependents, dep);

	dependent_add_dependencies (dep, pos);

	dep->flags |= DEPENDENT_IN_EXPR_LIST;
}

/**
 * dependent_unlink:
 * @dep : the dependent that changed
 * @pos: The optionally NULL position of the dependent.
 *
 * Removes the dependent from the workbook wide list of dependents.
 */
void
dependent_unlink (Dependent *dep, CellPos const *pos)
{
	Workbook *wb;

	g_return_if_fail (dep != NULL);

	if (dep->sheet == NULL)
		return;

	g_return_if_fail (dep->expression != NULL);
	g_return_if_fail (dep->flags & DEPENDENT_IN_EXPR_LIST);

	dep->flags &= ~DEPENDENT_IN_EXPR_LIST;

	wb = dep->sheet->workbook;

	dependent_drop_dependencies (dep, pos);
	wb->dependents = g_list_remove (wb->dependents, dep);

	/* An optimization to avoid an expensive list lookup */
	if (dep->flags & DEPENDENT_QUEUED_FOR_RECALC)
		dependent_unqueue_recalc (dep);
}

/**
 * dependent_unlink_sheet :
 * @sheet :
 *
 * An internal routine to remove all expressions associated with a given sheet
 * from the workbook wide expression list.  WARNING : This is a dangerous
 * internal function.  it leaves the cells in an invalid state.  It is intended
 * for use by sheet_destroy_contents.
 */
void
dependent_unlink_sheet (Sheet const *sheet)
{
	GList *ptr, *next, *queue;
	Workbook *wb;

	g_return_if_fail (sheet != NULL);
	g_return_if_fail (IS_SHEET (sheet));

	wb = sheet->workbook;
	queue = wb->dependents;
	for (ptr = queue; ptr != NULL ; ptr = next) {
		Dependent *dep = ptr->data;
		next = ptr->next;

		if (dep->sheet == sheet) {
			dep->flags &= ~DEPENDENT_IN_EXPR_LIST;
			queue = g_list_remove_link (queue, ptr);
			g_list_free_1 (ptr);
		}
	}
	wb->dependents = queue;
}

/*
 * dependent_queue_recalc:
 * @dep: the dependent that contains the expression needing recomputation.
 *
 * Queues the dependent @dep for recalculation.
 */
void
dependent_queue_recalc (Dependent *dep)
{
	Workbook *wb;

	g_return_if_fail (dep != NULL);

	if (dep->flags & DEPENDENT_QUEUED_FOR_RECALC)
		return;

#ifdef DEBUG_EVALUATION
	if (dependency_debugging > 2) {
		printf ("Queuing: ");
		dependent_debug_name (dep, stdout);
		puts ("");
	}
#endif
	wb = dep->sheet->workbook;
	wb->eval_queue = g_list_prepend (wb->eval_queue, dep);
	dep->flags |= DEPENDENT_QUEUED_FOR_RECALC;
}

void
dependent_queue_recalc_list (GList *list, gboolean freelist)
{
	Workbook *wb;
	GList *ptr = list;

	while (ptr) {
		Dependent *dep = ptr->data;
		ptr = ptr->next;

		if (dep->flags & DEPENDENT_QUEUED_FOR_RECALC)
			continue;

		g_return_if_fail (dep->sheet != NULL);

		/*
		 * Use the wb associated with the current dependent in case we
		 * have cross workbook depends
		 */
		wb = dep->sheet->workbook;
		wb->eval_queue = g_list_prepend (wb->eval_queue, dep);

		dep->flags |= DEPENDENT_QUEUED_FOR_RECALC;
	}

	if (freelist)
		g_list_free (list);
}

/*
 * dependent_unqueue_recalc:
 * @dep: the dependent to remove from the recomputation queue
 *
 * Removes a dependent that has been previously added to the recalc
 * queue.  Used internally when a dependent that was queued is changed or
 * removed.
 */
void
dependent_unqueue_recalc (Dependent *dep)
{
	Workbook *wb;

	g_return_if_fail (dep != NULL);

	if (!(dep->flags & DEPENDENT_QUEUED_FOR_RECALC))
		return;

	wb = dep->sheet->workbook;
	wb->eval_queue = g_list_remove (wb->eval_queue, dep);
	dep->flags &= ~DEPENDENT_QUEUED_FOR_RECALC;
}

/*
 * eval_unqueue_recalc_sheet:
 * @sheet : the sheet whose cells need to be unqueued.
 *
 * Remove all cells from the specified sheet from the recalc queue.
 */
void
dependent_unqueue_recalc_sheet (Sheet *sheet)
{
	GList *ptr, *next, *queue;
	Workbook *wb;

	g_return_if_fail (sheet != NULL);
	g_return_if_fail (IS_SHEET (sheet));

	wb = sheet->workbook;
	queue = wb->eval_queue;
	for (ptr = queue; ptr != NULL ; ptr = next) {
		Dependent *dep = ptr->data;
		next = ptr->next;

		if (dep->sheet == sheet) {
			dep->flags &= ~DEPENDENT_QUEUED_FOR_RECALC;
			queue = g_list_remove_link (queue, ptr);
			g_list_free_1 (ptr);
		}
	}
	wb->eval_queue = queue;
}

static GPtrArray *dep_classes = NULL;

/**
 * dependent_register_type :
 * @klass : A vtable
 *
 * Store the vtable and allocate an ID for a new class
 * of dependents.
 */
guint32
dependent_type_register (DependentClass const *klass)
{
	guint32 res;

	if (dep_classes == NULL) {
		/* Init with a pair of NULL classes so we can access directly */
		dep_classes = g_ptr_array_new ();
		g_ptr_array_add	(dep_classes, NULL);
		g_ptr_array_add	(dep_classes, NULL);
	}
	g_ptr_array_add	(dep_classes, (gpointer)klass);
	res = dep_classes->len-1;

	g_return_val_if_fail (res <= DEPENDENT_TYPE_MASK, res);

	return res;
}

/**
 * dependent_eval :
 * @dep : The dependent we are interested in.
 *
 * When the expression needs recalculation
 * this routine dispatches to the virtual handler.
 */
void
dependent_eval (Dependent *dep)
{
	if (dep->generation != dep->sheet->workbook->generation) {
		int const t = (dep->flags & DEPENDENT_TYPE_MASK);
		dep->generation = dep->sheet->workbook->generation;

		if (t == DEPENDENT_CELL) {
			cell_eval (DEP_TO_CELL (dep));
		} else {
			DependentClass *klass = g_ptr_array_index (dep_classes, t);

			g_return_if_fail (klass);
			(*klass->eval) (dep);
		}
	}
}

/**
 * dependent_set_expr :
 * @dep : The dependent we are interested in.
 * @expr : new expression.
 *
 * When the expression associated with a dependent needs to change
 * this routine dispatches to the virtual handler.
 */
void
dependent_set_expr (Dependent *dep, ExprTree *expr)
{
	int const t = (dep->flags & DEPENDENT_TYPE_MASK);

	if (t == DEPENDENT_CELL) {
		/*
		 * Explicitly do not check for array subdivision, we may be
		 * replacing the corner of an array.
		 */
		cell_set_expr_unsafe (DEP_TO_CELL (dep), expr, NULL);
	} else {
		DependentClass *klass = g_ptr_array_index (dep_classes, t);

		g_return_if_fail (klass);
		(*klass->set_expr) (dep, expr);
	}
}

/**
 * dependent_debug_name :
 * @dep : The dependent we are interested in.
 * @file : FILE * to print to.
 *
 * A useful little debugging utility. 
 */
void
dependent_debug_name (Dependent const *dep, FILE *out)
{
	int t;

	g_return_if_fail (dep != NULL);
	g_return_if_fail (out != NULL);

	t = (dep->flags & DEPENDENT_TYPE_MASK);

	if (dep->sheet != NULL)
		fprintf (out, "%s!", dep->sheet->name_quoted);
	else
		g_warning ("Invalid dep, missing sheet");

	if (t == DEPENDENT_CELL) {
		fprintf (out, "%s", cell_name (DEP_TO_CELL (dep)));
	} else {
		DependentClass *klass = g_ptr_array_index (dep_classes, t);

		g_return_if_fail (klass);
		(*klass->debug_name) (dep, out);
	}
}

/**
 * dependent_list_filter :
 * @input : A GList of dependents
 * @type : The type of dependent desired.
 *
 * Create a new list containint only dependents or the desired type.
 */
GList *
dependent_list_filter (GList *input, DependentFlags type)
{
	GList *ptr, *res = NULL;

	for (ptr = input; ptr != NULL ; ptr = ptr->next) {
		Dependent *dep = ptr->data;
		if ((dep->flags & DEPENDENT_TYPE_MASK) == type)
			res = g_list_prepend (res, dep);
	}
	g_list_free (input);
	return res;
}

