/* ps.c -- routines for producing PostScript output from xdatplot	*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"
#include <pwd.h>

#define	FONT_FAMILY	"Courier"

#ifdef	__STDC__
static	void	DSC_header(FILE *, int, Bool);
static	void	PSplot_data(FILE *, Bool);
static	void	PSplot_marks(FILE *, Bool);
static	void	PSplot_axes(FILE *, Bool);
static	void	put_PS_string(FILE *, String);
#else	/* __STDC__ */
static	void	DSC_header();
static	void	PSplot_data();
static	void	PSplot_marks();
static	void	PSplot_axes();
static	void	put_PS_string();
#endif	/* __STDC__ */

String	ps_formats[] = {
    "Landscape, large",
    "Landscape, small",
    "Portrait, large",
    "Portrait, small"
};
int	num_ps_formats = XtNumber(ps_formats);
#define	LANDSCAPE_LARGE	0
#define	LANDSCAPE_SMALL	1
#define	PORTRAIT_LARGE	2
#define	PORTRAIT_SMALL	3
static	String	format_commands[] = {
    "LANDSCAPE LARGE",
    "LANDSCAPE SMALL",
    "PORTRAIT LARGE",
    "PORTRAIT SMALL"
};

static	String	prolog = "
/xdatplotDict 20 dict def
xdatplotDict begin
/PAGEWIDTH 612 def
/PAGEHEIGHT 792 def
/MARGIN 18 def

% LANDSCAPE
/LANDSCAPE {
    initgraphics
    PAGEWIDTH 0 translate
    90 rotate
    /PAGEHEIGHT PAGEWIDTH /PAGEWIDTH PAGEHEIGHT def def
} def

% PORTRAIT
/PORTRAIT
{
    initgraphics
} def

% LARGE
/LARGE {} def

% SMALL
/SMALL {
    /MARGIN PAGEWIDTH MARGIN sub 4 div MARGIN add def
} def

% x1 y1 x2 y2 PAGESETUP
% The arguments are the bounding box.
/PAGESETUP {
    10 dict begin
    /y2 exch def
    /x2 exch def
    /y1 exch def
    /x1 exch def
    PAGEWIDTH MARGIN 2 mul sub x2 x1 sub abs div
    PAGEHEIGHT MARGIN 2 mul sub y2 y1 sub abs div
    2 copy lt {				% fit in width of Landscape page
	pop /SF exch def
	SF dup scale
	MARGIN SF div x1 sub
	PAGEHEIGHT SF div 2 div y1 y2 add 2 div sub
	translate
    }
    {
	exch pop /SF exch def
	SF dup scale
	PAGEWIDTH SF div 2 div x1 x2 add 2 div sub
	MARGIN SF div y1 sub
	translate
    } ifelse
    0.25 setlinewidth
    1 setlinecap
    1 setlinejoin
    0 setgray
    end
} def

% ENDPAGE
/ENDPAGE {
    end
    showpage
} def

% x y w h SETCLIP
% x and y are the upper left corner of the rectangle (i.e., Xwindows
% convention, not PostScript).  All subsequent drawing (until the next
% grestore) is clipped to the specified rectangle.
/SETCLIP {
    4 dict begin
    /h exch def
    /w exch def
    /y exch def
    /x exch def
    x y moveto
    w 0 rlineto
    0 h neg rlineto
    w neg 0 rlineto
    closepath
    clip newpath
    end
} def

% ascent descent /Font-family SETFONT
/SETFONT {
    exch /DESCENT exch def
    exch /ASCENT exch def
    findfont ASCENT DESCENT add scalefont setfont
} def

% x y (text) LRalign TBalign TEXT
% LRalign is -1, 0, or 1 for left, center, or right alignment.
% TBalign is -1, 0, or 1 for bottom, center, or top alignment.
/TEXT {
    10 dict begin
    /TBalign exch def
    /LRalign exch def
    /text exch def
    /y exch def
    /x exch def
    x y moveto
    %
    % compute x adjustment
    %
    text stringwidth pop
    -2 div LRalign 1 add mul
    %
    % compute y adjustment
    %
    TBalign -1 eq {
	DESCENT
    } if
    TBalign 0 eq {
	DESCENT ASCENT sub 2 div
    } if
    TBalign 1 eq {
	ASCENT neg
    } if
    rmoveto
    text show
    end
} def

% x y w h RECTANGLE
% x and y are the upper left corner of the rectangle (i.e., Xwindows
% convention, not PostScript).
/RECTANGLE {
    4 dict begin
    /h exch def
    /w exch def
    /y exch def
    /x exch def
    x y moveto
    w 0 rlineto
    0 h neg rlineto
    w neg 0 rlineto
    closepath
    stroke
    end
} def

% x1 y1 x2 y2 LINE
/LINE {
    moveto lineto stroke
} def

% x1 y1 x2 y2 thickness THICKLINE
/THICKLINE {
    gsave
    setlinewidth
    moveto lineto stroke
    grestore
} def
";

/* PSplot -- write the current plot to the named file			*/
void
PSplot(fname, format, EPSF)
String		fname;
int		format;
Bool		EPSF;
{
    char	ebuf[LLEN];
    FILE	*os;

    if (EPSF) {
	PU_error("Encapsulated PostScript not yet available",
		 "file.html#PRINT");
	return;
    }
    if (NULL == (os = fopen(fname, "w"))) {
	sprintf(ebuf, "unable to open %s for printing", fname);
	PU_error(ebuf, "file.html#PRINT");
	return;
    }
    PSplot_stream(os, format, EPSF);
    fclose(os);
}

/* PSplot_stream -- write the current plot to the open stream		*/
void
PSplot_stream(os, format, EPSF)
FILE		*os;
int		format;
Bool		EPSF;
{
    register char	*s;

    if (EPSF) {
	PU_error("Encapsulated PostScript not yet available",
		 "file.html#POSTSCRIPT");
	return;
    }
    DSC_header(os, format, EPSF);
    fprintf(os, "%%%%BeginProlog\n");
    for(s = prolog; '\0' != *s; s++) putc(*s, os);
    fprintf(os, "%%%%EndProlog\n");
    fprintf(os, "%%%%Page: \"1\" 1\n");
    fprintf(os, "%%%%BeginPageSetup\n");
    fprintf(os, "%s\n", format_commands[format]);
    fprintf(os, "%d %d %d %d PAGESETUP\n", 0, 0, PWW, PWH);
    fprintf(os, "%%%%EndPageSetup\n");
    PSplot_data(os, EPSF);
    PSplot_marks(os, EPSF);
    PSplot_axes(os, EPSF);
    fprintf(os, "ENDPAGE\n");
    fprintf(os, "%%%%EndPage: \"1\" 0\n");
    fprintf(os, "%%%%Trailer\n");
}

static void
DSC_header(os, format, EPSF)
FILE		*os;
int		format;
Bool		EPSF;
{
    time_t		tm;
    String		s;
    struct passwd	*pswd;

    fprintf(os, "%%!PS-Adobe-3.0\n");
    fprintf(os, "%%%%BoundingBox: %d %d %d %d\n", 0, 0, PRW, PRH);
    fprintf(os, "%%%%Creator: xdatplot %s\n", VERSION);
    tm = time(NULL);
    s = ctime(&tm);
    if ('\n' == s[strlen(s) - 1]) s[strlen(s) - 1] = '\0';
    fprintf(os, "%%%%CreationDate: %s\n", s);
    pswd = getpwuid(getuid());
    fprintf(os, "%%%%For: %s \"%s\"\n", pswd->pw_name, pswd->pw_gecos);
    switch(format) {
    case LANDSCAPE_LARGE:
    case LANDSCAPE_SMALL:
	s = "Landscape";
	break;

    case PORTRAIT_LARGE:
    case PORTRAIT_SMALL:
	s = "Portrait";
	break;
	
    default:
	error("DSC_header: can't happen");
    }
    fprintf(os, "%%%%Orientation: %s\n", s);
    fprintf(os, "%%%%Pages: 1\n");
    fprintf(os, "%%%%DocumentFonts: %s\n", FONT_FAMILY);
    fprintf(os, "%%%%EndComments\n");
}

static void
PSplot_data(os, EPSF)
FILE		*os;
Bool		EPSF;
{
    register TIME	i;

    fprintf(os, "gsave\n");
    fprintf(os, "%d %d %d %d SETCLIP\n", PRX, PWH - PRY, PRW, PRH);
    for(i=app_data.tl+1; i<TR; i++) {
	fprintf(os, "%g %g %g %g LINE\n",
		(double) Map_TIME_to_Dim_c(i-1),
		(double) PWH - Map_VOLTAGE_to_Dim_c(Vf(i-1)),
		(double) Map_TIME_to_Dim_c(i),
		(double) PWH - Map_VOLTAGE_to_Dim_c(Vf(i)));
	if (debug) fflush(os);
    }
    fprintf(os, "grestore\n");
}

static void
PSplot_marks(os, EPSF)
FILE		*os;
Bool		EPSF;
{
    MARK		*mp;
    int			n;
    register int	i;
    double		x, y;
    double		st, en;

    n = lookup_interval(TL, TR, &mp);
    if (n > 0) {
	fprintf(os, "gsave\n");
	fprintf(os, "%d %d %d %d SETCLIP\n", PRX, PWH - PRY, PRW, PRH);
	fprintf(os, "%d %d /%s SETFONT\n",
		PFINFO->ascent,
		PFINFO->descent,
		FONT_FAMILY);
	st = -MARK_SIZE / 2;
	en = st + MARK_SIZE;
	for(i=0; i<n; i++) {
	    x = Map_MARK_to_Dim_c(mp[i].t);
	    y = PWH - Map_VOLTAGE_to_Dim_c(Vf(Map_MARK_to_TIME(mp[i].t)));
	    fprintf(os, "%g %g %g %g %d THICKLINE\n",
		    (double) (x + st), (double) (y + st),
		    (double) (x + en), (double) (y + en),
		    MARK_THICKNESS);
	    fprintf(os, "%g %g %g %g %d THICKLINE\n",
		    (double) (x + st), (double) (y + en),
		    (double) (x + en), (double) (y + st),
		    MARK_THICKNESS);
	    y = PWH - (PRY + 2);
	    fprintf(os, "%g %g (%c) 0 1 TEXT\n", x, y, mp[i].val);
	    if (debug) fflush(os);
	}
	fprintf(os, "grestore\n");
    }
}

static void
PSplot_axes(os, EPSF)
FILE		*os;
Bool		EPSF;
{
    register int	i;
    register double	tp;
    TICS		tics;
    double		x, y;
    double		xmin;
    char		lbuf[LLEN];

    fprintf(os, "gsave\n");
    fprintf(os, "%d %d /%s SETFONT\n",
	    PFINFO->ascent,
	    PFINFO->descent,
	    FONT_FAMILY);
    fprintf(os, "%d %d %d %d RECTANGLE\n",
	    PRX-1, PWH - (PRY-1), PRW+1, PRH+1);
    if (debug) fflush(os);
    /*
     * put tic marks on the t axis
     */
    y = PWH - (PRY + PRH + 2);
    tics.tcmin = T_TICS;
    tics.lo = Map_TIME_to_t(TL + V_TICLEN / TIME_to_Dim_sf);
    tics.hi = Map_TIME_to_t(TR - V_TICLEN / TIME_to_Dim_sf);
    find_tics(&tics);
    for(i = tics.first; i < tics.n + tics.first; i++) {
	tp = i * tics.space;
	x = Map_t_to_Dim_c(tp);
	fprintf(os, "%g %g (%g) 0 1 TEXT\n", x, y, tp);
	fprintf(os, "%g %g %g %g LINE\n",
		(double) Map_t_to_Dim_c(tp), (double) (PWH - PRY),
		(double) Map_t_to_Dim_c(tp), (double) (PWH - PRY - T_TICLEN));
	fprintf(os, "%g %g %g %g LINE\n",
		(double) Map_t_to_Dim_c(tp), (double) (PWH - PRY - PRH + 1),
		(double) Map_t_to_Dim_c(tp),
		(double) (PWH - PRY - PRH + 1 + T_TICLEN));
	if (debug) fflush(os);
    }
    /*
     * put tic marks on the v axis
     */
    tics.tcmin = V_TICS;
    tics.lo = Map_VOLTAGE_to_v(VB + T_TICLEN / VOLTAGE_to_Dim_sf);
    tics.hi = Map_VOLTAGE_to_v(VT - T_TICLEN / VOLTAGE_to_Dim_sf);
    find_tics(&tics);
    xmin = PRX;
    for(i = tics.first; i < tics.n + tics.first; i++) {
	tp = i * tics.space;
	x = PRX - PFINFO->min_bounds.width;
	sprintf(lbuf, "%g", tp);
	xmin = min(xmin, x - XTextWidth(PFINFO, lbuf, strlen(lbuf)));
	if (x < 0) x = 0;
	fprintf(os, "%g %g (%s) 1 0 TEXT\n",
		x, (double) (PWH - Map_v_to_Dim_c(tp)), lbuf);
	fprintf(os, "%g %g %g %g LINE\n",
		(double) PRX, (double) (PWH - Map_v_to_Dim_c(tp)),
		(double) (PRX + V_TICLEN), (double) (PWH - Map_v_to_Dim_c(tp)));
	fprintf(os, "%g %g %g %g LINE\n",
		(double) (PRX + PRW - 1),
		(double) (PWH - Map_v_to_Dim_c(tp)),
		(double) (PRX + PRW - 1 - V_TICLEN),
		(double) (PWH - Map_v_to_Dim_c(tp)));
	if (debug) fflush(os);
    }
    /*
     * place all the various headers
     */
    x = PRX + PRW/2;
    y = PWH - PRY - PRH - 2 - PFINFO->ascent - PFINFO->descent;
    y = min(y, PWH);
    fprintf(os, "%g %g ", x, y);
    put_PS_string(os, T_UNITS);
    fprintf(os, " 0 1 TEXT\n");
	
    x = xmin - PFINFO->min_bounds.width;
    x = max(x, 0);
    y = PWH - PRY - PRH/2;
    y = min(y, PWH);
    fprintf(os, "%g %g ", x, y);
    put_PS_string(os, V_UNITS);
    fprintf(os, " 1 0 TEXT\n");
	
    x = PRX + PRW/2;
    y = PWH - PRY + 2;
    y = min(y, PWH);
    fprintf(os, "%d %d /%s SETFONT\n",
	    PHFINFO->ascent,
	    PHFINFO->descent,
	    FONT_FAMILY);
    fprintf(os, "%g %g ", x, y);
    put_PS_string(os, DATA_FILE);
    fprintf(os, " 0 -1 TEXT\n");
    if (debug) fflush(os);
    fprintf(os, "grestore\n");
}

static void
put_PS_string(os, s)
FILE		*os;
String		s;
{
    register int	c;

    putc('(', os);
    while('\0' != (c = *s++)) {
	if (isprint(c) && (c != '(') && (c != ')') && (c != '\\')) {
	    putc(c, os);
	}
	else {
	    fprintf("\\%03o", c);
	}
    }
    putc(')', os);
}
