/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.Type;
import org.gnu.javagnome.Handle;

/**
 * A Viewport is a container that does not resize to fit in the
 * space allocated to it. Instead, it adds scrollbars as necessary so
 * you can view any part of the window.
 */
public class Viewport extends Bin 
{
	/**
	 * Construct a new Viewport with the given Adjustments.
	 * 
	 * @param hadj The horizontal Adjustment to use for the Viewport.
	 * @param vadj The vertical Adjustment to use for the Viewport.
	 */
	public Viewport(Adjustment hadj, Adjustment vadj) {
		super(init(hadj, vadj));
	}
	
	private static Handle init(Adjustment hadj, Adjustment vadj) {
	    Handle horizontal = null;
	    Handle vertical = null;
		if (null != hadj) {
			horizontal = hadj.getHandle();
		}
		if (null != vadj) {
			vertical = vadj.getHandle();
		}
		return gtk_viewport_new(horizontal, vertical);
	}
	
	/**
	 * Construct a Viewport using a handle to a native resource.
	 */
	public Viewport(Handle handle) {
	    super(handle);
	}

	/**
	 * Returns the horizontal Viewport's Adjustment.  This can be used
	 * to connect the horizontal scrollbar to the child Widget's 
	 * horizontal scroll functionality.
	 * 
	 * @return The horizontal Viewport's Adjustment.
	 */
	public Adjustment getHAdjustment() {
	    Handle hndl = gtk_viewport_get_hadjustment(getHandle());
		return Adjustment.getAdjustment(hndl);
	}

	/**
	 * Sets the horizontal Viewport's Adjustment.
	 * 
	 * @param hadj The horizontal Adjustment.
	 */
	public void setHAdjustment(Adjustment hadj) {
		Viewport.gtk_viewport_set_hadjustment(getHandle(), hadj.getHandle());
	}

	/**
	 * Returns the vertical Viewport's Adjustment.  This can be used
	 * to connect the vertical scrollbar to the child Widget's 
	 * vertical scroll functionality.
	 * 
	 * @return The vertical Viewport's Adjustment.
	 */
	public Adjustment getVAdjustment() {
	    Handle hndl = gtk_viewport_get_vadjustment(getHandle());
		return Adjustment.getAdjustment(hndl);
	}

	/**
	 * Sets the vertical Viewport's Adjustment.
	 * 
	 * @param vadj The vertical Adjustment.
	 */
	public void setVAdjustment(Adjustment vadj) {
		Viewport.gtk_viewport_set_vadjustment(getHandle(), vadj.getHandle());
	}
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_viewport_get_type());
	}

    native static final protected int gtk_viewport_get_type ();
    native static final protected Handle gtk_viewport_new (Handle hadjustment, Handle vadjustment);
    native static final protected Handle gtk_viewport_get_hadjustment (Handle viewport);
    native static final protected void gtk_viewport_set_hadjustment (Handle viewport, Handle adjustment);
    native static final protected Handle gtk_viewport_get_vadjustment (Handle viewport);
    native static final protected void gtk_viewport_set_vadjustment (Handle viewport, Handle adjustment);
    native static final protected void gtk_viewport_set_shadow_type (Handle viewport, int type);
    native static final protected int gtk_viewport_get_shadow_type (Handle viewport);
}

