/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.atk;

import org.gnu.glib.GObject;

/**
 * This class is the primary class for accessibility support via the Accessibility TookKit (ATK).
 * Objects which are instances of AtkObject are queried for properties which relate basic
 * properties of a UI component such as name and description.
 */
public class AtkObject extends GObject 
{
	/**
	 * Construct a new AtkObject using the handle returned
	 * by a call to the native layer.
	 * 
	 * @param handle The handle to the native resource.
	 */
	public AtkObject(int handle) {
		super(handle);
	}
	
	/**
	 * Gets the accessible name of the accessible.
	 * 
	 * @returns A string representing the accessible name of the accessible.
	 */
	public String getName() {
		return atk_object_get_name(getHandle());
	}
	
	/**
	 * Sets the accessible name of the accessible.
	 * 
	 * @param name The accessible name.
	 */
	public void setName(String name) {
		atk_object_set_name(getHandle(), name);
	}
	
	/**
	 * Gets the accessible description of the accessible.
	 * 
	 * @returns A string representing the accessible description of the accessible.
	 */
	public String getDescription() {
		return atk_object_get_description(getHandle());
	}
	
	/**
	 * Sets the accessible description of the accessible.
	 * 
	 * @param description The accessible description.
	 */
	public void setDescription(String description) {
		atk_object_set_description(getHandle(), description);
	}
	
	/**
	 * Get the accessible parent of the accessible.
	 * 
	 * @return An AtkObject representing the accessible parent of the accessible.
	 */
	public AtkObject getParent() {
		int hndl = atk_object_get_parent(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (AtkObject)obj;
		return new AtkObject(hndl);
	}
	
	/**
	 * Sets the accessible parent of the accessible.
	 * 
	 * @param parent The accessible parent.
	 */
	public void setParent(AtkObject parent) {
		atk_object_set_parent(getHandle(), parent.getHandle());
	}
	
	/**
	 * Returns the number of accessible children of the accessible.
	 * 
	 * @return the number of accessible children.
	 */
	public int getNumChildren() {
		return atk_object_get_n_accessible_children(getHandle());
	}
	
	/**
	 * Returns the specified child of the object.  The index is 0 based.
	 * 
	 * @param index The position of the child starting at 0.
	 * @return An AtkObject that is the child at the specified position.
	 */
	public AtkObject getChild(int index) {
		int hndl = atk_object_ref_accessible_child(getHandle(), index);
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (AtkObject)obj;
		return new AtkObject(hndl);
	}
	
	/**
	 * Gets the role of the accessible.
	 * 
	 * @return A Role which is the role of the accessible.
	 */
	public Role getRole() {
		return Role.intern(atk_object_get_role(getHandle()));
	}
	
	/**
	 * Sets the role of the accessible.
	 * 
	 * @param role The role to be set.
	 */
	public void setRole(Role role) {
		atk_object_set_role(getHandle(), role.getValue());
	}
	
    /****************************************
     * BEGINNING OF GENERATED CODE
     ****************************************/
    native static final protected int atk_object_get_type ();
    native static final protected String atk_object_get_name (int accessible);
    native static final protected String atk_object_get_description (int accessible);
    native static final protected int atk_object_get_parent (int accessible);
    native static final protected int atk_object_get_n_accessible_children (int accessible);
    native static final protected int atk_object_ref_accessible_child (int accessible, int i);
    native static final protected int atk_object_ref_relation_set (int accessible);
    native static final protected int atk_object_get_role (int accessible);
    native static final protected int atk_object_get_layer (int accessible);
    native static final protected int atk_object_get_mdi_zorder (int accessible);
    native static final protected int atk_object_get_index_in_parent (int accessible);
    native static final protected void atk_object_set_name (int accessible, String name);
    native static final protected void atk_object_set_description (int accessible, String description);
    native static final protected void atk_object_set_parent (int accessible, int parent);
    native static final protected void atk_object_set_role (int accessible, int role);
    native static final protected void atk_object_remove_property_change_handler (int accessible, 
        int handlerId);
    native static final protected void atk_object_notify_state_change (int accessible, int state, 
        boolean value);
    native static final protected void atk_object_initialize (int accessible, int data);
    /****************************************
     * END OF GENERATED CODE
     ****************************************/
}

