/* rfiles.c -- read and write resource files				*/
/*
 * Copyright (c) 1994  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */
#include "rfiles.h"
#include "config.h"

#define	Nfree(p)	((NULL == (p)) || (XtFree((XtPointer) p), (p) = NULL))
#define	WHITE	" \t\n"			/* whitespace chars		*/

/* write_resource_file -- write a structure as a resource file
 *
 * Returns NULL if there was no error, an error message otherwise.
 * The error message, if any, will be in space newly allocated from
 * the heap.
 */
String
write_resource_file(fname, structure, widget, name, list, nlist)
String		fname;
XtPointer	structure;
Widget		widget;
String		name;
XtResourceList	list;
int		nlist;
{
    char		lbuf[LLEN];
    FILE		*stream;
    String		s;

    if (NULL == (stream = fopen(fname, "w"))) {
	sprintf(lbuf, "unable to open %s", fname);
	return(XtNewString(lbuf));
    }
    s = write_resource_stream(stream, structure, widget, name, list, nlist);
    fclose(stream);
    return(s);
}

String
write_resource_stream(stream, structure, widget, name, list, nlist)
FILE		*stream;
XtPointer	structure;
Widget		widget;
String		name;
XtResourceList	list;
int		nlist;
{
    char		lbuf[LLEN];
    register int	i;
    String		val;

    for(i=0; i<nlist; i++) {
	sprintf(lbuf, "%s%s", name, list[i].resource_name);
	val = resource_to_string(widget, structure, &list[i]);
	if (NULL != val) {
	    put_resource_line(stream, lbuf, val);
	}
    }
    return(NULL);
}

void
faddc(l, s, len, n, c)
String		*l;
String		*s;
int		*len;
int		*n;
int		c;
{
    int			d = (*s) - (*l);

    if (*len < *n) {
	*(*s)++ = c;
	(*len)++;
	return;
    }
    *l = (String) XtRealloc(*l, *n + LLEN);
    *n += LLEN;
    *s = *l + d;
    *(*s)++ = c;
    (*len)++;
}

void
put_resource_line(stream, res, val)
FILE	*stream;
String	res;
String	val;
{
    String		s = escape_resource(val);

    put_string(stream, res);
    putc(':', stream);
    putc('\t', stream);
    put_string(stream, s);
    putc('\n', stream);
    XtFree(s);
}

Bool
parse_resource_line(line, wid, res, val)
String		line;
String		*wid;
String		*res;
String		*val;
{
    String		colon;
    String		s;
    String		t;

    if (line[strspn(line, WHITE)] == '\0') {
	*wid = *res = *val = NULL;	/* blank (or comment) line	*/
	return(FALSE);
    }
    if (NULL == (colon = STRCHR(line, ':'))) return(TRUE);
    *val = XtNewString(colon + 1 + strspn(colon+1, WHITE));
    *wid = (String) XtMalloc(colon - line + 1);
    strncpy(*wid, line, colon - line);
    (*wid)[colon-line] = '\0';
    s = STRRCHR(*wid, '.');
    t = STRRCHR(*wid, '*');
    if ((NULL != s) || (NULL != t)) {
	if (t > s) s = t;
	*s++ = '\0';
	*res = XtNewString(s);
    }
    else {
	*res = *wid;
	*wid = XtNewString("*");
    }
    return(FALSE);
}

/* escape_resource -- escape special chars in resource values
 *
 * String	in;
 * String	out;
 * out = escape_resource(in);
 *
 * The return value points to space newly allocated from the heap.
 */
String
escape_resource(in)
String	in;
{
    String		l = XtMalloc(LLEN);
    String		s;
    int			len;
    int			n = LLEN;
    int			c;

    s = l; len = 0;
    while('\0' != (c = *in++)) {
	switch(c) {
	case '\n':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, 'n');
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, '\n');
	    break;

	case '\a':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, 'a');
	    break;

	case '\b':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, 'b');
	    break;

	case '\f':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, 'f');
	    break;

	case '\r':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, 'r');
	    break;

	case '\t':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, 't');
	    break;

	case '\v':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, 'v');
	    break;

	case '\"':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, '\"');
	    break;

	case '\'':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, '\'');
	    break;

	case ' ':
	    addc(l, s, len, n, ' ');
	    break;

	case '\\':
	    addc(l, s, len, n, '\\');
	    addc(l, s, len, n, '\\');
	    break;

	default:
	    if (isgraph(c)) {
		addc(l, s, len, n, c);
	    }
	    else {
		addc(l, s, len, n, '\\');
		addc(l, s, len, n, '0');
		addc(l, s, len, n, '0');
		addc(l, s, len, n, '0');
		sprintf(s-3, "%03o", 0xff & c);
	    }
	    break;
	}
    }
    addc(l, s, len, n, '\0');
    l = (String) XtRealloc((XtPointer) l, len);
    return(l);
}

/* resource_to_string -- convert structure member to a string
 *
 * Widget	widget;
 * XtPointer	structure;
 * XtResource	resource;
 * String	s;
 * s = resource_to_string(widget, structure, resource);
 *
 * structure points to the base of the structure that the offset in
 * resource refers to.  s converts the value to a string, or returns
 * NULL on error.  The return value points to a static area and should
 * not be freed or changed.
 *
 * The widget argument is required only because the Xt conversion
 * routines need one (and crash if you pass NULL).  This is
 * unfortunate, since many conversions don't need a display,
 * application context, etc, and yet you have to drag in the whole
 * toolkit and connect to the display to use them.  Someday perhaps
 * I'll extract the relevant source code from the Xt libraries.
 */
String
resource_to_string(widget, structure, resource)
Widget		widget;
XtPointer	structure;
XtResource	*resource;
{
    String		type;
    String		val;

    val = cvt_to_string(widget, structure, resource->resource_offset,
			resource->resource_size, resource->resource_type);
    return(val);
}

/* string_to_resource -- convert string to structure member
 *
 * Widget	widget;
 * XtPointer	structure;
 * XtResource	resource;
 * String	s;
 * Bool		ret;
 * ret = resource_to_string(widget, structure, resource, s);
 *
 * structure points to the base of the structure that the offset in
 * resource refers to.  The string in s is converted and stored in the
 * structure.  string to resource returns True on error, otherwise
 * False.
 *
 * string_to_resource calls cvt_from_string with a freestrings
 * argument of True.  See cvt_from_string for an explanation.
 *
 * The widget argument is required only because the Xt conversion
 * routines need one (and crash if you pass NULL).  This is
 * unfortunate, since many conversions don't need a display,
 * application context, etc, and yet you have to drag in the whole
 * toolkit and connect to the display to use them.  Someday perhaps
 * I'll extract the relevant source code from the Xt libraries. */
Bool
string_to_resource(widget, structure, resource, s)
Widget		widget;
XtPointer	structure;
XtResource	*resource;
String		s;
{
    String		type;
    Bool		ret;

    ret = cvt_from_string(s, widget, structure, resource->resource_offset,
			  resource->resource_size, resource->resource_type,
			  True);
    return(ret);
}

String
string_from_db(db, name, class)
XrmDatabase	db;
String		name;
String		class;
{
    XrmValue		val;
    String		type;

    if (
	!XrmGetResource(db, name, class, &type, &val) ||
	(0 != strcmp(XtRString, type))
    ) return(NULL);
    return((String) val.addr);
}

/* read_resource_file -- read a structure from a resource file
 *
 * Returns NULL if there was no error, an error message otherwise.
 * The error message, if any, will be in space newly allocated from
 * the heap.
 */
String
read_resource_file(fname, structure, widget, name, class, list, nlist)
String		fname;
XtPointer	structure;
Widget		widget;
String		name;
String		class;
XtResourceList	list;
int		nlist;
{
    register int	i;
    char		nbuf[LLEN];
    char		cbuf[LLEN];
    String		s;
    XrmDatabase		db;

    db = XrmGetFileDatabase(fname);
    s = read_resource_db(db, structure, widget, name, class, list, nlist);
    XrmDestroyDatabase(db);
    return(s);
}

String
read_resource_db(db, structure, widget, name, class, list, nlist)
XrmDatabase	db;
XtPointer	structure;
Widget		widget;
String		name;
String		class;
XtResourceList	list;
int		nlist;
{
    register int	i;
    char		nbuf[LLEN];
    char		cbuf[LLEN];
    String		s;

    for(i=0; i<nlist; i++) {
	sprintf(nbuf, "%s.%s", name, list[i].resource_name);
	sprintf(cbuf, "%s.%s", class, list[i].resource_class);
	s = string_from_db(db, nbuf, cbuf);
	if (NULL == s) break;
	if (string_to_resource(widget, structure, &list[i], s)) break;
    }
    if (i < nlist) {
	sprintf(cbuf, "%s not found", nbuf);
	return(XtNewString(cbuf));
    }
    return(NULL);
}

/* cvt_to_string -- use Xt conversion to convert arbitrary type to string
 *
 * Widget	widget;
 * XtPointer	structure;
 * Cardinal	offset;
 * Cardinal	size;
 * String	type;
 * String	string;
 * string = cvt_to_string(widget, structure, offset, size, type);
 *
 * structure points to the structure within which the value to be
 * converted resides, and offset is its byte offset from the base of
 * the structure.  (If the value is not within a structure, pass its
 * address as structure and 0 as offset.)  size is its size, type its
 * type.  widget is needed for the call to XtConvertAndStore.  (NULL
 * will not work, alas.)  the return value points to a string in
 * internal static storage, or NULL if there was an error.
 */
String
cvt_to_string(widget, structure, offset, size, type)
Widget		widget;
XtPointer	structure;
Cardinal	offset;
Cardinal	size;
String		type;
{
    XrmValue		from;
    XrmValue		to;
    Bool		ret;

    from.addr = (caddr_t) structure;
    from.addr += offset;
    if (0 == strcmp(type, XtRString))
	return(*((String *) from.addr));
    from.size = size;
    to.addr = NULL;
    ret = XtConvertAndStore(widget, type, &from, XtRString, &to);
    return(ret ? (String) to.addr : NULL);
}

/* cvt_from_string -- Xt conversion from string to arbitrary type
 *
 * String	string;
 * Widget	widget;
 * XtPointer	structure;
 * Cardinal	offset;
 * Cardinal	size;
 * String	type;
 * Bool		freestrings
 * Bool		failed;
 * failed = cvt_from_string(string, widget, structure, offset, size,
 *			    type, freestrings);
 *
 * Arguments are like cvt_to_string, except that string is the input
 * string.  Returns True on error, False on success.
 *
 * If type is XtRString, structure and offset are assumed to point to
 * a variable of type String, i.e. a pointer to the actual string.  If
 * freestrings is True and this pointer is not NULL on call, is
 * assumed to point to a string in space allocated from the heap,
 * which is freed using XtFree.  It is then replaced with a pointer to
 * a copy of the input string, newly allocated from the heap.  If
 * freestrings is False, the pointer is simply replaced with s.
 */
Bool
cvt_from_string(s, widget, structure, offset, size, type, fstr)
String		s;
Widget		widget;
XtPointer	structure;
Cardinal	offset;
Cardinal	size;
String		type;
Bool		fstr;
{
    XrmValue		from;
    XrmValue		to;
    Bool		ret;

    to.addr = (caddr_t) structure;
    to.addr += offset;
    if (0 == strcmp(type, XtRString)) {
	if (fstr) {
	    Nfree(*((String *) to.addr));
	    *((String *) to.addr) = XtNewString(s);
	}
	else {
	    *((String *) to.addr) = s;
	}	    
	return(False);
    }
    to.size = size;
    from.addr = s;
    from.size = strlen(s) + 1;
    ret = !XtConvertAndStore(widget, XtRString, &from, type, &to);
    return(ret);
}

/* escape -- interpret escapes in quoted strings
 *
 * String	s;
 * int		c;
 * if ('\\' == (c = *s++)) c = escape(&s);
 *
 * escape, used as above, will return the next character in a quoted
 * string.  If the escape sequence produces no char (\<newline> or
 * \<null>), escape returns EOF.  s points after the end of the escape
 * sequence on return.
 */
int
escape(sp)
String	*sp;
{
    int			c;
    int			d;

    switch(c = *(*sp)++) {
    case 'x':				/* hex escape			*/
	c = 0;
	while(isxdigit(d = *(*sp)++)) {
	    c *= 0x10;
	    c += isdigit(d) ? (d - '0') : (tolower(d) - 'a' + 0xa);
	}
	(*sp)--;
	break;

    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
	(*sp)--;
	c = 0;
	while(('0' <= (d = *(*sp)++)) && ('7' >= d)) {
	    c *= 010;
	    c += d - '0';
	}
	(*sp)--;
	break;

    case 'a':
	c = '\a';
	break;

    case 'b':
	c = '\b';
	break;

    case 'f':
	c = '\f';
	break;

    case 'n':
	c = '\n';
	break;

    case 'r':
	c = '\r';
	break;

    case 't':
	c = '\t';
	break;

    case 'v':
	c = '\v';
	break;

    case '\0':
    case '\n':
	c = EOF;
	break;

    case EOF:
    default:
	break;
    }
    return(c);
}

/* quoted_string -- interpret a quoted string
 *
 * String	s;
 * int		q;
 * String	val;
 * val = quoted_string(&s, q);
 *
 * On call, s points to the char AFTER the initial quote of a quoted
 * string.  q is the character that will terminate the string
 * (normally the same as the initial quote).  quoted_string leaves s
 * pointing after the terminator, and returns a pointer to a string
 * newly allocated from the heap.
 */
String
quoted_string(sp, q)
String	*sp;
int	q;
{
    register int	c;
    String		s = *sp;
    String		buf = XtNewString("");
    String		t;
    int			len = 0;
    int			n = 0;

    t = buf;
    while(('\0' != *s) && (q != (c = *s++))) {
	if ('\\' == c) c = escape(&s);
	if (EOF != c) addc(buf, t, len, n, c);
    }
    addc(buf, t, len, n, '\0');
    *sp = s;
    buf = (String) XtRealloc(buf, len);
    return(buf);
}
