/* children.c -- handle child xdatplot windows				*/
/*
 * Copyright (c) 1994  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */
#include "xdatplot.h"
#include "rfiles.h"

typedef	struct CQUEUE {
    struct CQUEUE *cq_nxt;		/* next command in queue	*/
    CHILD	*cq_ch;			/* receiving child		*/
    Bool	cq_sent;		/* True if already sent		*/
    int		cq_id;			/* id # of this command		*/
    String	cq_txt;			/* text of the command		*/
}		CQUEUE;

#ifdef	__STDC__
static	String	parse_cmd(String, String **, int *);
static	String	add_resources(String **, int *);
static	String	add_file(String **, int *);
static	void	destroy_argv(String *);
static	void	child_ex(XtPointer, int *, XtInputId *);
static	void	child_in(XtPointer, int *, XtInputId *);
static	void	child_out(XtPointer, int *, XtInputId *);
static	void	delete_child(CHILD *);
static	void	delete_cqueue(CQUEUE **);
static	void	add_cqueue(CQUEUE *);
static	String	send_command(CHILD *, String);
static	String	send_commands(String);
static	String	getrep(String, int, FILE *);
#else	/* __STDC__ */
static	String	parse_cmd();
static	String	add_resources();
static	String	add_file();
static	void	destroy_argv();
static	void	child_ex();
static	void	child_in();
static	void	child_out();
static	void	delete_child();
static	void	delete_cqueue();
static	void	add_cqueue();
static	String	send_command();
static	String	send_commands();
static	String	getrep();
#endif	/* __STDC__ */

static	String	err_msg = NULL;		/* "static" messages		*/
static	int	cmdid = 0;		/* for command id's		*/
static	CQUEUE	*cmd_q = NULL;		/* head of command queue...	*/
static	CQUEUE	**cqtail = &cmd_q;	/* ...and tail			*/

String
start_child()
{
    String		msg;
    String		cmd;
    String		*argv;
    int			alen;
    CHILD		*cp;
    int			pipes[2];

    cp = XtNew(CHILD);
    cp->c_nxt = NULL;
    cp->c_in = cp->c_out = NULL;
    if (
	(NULL != (msg = parse_cmd(app_data.xdatplot_command, &argv, &alen))) ||
	(NULL != (msg = add_file(&argv, &alen))) ||
	(NULL != (msg = add_resources(&argv, &alen)))
    ) {
	destroy_argv(argv);
	return(msg);
    }
    if (debug) {
	int		i;

	dprintf("command to start child:\n");
	for(i=0; i<alen; i++) dprintf1("    %s\n", argv[i]);
    }
    cmd = argv[0];
    cp->c_pid = run2(cmd, argv, pipes);
    destroy_argv(argv);
    if (EOF == cp->c_pid) {
	return("unable to start new xdatplot");
    }
    cp->c_in = fdopen(pipes[0], "w");
    setbuf(cp->c_in, NULL);
    cp->c_out = fdopen(pipes[1], "r");
    setbuf(cp->c_out, NULL);
    cp->c_nxt = app_data.children;
    app_data.children = cp;
    cp->c_ioid = NO_INPUTID;
    cp->c_ieid = AppAddInput(app, pipes[0], XtInputExceptMask, child_ex, cp);
    cp->c_oiid = AppAddInput(app, pipes[1], XtInputReadMask, child_in, cp);
    cp->c_oeid = AppAddInput(app, pipes[1], XtInputExceptMask, child_ex, cp);
    if (NULL != (msg = child_plot())) return(msg);
    return(NULL);
}

static String
parse_cmd(cmd, avp, alp)
String	cmd;
String	**avp;
int	*alp;
{
    register String	s;
    register int	n;
    int			q;
    String		*ap;

    ap = XtNew(String);
    n = 0;
    while(isspace(*cmd)) cmd++;
    while('\0' != *cmd) {
	if (('\"' == *cmd) || ('\'' == *cmd)) {
	    q = *cmd++;
	    ap[n++] = quoted_string(&cmd, q);
	}
	else {
	    for(s = cmd; ('\0' != *s) && !isspace(*s); s++);
	    ap[n] = XtMalloc(s - cmd + 1);
	    strncpy(ap[n], cmd, s - cmd);
	    ap[n++][s - cmd] = '\0';
	    cmd = s;
	}
	ap = (String *) Realloc(ap, (n+1) * sizeof(String));
	while(isspace(*cmd)) cmd++;
    }
    ap[n] = NULL;
    *avp = ap;
    *alp = n;
    return(NULL);
}

/* add_resources -- add resources to command line			*/
static String
add_resources(avp, alp)
String	**avp;
int	*alp;
{
    String		msg;
    String		val;
    String		s = app_data.child_resources;
    String		t;
    String		u = XtNewString("");
    int			ulen = 0;
    String		*ap = *avp;
    int			n = *alp;

    if (NULL == s) return(NULL);
    u = XtNewString("");
    while(isspace(*s)) s++;
    while('\0' != *s) {
	for(t=s; ('\0' != *t) && !isspace(*t); t++);
	if (t-s >= ulen) {
	    ulen = t-s + 1;
	    u = Realloc(u, ulen);
	}
	strncpy(u, s, t-s);
	u[t-s] = '\0';
	s = t;
	while(isspace(*s)) s++;
	if (NULL != (msg = get_resource_val(u, &val))) {
	    Free(u);
	    return(msg);
	}
	val = escape_resource(val);
	ap = (String *) Realloc(ap, (n+3) * sizeof(String));
	ap[n++] = XtNewString("-xrm");
	ap[n] = (String) XtMalloc(strlen(u) + 2 + strlen(val) + 1);
	strcpy(ap[n], u);
	strcat(ap[n], ": ");
	strcat(ap[n], val);
	n++;
	Free(val);
	ap[n] = NULL;
    }
    Free(u);
    *avp = ap;
    *alp = n;
    return(NULL);
}

static String
add_file(avp, alp)
String	**avp;
int	*alp;
{
    String		*ap = *avp;
    int			n = *alp;

    if (NULL != app_data.xdp_file) {
	ap = (String *) Realloc(ap, (n+3) * sizeof(String));
	ap[n++] = XtNewString("-file");
	ap[n++] = XtNewString(app_data.xdp_file);
	ap[n] = NULL;
    }
    else if (NULL != app_data.data_file) {
	ap = (String *) Realloc(ap, (n+3) * sizeof(String));
	ap[n++] = XtNewString("-file");
	ap[n++] = XtNewString(app_data.data_file);
	ap[n] = NULL;
    }
    else {}
    *avp = ap;
    *alp = n;
    return(NULL);
}

static void
destroy_argv(argv)
String	*argv;
{
    String		*av = argv;

    for(av = argv; NULL != *av; av++) Free(*av);
    Free(argv);
}

String
child_file()
{
    char		cbuf[LLEN];
    register CHILD	*cp;
    String		name;
    String		msg;

    if (
	(NULL == (name = app_data.xdp_file)) &&
	(NULL == (name = app_data.data_file))
    ) return(NULL);
    sprintf(cbuf, "NewFile %s", name);
    return(send_commands(cbuf));
}

String
child_plot()
{
    char		cbuf[LLEN];
    Dimension		w;

    if (NULL == app_data.children) return(NULL);
    XtVaGetValues(plot, XmNwidth, &w, NULL);
    sprintf(cbuf, "SizeZoomScrollT %d %.17g %.17g", (int) w,
	    (double) t_to_Dim_sf, (double) Map_TIME_to_t_c(TL));
    send_commands(cbuf);
    return(NULL);
}

String
child_cursor()
{
    char		cbuf[LLEN];

    if (NULL == app_data.children) return(NULL);
    sprintf(cbuf, "SetCursor %.17g", (double) Map_TIME_to_t(CURSOR));
    send_commands(cbuf);
    return(NULL);
}

String
child_clr_cursor()
{
    char		cbuf[LLEN];

    if (NULL == app_data.children) return(NULL);
    sprintf(cbuf, "ClearCursor");
    send_commands(cbuf);
    return(NULL);
}

String
child_region()
{
    char		cbuf[LLEN];

    if (NULL == app_data.children) return(NULL);
    sprintf(cbuf, "SetCursor %.17g", (double) Map_TIME_to_t(CURSOR));
    send_commands(cbuf);
    sprintf(cbuf, "Region %.17g", (double) Map_TIME_to_t(CURSOR_END));
    send_commands(cbuf);
    return(NULL);
}

/*
 * I/O Exception from child: probably means it exited
 */
static void
child_ex(data, fdp, idp)
XtPointer	data;
int		*fdp;
XtInputId	*idp;
{
    CHILD		*cp = (CHILD *) data;

    delete_child(cp);
}

/*
 * input from child: delete completed command
 */
static void
child_in(data, fdp, idp)
XtPointer	data;
int		*fdp;
XtInputId	*idp;
{
    CHILD		*cp = (CHILD *) data;
    CQUEUE		**cqp;	
    char		lbuf[LLEN];
    char		ebuf[LLEN];
    int			id;
    String		s;

    if (NULL == getrep(lbuf, LLEN, cp->c_out)) {
	delete_child(cp);
	return;
    }
    id = strtol(lbuf, &s, 0);
    if (s == lbuf) {
	XtAppError(app, "Incorrect output from child");
	return;				/* not reached			*/
    }
    while(isspace(*s)) s++;
    if ((0 == strncmp(s, "OK", 2)) && isspace(s[2])) {}
    else if ((0 == strncmp(s, "ERR", 3)) && isspace(s[3])) {
	sprintf(ebuf, "Error from child:\n    %s", s);
	XtAppWarning(app, ebuf);
    }
    else {
	XtAppError(app, "Incorrect output from child");
	return;				/* not reached			*/
    }
    for(
	cqp = &cmd_q;
	NULL != *cqp;
	cqp = &(*cqp)->cq_nxt
    ) {
	if ((*cqp)->cq_id == id) break;
    }
    if (NULL == *cqp) {
	XtAppError(app, "Incorrect output from child");
	return;				/* not reached			*/
    }
    delete_cqueue(cqp);
}

/*
 * child ready for output: send a command
 */
static void
child_out(data, fdp, idp)
XtPointer	data;
int		*fdp;
XtInputId	*idp;
{
    CHILD		*cp = (CHILD *) data;
    CQUEUE		*cq;

    for(cq=cmd_q; NULL != cq; cq = cq->cq_nxt) {
	if (!cq->cq_sent && (cp == cq->cq_ch)) break;
    }
    if (NULL == cq) {			/* nothing: delete handler	*/
	XtRemoveInput(cp->c_ioid);
	cp->c_ioid = NO_INPUTID;
    }
    else {				/* found something: send it	*/
	dprintf1("%s\n", cq->cq_txt);
	fprintf(cp->c_in, "%d %s\n", cq->cq_id, cq->cq_txt);
	fflush(cp->c_in);
	cq->cq_sent = True;
    }
}

static void
delete_child(cp)
CHILD	*cp;
{
    CHILD		**cpp;
    CQUEUE		*cq;
    CQUEUE		**cqp;

    XtRemoveInput(cp->c_ieid);		/* turn off all I/O		*/
    if (NO_INPUTID != cp->c_ioid) XtRemoveInput(cp->c_ioid);
    XtRemoveInput(cp->c_oeid);
    XtRemoveInput(cp->c_oiid);
    fclose(cp->c_in);
    fclose(cp->c_out);
    for(				/* remove queued cmds		*/
	cqp = &cmd_q;
	NULL != *cqp;
    ) {
	if (cp == (*cqp)->cq_ch) {
	    delete_cqueue(cqp);
	}
	else {
	    cqp = &(*cqp)->cq_nxt;
	}
    }
    for(				/* delete this child from list	*/
	cpp = &app_data.children;
	(NULL != *cpp) && (cp != *cpp);
	cpp = &(*cpp)->c_nxt
    );
    if (cp != *cpp) error("child_ex: can't happen");
    *cpp = cp->c_nxt;
    Free(cp);
    wait();				/* reap it			*/
}

static void
delete_cqueue(cqp)
CQUEUE	**cqp;
{
    CQUEUE		*cq = *cqp;

    *cqp = cq->cq_nxt;
    if (NULL == *cqp) cqtail = cqp;
    Free(cq->cq_txt);
    Free(cq);
}

static void
add_cqueue(cq)
CQUEUE	*cq;
{
    *cqtail = cq;
    cq->cq_nxt = NULL;
    cqtail = &cq->cq_nxt;
}

static String
send_commands(cmd)
String	cmd;
{
    String		msg;
    CHILD		*cp;

    for(
	cp = app_data.children;
	NULL != cp;
	cp = cp->c_nxt
    ) {
	if (NULL != (msg = send_command(cp, cmd)))
	    return(msg);
    }
    return(NULL);
}

static String
send_command(cp, cmd)
CHILD	*cp;
String	cmd;
{
    CQUEUE		*cq;

    cq = XtNew(CQUEUE);
    cq->cq_nxt = NULL;
    cq->cq_ch = cp;
    cq->cq_sent = False;
    cq->cq_id = ++cmdid;
    cq->cq_txt = XtNewString(cmd);
    add_cqueue(cq);
    if (NO_INPUTID == cp->c_ioid) {
	cp->c_ioid = AppAddInput(app, fileno(cp->c_in),
				 XtInputWriteMask, child_out, cp);
    }
    return(NULL);
}

/* getrep -- get reply
 *
 * Called like fgets.  Always flushes input to newline or endfile.
 */
static String
getrep(buf, len, st)
String	buf;
int	len;
FILE	*st;
{
    register int	c;

    buf = fgets(buf, len, st);
    if (
	(NULL == buf) ||
	'\n' == buf[strlen(buf)-1]
    ) return(buf);
    while((EOF != (c = getc(st))) && ('\n' != c));
    return(buf);
}
