/*
 *	@(#)tocsin.c	1.16	08/25/98	- Doug Hughes, Auburn University College of Engineering
 */
#include <stdio.h>
#include <stropts.h>
#include <errno.h>
#include <netdb.h>
#include <poll.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/sockio.h>
#include <sys/syslog.h>
#include <net/if.h>
#include <memory.h>
#include <netinet/in.h>
#include <string.h>
#ifdef SVR4
# define DEVICE "/dev/le"
# include <sys/dlpi.h>
# include <sys/pfmod.h>
# include <sys/bufmod.h>
# include <sys/ethernet.h>
# include <sys/fcntl.h>
# define OFFSET 0
# define PACKETFILTER "pfmod"
# define PUSHFILTER PFIOCSETF
# define BUFDEVICE "bufmod"
# define BUFTIMEOUT SBIOCSTIME
# define BUFCHUNK SBIOCSCHUNK
# define BUFSNAP SBIOCSSNAP
# define DLPI_DEFAULTSAP	0x800	/* IP protocol */
#else
# include <netinet/if_ether.h>
# include <net/nit.h>
# include <net/nit_if.h>
# include <net/nit_pf.h>
# include <net/nit_buf.h>
# include <net/packetfilt.h>
# define OFFSET 8
# define DEVICE "/dev/nit"
# define PACKETFILTER "pf"
# define BUFDEVICE "nbuf"
# define PUSHFILTER NIOCSETF
# define BUFTIMEOUT NIOCSTIME
# define BUFCHUNK NIOCSCHUNK
# define BUFSNAP NIOCSSNAP
#endif

static struct _snoop_file_header {
	char id[8];
	int version;
	int datalink;
} SnoopFileHeader = { "snoop", 2, 4 };

typedef struct _pkt_header {
	unsigned original_length;
	unsigned included_length;
	unsigned record_length;
	unsigned cumulative_drops;
	struct timeval timestamp;
} PktHeader;

#define MAX_ADDR_SIZE 256

typedef struct _scans {
	u_short pnum;
	char *pname;
} Scanport;


/* Make customizations here */

/* Total number of ports to watch for SYN's on */
#define NPORTS 64

/****************************/

/* Global defines */
#define VALIDARGS "i:dho:"
/*
 * Global variables
 */
static int nports=0;					/* number of ports currently defined */
static Scanport scanarray[NPORTS];		/* port structures */
extern int errno;
extern int optind, opterr;
extern char *optarg;

static void usage()
{
	printf("usage: tcptrap [-i <interface>] [-o <mfile>] [-d] [-h] service [service] [service] [...]\n\
	-d dump packet in hex (debug mode)\n\
	-o <file> dump packet in snoop format to <file>\n\
	-h this message\n\
	-i <interface> (le0 is default)\n");
	exit(0);
}

/* Go into background */
static void daemon(int debug)
{
	long bg;

	if (!debug) {
		if ((bg = fork()) < 0) {
			perror("couldn't go into background");
			exit(1);
		} else if (bg > 0) {	/* parent, exit */
			exit(0);
		}
		close(0);
		close(1);
		close(2);
		setpgrp(0, getpid());
		setsid();
	}
}

/* Handle DLPI stuff */
static int strioctl(fd, cmd, timout, len, dp)
int     fd;
int     cmd;
int     timout;
int     len;
char    *dp;
{
	struct  strioctl  sioc;
	int     rc;

	sioc.ic_cmd = cmd;
	sioc.ic_timout = timout;
	sioc.ic_len = len;
	sioc.ic_dp = dp;
	rc = ioctl (fd, I_STR, &sioc);

	if (rc < 0)
		return (rc);
	else
		return (sioc.ic_len);
}


/*
 * This routine initializes the nit interface.
 */
static int init_nit(char *interface)
{
	int s, i;
	int fd;
	struct strioctl si;
	struct ifconf ic;
	struct ifreq ifr, *ifp;
	u_char if_buf[BUFSIZ];
	struct timeval timeout;
	int chunksize = 1024;
	int truncation = PKTSIZE;
	struct packetfilt pf;
	register u_short *fwp = pf.Pf_Filter;
	struct sockaddr_in saddr, *netmask;	/* interface address and netmask */
	u_long if_flags;
	int offset;					/* offset in shorts from beginning of packet */
#ifdef SVR4
	int instance;
	char devname[9]="/dev/";
	int ppa=0;
	int sap=DLPI_DEFAULTSAP;
#else
	char *devname="/dev/nit";
#endif

#ifdef SVR4
	/* displacement from beggining of the packet in u_short units */

	instance = 0;
	while (instance++ < 7) {
		if ((interface[instance] - '0' >= 0) && (interface[instance] - '0' <=9))
			break;
	}

	strncat(devname, interface, instance);
#endif

	if ((fd = open(devname, O_RDWR)) < 0) {
		perror("Failed to open streams interface");
		exit(1);
	}
	/*
	 * Configure the nit device, binding it to the proper
	 * underlying interface.
	 */

#ifdef SVR4
	ppa = atoi(interface + instance);
	/* attach */
	if (!AttachDevice(fd, ppa)) {
		perror("attaching device");
		exit(-1);
	}
	/* bind */
	if (!BindProtocol(fd, sap, 0, DL_CLDLS, 0, 0)) {
		fprintf(stderr, "Bad interface name - %s\n", interface);
		perror("binding device");
		exit(-1);
	}
#else
	strncpy(ifr.ifr_name, interface, sizeof(ifr.ifr_name));
	ifr.ifr_name[sizeof ifr.ifr_name - 1] = '\0';
	si.ic_cmd = NIOCBIND;
	si.ic_timout = 0;
	si.ic_len = sizeof ifr;
	si.ic_dp = (char *)&ifr;
	if (ioctl(fd, I_STR, (char *)&si) < 0) {
		fprintf(stderr, "Bad interface name - %s\n", interface);
		perror("ioctl NIOCBIND");
		exit(1);
	}

#endif

	s = socket(AF_INET, SOCK_DGRAM, 0);

	/* Get MY address list */
	memset((char *) &ic, 0, sizeof(ic));
	ic.ifc_len = BUFSIZ;
	ic.ifc_buf = (caddr_t) if_buf;
	if (ioctl(s, SIOCGIFCONF, &ic) < 0) {
		perror("ioctl SIOCGIFCONF");
		exit(1);
	}
	for (i = 0;; i++) {
		if (strcmp(ic.ifc_ifcu.ifcu_req[i].ifr_name, interface) == 0)
			break;
	}
	ifp = &ic.ifc_ifcu.ifcu_req[i];
	memcpy((char *) &saddr, 
				(char *) &(ifp->ifr_ifru.ifru_addr), 
				sizeof (struct sockaddr));
	
	if (ioctl(s, SIOCGIFNETMASK, (caddr_t) ifp) < 0) {
		perror("ioctl SIOCGIFNETMASK");
		exit(1);
	}
	netmask = (struct sockaddr_in *) &(ifp->ifr_ifru.ifru_addr);
	
	/* Mask netmask with address to give network */
	saddr.sin_addr.s_addr &= netmask->sin_addr.s_addr;
	close(s);

	/* set the interface flags - promiscuous mode */
#ifdef SVR4
	PromMode(fd, DL_PROMISC_PHYS);

	if (strioctl(fd, DLIOCRAW, -1, 0, NULL) < 0) {	/* set raw mode */
		perror("DLIORAW");
		exit(-1);
	}
#else

	si.ic_cmd = NIOCSFLAGS;
	if_flags = NI_PROMISC;			/* promiscous and dropped pkts*/
	si.ic_len = sizeof(if_flags);
	si.ic_dp = (char *)&if_flags;
	if (ioctl(fd, I_STR, (char*)&si) < 0) {
		perror("setting flags(NIOCSFLAGS)");
		exit(1);
	}
#endif

	/* set message discard mode */
	if (ioctl(fd, I_SRDOPT, (char*)RMSGD) < 0) {
		perror("RMSGD");
		exit(1);
	}


#ifdef SVR4
	offset = 23;
#else
	offset = 23;
#endif

	/*
	 * Setup packet filter on buffer
	 */
	memset((char *) &pf, 0, sizeof(pf));
	if (ioctl(fd, I_PUSH, PACKETFILTER) < 0) {
		perror("couldn't push packet filter");
		exit(1);
	}

#ifdef TCP_ONLY
	/* Code bits match SYN  and proto = TCP */
	*fwp++ = ENF_PUSHWORD + offset - 12;
	*fwp++ = ENF_PUSHLIT | ENF_AND;	/* only need lower byte */
	*fwp++ = htons(0x00ff);
	*fwp++ = ENF_PUSHLIT | ENF_EQ;
	*fwp++ = htons(0x0006);
#elif SYN_ONLY
	*fwp++ = ENF_PUSHWORD + offset;
	*fwp++ = ENF_PUSHLIT | ENF_AND;
	*fwp++ = htons(0x0002);
	*fwp++ = ENF_PUSHLIT | ENF_EQ;
	*fwp++ = htons(0x0002);
	*fwp++ = ENF_NOPUSH | ENF_AND;
#else /* neither - but need to put a one here  for later and */
	*fwp++ = ENF_PUSHLIT | ENF_NOP;
	*fwp++ = 1;
#endif
#ifdef DEST_ONLY
	/* only packets matching this dest net */
	*fwp++ = ENF_PUSHWORD + offset - 8;
	*fwp++ = ENF_PUSHLIT | ENF_AND;
	*fwp++ = htons(netmask->sin_addr.S_un.S_un_w.s_w1);
	*fwp++ = ENF_PUSHLIT | ENF_EQ;
	*fwp++ = htons(saddr.sin_addr.S_un.S_un_w.s_w1);
	*fwp++ = ENF_NOPUSH | ENF_AND;
	/* and lower word of dest net*/
	*fwp++ = ENF_PUSHWORD + offset - 7;
	*fwp++ = ENF_PUSHLIT | ENF_AND;
	*fwp++ = htons(netmask->sin_addr.S_un.S_un_w.s_w2);	/* & netmask */
	*fwp++ = ENF_PUSHLIT | ENF_EQ;
	*fwp++ = htons(saddr.sin_addr.S_un.S_un_w.s_w2);	/* = lower short */
	*fwp++ = ENF_NOPUSH | ENF_AND;
#endif
	/* Now filter for each of the ports */
	for (i = 0; i < nports; i++) {
		*fwp++ = ENF_PUSHWORD + offset - 5;
		*fwp++ = ENF_PUSHLIT | ENF_EQ;
		*fwp++ = htons(scanarray[i].pnum);
		if (i > 0)
			*fwp++ = ENF_NOPUSH | ENF_OR;
	}
	*fwp++ = ENF_NOPUSH | ENF_AND;
	/* We don't care about continuation fragments */
	*fwp++ = ENF_PUSHWORD + offset - 13;
	*fwp++ = ENF_PUSH00FF | ENF_AND;
	*fwp++ = ENF_PUSHLIT | ENF_EQ;
	*fwp++ = 0;
	*fwp++ = ENF_NOPUSH | ENF_AND;
#ifndef NO_IP_OPTIONS
	/* if header length > 5, accept - it's suspicious */
	*fwp++ = ENF_PUSHWORD + offset - 16;			/* get HLEN, vers, etc */
	*fwp++ = ENF_PUSHLIT | ENF_AND;
	*fwp++ = htons(0x0f00);
	*fwp++ = ENF_PUSHLIT | ENF_GT;
	*fwp++ = htons(0x0500);
	*fwp++ = ENF_NOPUSH | ENF_OR;
#endif
	pf.Pf_FilterLen = fwp - &pf.Pf_Filter[0];

#ifndef SYSV
	if (ioctl(fd, PUSHFILTER, (struct packetfilt *) &pf) < 0) {
		perror("pushing packet filter");
		exit(1);
	}
#else
	si.ic_cmd = PUSHFILTER;
	si.ic_timout = 0;
	si.ic_len = sizeof (pf);
	si.ic_dp = (char *)&pf;
	if (ioctl(fd, I_STR, (char *) &si) < 0) {
		perror("pushing packet filter");
		exit(1);
	}
#endif SYSV
	
	/* Now load bufmod */
	ioctl(fd, I_PUSH, BUFDEVICE);

	si.ic_timout = INFTIM;
    timeout.tv_sec = 1;
    timeout.tv_usec = 0;
    si.ic_cmd = BUFTIMEOUT;
    si.ic_len = sizeof(timeout);
    si.ic_dp = (char*)&timeout;
    if (ioctl(fd, I_STR, (char*)&si) < 0) {
        perror("pushing timeout");
        exit(-1);
    }

#ifdef SVR4
	/* Set IF flags */
	si.ic_cmd = SBIOCSFLAGS;
	si.ic_timout = INFTIM;
	if_flags = SB_NO_HEADER | SB_NO_DROPS;		/* don't need header garbage */
	si.ic_len = sizeof if_flags;
	si.ic_dp = (char*) &if_flags;
	if (ioctl(fd, I_STR, (char *) &si) <0) {
		perror("setting flags");
		exit(1);
	}
# endif SYSV

	/* set the chunksize */
	si.ic_cmd = BUFCHUNK;
	si.ic_timout = INFTIM;
	si.ic_len = sizeof(chunksize);
	si.ic_dp = (char *) &chunksize;
	if (ioctl(fd, I_STR, (char*)&si) < 0) {
		perror("tocsin: setting chunksize");
		exit(-1);
	}

	/* Set packets to see only first 48 bytes */
	si.ic_cmd = BUFSNAP;
	si.ic_timout = INFTIM;
	si.ic_len = sizeof(truncation);
	si.ic_dp = (char *) &truncation;
	if (ioctl(fd, I_STR, (char *) &si) < 0) {
		perror("tocsin: setting truncation");
		exit(-1);
	}

	/*
	 * Flush any garbage messages which has arrived
	 */
	if (ioctl(fd, I_FLUSH, (char *)FLUSHR) < 0) {
		perror("I_FLUSH");
		exit(1);
	}

	return(fd);
}

typedef union _Readbuf {
	u_char	c[ETHERMTU];
	u_short i[ETHERMTU/2];
} Readbuf;

#define cbuf readbuf.c
#define ibuf readbuf.i

main(int argc, char *argv[])
{
	Scanport *portmap[65536];
	Readbuf readbuf;
	struct servent *sp;				/* service entry struct */
	char *srv;						/* service name */
	char haddr[MAX_ADDR_SIZE + 1];	/* host address IP in ascii */
	register int portno;			/* port of service */
	register int ret;
	register int i;
	register int opt_offset=0;		/* offset if options are on */
	register int hlen;				/* packet header length */
	int fd;							/* network file descriptor */
	int hid;						/* host id */
	char logbuf[BUFSIZ];			/* Send syslog out */
	struct pollfd pfds;				/* for polling interface */
	register int c;					/* used by getopts */
	register struct hostent *hp;	/* hostent structure */
	struct in_addr dest;			/* destination host/network */
	PktHeader hdr;					/* the packet header structure */
	int pad;						/* variable padding */
	char padding[3] = { 0, 0, 0};	/* up to 3 bytes of padding */
	char ptype[30];					/* packet type and flags */
	char ip_opts[20] = {NULL};		/* IP options set */
	
/* command line options */
	int dump = 0;					/* dump packets */
	int ofile = 0;					/* output file */
	char interface[8]="le0";		/* interface from which to listen */

/* parse args */

	while ((c = getopt(argc, argv, VALIDARGS)) != -1) {
		switch(c) {
		case 'h':
			usage();
			break;
		case 'i':
			strcpy(interface, optarg);
			break;
		case 'd':
			dump = 1;
			break;
		case 'o':
			if ((ofile = open(strdup(optarg), O_RDWR|O_CREAT|O_TRUNC), 0644) < 0) {
				perror("Couldn't open output file for writing");
				exit(1);
			}
			write(ofile, (char *) &SnoopFileHeader, sizeof(SnoopFileHeader));
			break;
		default: usage();
		}
	}
	for ( ; optind > 1 ; optind --) {
		argc--;
		argv++;
	}

	daemon(dump);

	/* Collect service names to watch for */
	while (argc > 1) {
		srv = argv[1];
		if (isdigit(srv[0])) {
			if ((sp = getservbyport(atoi(srv), NULL)) == NULL) {
				if (dump) {
					printf("Warning: service name undefined for %s\n", srv);
				}
				scanarray[nports].pname = strdup(srv);
			} else 
				scanarray[nports].pname = strdup(sp->s_name);

			portno = atoi(srv);
			scanarray[nports].pnum = portno;
		} else {
			if ((sp = getservbyname(srv, NULL)) == (struct servent *) NULL) {
				if (dump) {
					printf("skipping unknown service %s\n", srv);
				}
				goto cont;
			}
			portno = sp->s_port;
			scanarray[nports].pnum = sp->s_port;
			scanarray[nports].pname = strdup(srv);
		}
		if (dump)
			printf("matching SYN requests to port %d\n", portno);

		portmap[portno] = &scanarray[nports];

		nports++;
cont:	argc--; argv++;
	}

	openlog("tocsin", 0, LOG_PID, LOG_LEVEL);

	fd = init_nit(interface);

	pfds.fd = fd;
	pfds.events = POLLIN;

	for(;;) {
		ret = poll(&pfds, 1, INFTIM);
		if (pfds.revents != POLLIN || ret == EINTR) {
			if (dump)
				printf("interrupted poll\n");
			continue;
		}
		ret = read(fd, cbuf, BUFSIZ);
		if (ret == 0) { 
			if (dump)
				printf("Read got nothing\n");
			continue;
		}
		if (ofile) {
			hdr.original_length = ret;
			hdr.included_length = ret;
			hdr.record_length = ret + 24;
			/* must take into account pad data */
			if ((pad = hdr.record_length % 4) != 0) 
				hdr.record_length += (4-pad);
			hdr.cumulative_drops = 0;
			gettimeofday(&hdr.timestamp, NULL);
			write(ofile, (char *) &hdr, 24);
			write(ofile, cbuf, ret);
			if (pad != 0) /* now write out pad if needed */
				write(ofile, padding, 4-pad);
		}
		hlen = cbuf[OFFSET+14] & 0x0f;	/* get packet header length */
		opt_offset = (hlen > 5) ? (hlen - 5) * 4 : 0;

		/* Get source address */
		memcpy((char *) &hid, (char *) &(cbuf[26+OFFSET]), 4);
		if ((hp = gethostbyaddr((char *)&hid, 4, AF_INET)) == NULL)
			strncpy(haddr, (char *) inet_ntoa(cbuf+26+OFFSET), 18);
		else
			strncpy(haddr, hp->h_name, MAX_ADDR_SIZE);

		memset(ptype, 0, sizeof(ptype));
		portno = 0;		/* don't print port info */
		/* determine IP type */
		switch(cbuf[23+OFFSET]) {
			case 1: /* ICMP */
				strcpy (ptype, "ICMP");
				break;
			case 6:	/* TCP */
				strcpy (ptype, "TCP: ");
				portno++;
				break;
			case 17: /* UDP */
				strcpy (ptype, "UDP");
				portno++;
				break;
			case 47: /* GRE */
				strcpy (ptype, "GRE");
				break;
			default:
				sprintf(ptype, "%d", cbuf[23+OFFSET]);
				break;
		}

		if (cbuf[23+OFFSET+opt_offset] == 6)	{ /* FLAGS only relevant for TCP packets */
			register int flags;						/* packet flags */
			register int i = 0;

			strcat(ptype, "(");
			flags = cbuf[47+OFFSET+opt_offset] & 0x3f;
			if (flags & 0x20) strcat(ptype, "URG|");
			if (flags & 0x10) strcat(ptype, "ACK|");
			if (flags & 0x8) strcat(ptype, "PSH|");
			if (flags & 0x4) strcat(ptype, "RST|");
			if (flags & 0x2) strcat(ptype, "SYN|");
			if (flags & 0x1) strcat(ptype, "FIN)");
				
			ptype[strlen(ptype)-1] = ')';

		}

		/* Check for IP Options set in header */
		memset(ip_opts, 0, sizeof(ip_opts));
		if (opt_offset > 0) {
			i = cbuf[34+OFFSET];	/* first IP option */
			sprintf(ip_opts, "(IP_OPTIONS! [%d,%d])", i >> 5 & 3, i & 0x1f);
		}

		memcpy((char *) &dest.s_addr, cbuf+30+OFFSET, 4);
		if (portno)
			sprintf(logbuf, "ALERT: host %s probing service: %s [%s] @ %s %s\n",
				haddr, portmap[ibuf[18+OFFSET/2+opt_offset/2]]->pname, ptype,
				inet_ntoa(dest), ip_opts);
		else 
			sprintf(logbuf, "ALERT: host %s IP (%s) probing %s %s\n",
				haddr, ptype, inet_ntoa(dest), ip_opts);
			
		syslog(LOG_LEVEL, logbuf);
		if (dump) {
			for (i=0;i<ret;i+=2) {
				if ((i % 16) == 0)
					printf("\n");
				printf("%2.2x%2.2x ", cbuf[i], cbuf[i+1]);
			}
			printf("\n");
		}
	}

}
