/*
 *  @(#)TestCorrelate.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.autodoc.v1.testserver;


import net.sourceforge.groboutils.util.throwable.v1.ThrowableParser;
import net.sourceforge.groboutils.util.throwable.v1.StackTraceLineParser;

import org.apache.log4j.Logger;


/**
 * Helper method to aid in the use of Monitors and TestData.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since      March 17, 2002
 * @version    $Date: 2003/02/10 22:52:13 $
 */
public abstract class TestCorrelate
{
    private static final Logger LOG = Logger.getLogger( TestCorrelate.class );
    
    private MonitorFinder finder;
    private Class owner;
    
    
    public TestCorrelate( Class owner, MonitorFinder finder )
    {
        if (finder == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        this.owner = owner;
        this.finder = finder;
    }
    
    
    
    /**
     * Helper method to discover the last method from the owning class in
     * the current stack.  If the method is not in the stack, then <tt>null</tt>
     * is returned.  This can be very slow.
     */
    protected String findClassMethodFromStack()
    {
        Throwable t = new Throwable();
        t.fillInStackTrace();
        
        ThrowableParser tp = new ThrowableParser( t );
        for (StackTraceLineParser stlp = tp.next(); stlp != null;
                stlp = tp.next() )
        {
            if (stlp.getClassName().equals( getOwnerName() ))
            {
                String methodName = stlp.getMethodName();
                LOG.debug( "Owning method for stack trace is '"+methodName+
                    "'", t );
                return methodName;
            }
        }
        
        // could not be found
        return null;
    }
    
    
    /**
     * Uses the owner class and the <tt>findClassMethodFromStack</tt>
     * result to crate a TestInfo instance.
     */
    protected TestInfo createTestInfoFromStack()
    {
        return new DefaultTestInfo( getOwnerName(),
            findClassMethodFromStack() );
    }
    
    
    /**
     * Use the given method name and the known owner class to create a new
     * <tt>TestInfo</tt> instance.
     */
    protected TestInfo createTestInfo( String methodName )
    {
        return new DefaultTestInfo( getOwnerName(), methodName );
    }
    
    
    protected MonitorFinder getFinder()
    {
        return this.finder;
    }
    
    
    /**
     * @param info the test info to find its corresponding TestData.
     * @return the TestData associated with the info, or <tt>null</tt> if
     *      it was not found.
     */
    protected TestData getTestData( TestInfo info )
    {
        // this could be hacky...
        try
        {
            return getFinder().getMonitor().getTestData( info );
        }
        catch (IllegalArgumentException iae)
        {
            LOG.warn( "Test "+info+" caused error w/ monitor", iae );
        }
        catch (IllegalStateException ise)
        {
            LOG.warn( "Test "+info+" caused error w/ monitor", ise );
        }
        return null;
    }
    
    
    protected Class getOwner()
    {
        return this.owner;
    }
    
    
    protected String getOwnerName()
    {
        Class c = getOwner();
        String res;
        if (c == null)
        {
            res = null;
        }
        else
        {
            res = c.getName();
        }
        return res;
    }
}

