/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is MacroTracker Bug Tracking System Source Code
 *
 * The Initial Developer of the Original Code is
 * R.J. Keller.
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */
package org.mozdev.MacroTracker.classes;

import java.util.*;
import java.io.Serializable;
import org.mozdev.MacroTracker.classes.*;

import com.trfenv.rlk.Debug;

/**
 *A Bug Database that can be saved to a file using the
 *MacroTracker Database format.
 *@author R.J. Keller <rlk@trfenv.com>
 */
public class MacroTrackerDatabase implements Serializable, iBugDatabase
{
	private List mProducts;
	private List mUsers;
	private List mBugs;

	/** The name of the database.*/
	private String mName;
	
	/**
	 *Returns the name of the database.
	 */
	public String getName()
	{
		return mName;
	}
	
	/**
	 *Sets the name of the BugDatabase
	 */
	public void setName(String aName)
	{
		mName = aName;
	}

	public MacroTrackerDatabase()
	{
		mProducts = new ArrayList();
		mUsers = new LinkedList();
		mBugs = new ArrayList();

		Debug.createObject(this);
	}

	//--BUGS--//
	/**
	 *Adds a bug to the bug database.
	 *@param bug The bug to add to the database.
	 */
	public void addBug(Bug bug)
	{
		mBugs.add(bug);
	}

	/**
	 *Returns whether or not the user taken in is a registered MacroTracker user.
	 *@param user The user to ensure is a valid user.
	 */
	public boolean isValidUser( User user )
	{
		User currentUser;

		for (int i = 0; i < mUsers.size(); i++)
		{
			currentUser = (User)mUsers.get(i);

			if (currentUser == user)
				return true;
		}

		return false;
	}

	/**
	 *Returns whether or not the user's email taken in is a valid MacroTracker user.
	 *@param email The email address of the user to validate.
	 */
	public boolean isValidUser( String email )
	{
		User currentUser;

		for (int i = 0; i < mUsers.size(); i++)
		{
			currentUser = (User)mUsers.get(i);

			if (currentUser.getEmail() == email)
				return true;
		}

		return false;
	}

	/**
	 *Returns how many bugs are in the bug database.
	 */
	public int bugCount()
	{
		return mBugs.size();
	}

	/**
	 *Returns a bug in the bug database.
	 *@param index The bug number of the bug to return.
	 */
	public Bug getBug(int index)
	{
		if (index >= 0 && index < bugCount())
		{
			return (Bug)mBugs.get(index);
		}
		else
		{
			Debug.printException(new IllegalArgumentException("Index is out of bounds."));
			return null;
		}
	}

	/**
	 *Removes a bug from the bug database.
	 *@param index The bug number of the bug to remove from the database<br>(precondition: 0 <= index < bugCount())
	 */
	public void removeBug(int index)
	{
		if (index >= 0 && index < bugCount())
		{
			mBugs.remove(index);
		}
		Debug.printException(new IllegalArgumentException("Index is out of bounds."));
	}

	//--USERS--//
	/**
	 *Adds a user to the bug database.
	 */
	public void addUser(User usr)
	{
		mUsers.add( usr );
	}
	
	/**
	 *Removes a user from the bug database.
	 */
	public void removeUser(String aUser)
	{
		for (int i = 0; i < mUsers.size(); i++)
		{
			User usr = (User)mUsers.get(i);
			if (usr.getUsername().equals(aUser))
				mUsers.remove(i);
		}
	}

	/**
	 *The number of users in the BugDatabase.
	 */
	public int userCount()
	{
		return mUsers.size();
	}

	/**
	 *Returns the user by their ID number.
	 *@param index The ID number of the user to return.
	 */
	public User getUser(int index)
	{
		if (index >= 0 && index < userCount())
		{
			return (User)mUsers.get(index);
		}
		else
		{
			Debug.printException(new IllegalArgumentException("Index is out of bounds."));

			return null;
		}
	}
	
	/*
	 *Class used by the userIterator() method below that creates
	 *a new iterator to iterate through all the users in the
	 *database
	 */
	private class userIterator implements Iterator
	{
		private int mIndex;
		
		public userIterator()
		{
			mIndex = 0;
		}
		
		public boolean hasNext()
		{
			return mIndex != userCount();
		}
		
		public Object next()
		{
			User usr = getUser(mIndex);
			mIndex++;
			return usr;
		}
		
		public void remove()
		{
			removeUser(mIndex-1);
		}
	}
	
	/**
	 *Returns an iterator that will iterate through all of the users
	 *in the BugDatabase.
	 */
	public Iterator userIterator()
	{
		return new userIterator();
	}

	/**
	 *Returns the user with the given email.
	 *@param email The email address of the User to return.<br>(precondition: isValidUser(email) == true)
	 *@return a User object that corresponds with the email address taken in.
	 */
	public User getUser(String email)
	{
		User usr;

		for (int i = 0; i < mUsers.size(); i++)
		{
			usr = (User)mUsers.get(i);

			if (usr.getEmail() == email)
				return usr;
		}
		Debug.printException(new IllegalArgumentException("Index is out of bounds."));

		return null;
	}

	/**
	 *Removes a user from the bug database.
	 *@param index The user ID of the user to remove from the bug database.
	 */
	public void removeUser(int index)
	{
		if (index >= 0 && index < userCount())
		{
			mUsers.remove(index);
		}
		Debug.printException(new IllegalArgumentException("Index is out of bounds."));
	}

	//--PRODUCTS--//
	/**
	 *Adds a Product to the BugDatabase. A Product is a product released from the company that they would like
	 *MacroTracker to report bugs or issues for.
	 *@param productToAdd the product to add to the BugDatabase.
	 */
	public void addProduct(Product productToAdd)
	{
		mProducts.add(productToAdd);
	}

	/**
	 *Returns how many product's are held in this MacroTracker database.
	 */
	public int productCount()
	{
		return mProducts.size();
	}

	/**
	 *Returns the product located at the specified index in the Product database.
	 *@param index The index of the Product you want returned.<br>(precondition: 0 <= index < productCount())
	 */
	public Product getProduct(int index)
	{
		if (index >= 0 && index < mProducts.size())
		{
			return (Product)mProducts.get(index);
		}
		else
		{
			Debug.printException(new IllegalArgumentException("Index is out of bounds."));

			return null;
		}
	}

	/**
	 *Returns a Product based on the productName taken in.
	 */
	public Product getProduct(String productName)
	{
		Product product;

		for (int i = 0; i < mProducts.size(); i++)
		{
			product = (Product)mProducts.get(i);

			if (product.name == productName)
				return product;
		}
		Debug.printException(new IllegalArgumentException("Index is out of bounds."));

		return null;
	}

	/**
	 *Returns an iterator that can iterate through
	 *all of the Product's in this database.
	 */
	public Iterator productIterator()
	{
		return new productIterator();
	}
	
	private class productIterator implements Iterator
	{
		int mIndex;
		
		public productIterator()
		{
			mIndex = 0;
		}
		
		public boolean hasNext()
		{
			return mIndex != productCount();
		}
		
		public void remove()
		{
			removeProduct(mIndex-1);
		}
		
		public Object next()
		{
			return getProduct(mIndex++);
		}
	}

	/**
	 *Removes a Product that bugs are tracked for from the BugDatabase. This product must not have any bugs
	 *using that product or else this function will return false without the Product being removed.
	 *@return Returns whether or not the Product was successfully removed.
	 */
	public boolean removeProduct(int index)
	{
		if (index >= 0 && index < productCount())
		{
			Product productToRemove = (Product)mProducts.get(index);
			Bug currentBug;

			for (int i = 0; i <= mBugs.size(); i++)
			{
				currentBug = (Bug)mBugs.get(i);

				//If this Product is still used in Bugs in the BugDatabase, return false and don't remove the Product, or
				//else problems will arise when viewing/modifying those bugs (since the object reference was destroyed).
				if (currentBug.getProduct() == productToRemove)
					return false;
			}

			mProducts.remove(index);
			return true;
		}
		else
		{
			Debug.printException(new IllegalArgumentException("Index is out of bounds."));
			return false;
		}
	}
	
	/**
	 *Removes a product when the name of the product is taken in. This product must not have any
	 *bugs using that product or else the product will not be removed from the Bug Database.
	 */
	public void removeProduct(String aProduct)
	{
		for (int i = 0; i < mProducts.size(); i++)
		{
			Product curProduct = (Product)mProducts.get(i);
		
			if (curProduct.name.equals(aProduct))
			{
				removeProduct(i);
				return;
			}
		}
	}
}
