/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Contributor(s):
 *   Original code is from mozilla 1.7RC1
 *   Added JRex changes: Rich Giuli <richard.giuli@sri.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the NPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the NPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package org.mozilla.jrex.liveconnect;

import org.mozilla.jrex.exception.JRexException;
import netscape.javascript.JSObject;
import netscape.javascript.JSException;

/**
 * JSObject allows Java to manipulate objects that are
 * defined in JavaScript.
 * Values passed from Java to JavaScript are converted as
 * follows:<ul>
 * <li>JSObject is converted to the original JavaScript object
 * <li>Any other Java object is converted to a JavaScript wrapper,
 *   which can be used to access methods and fields of the java object.
 *   Converting this wrapper to a string will call the toString method
 *   on the original object, converting to a number will call the
 *   doubleValue method if possible and fail otherwise.  Converting
 *   to a boolean will try to call the booleanValue method in the
 *   same way.
 * <li>Java arrays are wrapped with a JavaScript object that understands
 *   array.length and array[index]
 * <li>A Java boolean is converted to a JavaScript boolean
 * <li>Java byte, char, short, int, long, float, and double are converted
 *   to JavaScript numbers
 * </ul>
 * Values passed from JavaScript to Java are converted as follows:<ul>
 * <li>objects which are wrappers around java objects are unwrapped
 * <li>other objects are wrapped with a JSObject
 * <li>strings, numbers and booleans are converted to String, Double,
 *   and Boolean objects respectively
 * </ul>
 * This means that all JavaScript values show up as some kind
 * of java.lang.Object in Java.  In order to make much use of them,
 * you will have to cast them to the appropriate subclass of Object,
 * e.g. <code>(String) window.getMember("name");</code> or
 * <code>(JSObject) window.getMember("document");</code>.
 */
final class JRexJSObject extends JSObject {
    /* the internal object data */
    private int                               internal;
    private int                               jrexLCSession;

    /**
     * it is illegal to construct a JSObject manually
     */
    private JRexJSObject(int jsobj_addr, int jrexLCSession) {
        internal = jsobj_addr;
        this.jrexLCSession = jrexLCSession;
    }

    /**
     * Retrieves a named member of a JavaScript object.
     * Equivalent to "this.<i>name</i>" in JavaScript.
     */
    public Object getMember(String name) {
        try {
            return JRexGetMember(name);
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    private native Object JRexGetMember(String name) throws JRexException;

    /**
     * Retrieves an indexed member of a JavaScript object.
     * Equivalent to "this[<i>index</i>]" in JavaScript.
     */
//    public Object		getMember(int index) { return getSlot(index); }
    public Object getSlot(int index) {
        try {
            return JRexGetSlot(index);
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    private native Object JRexGetSlot(int index) throws JRexException;

    /**
     * Sets a named member of a JavaScript object.
     * Equivalent to "this.<i>name</i> = <i>value</i>" in JavaScript.
     */
    public void setMember(String name, Object value) {
        try {
            JRexSetMember(name, value);
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    public native void JRexSetMember(String name, Object value) throws JRexException;

    /**
     * Sets an indexed member of a JavaScript object.
     * Equivalent to "this[<i>index</i>] = <i>value</i>" in JavaScript.
     */
//    public void 		setMember(int index, Object value) {
//        setSlot(index, value);
//    }
    public void setSlot(int index, Object value) {
        try {
            JRexSetSlot(index, value);
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    public native void JRexSetSlot(int index, Object value) throws JRexException;

    /**
     * Removes a named member of a JavaScript object.
     */
    public void removeMember(String name) {
        try {
            JRexRemoveMember(name);
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    public native void JRexRemoveMember(String name) throws JRexException;

    /**
     * Calls a JavaScript method.
     * Equivalent to "this.<i>methodName</i>(<i>args</i>[0], <i>args</i>[1], ...)" in JavaScript.
     */
    public Object call(String methodName, Object args[]) {
        try {
            // In mozilla debug, an error occurs if an empty object array is passed
            if (args != null && args.length == 0) {
                args = null;
            }
            return JRexCall(methodName, args);
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    public native Object JRexCall(String methodName, Object args[]) throws JRexException;

    /**
     * Evaluates a JavaScript expression. The expression is a string
     * of JavaScript source code which will be evaluated in the context
     * given by "this".
     */
    public Object eval(String s) {
        try {
            return JRexEval(s);
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    public native Object JRexEval(String s) throws JRexException;

    /**
     * Converts a JSObject to a String.
     */
    public String toString() {
        try {
            return JRexToString();
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    public native String JRexToString() throws JRexException;

    // should use some sort of identifier rather than String
    // is "property" the right word?
  //    native String[]                         listProperties();

    public static JRexJSObject	getWindowForSession(JRexLiveConnectSessionImpl lcSession) throws JRexException {
        if (lcSession == null) {
            throw new IllegalArgumentException();
        }
        JRexJSObject ret = GetWindowForSession(lcSession);
        if (ret == null) {
            throw new JRexException("Failed to create an instance of " + JRexJSObject.class.getName());
        }
        ret.jrexLCSession = lcSession.getSessionID();
        return ret;
    }

    private static native JRexJSObject GetWindowForSession(JRexLiveConnectSessionImpl lcSession) throws JSException;

    /**
     * Finalization decrements the reference count on the corresponding
     * JavaScript object.
     */
    protected void	finalize() {
        // The native code only uses internal for JRexLiveConnectSession.
        FinalizeLCSession(internal);
    }

    protected native void	FinalizeLCSession(int internal);

    /**
     * Override java.lang.Object.equals() because identity is not preserved
     * with instances of JSObject.
     */
    public boolean equals(Object obj) {
        try {
            return JRexEquals(obj);
        } catch (JRexException e) {
            JSException jsex = new JSException(e.getMessage());
            jsex.initCause(e);
            throw jsex;
        }
    }

    public native boolean JRexEquals(Object obj) throws JRexException;
}