/*$Id: RulesParser.cpp,v 1.15 2006/09/03 19:00:50 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *  Copyright (C) 2003 Brian Caswell <bmc@snort.org>
 *  Copyright (C) 2003 Michael J. Pomraning <mjp@securepipe.com>
 *  Copyright (C) 2003 Sourcefire, Inc
 *  Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "Common.h"
#include "RulesParser.h"
#include "RuleSyntax.h"
#include "Wrap.h"
#include "Error.h"



#ifndef HAVE_VSNPRINTF

#ifdef HAVE__VSNPRINTF /*MSVC defines this*/
int (*vsnprintf)(char *, size_t, const char *, va_list) = _vsnprintf;
int (*snprintf)(char *, size_t, const char *, ...) = _snprintf;
#else /*HAVE__VSNPRINTF*/
#error "vsnprintf not defined"
#endif /*HAVE__VSNPRINTF*/

#else
#endif /*HAVE_VSNPRINTF*/


/*
 *@error - set to not null if there are some problems compiling 
 *         regular expressions that describe rule syntax. This
 *         generally shouldn't occur, but maybe is possible if
 *         there are problems with pcre library.
 */
RulesParser::RulesParser(char **error)
{
	memset(this, 0, sizeof(*this));
	*error = NULL;
	compileREs();
	if (strlen(errmsg)){
		*error = errmsg;
		freeREs();
	}
};

RulesParser::~RulesParser()
{
	freeREs();
};

void RulesParser::err_append(const char *fmt, ...)
{
	char buf[ERRBUFLEN];
	va_list ap;
	va_start(ap, fmt);
	vsnprintf(buf, ERRBUFLEN, fmt, ap);
	va_end(ap);
	int l;
	
	if ((l = ERRBUFLEN - strlen(errmsg)) > 1)
		strncat(errmsg, buf, l);
	if ((l = ERRBUFLEN - strlen(errmsg)) > 1)
		strncat(errmsg, "\n", l);
	errmsg[ERRBUFLEN - 1] = 0;
}

	
bool RulesParser::emptyString(char *p)
{
	for(; *p; p++)
		if (!isspace(*p))
			return false;
	return true;
}


bool RulesParser::comentedString(char *p)
{
	for(; *p; p++)
		if (!isspace(*p)){
			if (*p == '#')
				return true;
			return false;
		}
	return true;
}

int RulesParser::compileRE(pcre **re, const char *pattern, const char *name)
{
	const char *error;
	int erroffset;

	*re = pcre_compile(pattern,              /* the pattern */
			   0,                    /* default options */
			   &error,               /* for error message */
			   &erroffset,           /* for error offset */
			   NULL);                /* use default character tables */
	
	if (error){
		err_append("Fatal error: '%s' invalid %s rule syntax at position %d: %s",
			   pattern, name, erroffset, error);
		return -1;
	}

	
	return 0;
}

int RulesParser::compileREs()
{
	
	if (compileRE(&ruleRE, rulePattern, "general") ||
	    compileRE(&msgRE, msgPattern, "msg") ||
	    compileRE(&referenceRE, referencePattern, "reference") ||
	    compileRE(&fidRE, fidPattern, "fid") ||
	    compileRE(&revRE, revPattern, "rev") ||
	    compileRE(&headerRE, headersRegPattern, "headers_re") ||
	    compileRE(&bodyRE, bodyRegPattern, "body_re") ||
	    compileRE(&urlRE, urlRegPattern, "url_re") ||
	    compileRE(&urlContentRE, urlContentPattern, "url_content") ||
	    compileRE(&headersContentRE, headersContentPattern, "headers_content") ||
	    compileRE(&bodyContentRE, bodyContentPattern, "body_content")
	    )
		return -1;	
	return 0;
}

void RulesParser::freeRE(pcre **re)
{
	if (*re){
		free(*re);
		*re = NULL;
	}
}


/*lots of Snort code*/
bool RulesParser::parseRE(char *data, const char *name, 
			  RE *result, char *errbuf, int errbuflen)
{
    const char *error;
    char *re, *free_me = 0;
    char *opts;
    char delimit = '/';
    int erroffset;
    int compile_flags = 0;
    
    if(data == NULL) 
    {
	    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
		     "re requires a regular expression",
		     name, data);
	    goto error;
    }

    if(!(free_me = strdup(data)))
    {
	    snprintf(errbuf, errbuflen, "Out of memory");
	    goto error;
    }
    re = free_me;


    /* get rid of starting and ending whitespace */
    while (isspace((int)re[strlen(re)-1])) re[strlen(re)-1] = '\0';
    while (isspace((int)*re)) re++;

    
    /* 'm//' or just '//' */
        
    if(*re == 'm')
    {
	    re++;
	    if(! *re) goto syntax;
        
	    /* Space as a ending delimiter?  Uh, no. */
	    if(isspace((int)*re)) goto syntax;
	    /* using R would be bad, as it triggers RE */
	    if(*re == 'R') goto syntax;   
	    
	    delimit = *re;
    } 
    else if(!(*re == delimit))
	    goto syntax;

    TRACE("DELIMETER %c", *re);
    /* find ending delimiter, trim delimit chars */
    opts = strrchr(re, delimit);
    if(!((opts - re) > 1)) /* empty regex(m||) or missing delim not OK */
        goto syntax;

    re++;
    *opts++ = '\0';

    /* process any /regex/ismxR options */
    while(*opts != '\0') {
	    switch(*opts) {
	    case 'i':  compile_flags |= PCRE_CASELESS;            break;
	    case 's':  compile_flags |= PCRE_DOTALL;              break;
	    case 'm':  compile_flags |= PCRE_MULTILINE;           break;
	    case 'x':  compile_flags |= PCRE_EXTENDED;            break;
            
		    /* 
		     * these are pcre specific... don't work with perl
		     */ 
	    case 'A':  compile_flags |= PCRE_ANCHORED;            break;
	    case 'E':  compile_flags |= PCRE_DOLLAR_ENDONLY;      break;
	    case 'G':  compile_flags |= PCRE_UNGREEDY;            break;
		    
	    default:
		    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
			     "unknown re option '%c'",
			     name, data, *opts);
		    goto error;
	    }
	    opts++;
    }

    
    /* now compile the re */
    TRACE("pcre: compiling %s", re);
    
    result->re_compiled = pcre_compile(re, compile_flags, &error, &erroffset, NULL);

    if(result->re_compiled == NULL){
	    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s' at position %d: %s",
		     name, data, erroffset, error);
	    goto error;
    }


    /* now study it... */
    result->re_extra = pcre_study(result->re_compiled, 0, &error);
    result->pattern = data;
    
    free(free_me);

    return true;

 syntax:
    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': unable to parse regular expression",
	     name, data);
 error:
    if(free_me) 
	    free(free_me);
    return false;
}


/****************************************************************************
 *
 * Slightly modified ParsePattern(char * ...) function from Snort 
 * ./detection-plugins/sp_pattern_match.c 
 * Author thinks this code is ugly, it works - good enough for me
 ***************************************************************************/
bool RulesParser::parseContent(char *rule, const char *name,
			       PatternMatchData *ds_idx, char *errbuf, int errbuflen)
{
    /*TODO: it was unsigned in original code*/
    char tmp_buf[2048];

    /* got enough ptrs for you? */
    char *start_ptr;
    char *end_ptr;
    char *idx;
    char *dummy_idx;
    char *dummy_end;
    char *tmp;
    char hex_buf[3];
    unsigned int dummy_size = 0;
    int size;
    int hexmode = 0;
    int hexsize = 0;
    int pending = 0;
    int cnt = 0;
    int literal = 0;
    //int exception_flag = 0;
    
    /* clear out the temp buffer */
    memset(tmp_buf, 0, 2048);
    
    if(rule == NULL){	    
	    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s' empty content",
		     name, rule);
	    return false;
    }

    while(isspace((int)*rule))
        rule++;

    /*not supported
    if(*rule == '!')
    {
        exception_flag = 1;
    }
    */

    /* find the start of the data */
    start_ptr = strchr(rule, '"');

    if(start_ptr == NULL)
    {
	    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
		     "content data needs to be enclosed in quotation marks (\")",
		     name, rule);
	    return false;
    }

    /* move the start up from the beggining quotes */
    start_ptr++;

    /* find the end of the data */
    end_ptr = strrchr(start_ptr, '"');

    if(end_ptr == NULL)
    {
	    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
		     "content data needs to be enclosed in quotation marks (\")",
		     name, rule);
	    return false;
    }

    /* Move the null termination up a bit more */
    *end_ptr = '\0';

    /* Is there anything other than whitespace after the trailing
     * double quote? */
    tmp = end_ptr + 1;
    while (*tmp != '\0' && isspace ((int)*tmp))
        tmp++;

    if (strlen (tmp) > 0)
    {
	    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
		     "missing semicolon after trailing double quote.",
		     name, rule);
	    return false;
    }

    /* how big is it?? */
    size = end_ptr - start_ptr;

    /* uh, this shouldn't happen */
    FK_ASSERT(size > 0);

    /* set all the pointers to the appropriate places... */
    idx = start_ptr;

    /* set the indexes into the temp buffer */
    dummy_idx = tmp_buf;
    dummy_end = (dummy_idx + size);

    /* why is this buffer so small? */
    memset(hex_buf, 0, 3);
    memset(hex_buf, '0', 2);

    /* BEGIN BAD JUJU..... */
    while(idx < end_ptr)
    {
        TRACE("processing char: %c", *idx);
        switch(*idx)
        {
            case '|':
                TRACE("Got bar... ");
                if(!literal)
                {
                    TRACE("not in literal mode... ");
                    if(!hexmode)
                    {
                        TRACE("Entering hexmode");
                        hexmode = 1;
                    }
                    else
                    {
                        TRACE("Exiting hexmode");

                        /*
                        **  Hexmode is not even.
                        */
                        if(!hexsize || hexsize % 2)
                        {
				snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
					 "content hexmode argument has invalid "
					 "number of hex digits.  The argument must "
					 "contain a full even byte string.",
					 name, rule);
				return false;
                        }

                        hexmode = 0;
                        pending = 0;
                    }

                    if(hexmode)
                        hexsize = 0;
                }
                else
                {
                    TRACE("literal set, Clearing");
                    literal = 0;
                    tmp_buf[dummy_size] = start_ptr[cnt];
                    dummy_size++;
                }

                break;

            case '\\':
		    TRACE("Got literal char... ");

                if(!literal)
                {
                    /* Make sure the next char makes this a valid
                     * escape sequence.
                     */
                    if (idx [1] != '\0' && strchr ("\\\":;", idx [1]) == NULL)
                    {
			    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
				     "bad escape sequence starting with '%s'",
				     name, rule, idx);
			    return false;
                    }

                    TRACE("Setting literal");

                    literal = 1;
                }
                else
                {
                    TRACE("Clearing literal");
                    tmp_buf[dummy_size] = start_ptr[cnt];
                    literal = 0;
                    dummy_size++;
                }

                break;
            case '"':
                if (!literal) {
			snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
				 "Non-escaped '\"' character",
				 name, rule);
			return false;
                }
                /* otherwise process the character as default */
            default:
                if(hexmode)
                {
                    if(isxdigit((int) *idx))
                    {
                        hexsize++;

                        if(!pending)
                        {
                            hex_buf[0] = *idx;
                            pending++;
                        }
                        else
                        {
                            hex_buf[1] = *idx;
                            pending--;

                            if(dummy_idx < dummy_end)
                            {                            
			            tmp_buf[dummy_size] = (unsigned char) 
                                    strtol(hex_buf, (char **) NULL, 16)&0xFF;

                                dummy_size++;
                                memset(hex_buf, 0, 3);
                                memset(hex_buf, '0', 2);
                            }
                            else
                            {
				    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax: "
					     "buffer overflow, make a smaller "
					     "pattern please! (Max size = 2048)",
					     name);
				    return false;
                            }
                        }
                    }
                    else
                    {
                        if(*idx != ' ')
                        {
				snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
					     "What is this "
					 "\"%c\"(0x%X) doing in your binary "
					 "buffer?  Valid hex values only please"
					 "(0x0 - 0xF) Position: %d",
					 name, rule, (char) *idx, (char) *idx, cnt);
				return false;
                        }
                    }
                }
                else
                {
                    if(*idx >= 0x1F && *idx <= 0x7e)
                    {
                        if(dummy_idx < dummy_end)
                        {
                            tmp_buf[dummy_size] = start_ptr[cnt];
                            dummy_size++;
                        }
                        else
                        {
				snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
					 "dummy buffer overflow", name, rule);
				return false;
                        }

                        if(literal)
                        {
                            literal = 0;
                        }
                    }
                    else
                    {
                        if(literal)
                        {
                            tmp_buf[dummy_size] = start_ptr[cnt];
                            dummy_size++;
                            TRACE("Clearing literal");
                            literal = 0;
                        }
                        else
                        {
				snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
					 "character value out of range, try a binary buffer",
					 name, rule);
				return false;
                        }
                    }
                }

                break;
        }

        dummy_idx++;
        idx++;
        cnt++;
    }
    /* ...END BAD JUJU */

    /* error prunning */

    if (literal) {
	    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
		     "backslash escape is not completed", name, rule);
	    return false;
    }
    if (hexmode) {
	    snprintf(errbuf, errbuflen, "Incorrect '%s' syntax '%s': "
		     "hexmodee is not completed", name, rule);
	    return false;
    }

    
    if((ds_idx->pattern_buf = (unsigned char *) calloc(dummy_size+1, sizeof(char))) 
       == NULL){
	    snprintf(errbuf, errbuflen,("Out of memory"));;
	    return false;
    }

    memcpy(ds_idx->pattern_buf, tmp_buf, dummy_size);

    ds_idx->pattern_size = dummy_size;

    //ds_idx->exception_flag = exception_flag;

    return true;
}

void RulesParser::freeREs()
{
	freeRE(&ruleRE);
	freeRE(&msgRE);
	freeRE(&referenceRE);
	freeRE(&fidRE);
	freeRE(&revRE);
	freeRE(&headerRE);
	freeRE(&bodyRE);
	freeRE(&urlRE);
}


void RulesParser::remove_substring(char *str, unsigned int b, unsigned int e)
{
	if (e == strlen(str))
		str[b] = 0;
	else
		memmove(str + b, str + e, strlen(str) - e + 1);
}


bool RulesParser::parseReference(char *options, Rule *rule)
{
	int rc;	
	int ovector[OVECCOUNT];	
	reference ref;


	rc = pcre_exec(referenceRE, NULL, options, strlen(options), 
			       0, 0, ovector, OVECCOUNT);
	if (rc < 0)
		return false;
		
	FK_ASSERT(rc == 3);
		
	Pcre_get_substring(options, ovector, rc, 1, (const char **)&ref.system);
	Pcre_get_substring(options, ovector, rc, 2, (const char **)&ref.ruleId);
	rule->references.push_back(ref);	
	remove_substring(options, ovector[0], ovector[1]);
	return true;
}

char *RulesParser::parseOption(pcre *re, char *options)
{
	int rc;	
	int ovector[OVECCOUNT];	
	char *buf;
	rc = pcre_exec(re, NULL, options, strlen(options), 
		       0, 0, ovector, OVECCOUNT);	
	if (rc < 0)
		return NULL;
	
	FK_ASSERT(rc == 2);
	
	Pcre_get_substring(options, ovector, rc, 1, (const char **)&buf);
	remove_substring(options, ovector[0], ovector[1]);
	return buf;
}

char *RulesParser::parseContentOption(pcre *re, char *options, int *nocase)
{
	int rc;	
	int ovector[OVECCOUNT];	
	char *buf;
	
	FK_ASSERT(nocase);
	rc = pcre_exec(re, NULL, options, strlen(options), 
		       0, 0, ovector, OVECCOUNT);	
	if (rc < 0){
		return NULL;
	}
	
	if (rc == 3){
		TRACE("nocase set");
		*nocase = 1;
	}
	else{
		TRACE("nocase not set");
		*nocase = 0;
	}
	
	Pcre_get_substring(options, ovector, rc, 1, (const char **)&buf);
	remove_substring(options, ovector[0], ovector[1]);
	return buf;
}

char *RulesParser::parseOptions(char *options, Rule *rule)
{	
	char *res;
	static char errbuf[ERRBUFLEN];
	RE re;
	PatternMatchData cont;
	
	TRACE("PARSING OPTIONS %s", options);
	rule->msg = parseOption(msgRE, options);

	if (!(res = parseOption(fidRE, options))){
		rule->fid = 0;
	}
	else{
		rule->fid = atoi(res);
		free(res);
	}
	
	if (!(res = parseOption(revRE, options))){
		rule->rev = 0;
	}
	else{
		rule->rev = atoi(res);
		free(res);
	}
	
	while ((res = parseOption(urlRE, options))){
		if (!parseRE(res, "url_re:", &re, errbuf, ERRBUFLEN)){
			free(res);
			return errbuf;
		}		
		rule->urlREs.push_back(re);
	}
	while ((res = parseOption(headerRE, options))){
		if (!parseRE(res, "headers_re:", &re, errbuf, ERRBUFLEN)){
			free(res);
			return errbuf;
		}
		rule->headersREs.push_back(re);
	}
	
	while ((res = parseOption(bodyRE, options))){
		if (!parseRE(res, "body_re:", &re, errbuf, ERRBUFLEN)){
			free(res);
			return errbuf;
		}
		rule->bodyREs.push_back(re);		
	}	
	
	while ((res = parseContentOption(urlContentRE, options, &cont.nocase))){
		
		if (!parseContent(res, "url_content:", &cont, errbuf, ERRBUFLEN)){
			free(res);
			return errbuf;
		}
		cont.parent = rule;
		rule->url_content.push_back(cont);
	}
	
	while ((res = parseContentOption(headersContentRE, options, &cont.nocase))){
		
		if (!parseContent(res, "headers_content:", &cont, errbuf, ERRBUFLEN)){
			free(res);
			return errbuf;
		}
		cont.parent = rule;
		rule->headers_content.push_back(cont);
	}
	
	while ((res = parseContentOption(bodyContentRE, options, &cont.nocase))){
		
		if (!parseContent(res, "body_content:", &cont, errbuf, ERRBUFLEN)){
			free(res);
			return errbuf;
		}
		cont.parent = rule;
		rule->body_content.push_back(cont);
	}

	while (parseReference(options, rule))
		;
	
	TRACE("MSG = %s", rule->msg);
	TRACE("FID = %d", rule->fid);
	TRACE("REV = %d", rule->rev);
	TRACE("References");
	list<RE>::iterator it;
	list<reference>::iterator it2;
	
	for(it2 = rule->references.begin(); it2 != rule->references.end(); ++it2)
		TRACE("\t %s : %s", it2->system, it2->ruleId);
	TRACE("Url rules");
	for(it = rule->urlREs.begin(); it != rule->urlREs.end(); ++it)
		TRACE("\t %s", it->pattern);
	TRACE("Header rules");
	for(it = rule->headersREs.begin(); it != rule->headersREs.end(); ++it)
		TRACE("\t %s", it->pattern);
	TRACE("Body rules");
	for(it = rule->bodyREs.begin(); it != rule->bodyREs.end(); ++it)
		TRACE("\t %s", it->pattern);	
	
	TRACE("OPTIONS LEFT: %s", options);
	if (!emptyString(options)){
		snprintf(errbuf, ERRBUFLEN, "Incorrect option syntax: %s", options);
		return errbuf;
	}
	return NULL;
}

Rule* RulesParser::parseRule(const char *rulebuf, char **error)
{
	Rule *rule;
	static char errbuf[ERRBUFLEN];
	int rc, i;	
	int ovector[OVECCOUNT];
	char *erropt, *options;

	FK_ASSERT(error);

	rule = new Rule();
	if (!rule){
		snprintf(errbuf, ERRBUFLEN, "Not enough memory");
		*error = errbuf;
		return NULL;
	}
	errbuf[0] = 0;
	
		
	rc = pcre_exec(ruleRE,         /* the compiled pattern */
		       NULL,           /* no extra data - didn't study the pattern */
		       rulebuf,        /* the subject string */
		       strlen(rulebuf),/* the length of the subject */
		       0,              /* start at offset 0 in the subject */
		       0,              /* default options */
		       ovector,        /* output vector for substring information */
		       OVECCOUNT);     /* number of elements in the output vector */
	
	if (rc < 0){
		switch(rc){
		case PCRE_ERROR_NOMATCH: 
			snprintf(errbuf, ERRBUFLEN, "Rule has to have form: action (options)");
			break;
		default: 
			/*some other error*/
			snprintf(errbuf, ERRBUFLEN, "Matching error %d\n", rc); 
			break;
		}
		goto error;
	}		
	
	/* Match succeded */		
	FK_ASSERT(rc == 3);
			
	TRACE("Match succeeded");
	Pcre_get_substring(rulebuf, ovector, rc, 2, (const char **)&options);
		
	erropt = parseOptions(options, rule);
	free(options);
		
	if (erropt){
		snprintf(errbuf, ERRBUFLEN, "%s ", erropt);
		goto error;
	}
	
	Pcre_get_substring(rulebuf, ovector, rc, 1, (const char **)&rule->action);
	
	for(i = 0; i < NACTIONS; ++i)
		if (strcmp(rule->action, knownActions[i]) == 0){
			rule->actionID = i;
			break;
		}
	
	if (i == NACTIONS){
		snprintf(errbuf, ERRBUFLEN, "Unknown action %s ", rule->action);
		goto error;
	}
		
	TRACE("action: %s", rule->action);
	TRACE("-----------------------------");
	if (strlen(errbuf) == 0){
		*error = NULL;
		return rule;
	}
 error:
	rule->reset();
	delete rule;
	*error = errbuf;
	return NULL;
}

void RulesParser::parseLine(Rules *rules, char *line, int linenr)
{
	char *error;
	Rule *newRule;
	
		
	if (comentedString(line))
		return;
	
	newRule = parseRule(line, &error);
	
	if (error){
		err_append("Error at line %d: %s", linenr, error);
		TRACE("error = %s", error);
	}
	else
		rules->addRule(newRule);
	
	return;
}

/*
  Read rules from a given string.
  @rules - Rules object to which succesfully read rules are added
  @input - file pointer to read rules from
  @error - pointer to an error message buffer or NULL if there is no error.
           Error message is a static buffer which shouldn't
           be freed. If some rules have incorrect syntax, this 
x	   routine creates suitable error message, but rules
	   that are correct are returned anyway.
  @return - 
 */
void RulesParser::parseRules(Rules *rules, const char *input, char **error)
{
	int linenr = 0;
	char buf[RULEBUFLEN];
	int idx = 0;
	int done = 0;
	errmsg[0] = 0;
	*error = errmsg;	
	
	if (!input){
		err_append("input string not specified");
		return;
	}
	
	if (!ruleRE){
		err_append("Fatal error: ruleRE object wasn't successfully created");
		return;
	}

	while(!done){	
		linenr++;	
		for(;input[idx] && input[idx] != '\n' && idx < RULEBUFLEN; idx++)
			buf[idx] = input[idx];
		if (!input[idx])
			done = 1;
		if (idx == RULEBUFLEN){
			err_append("Line %d too long", linenr);
			break;
		}
		buf[idx] = 0;
		parseLine(rules, buf, linenr);
		input += idx + 1;
		idx = 0;
	}
	
	       
	if (strlen(errmsg) == 0)
		*error = NULL;
}


