/*$Id: Firekeeper.cpp,v 1.19 2006/08/20 16:39:54 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "nsCOMPtr.h"
#include "nsXPCOM.h"
#include "nsMemory.h"
#include "nsCRT.h"
#include "nsStringAPI.h"
#include "nsEmbedString.h"

#include "nsIHttpProtocolHandler.h"
#include "nsIComponentManager.h"

#include "nsIServiceManager.h"

#include "Firekeeper.h"
#include "fkHttpHandlerFactory.h"
#include "Error.h"
#include "Common.h"

static const nsIID kIComponentRegistrarIID = NS_ICOMPONENTREGISTRAR_IID;
static const nsCID kfkHttpHandlerCID  = FK_HTTPHANDLER_CID;
static const nsCID kfkHttpsHandlerCID = FK_HTTPSHANDLER_CID;

Rules *rules = nsnull;
nsCOMPtr<fkIJudge> fkJudge = nsnull;
Firekeeper *firekeeper = nsnull;

/*TODO: this should be configurable*/
const char *rulesOrder[]={"pass", "drop", "alert"};

Firekeeper::Firekeeper(): httpHandler(nsnull), httpsHandler(nsnull)
{	
	TRACE("constructor %08x", this);
}

Firekeeper::~Firekeeper()
{
	TRACE("destructor %08x", this);
	firekeeper = nsnull;
}

NS_IMPL_ISUPPORTS2(Firekeeper, fkIFirekeeper,
		   nsIObserver)


Firekeeper*
Firekeeper::GetInstance()
{
	TRACE("");
	if (firekeeper == nsnull){
		firekeeper = new Firekeeper();
		if (!firekeeper)
			return nsnull;

		/*don't destroy firekeeper*/
		NS_ADDREF(firekeeper);	
			
		nsresult rv = firekeeper->init();
		if (NS_FAILED(rv))
			return nsnull;
	}
	
	NS_IF_ADDREF(firekeeper);
	return firekeeper;
}
 

NS_IMETHODIMP 
Firekeeper::init()
{
	TRACE("Firekeeper init() called");
	rules = nsnull;	
	rules = new Rules();
	if (!rules || rules->init() < 0)
		return NS_ERROR_OUT_OF_MEMORY;		
	

	nsCOMPtr<nsIServiceManager> servman; 
	nsresult rv = NS_GetServiceManager(getter_AddRefs(servman)); 
	if (NS_FAILED(rv)){
		TRACE("Can't get ServiceManager");
		return rv;
	}

	nsCOMPtr<nsIComponentManager> compmgr; 
	rv = NS_GetComponentManager(getter_AddRefs(compmgr)); 
	if (NS_FAILED(rv)){
		TRACE("Can't get ComponentManager");
		return rv;
	}
	
	nsCOMPtr<nsIComponentRegistrar> compreg;	
	rv = compmgr->QueryInterface(kIComponentRegistrarIID, getter_AddRefs(compreg));
	if (NS_FAILED(rv)){
		TRACE("Can't get ComponentRegistrar");
		return rv;
	}

	/*register new HTTP handler*/
	nsCOMPtr<nsIHttpProtocolHandler> oldHandler;
	rv = servman->GetServiceByContractID(HTTPHANDLER_CONTRACTID,
					     NS_GET_IID(nsIHttpProtocolHandler),
					     getter_AddRefs(oldHandler));
	if (NS_FAILED(rv))
		return rv;
	
	if (!oldHandler){
		TRACE("oldHandler == NULL");
		return NS_ERROR_FAILURE;
	}
	httpHandler = new fkHttpHandler();
	if (!httpHandler){
		TRACE("Can't create httpHandler");
		return NS_ERROR_OUT_OF_MEMORY;
	}
	rv = httpHandler->init(oldHandler);
	if (NS_FAILED(rv)){
		TRACE("Can't init http handler");
		return rv;
	}	
	NS_ADDREF(httpHandler);
	
	fkHttpHandlerFactory *httpHandlerFactory = new fkHttpHandlerFactory(httpHandler);
	if (!httpHandlerFactory){
		TRACE("Can't create httpHandlerFactory");
		return NS_ERROR_OUT_OF_MEMORY;
	}
	NS_ADDREF(httpHandlerFactory);
	
	rv = compreg->RegisterFactory(kfkHttpHandlerCID,
				      "fkHttpHandler",
				      HTTPHANDLER_CONTRACTID,
				      httpHandlerFactory);
	if (NS_FAILED(rv)){
		TRACE("Can't register factory");
		return rv;
	}

	TRACE("http handler replaced");

	
	/*register new HTTPS handler*/
	nsCOMPtr<nsIHttpProtocolHandler> oldHttpsHandler;
	rv = servman->GetServiceByContractID(HTTPSHANDLER_CONTRACTID,
					     NS_GET_IID(nsIHttpProtocolHandler),
					     getter_AddRefs(oldHttpsHandler));
	if (NS_FAILED(rv))
		return rv;
	
	if (!oldHttpsHandler){
		TRACE("oldHttpsHandler == NULL");
		return NS_ERROR_FAILURE;
	}	
	httpsHandler = new fkHttpHandler();
	if (!httpsHandler){
		TRACE("Can't create httpsHandler");
		return NS_ERROR_OUT_OF_MEMORY;
	}
	rv = httpsHandler->init(oldHttpsHandler);
	if (NS_FAILED(rv)){
		TRACE("Can't init https handler");
		return rv;
	}
	
	NS_ADDREF(httpsHandler);
	
	fkHttpHandlerFactory *httpsHandlerFactory = new fkHttpHandlerFactory(httpsHandler);
	if (!httpsHandlerFactory){
		TRACE("Can't create httpsHandlerFactory");
		return NS_ERROR_OUT_OF_MEMORY;
	}
	NS_ADDREF(httpsHandlerFactory);
	
	rv = compreg->RegisterFactory(kfkHttpsHandlerCID,
				      "fkHttpsHandler",
				      HTTPSHANDLER_CONTRACTID,
				      httpsHandlerFactory);
	if (NS_FAILED(rv)){
		TRACE("Can't register factory");
		return rv;
	}

	TRACE("https handler replaced");
	return NS_OK;
}

NS_IMETHODIMP 
Firekeeper::AddRule(const char *rule, char **errmsg)
{	
	char *error = nsnull;
	if (!errmsg)
		return NS_ERROR_NULL_POINTER;
	
	/*TODO: don't create parser here*/
	RulesParser p(&error);
	if (error){
		err_msg(error);
		return NS_ERROR_FAILURE;
	}
	Rule *newRule =  p.parseRule(rule, &error);
	
	if (error){
		TRACE("error %s", error);
		*errmsg = nsCRT::strdup(error);	
		if (!*errmsg)
			return NS_ERROR_OUT_OF_MEMORY;

		/*can't return NS_ERROR_FAILURE here it would
		  make errmsg variable inaccessible in javascript*/
		return NS_OK;
	}

	rules->addRule(newRule);	
	
	return NS_OK;
}

/* void addRules (in string rules, out string errmsg); */
NS_IMETHODIMP 
Firekeeper::AddRules(const char *rulesbuf, char **errmsg)
{
	char *error;
	if (!errmsg)
		return NS_ERROR_NULL_POINTER;
	
	RulesParser p(&error);
	if (error){
		err_msg(error);
		return NS_ERROR_FAILURE;
	}
	p.parseRules(rules, rulesbuf, &error);
	
	if (error){
		TRACE("error %s", error);
		*errmsg = nsCRT::strdup(error);	
		if (!*errmsg)
			return NS_ERROR_OUT_OF_MEMORY;
		
		/*even when there were parsing errors return NS_OK*/
	}	
	
	return NS_OK;
}


/* void setJudge (in fkIJudge judge); */
NS_IMETHODIMP 
Firekeeper::SetJudge(fkIJudge *judge)
{
	fkJudge = judge;
	return NS_OK;
}


NS_IMETHODIMP 
Firekeeper::Observe(nsISupports *subject, const char* topic, const PRUnichar* data) 
{
	if (!topic || strcmp(topic, "xpcom-startup") != 0)
		return NS_OK;	
	return NS_OK;
}

/* void enable (in boolean tracing); */
NS_IMETHODIMP 
Firekeeper::Enable(PRBool tracingEnabled)
{
	httpHandler->enableTracing(tracingEnabled);
	return NS_OK;
}

/* void isEnabled (out boolean tracing); */
NS_IMETHODIMP 
Firekeeper::IsEnabled(PRBool *tracingEnabled)
{
	if (!tracingEnabled)
		return NS_ERROR_NULL_POINTER;
	*tracingEnabled = httpHandler->isTracingEnabled();
	return NS_OK;
}

/* void resetRules (); */
NS_IMETHODIMP 
Firekeeper::ResetRules()
{
	if (rules == nsnull)
		return NS_ERROR_FAILURE;
	
	rules->resetRules();
	return NS_OK;
}
