/******************************** -*- C -*- ****************************
 *
 *	sbrk-like behavior for separate mmap'ed regions
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 2000, 2001, 2002 Software Foundation, Inc.
 * Written by Paolo Bonzini (redisorganization of GNU mmalloc).
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 ***********************************************************************/


#include "gst.h"
#include "gstpriv.h"

#include <signal.h>
#include <setjmp.h>
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>		/* dpx/2 needs this after sys/types.h */
#include <string.h>

#ifdef WIN32
# include <windows.h>
#else
# include <sys/mman.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef HAVE_STDDEF_H
# include <stddef.h>
#endif

#ifdef HAVE_LIMITS_H
# include <limits.h>
#endif

#if defined(MAP_ANONYMOUS) && !defined(MAP_ANON)
# define MAP_ANON MAP_ANONYMOUS
#endif

#ifndef MAP_FAILED
# define MAP_FAILED ((char *) -1)
#endif

struct heap;

/* We have three possible implementations: Win32, mmap-ing /dev/zero
   or using anonymous mmaps, and anonymous mmaps with MAP_NORESERVE.  */
typedef struct heap_implementation {
  mst_Boolean (*reserve) (struct heap *, size_t);
  void (*release) (struct heap *);
  PTR (*map_in) (struct heap *, size_t);
  void (*map_out) (struct heap *, size_t);
} heap_implementation;

#if defined(WIN32)
static mst_Boolean win32_reserve (struct heap *, size_t);
static void	   win32_release (struct heap *);
static PTR	   win32_map_in (struct heap *, size_t);
static void	   win32_map_out (struct heap *, size_t);
struct heap_implementation heap_impl_tab[] = {
  { win32_reserve, win32_release, win32_map_in, win32_map_out },
  { NULL, NULL, NULL, NULL }
};
#else /* !WIN32 */
# if defined (MAP_ANON) && defined (MAP_NORESERVE)
static mst_Boolean noreserve_reserve (struct heap *, size_t);
static void	   noreserve_release (struct heap *);
static PTR	   noreserve_map_in (struct heap *, size_t);
static void	   noreserve_map_out (struct heap *, size_t);
#endif
static mst_Boolean anon_mmap_reserve (struct heap *, size_t);
static void	   anon_mmap_release (struct heap *);
static PTR	   anon_mmap_map_in (struct heap *, size_t);
static void	   anon_mmap_map_out (struct heap *, size_t);
struct heap_implementation heap_impl_tab[] = {
# if defined (MAP_ANON) && defined (MAP_NORESERVE)
  { noreserve_reserve, noreserve_release, noreserve_map_in, noreserve_map_out },
# endif
  { anon_mmap_reserve, anon_mmap_release, anon_mmap_map_in, anon_mmap_map_out },
  { NULL, NULL, NULL, NULL }
};
#endif /* !WIN32 */


/* Internal structure that defines the format of the heap descriptor.
   This gets written to the base address of the region that we are
   managing. */
struct heap
{
  struct heap_implementation *impl;

  size_t areasize;

  /* The base address of the memory region for this malloc heap.  This
     is the location where the bookkeeping data for mmap and for malloc
     begins. */
  char *base;

  /* The current location in the memory region for this malloc heap
     which represents the end of memory in use. */
  char *breakval;

  /* The end of the current memory region for this malloc heap.  This
     is the first location past the end of mapped memory. */
  char *top;
};

/* This is the internal function for heap_sbrk which receives a struct
   heap instead of the pointer to the base location available to
   clients. */
static PTR heap_sbrk_internal (struct heap *hdp,
			       int size);

/* Cache pagesize-1 for the current host machine.  Note that if the
   host does not readily provide a getpagesize() function, we need to
   emulate it elsewhere, not clutter up this file with lots of kluges
   to try to figure it out. */
static size_t pageround, pagesize;
#ifndef HAVE_GETPAGESIZE
extern int getpagesize ();
#endif

#define PAGE_ALIGN(addr) ((PTR) (((long)(addr) + pageround) & ~pageround))

/* We allocate extra pages for the heap descriptor and answer an
   address that is HEAP_DELTA bytes past the actual beginning of the
   allocation. */
#define HEAP_DELTA	 ((long) PAGE_ALIGN(sizeof (struct heap)))



heap
_gst_heap_create (int size)
{
  struct heap mtemp;
  struct heap *hdp;
  heap hd;

  if (!pageround)
    {
      pagesize = getpagesize ();
      pageround = pagesize - 1;
    }

  /* We start off with the heap descriptor allocated on the stack,
     until we build it up enough to call heap_sbrk_internal() to
     allocate the first page of the region and copy it there.  Ensure
     that it is zero'd and then initialize the fields that we know
     values for. */

  hdp = &mtemp;
  memzero ((char *) hdp, sizeof (mtemp));
  hdp->areasize = size;

  for (hdp->impl = heap_impl_tab;
       hdp->impl->reserve && !hdp->impl->reserve (hdp, size);
       hdp->impl++);

  if (!hdp->impl->reserve)
    {
      errno = ENOMEM;
      return (NULL);
    }

  /* Now try to map in the first page, copy the heap descriptor
     structure there, and arrange to return a pointer to this new copy. 
     If the mapping fails, then close the file descriptor if it was
     opened by us, and arrange to return a NULL. */

  hdp->top = hdp->breakval = hdp->base;
  if ((hdp = heap_sbrk_internal (hdp, HEAP_DELTA)) != NULL)
    {
      hd = ((char *) hdp) + HEAP_DELTA;
      memcpy (hdp, &mtemp, sizeof (mtemp));
    }
  else
    {
      hd = NULL;
      hdp->impl->release (hdp);
    }

  return hd;
}

heap
_gst_heap_destroy (heap hd)
{
  struct heap mtemp;

  if (hd != NULL)
    {
      mtemp = *(struct heap *) (hd - HEAP_DELTA);

      /* Now unmap all the pages associated with this region by asking
         for a negative increment equal to the current size of the
         region. */

      if ((heap_sbrk_internal (&mtemp, mtemp.base - mtemp.top)) == NULL)
	/* Update the original heap descriptor with any changes */
	*(struct heap *) (hd - HEAP_DELTA) = mtemp;
      else
	{
	  mtemp.impl->release (&mtemp);
	  hd = NULL;
	}
    }

  return (hd);
}

PTR
_gst_heap_sbrk (heap hd,
		size_t size)
{
  struct heap *hdp;

  hdp = (struct heap *) (hd - HEAP_DELTA);
  return heap_sbrk_internal (hdp, size);
}

PTR
heap_sbrk_internal (struct heap * hdp,
		    int size)
{
  char *result = NULL;
  size_t mapbytes;		/* Number of bytes to map */
  char *moveto;			/* Address where we wish to move "break 
				   value" to */
  char *mapto;			/* Address we actually mapped to */

  if (size == 0)
    /* Just return the current "break" value. */
    result = hdp->breakval;

  else if (size < 0)
    {
      /* We are deallocating memory.  If the amount requested would
         cause us to try to deallocate back past the base of the mmap'd 
         region then do nothing, and return NULL.  Otherwise,
         deallocate the memory and return the old break value. */
      if (hdp->breakval + size >= hdp->base)
	{
	  result = (PTR) hdp->breakval;
	  hdp->breakval += size;
	  moveto = PAGE_ALIGN (hdp->breakval);
	  hdp->impl->map_out (hdp, (size_t) (hdp->top - moveto));
	  hdp->top = moveto;
	}
    }
  else if (hdp->breakval + size > hdp->top)
    {
      moveto = PAGE_ALIGN (hdp->breakval + size);
      mapbytes = moveto - hdp->top;
      mapto = hdp->impl->map_in (hdp, mapbytes);
      if (!mapto)
	errno = ENOMEM;
      else
	{
	  hdp->top = moveto;
	  result = (PTR) hdp->breakval;
	  hdp->breakval += size;
	}
    }
  else
    {
      result = (PTR) hdp->breakval;
      hdp->breakval += size;
    }

  return (result);
}


/* Implementation of the four basic primitives for Win32.  */

#ifdef WIN32

mst_Boolean
noreserve_reserve (struct heap *hdp, size_t size)
{
  hdp->base = VirtualAlloc(NULL, size), MEM_RESERVE, PAGE_NOACCESS);
  return hdp->base != NULL;
}

void
noreserve_release (struct heap *hdp)
{
  VirtualFree(hdp->base, hdp->areasize, MEM_RELEASE);
}

PTR
noreserve_map_in (struct heap *hdp, size_t size)
{
  return VirtualAlloc (hdp->top, size, MEM_COMMIT, PAGE_EXECUTE_READWRITE);
}

void
noreserve_map_out (struct heap *hdp, size_t size);
{
  VirtualFree(PAGE_ALIGN (hdp->breakval), size, MEM_DECOMMIT);
}

#else /* !WIN32 */

#if defined (MAP_ANON) && defined (MAP_NORESERVE)
/* Implementation of the four basic primitives when MAP_NORESERVE
   is available.  */

mst_Boolean
noreserve_reserve (struct heap *hdp, size_t size)
{
  hdp->base = mmap (NULL, size, PROT_NONE,
		    MAP_ANON | MAP_PRIVATE | MAP_NORESERVE,
		    -1, 0);
  return hdp->base != MAP_FAILED;
}

void
noreserve_release (struct heap *hdp)
{
  munmap (hdp->base, hdp->areasize);
}

PTR
noreserve_map_in (struct heap *hdp, size_t size)
{
  PTR addr = mmap (hdp->top, size,
	PROT_READ | PROT_WRITE | PROT_EXEC,
	MAP_ANON | MAP_PRIVATE | MAP_FIXED,
	-1, 0);
  return (addr == MAP_FAILED) ? NULL : addr;
}

void
noreserve_map_out (struct heap *hdp, size_t size)
{
  mmap (PAGE_ALIGN (hdp->breakval), size,
	PROT_NONE, MAP_ANON | MAP_PRIVATE | MAP_NORESERVE | MAP_FIXED,
	-1, 0);
}
#endif

/* Implementation of the four basic primitives when MAP_NORESERVE
   is not available.  */

#ifndef MAP_ANON
static int fd;
#endif
static char *baseaddr;
static PTR find_heap_base (void);

mst_Boolean
anon_mmap_reserve (struct heap *hdp, size_t size)
{
  if (!baseaddr)
    {
#ifndef MAP_ANON
      /* Open the file descriptor we need for anonymous mmaps */
      fd = open ("/dev/zero", O_RDWR);
#endif
      baseaddr = find_heap_base ();
    }

  hdp->base = baseaddr;
  baseaddr += size;
  return true;
}

void
anon_mmap_release (struct heap *hdp)
{
  if (baseaddr == hdp->base + hdp->areasize)
    baseaddr = hdp->base;
}

PTR
anon_mmap_map_in (struct heap *hdp, size_t size)
{
  PTR addr;
#ifndef MAP_ANON
  addr = mmap (hdp->top, size,
	PROT_READ | PROT_WRITE | PROT_EXEC,
	MAP_PRIVATE | MAP_FIXED,
	fd, 0);
#else
  addr = mmap (hdp->top, size,
	PROT_READ | PROT_WRITE | PROT_EXEC,
	MAP_ANON | MAP_PRIVATE | MAP_FIXED,
	-1, 0);
#endif
  return (addr == MAP_FAILED) ? NULL : addr;
}

void
anon_mmap_map_out (struct heap *hdp, size_t size)
{
  munmap (PAGE_ALIGN (hdp->breakval), (size));
}

/* This is hairy and a hack.  We have to find a place for our heaps... */

/* This signal handler is used if it is the only means to decide if
   a page is mapped into memory.  We intercept SIGSEGV and decide
   that the heap can be allocated at a given location only if we
   receive a signal.

   We also try access(2) and EFAULT, but it is not ensured that it
   works because the whole EFAULT business is quite unclear; however
   it is worth doing because debugging is way more painful if we
   have SIGSEGV's as part of the normal initialization sequence. */

static RETSIGTYPE not_mapped (int sig);
static jmp_buf already_mapped;

RETSIGTYPE
not_mapped (sig)
     int sig;
{
  _gst_set_signal_handler (sig, not_mapped);
  longjmp (already_mapped, 1);
}

#ifndef MAP_ANON
#define ANON_MMAP(addr) \
  mmap ((addr), pagesize, \
	PROT_READ | PROT_WRITE | PROT_EXEC, \
	MAP_PRIVATE | ((addr) ? MAP_FIXED : 0), \
	fd, 0)
#else
#define ANON_MMAP(addr) \
  mmap ((addr), pagesize, \
	PROT_READ | PROT_WRITE | PROT_EXEC, \
	MAP_ANON | MAP_PRIVATE | ((addr) ? MAP_FIXED : 0), \
	-1, 0)
#endif

PTR
find_heap_base (void)
{
  volatile char *mmapGuess, *higher, *lower;	/* reference addresses */
  volatile char *first = NULL, *second;	/* probed addresses */
  volatile const int *step;

  static const int steps[] = {
    true, 256, 256, 0,		/* try 256 Mb after the higher address */
    true, 128, 256, 0,		/* try 128 Mb after the higher address */
    true, 64, 256, 0,		/* try 64 Mb after the higher address */
    false, 256, 256, 512,	/* try 256 Mb after the lower address */
    false, 128, 256, 384,	/* try 128 Mb after the lower address */
    false, 64, 256, 320,	/* try 64 Mb after the lower address */
    true, 32, 256, 0,		/* try 32 Mb after the higher address */
    true, 32, 128, 0,		/* again, for a smaller heap */
    false, 64, 128, 192,	/* this has a smaller heap too */
    -1
  };

  volatile int test, *testPtr = &test;
  volatile SigHandler oldSegvHandler;

  /* Placate GNU C's warnings about clobbered variables */
  testPtr = (volatile int *) &higher;
  testPtr = (volatile int *) &lower;
  testPtr = (volatile int *) &first;
  testPtr = (volatile int *) &step;
  testPtr = (volatile int *) &test;

  /* Install custom signal handlers to detect failed memory accesses */
  oldSegvHandler = _gst_set_signal_handler (SIGSEGV, not_mapped);

  /* Get two reference addresses which we will base ourselves on */
  mmapGuess = higher = ANON_MMAP (NULL);
  lower = sbrk (0);

  if (higher < lower)
    {
      higher = lower;
      lower = mmapGuess;
    }

  /* Now try each of the possibilities... */
  for (step = steps; *step > -1; step += 4)
    {
      if (higher > lower + (step[3] << 20))
	{
	  first =
	    PAGE_ALIGN ((step[0] ? higher : lower) + (step[1] << 20));
	  second = PAGE_ALIGN (first + (step[2] << 20));

	  /* Try reading the two locations */
	  if (setjmp (already_mapped) == 0)
	    {
	      errno = 0;
	      access ((char *) first, F_OK);
	      if (errno != EFAULT)
		{
		  *testPtr = *first;
	          continue;
		}
	    }
	  if (setjmp (already_mapped) == 0)
	    {
	      errno = 0;
	      access ((char *) second, F_OK);
	      if (errno != EFAULT)
		{
		  *testPtr = *second;
	          continue;
		}
	    }

	  /* Try mmap-ing them */
	  if (ANON_MMAP ((char *) first) == MAP_FAILED)
	    continue;

	  munmap ((char *) first, pagesize);
	  if (ANON_MMAP ((char *) second) == MAP_FAILED)
	    continue;

	  /* Were not readable and could be mmap-ed.  We're done. */
	  munmap ((char *) second, pagesize);
	  break;
	}
    }

  /* Restore things... */
  _gst_set_signal_handler (SIGSEGV, oldSegvHandler);

  munmap ((char *) mmapGuess, pagesize);

  if (first == NULL || *step == -1)
    {
      _gst_errorf ("Could not find a place for the Smalltalk heaps.");
      return (NULL);
    }
  else
    return (PTR) (first);
}
#endif /* !WIN32 */
