/**
 * Copyright (c) 2014,2018 Contributors to the Eclipse Foundation
 * 
 * See the NOTICE file(s) distributed with this work for additional
 * information regarding copyright ownership.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.smarthome.model.script.interpreter;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smarthome.core.items.Item;
import org.eclipse.smarthome.core.items.ItemNotFoundException;
import org.eclipse.smarthome.core.items.ItemRegistry;
import org.eclipse.smarthome.core.library.types.QuantityType;
import org.eclipse.smarthome.core.types.Type;
import org.eclipse.smarthome.model.script.engine.ScriptError;
import org.eclipse.smarthome.model.script.engine.ScriptExecutionException;
import org.eclipse.smarthome.model.script.lib.NumberExtensions;
import org.eclipse.smarthome.model.script.scoping.StateAndCommandProvider;
import org.eclipse.smarthome.model.script.script.QuantityLiteral;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmIdentifiableElement;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.xbase.XAbstractFeatureCall;
import org.eclipse.xtext.xbase.XCastedExpression;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XFeatureCall;
import org.eclipse.xtext.xbase.XMemberFeatureCall;
import org.eclipse.xtext.xbase.interpreter.IEvaluationContext;
import org.eclipse.xtext.xbase.interpreter.impl.XbaseInterpreter;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociations;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.typesystem.IBatchTypeResolver;
import org.eclipse.xtext.xbase.typesystem.IResolvedTypes;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReference;

/**
 * The script interpreter handles ESH specific script components, which are not known
 * to the standard Xbase interpreter.
 * 
 * @author Kai Kreuzer - Initial contribution and API
 * @author Oliver Libutzki - Xtext 2.5.0 migration
 */
@SuppressWarnings("restriction")
public class ScriptInterpreter extends XbaseInterpreter {
  @Inject
  private ItemRegistry itemRegistry;
  
  @Inject
  private StateAndCommandProvider stateAndCommandProvider;
  
  @Inject
  private IBatchTypeResolver typeResolver;
  
  @Inject
  @Extension
  private IJvmModelAssociations _iJvmModelAssociations;
  
  @Override
  protected Object _invokeFeature(final JvmField jvmField, final XAbstractFeatureCall featureCall, final Object receiver, final IEvaluationContext context, final CancelIndicator indicator) {
    Object _xblockexpression = null;
    {
      final EObject sourceElement = IterableExtensions.<EObject>head(this._iJvmModelAssociations.getSourceElements(jvmField));
      Object _xifexpression = null;
      if ((sourceElement != null)) {
        Object _xblockexpression_1 = null;
        {
          final Object value = context.getValue(QualifiedName.create(jvmField.getSimpleName()));
          Object _elvis = null;
          if (value != null) {
            _elvis = value;
          } else {
            Object _xblockexpression_2 = null;
            {
              final String fieldName = jvmField.getSimpleName();
              Object _elvis_1 = null;
              Type _stateOrCommand = this.getStateOrCommand(fieldName);
              if (_stateOrCommand != null) {
                _elvis_1 = _stateOrCommand;
              } else {
                Item _item = this.getItem(fieldName);
                _elvis_1 = _item;
              }
              _xblockexpression_2 = _elvis_1;
            }
            _elvis = _xblockexpression_2;
          }
          _xblockexpression_1 = _elvis;
        }
        _xifexpression = _xblockexpression_1;
      } else {
        _xifexpression = super._invokeFeature(jvmField, featureCall, receiver, context, indicator);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  @Override
  protected Object invokeFeature(final JvmIdentifiableElement feature, final XAbstractFeatureCall featureCall, final Object receiverObj, final IEvaluationContext context, final CancelIndicator indicator) {
    try {
      Object _xblockexpression = null;
      {
        if (((feature != null) && feature.eIsProxy())) {
          if ((featureCall instanceof XMemberFeatureCall)) {
            final XExpression expression = ((XMemberFeatureCall)featureCall).getMemberCallTarget();
            IResolvedTypes _resolveTypes = this.typeResolver.resolveTypes(expression);
            LightweightTypeReference _actualType = null;
            if (_resolveTypes!=null) {
              _actualType=_resolveTypes.getActualType(expression);
            }
            String _identifier = null;
            if (_actualType!=null) {
              _identifier=_actualType.getIdentifier();
            }
            final String type = _identifier;
            String _concreteSyntaxFeatureName = ((XMemberFeatureCall)featureCall).getConcreteSyntaxFeatureName();
            String _plus = ("\'" + _concreteSyntaxFeatureName);
            String _plus_1 = (_plus + "\' is not a member of \'");
            String _plus_2 = (_plus_1 + type);
            String _plus_3 = (_plus_2 + "\'");
            ScriptError _scriptError = new ScriptError(_plus_3, featureCall);
            throw new ScriptExecutionException(_scriptError);
          } else {
            if ((featureCall instanceof XFeatureCall)) {
              String _concreteSyntaxFeatureName_1 = ((XFeatureCall)featureCall).getConcreteSyntaxFeatureName();
              String _plus_4 = ("The name \'" + _concreteSyntaxFeatureName_1);
              String _plus_5 = (_plus_4 + 
                "\' cannot be resolved to an item or type");
              ScriptError _scriptError_1 = new ScriptError(_plus_5, featureCall);
              throw new ScriptExecutionException(_scriptError_1);
            } else {
              String _concreteSyntaxFeatureName_2 = featureCall.getConcreteSyntaxFeatureName();
              String _plus_6 = ("Unknown variable or command \'" + _concreteSyntaxFeatureName_2);
              String _plus_7 = (_plus_6 + "\'");
              ScriptError _scriptError_2 = new ScriptError(_plus_7, featureCall);
              throw new ScriptExecutionException(_scriptError_2);
            }
          }
        }
        _xblockexpression = super.invokeFeature(feature, featureCall, receiverObj, context, indicator);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  protected Type getStateOrCommand(final String name) {
    Iterable<Type> _allTypes = this.stateAndCommandProvider.getAllTypes();
    for (final Type type : _allTypes) {
      String _string = type.toString();
      boolean _equals = Objects.equal(_string, name);
      if (_equals) {
        return type;
      }
    }
    return null;
  }
  
  protected Item getItem(final String name) {
    try {
      return this.itemRegistry.getItem(name);
    } catch (final Throwable _t) {
      if (_t instanceof ItemNotFoundException) {
        return null;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Override
  protected boolean eq(final Object a, final Object b) {
    if (((a instanceof Type) && (b instanceof Number))) {
      return NumberExtensions.operator_equals(((Type) a), ((Number) b));
    } else {
      if (((a instanceof Number) && (b instanceof Type))) {
        return NumberExtensions.operator_equals(((Type) b), ((Number) a));
      } else {
        return super.eq(a, b);
      }
    }
  }
  
  @Override
  public Object _assignValueTo(final JvmField jvmField, final XAbstractFeatureCall assignment, final Object value, final IEvaluationContext context, final CancelIndicator indicator) {
    Object _xblockexpression = null;
    {
      final EObject sourceElement = IterableExtensions.<EObject>head(this._iJvmModelAssociations.getSourceElements(jvmField));
      Object _xifexpression = null;
      boolean _notEquals = (!Objects.equal(sourceElement, null));
      if (_notEquals) {
        Object _xblockexpression_1 = null;
        {
          context.assignValue(QualifiedName.create(jvmField.getSimpleName()), value);
          _xblockexpression_1 = value;
        }
        _xifexpression = _xblockexpression_1;
      } else {
        _xifexpression = super._assignValueTo(jvmField, assignment, value, context, indicator);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  @Override
  protected Object doEvaluate(final XExpression expression, final IEvaluationContext context, final CancelIndicator indicator) {
    if ((expression == null)) {
      return null;
    } else {
      if ((expression instanceof QuantityLiteral)) {
        return this.doEvaluate(((QuantityLiteral) expression), context, indicator);
      } else {
        return super.doEvaluate(expression, context, indicator);
      }
    }
  }
  
  protected Object doEvaluate(final QuantityLiteral literal, final IEvaluationContext context, final CancelIndicator indicator) {
    String _value = literal.getValue();
    String _plus = (_value + " ");
    String _value_1 = literal.getUnit().getValue();
    String _plus_1 = (_plus + _value_1);
    return QuantityType.valueOf(_plus_1);
  }
  
  @Override
  public Object _doEvaluate(final XCastedExpression castedExpression, final IEvaluationContext context, final CancelIndicator indicator) {
    try {
      try {
        return super._doEvaluate(castedExpression, context, indicator);
      } catch (final Throwable _t) {
        if (_t instanceof RuntimeException) {
          final RuntimeException e = (RuntimeException)_t;
          Throwable _cause = e.getCause();
          if ((_cause instanceof ClassCastException)) {
            final Object result = this.internalEvaluate(castedExpression.getTarget(), context, indicator);
            String _qualifiedName = castedExpression.getType().getType().getQualifiedName();
            String _plus = ((("Could not cast " + result) + " to ") + _qualifiedName);
            ScriptError _scriptError = new ScriptError(_plus, castedExpression);
            throw new ScriptExecutionException(_scriptError);
          } else {
            throw e;
          }
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
