/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd, node*/
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) { //$NON-NLS-0$
        define('orion/Deferred',factory);
    } else if (typeof exports === "object") { //$NON-NLS-0$
        module.exports = factory();
    } else {
        root.orion = root.orion || {};
        root.orion.Deferred = factory();
    }
}(this, function() {
    var queue = [],
        running = false;

    function run() {
        var fn;
        while ((fn = queue.shift())) {
            fn();
        }
        running = false;
    }

	var runAsync = (function() {
		if (typeof process !== "undefined" && typeof process.nextTick === "function") {
			var nextTick = process.nextTick;
    		return function() {
    			nextTick(run);
    		};
		} else if (typeof MutationObserver === "function") {
			var div = document.createElement("div");
			var observer = new MutationObserver(run);
			observer.observe(div, {
            	attributes: true
        	});
        	return function() {
        		div.setAttribute("class", "_tick");
        	};
		}
		return function() {
			setTimeout(run, 0);
		};
	})();

    function enqueue(fn) {
        queue.push(fn);
        if (!running) {
            running = true;
            runAsync();
        }
    }

    function noReturn(fn) {
        return function(result) {
            fn(result);
        };
    }
    
    function settleDeferred(fn, result, deferred) {
    	try {
    		var listenerResult = fn(result);
    		var listenerThen = listenerResult && (typeof listenerResult === "object" || typeof listenerResult === "function") && listenerResult.then;
    		if (typeof listenerThen === "function") {
    			if (listenerResult === deferred.promise) {
    				deferred.reject(new TypeError());
    			} else {
    				var listenerResultCancel = listenerResult.cancel;
    				if (typeof listenerResultCancel === "function") {
    					deferred._parentCancel = listenerResultCancel.bind(listenerResult);
    				} else {
    					delete deferred._parentCancel;
    				}
    				listenerThen.call(listenerResult, noReturn(deferred.resolve), noReturn(deferred.reject), noReturn(deferred.progress));
    			}
    		} else {
    			deferred.resolve(listenerResult);
    		}
    	} catch (e) {
    		deferred.reject(e);
    	}
    }


    /**
     * @name orion.Promise
     * @class Interface representing an eventual value.
     * @description Promise is an interface that represents an eventual value returned from the single completion of an operation.
     *
     * <p>For a concrete class that implements Promise and provides additional API, see {@link orion.Deferred}.</p>
     * @see orion.Deferred
     * @see orion.Deferred#promise
     */
    /**
     * @name then
     * @function
     * @memberOf orion.Promise.prototype
     * @description Adds handlers to be called on fulfillment or progress of this promise.
     * @param {Function} [onResolve] Called when this promise is resolved.
     * @param {Function} [onReject] Called when this promise is rejected.
     * @param {Function} [onProgress] May be called to report progress events on this promise.
     * @returns {orion.Promise} A new promise that is fulfilled when the given <code>onResolve</code> or <code>onReject</code>
     * callback is finished. The callback's return value gives the fulfillment value of the returned promise.
     */
    /**
     * Cancels this promise.
     * @name cancel
     * @function
     * @memberOf orion.Promise.prototype
     * @param {Object} reason The reason for canceling this promise.
     * @param {Boolean} [strict]
     */

    /**
     * @name orion.Deferred
     * @borrows orion.Promise#then as #then
     * @borrows orion.Promise#cancel as #cancel
     * @class Provides abstraction over asynchronous operations.
     * @description Deferred provides abstraction over asynchronous operations.
     *
     * <p>Because Deferred implements the {@link orion.Promise} interface, a Deferred may be used anywhere a Promise is called for.
     * However, in most such cases it is recommended to use the Deferred's {@link #promise} field instead, which exposes a 
     * simplified, minimally <a href="https://github.com/promises-aplus/promises-spec">Promises/A+</a>-compliant interface to callers.</p>
     */
    function Deferred() {
        var result, state, listeners = [],
            _this = this;

        function notify() {
            var listener;
            while ((listener = listeners.shift())) {
                var deferred = listener.deferred;
                var methodName = state === "fulfilled" ? "resolve" : "reject"; //$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$
                var fn = listener[methodName];
                if (typeof fn === "function") { //$NON-NLS-0$
                	settleDeferred(fn, result, deferred);
                } else {
                    deferred[methodName](result);
                }
            }
        }

        function _reject(error) {
            delete _this._parentCancel;
            state = "rejected";
            result = error;
            if (listeners.length) {
                enqueue(notify);
            }
        }

        function _resolve(value) {
            function once(fn) {
                return function(result) {
                    if (!state || state === "assumed") {
                          fn(result);
                    }
                };
            }
            delete _this._parentCancel;
            try {
                var valueThen = value && (typeof value === "object" || typeof value === "function") && value.then;
                if (typeof valueThen === "function") {
                    if (value === _this) {
                        _reject(new TypeError());
                    } else {
                        state = "assumed";
                        var valueCancel = value && value.cancel;
                        if (typeof valueCancel !== "function") {
                            var deferred = new Deferred();
                            value = deferred.promise;
                            try {
                                valueThen(deferred.resolve, deferred.reject, deferred.progress);
                            } catch (thenError) {
                                deferred.reject(thenError);
                            }
                            valueCancel = value.cancel;
                            valueThen = value.then;
                        }
                        result = value;
                        valueThen.call(value, once(_resolve), once(_reject));
                        _this._parentCancel = valueCancel.bind(value);
                    }
                } else {
                    state = "fulfilled";
                    result = value;
                    if (listeners.length) {
                        enqueue(notify);
                    }
                }
            } catch (error) {
                once(_reject)(error);
            }
        }

        function cancel() {
            var parentCancel = _this._parentCancel;
            if (parentCancel) {
                delete _this._parentCancel;
                parentCancel();
            } else if (!state) {
                var cancelError = new Error("Cancel");
                cancelError.name = "Cancel";
                _reject(cancelError);
            }
        }


        /**
         * Resolves this Deferred.
         * @name resolve
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} value
         * @returns {orion.Promise}
         */
        this.resolve = function(value) {
            if (!state) {
                _resolve(value);
            }
            return _this;
        };

        /**
         * Rejects this Deferred.
         * @name reject
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} error
         * @param {Boolean} [strict]
         * @returns {orion.Promise}
         */
        this.reject = function(error) {
            if (!state) {
                _reject(error);
            }
            return _this;
        };

        /**
         * Notifies listeners of progress on this Deferred.
         * @name progress
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} update The progress update.
         * @returns {orion.Promise}
         */
        this.progress = function(update) {
            if (!state) {
                listeners.forEach(function(listener) {
                    if (listener.progress) {
                        try {
                            listener.progress(update);
                        } catch (ignore) {
                            // ignore
                        }
                    }
                });
            }
            return _this.promise;
        };

        this.cancel = function() {
            if (_this._parentCancel) {
                setTimeout(cancel, 0);
            } else {
                cancel();
            }
            return _this;
        };

        // Note: "then" ALWAYS returns before having onResolve or onReject called as per http://promises-aplus.github.com/promises-spec/
        this.then = function(onFulfill, onReject, onProgress) {
        	var deferred = new Deferred();
            deferred._parentCancel = _this.promise.cancel;
            listeners.push({
                resolve: onFulfill,
                reject: onReject,
                progress: onProgress,
                deferred: deferred
            });
            if (state === "fulfilled" || state === "rejected") {
                enqueue(notify);
            }
            return deferred.promise;
        };

        /**
         * The promise exposed by this Deferred.
         * @name promise
         * @field
         * @memberOf orion.Deferred.prototype
         * @type orion.Promise
         */
        this.promise = {
            then: _this.then,
            cancel: _this.cancel
        };
    }

    /**
     * Returns a promise that represents the outcome of all the input promises.
     * <p>When <code>all</code> is called with a single parameter, the returned promise has <dfn>eager</dfn> semantics,
     * meaning that if any input promise rejects, the returned promise immediately rejects, without waiting for the rest of the
     * input promises to fulfill.</p>
     *
     * To obtain <dfn>lazy</dfn> semantics (meaning the returned promise waits for every input promise to fulfill), pass the
     * optional parameter <code>optOnError</code>.
     * @name all
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {orion.Promise[]} promises The input promises.
     * @param {Function} [optOnError] Handles a rejected input promise. <code>optOnError</code> is invoked for every rejected
     * input promise, and is passed the reason the input promise was rejected. <p><code>optOnError</code> can return a value, which
     * allows it to act as a transformer: the return value serves as the final fulfillment value of the rejected promise in the 
     * results array generated by <code>all</code>.
     * @returns {orion.Promise} A new promise. The returned promise is generally fulfilled to an <code>Array</code> whose elements
     * give the fulfillment values of the input promises. <p>However, if an input promise rejects and eager semantics is used, the 
     * returned promise will instead be fulfilled to a single error value.</p>
     */
    Deferred.all = function(promises, optOnError) {
        var count = promises.length,
            result = [],
            rejected = false,
            deferred = new Deferred();

        deferred.then(undefined, function() {
            rejected = true;
            promises.forEach(function(promise) {
                if (promise.cancel) {
                    promise.cancel();
                }
            });
        });

        function onResolve(i, value) {
            if (!rejected) {
                result[i] = value;
                if (--count === 0) {
                    deferred.resolve(result);
                }
            }
        }

        function onReject(i, error) {
            if (!rejected) {
                if (optOnError) {
                    try {
                        onResolve(i, optOnError(error));
                        return;
                    } catch (e) {
                        error = e;
                    }
                }
                deferred.reject(error);
            }
        }

        if (count === 0) {
            deferred.resolve(result);
        } else {
            promises.forEach(function(promise, i) {
                promise.then(onResolve.bind(undefined, i), onReject.bind(undefined, i));
            });
        }
        return deferred.promise;
    };

    /**
     * Applies callbacks to a promise or to a regular object.
     * @name when
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {Object|orion.Promise} value Either a {@link orion.Promise}, or a normal value.
     * @param {Function} onResolve Called when the <code>value</code> promise is resolved. If <code>value</code> is not a promise,
     * this function is called immediately.
     * @param {Function} onReject Called when the <code>value</code> promise is rejected. If <code>value</code> is not a promise, 
     * this function is never called.
     * @param {Function} onProgress Called when the <code>value</code> promise provides a progress update. If <code>value</code> is
     * not a promise, this function is never called.
     * @returns {orion.Promise} A new promise.
     */
    Deferred.when = function(value, onResolve, onReject, onProgress) {
        var promise, deferred;
        if (value && typeof value.then === "function") { //$NON-NLS-0$
            promise = value;
        } else {
            deferred = new Deferred();
            deferred.resolve(value);
            promise = deferred.promise;
        }
        return promise.then(onResolve, onReject, onProgress);
    };

    return Deferred;
}));
/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/EventTarget',[],function() {
	/**
	 * Creates an Event Target
	 *
	 * @name orion.EventTarget
	 * @class Base for creating an Orion event target
	 */
	function EventTarget() {
		this._namedListeners = {};
	}

	EventTarget.prototype = /** @lends orion.EventTarget.prototype */
	{
		/**
		 * Dispatches a named event along with an arbitrary set of arguments. Any arguments after <code>eventName</code>
		 * will be passed to the event listener(s).
		 * @param {Object} event The event to dispatch. The event object MUST have a type field
		 * @returns {boolean} false if the event has been canceled and any associated default action should not be performed
		 * listeners (if any) have resolved.
		 */
		dispatchEvent: function(event) {
			if (!event.type) {
				throw new Error("unspecified type");
			}
			var listeners = this._namedListeners[event.type];
			if (listeners) {
				listeners.forEach(function(listener) {
					try {
						if (typeof listener === "function") {
							listener(event);
						} else {
							listener.handleEvent(event);
						}
					} catch (e) {
						if (typeof console !== 'undefined') {
							console.log(e); // for now, probably should dispatch an ("error", e)
						}
					}			
				});
			}
			return !event.defaultPrevented;
		},

		/**
		 * Adds an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		addEventListener: function(eventName, listener) {
			if (typeof listener === "function" || listener.handleEvent) {
				this._namedListeners[eventName] = this._namedListeners[eventName] || [];
				this._namedListeners[eventName].push(listener);
			}
		},

		/**
		 * Removes an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		removeEventListener: function(eventName, listener) {
			var listeners = this._namedListeners[eventName];
			if (listeners) {
				for (var i = 0; i < listeners.length; i++) {
					if (listeners[i] === listener) {
						if (listeners.length === 1) {
							delete this._namedListeners[eventName];
						} else {
							listeners.splice(i, 1);
						}
						break;
					}
				}
			}
		}
	};
	EventTarget.prototype.constructor = EventTarget;
	
	EventTarget.attach = function(obj) {
		var eventTarget = new EventTarget();
		obj.dispatchEvent = eventTarget.dispatchEvent.bind(eventTarget);
		obj.addEventListener = eventTarget.addEventListener.bind(eventTarget);
		obj.removeEventListener = eventTarget.removeEventListener.bind(eventTarget);
	};
	
	return EventTarget;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd, node*/
/* eslint-disable missing-nls */
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) {
        define('orion/plugin',["orion/Deferred", "orion/EventTarget"], factory);
    } else if (typeof exports === "object") {
        module.exports = factory(require("orion/Deferred"), require("orion/EventTarget"));
    } else {
        root.orion = root.orion || {};
        root.orion.PluginProvider = factory(root.orion.Deferred, root.orion.EventTarget);
    }
}(this, function(Deferred, EventTarget) {

    function _equal(obj1, obj2) {
        var keys1 = Object.keys(obj1);
        var keys2 = Object.keys(obj2);
        if (keys1.length !== keys2.length) {
            return false;
        }
        keys1.sort();
        keys2.sort();
        for (var i = 0, len = keys1.length; i < len; i++) {
            var key = keys1[i];
            if (key !== keys2[i]) {
                return false;
            }
            var value1 = obj1[key],
                value2 = obj2[key];
            if (value1 === value2) {
                continue;
            }
            if (JSON.stringify(value1) !== JSON.stringify(value2)) {
                return false;
            }
        }
        return true;
    }

    function ObjectReference(objectId, methods) {
        this.__objectId = objectId;
        this.__methods = methods;
    }
    
    function PluginProvider(headers, serviceRegistry) {
        var _headers = headers;
        var _connected = false;

        var _currentMessageId = 0;
        var _currentObjectId = 0;
        var _currentServiceId = 0;

        var _requestReferences = {};
        var _responseReferences = {};
        var _objectReferences = {};
        var _serviceReferences = {};
        
        var _services;
        var _remoteServices = {};
        var _registry = serviceRegistry;
        var _connectCallback;
        
        var _ports = [];
        var _shared = false;
        
        var _target = null;
        if (typeof(window) === "undefined") {
            if (self.postMessage) {
                _target = self;
            } else {
                _shared = true;
            }
        } else if (window !== window.parent) {
            _target = window.parent;
        } else if (window.opener !== null) {
            _target = window.opener;
        }        

        function _publish(message, target) {
            target = target || _target;
            if (target) {
                if (typeof(ArrayBuffer) === "undefined") {
                    message = JSON.stringify(message);
                }
                if (target === self || _shared) {
                    target.postMessage(message);
                } else {
                    target.postMessage(message, "*");
                }
            }
        }
        var _notify = _publish;
        var _errHandler = function(evt){
        	_publish({method: "error", error: _serializeError(evt.error)});
        };
        addEventListener("error", _errHandler);
        
        var lastHeartbeat;
        var startTime = new Date().getTime();
        function log(state) {
            if (typeof(localStorage) !== "undefined" && localStorage.pluginLogging) {
            	console.log(state + "(" + (new Date().getTime() - startTime) + "ms)=" + self.location);
        	}
        }
        function heartbeat() {
            var time = new Date().getTime();
            // This timeout depends on the handshake timeout of the plugin registry. Update both accordingly.
            if (lastHeartbeat  && time - lastHeartbeat < 4000) return;
            lastHeartbeat = time;
            _publish({
                method: "loading"
            });
            log("heartbeat");
        }
        heartbeat();

        if (_shared) {
            self.addEventListener("connect", function(evt) {
                var port = evt.ports[0];
                _ports.push(port);
                if (_connected) {
                    var message = {
                        method: "plugin",
                        params: [_getPluginData()]
                    };
                    _publish(message, port);
                } else {
                    heartbeat();
                }
                port.addEventListener("message",  function(evt) {
                	_handleMessage(evt, port);
                });
                port.start();
            });
        }

        function _getPluginData() {
            var services = [];
            // we filter out the service implementation from the data
            Object.keys(_serviceReferences).forEach(function(serviceId) {
                var serviceReference = _serviceReferences[serviceId];
                services.push({
                    serviceId: serviceId,
                    names: serviceReference.names,
                    methods: serviceReference.methods,
                    properties: serviceReference.properties
                });
            });
            return {
            	updateRegistry: !!_registry,
                headers: _headers || {},
                services: services
            };
        }

        function _jsonXMLHttpRequestReplacer(name, value) {
            if (value && value instanceof XMLHttpRequest) {
                var status, statusText;
                try {
                    status = value.status;
                    statusText = value.statusText;
                } catch (e) {
                    // https://bugs.webkit.org/show_bug.cgi?id=45994
                    status = 0;
                    statusText = ""; //$NON-NLS-0
                }
                return {
                    status: status || 0,
                    statusText: statusText
                };
            }
            return value;
        }

        function _serializeError(error) {
            var result = error ? JSON.parse(JSON.stringify(error, _jsonXMLHttpRequestReplacer)) : error; // sanitizing Error object
            if (error instanceof Error) {
                result.__isError = true;
                result.message = result.message || error.message;
                result.name = result.name || error.name;
            }
            return result;
        }

        function _request(message, target) {
            target = target || _target;
            if (!target) {
                return new Deferred().reject(new Error("plugin not connected"));
            }

            message.id = String(_currentMessageId++);
            var d = new Deferred();
            _responseReferences[message.id] = d;
            d.then(null, function(error) {
                if (_connected && error instanceof Error && error.name === "Cancel") {
                    _notify({
                        requestId: message.id,
                        method: "cancel",
                        params: error.message ? [error.message] : []
                    }, target);
                }
            });

            var toStr = Object.prototype.toString;
            message.params.forEach(function(param, i) {
                if (toStr.call(param) === "[object Object]" && !(param instanceof ObjectReference)) {
                    var candidate, methods;
                    for (candidate in param) {
                        if (toStr.call(param[candidate]) === "[object Function]") {
                            methods = methods || [];
                            methods.push(candidate);
                        }
                    }
                    if (methods) {
                        var objectId = _currentObjectId++;
                        _objectReferences[objectId] = param;
                        var removeReference = function() {
                            delete _objectReferences[objectId];
                        };
                        d.then(removeReference, removeReference);
                        message.params[i] = new ObjectReference(objectId, methods);
                    }
                }
            });
            _notify(message, target);
            return d.promise;
        }

        function _throwError(messageId, error, target) {
            if (messageId || messageId === 0) {
                _notify({
                    id: messageId,
                    result: null,
                    error: error
                }, target);
            } else {
                console.log(error);
            }
        }

        function _callMethod(messageId, implementation, method, params, target) {
            params.forEach(function(param, i) {
                if (param && typeof param.__objectId !== "undefined") {
                    var obj = {};
                    param.__methods.forEach(function(method) {
                        obj[method] = function() {
                            return _request({
                                objectId: param.__objectId,
                                method: method,
                                params: Array.prototype.slice.call(arguments)
                            }, target);
                        };
                    });
                    params[i] = obj;
                }
            });
            var response = typeof messageId === "undefined" ? null : {
                id: messageId,
                result: null,
                error: null
            };
            try {
                var promiseOrResult = method.apply(implementation, params);
                if (!response) {
                    return;
                }

                if (promiseOrResult && typeof promiseOrResult.then === "function") {
                    _requestReferences[messageId] = promiseOrResult;
                    promiseOrResult.then(function(result) {
                        delete _requestReferences[messageId];
                        response.result = result;
                        _notify(response, target);
                    }, function(error) {
                        if (_requestReferences[messageId]) {
                            delete _requestReferences[messageId];
                            response.error = _serializeError(error);
                            _notify(response, target);
                        }
                    }, function() {
                        _notify({
                            responseId: messageId,
                            method: "progress",
                            params: Array.prototype.slice.call(arguments)
                        }, target);
                    });
                } else {
                    response.result = promiseOrResult;
                    _notify(response, target);
                }
            } catch (error) {
                if (response) {
                    response.error = _serializeError(error);
                    _notify(response, target);
                }
            }
        }

        function _handleMessage(evnt, target) {
            if (!_shared && evnt.source !== _target && typeof window !== "undefined") {
                return;
            }
            var data = evnt.data;
            var message = (typeof data !== "string" ? data : JSON.parse(data));
            try {
                if (message.method) { // request
                    var method = message.method,
                        params = message.params || [];
                    if ("serviceId" in message) {
                        var service = _serviceReferences[message.serviceId];
                        if (!service) {
                            _throwError(message.id, "service not found", target);
                        } else {
	                        service = service.implementation;
	                        if (method in service) {
	                            _callMethod(message.id, service, service[method], params, target);
	                        } else {
	                            _throwError(message.id, "method not found", target);
	                        }
                    	}
                    } else if ("objectId" in message) {
                        var object = _objectReferences[message.objectId];
                        if (!object) {
                            _throwError(message.id, "object not found", target);
                        } else if (method in object) {
                            _callMethod(message.id, object, object[method], params, target);
                        } else {
                            _throwError(message.id, "method not found", target);
                        }
                    } else if ("requestId" in message) {
                        var request = _requestReferences[message.requestId];
                        if (request && method === "cancel" && request.cancel) {
                            request.cancel.apply(request, params);
                        }
                    } else if ("responseId" in message) {
                        var response = _responseReferences[message.responseId];
                        if (response && method === "progress" && response.progress) {
                            response.progress.apply(response, params);
                        }
                    } else {
                        if ("plugin" === message.method) { //$NON-NLS-0$
                            var manifest = message.params[0];
                            _update({
                                services: manifest.services
                            });
                        } else {
                            throw new Error("Bad method: " + message.method);
                        }
                    }
                } else if (message.id) {
                    var deferred = _responseReferences[String(message.id)];
                    if (deferred) {
	                    delete _responseReferences[String(message.id)];
	                    if (message.error) {
	                        deferred.reject(message.error);
	                    } else {
	                        deferred.resolve(message.result);
	                    }
                    }
                }
            } catch (e) {
                console.log("Plugin._messageHandler " + e);
            }
        }        
        
        function _createServiceProxy(service) {
            var serviceProxy = {};
            if (service.methods) {
                service.methods.forEach(function(method) {
                    serviceProxy[method] = function() {
                        var message = {
                            serviceId: service.serviceId,
                            method: method,
                            params: Array.prototype.slice.call(arguments)
                        };
                        return _request(message);
                    };
                });

                if (serviceProxy.addEventListener && serviceProxy.removeEventListener && EventTarget) {
                    var eventTarget = new EventTarget();
                    var objectId = _currentObjectId++;
                    _objectReferences[objectId] = {
                        handleEvent: eventTarget.dispatchEvent.bind(eventTarget)
                    };
                    var listenerReference = new ObjectReference(objectId, ["handleEvent"]);

                    var _addEventListener = serviceProxy.addEventListener;
                    serviceProxy.addEventListener = function(type, listener) {
                        if (!eventTarget._namedListeners[type]) {
                            _addEventListener(type, listenerReference);
                        }
                        eventTarget.addEventListener(type, listener);
                    };
                    var _removeEventListener = serviceProxy.removeEventListener;
                    serviceProxy.removeEventListener = function(type, listener) {
                        eventTarget.removeEventListener(type, listener);
                        if (!eventTarget._namedListeners[type]) {
                            _removeEventListener(type, listenerReference);
                        }
                    };
                }
            }
            return serviceProxy;
        }

        function _createServiceProperties(service) {
            var properties = JSON.parse(JSON.stringify(service.properties));
            var objectClass = service.names || service.type || [];
            if (!Array.isArray(objectClass)) {
                objectClass = [objectClass];
            }
            properties.objectClass = objectClass;
            return properties;
        }

        function _registerService(service) {
        	if (!_registry) return;
            var serviceProxy = _createServiceProxy(service);
            var properties = _createServiceProperties(service);
            var registration = _registry.registerService(service.names || service.type, serviceProxy, properties);
            _remoteServices[service.serviceId] = {
                registration: registration,
                proxy: serviceProxy
            };
        }

        function _update(input) {
            var oldServices = _services || [];
            _services = input.services || [];

            if (!_equal(_services, oldServices)) {
	            var serviceIds = [];
				_services.forEach(function(service) {
					var serviceId = service.serviceId;
	                serviceIds.push(serviceId);
	                var remoteService = _remoteServices[serviceId];
	                if (remoteService) {
	                    if (_equal(service.methods, Object.keys(remoteService.proxy))) {
	                        var properties = _createServiceProperties(service);
	                        var reference = remoteService.registration.getReference();
	                        var currentProperties = {};
	                        reference.getPropertyKeys().forEach(function(_name) {
	                            currentProperties[_name] = reference.getProperty(_name);
	                        });
	                        if (!_equal(properties, currentProperties)) {
	                            remoteService.registration.setProperties(properties);
	                        }
	                        return;
	                    }
	                    remoteService.registration.unregister();
	                    delete _remoteServices[serviceId];
	                }
	                _registerService(service);
	            });
	            Object.keys(_remoteServices).forEach(function(serviceId) {
	                if (serviceIds.indexOf(serviceId) === -1) {
	                    _remoteServices[serviceId].registration.unregister();
	                    delete _remoteServices[serviceId];
	                }
	            });
           }
           
           if (_connectCallback) {
               _connectCallback();
               _connectCallback = null;
           }
        }

        this.updateHeaders = function(headers) {
            if (_connected) {
                throw new Error("Cannot update headers. Plugin Provider is connected");
            }
            _headers = headers;
        };

        this.registerService = function(names, implementation, properties) {
            if (_connected) {
                throw new Error("Cannot register service. Plugin Provider is connected");
            }

            if (typeof names === "string") {
                names = [names];
            } else if (!Array.isArray(names)) {
                names = [];
            }

            var method = null;
            var methods = [];
            for (method in implementation) {
                if (typeof implementation[method] === 'function') {
                    methods.push(method);
                }
            }
            _serviceReferences[_currentServiceId++] = {
                names: names,
                methods: methods,
                implementation: implementation,
                properties: properties || {},
                listeners: {}
            };
            heartbeat();
        };
        this.registerServiceProvider = this.registerService;

        this.connect = function(callback, errback) {
            if (_connected) {
                if (callback) {
                    callback();
                }
                return;
            }
            removeEventListener("error", _errHandler);
            var message = {
                method: "plugin",
                params: [_getPluginData()]
            };
            if (!_shared) {
                if (!_target) {
                    if (errback) {
                        errback("No valid plugin target");
                    }
                    return;
                }           
                addEventListener("message", _handleMessage, false);
                _publish(message);
            }
            if (typeof(window) !== "undefined") {
            	var head = document.getElementsByTagName("head")[0] || document.documentElement;
            	var title = head.getElementsByTagName("title")[0];
            	if (!title) {
	            	title = document.createElement("title");
	            	title.textContent = _headers ? _headers.name : '';
	            	head.appendChild(title);
	        	}
        	}

            _ports.forEach(function(port) {
                _publish(message, port);
            });
            _connected = true;
            if (_registry) {
            	_connectCallback = callback;
            } else {
	            if (callback) {
	                callback();
	            }
            }
        };

        this.disconnect = function() {
            if (_connected) {
                removeEventListener("message", _handleMessage);
                _ports.forEach(function(port) {
                    port.close();
                });
                _ports = null;
                _target = null;
                _connected = false;
            }
            // Note: re-connecting is not currently supported
        };            
    }
    
    return PluginProvider;
}));

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/arduino/arduinoPlugin',['orion/plugin'], function(PluginProvider) {

	function connect() {
		var headers = {
			name: "Orion Arduino Tool Support",
			version: "1.0",
			description: "This plugin provides Arduino tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-arduino", // TODO could not find a commonly-used value for this
					"extends": "text/x-csrc",
					name: "Arduino",
					extension: ["ino", "pde"]
				}
			] 
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/lib/syntax", [], function() {
	return {
		id: "orion.lib",
		grammars: [{
			id: "orion.lib",
			repository: {
				brace_open: {
					match: "{",
					name: "punctuation.section.block.begin"
				},
				brace_close: {
					match: "}",
					name: "punctuation.section.block.end"
				},
				bracket_open: {
					match: "\\[",
					name: "punctuation.section.bracket.begin"
				},
				bracket_close: {
					match: "\\]",
					name: "punctuation.section.bracket.end"
				},
				parenthesis_open: {
					match: "\\(",
					name: "punctuation.section.parens.begin"
				},
				parenthesis_close: {
					match: "\\)",
					name: "punctuation.section.parens.end"
				},
				operator: {
					match: "(\\+|-|!|=|>|<|&|(\\|\\|))+",
					name: "punctuation.operator"
				},
				doc_block: {
					begin: "/\\*\\*",
					end: "\\*/",
					name: "comment.block.documentation",
					patterns: [
						{
							match: "@(?:(?!\\*/)\\S)*",
							name: "meta.documentation.annotation"
						}, {
							match: "<[^\\s>]*>",
							name: "meta.documentation.tag"
						}, {
							match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
							name: "meta.annotation.task.todo",
							captures: {
								2: {name: "keyword.other.documentation.task"},
								4: {name: "comment.block"}
							}
						}
					]
				},
				number_decimal: {
					match: "\\b-?(?:\\.\\d+|\\d+\\.?\\d*)(?:[eE][+-]?\\d+)?\\b",
					name: "constant.numeric.number"
				},
				number_hex: {
					match: "\\b0[xX][0-9A-Fa-f]+\\b",
					name: "constant.numeric.hex"
				},
				string_doubleQuote: {
					match: '"(?:\\\\.|[^"])*"?',
					name: "string.quoted.double"
				},
				string_singleQuote: {
					match: "'(?:\\\\.|[^'])*'?",
					name: "string.quoted.single"
				},
				todo_comment_singleLine: {
					match: "(\\b)(TODO)(\\b)(.*)",
					name: "meta.annotation.task.todo",
					captures: {
						2: {name: "keyword.other.documentation.task"},
						4: {name: "comment.line"}
					}
				}
			}
		}, {
			id: "orion.c-like",
			repository: {
				comment_singleLine: {
					match: {match: "//.*", literal: "//"},
					name: "comment.line.double-slash",
					patterns: [
						{
							include: "orion.lib#todo_comment_singleLine"
						}
					]
				},
				comment_block: {
					begin: {match: "/\\*", literal: "/*"},
					end: {match: "\\*/", literal: "*/"}, 
					name: "comment.block",
					patterns: [
						{
							match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
							name: "meta.annotation.task.todo",
							captures: {
								2: {name: "keyword.other.documentation.task"},
								4: {name: "comment.block"}
							}
						}
					]
				}
			}
		}],
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-csrc/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"auto",
		"break",
		"case", "char", "const", "continue",
		"default", "double", "do",
		"else", "enum", "extern",
		"float", "for",
		"goto",
		"if", "inline", "int",
		"long",
		"register", "return",
		"short", "signed", "sizeof", "static", "struct", "switch",
		"typedef",
		"union", "unsigned",
		"void", "volatile",
		"while",
		"_Bool", "_Complex", "_Imaginary"
	];

	var directives = [
		"define", "elif", "else", "endif", "error", "ifdef",
		"ifndef", "if", "include", "line", "pragma", "undef"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.c",
		contentTypes: ["text/x-csrc", "text/x-c"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "#directive"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.c"
			}
		],
		repository: {
			directive: {
				match: "#\\s*(?:" + directives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.c"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(directives)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/c/cPlugin',['orion/plugin', 'orion/editor/stylers/text_x-csrc/syntax'], function(PluginProvider, mC) {

	function connect() {
		var headers = {
			name: "Orion C Tool Support",
			version: "1.0",
			description: "This plugin provides C tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-csrc",
					"extends": "text/plain",
					name: "C",
					extension: ["c", "h"]
				}
			] 
		});
		mC.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/application_javascript/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"class", "const",
		"debugger", "delete",
		"enum", "export", "extends",
		"function",
		"implements", "import", "in", "instanceof", "interface",
		"let",
		"new",
		"package", "private", "protected", "public",
		"static", "super",
		"typeof",
		"var", "void",
		"with"
	];
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "do",
		"else",
		"finally", "for",
		"if",
		"return",
		"switch",
		"throw", "try",
		"while",
		"yield"
	];
	var languageVariables = ["this"];
	var constants = [
		"false", "null", "true", "undefined"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.js",
		contentTypes: ["application/javascript"],
		patterns: [
			{
				begin: "'(?:\\\\.|[^\\\\'])*\\\\$",
				end: "^(?:$|(?:\\\\.|[^\\\\'])*('|[^\\\\]$))",
				name: "string.quoted.single.js"
			}, {
				begin: '"(?:\\\\.|[^\\\\"])*\\\\$',
				end: '^(?:$|(?:\\\\.|[^\\\\"])*("|[^\\\\]$))',
				name: "string.quoted.double.js"
			},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{
				match: "/(?![\\s\\*])(?:\\\\.|[^/])+/(?:[gim]{0,3})",
				name: "string.regexp.js"
			},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "#jsFunctionDef"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.js"
			},
			{
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.js"
			},
			{
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.js"
			},
			{
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.js"
			}
		],
		repository: {
			jsFunctionDef: {
				/*
				 * http://stackoverflow.com/questions/2008279/validate-a-javascript-function-name/2008444#2008444
				 * was referenced in the composition of the "begin" pattern below.
				 */
				begin: "(function)(\\s+[_$a-zA-Z\\xA0-\\uFFFF][_$a-zA-Z0-9\\xA0-\\uFFFF]*)?\\s*\\(",
				end: "\\)",
				captures: {
					1: {name: "keyword.operator.js"},
					2: {name: "entity.name.function.js"}
				},
				patterns: [
					{include: "orion.c-like#comment_singleLine"},
					{include: "orion.c-like#comment_block"},
					{
						match: "[^\\s,]+",
						name: "variable.parameter.js"
					}
				]
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(languageVariables).concat(constants)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/application_vnd.coffeescript/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/application_javascript/syntax"], function(mLib, mJS) {
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "do",
		"else",
		"finally", "for",
		"if",
		"loop",
		"return",
		"switch",
		"then", "throw", "try",
		"unless", "until",
		"when", "while",
		"yield"
	];
	var keywords = [
		"and", "arguments",
		"by",
		"class", "const",
		"debugger", "delete",
		"enum", "eval", "export", "extends",
		"function",
		"implements", "import", "instanceof", "interface", "in", "isnt", "is",
		"let",
		"native", "new", "not", "no",
		"off", "of", "on", "or",
		"package", "private", "protected", "public",
		"static", "super",
		"typeof",
		"var", "void",
		"with",
		"yes",
		"__bind", "__extends", "__hasProp", "__indexOf", "__slice",
	];
	var constants = [
		"false", "null", "true", "undefined"
	];
	var languageVariables = ["this"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push({
		id: "orion.coffeescript",
		contentTypes: ["application/vnd.coffeescript"],
		patterns: [
			{include: "#comment_block"},
			{include: "#embedded_js"},
			{include: "#string_singleline"},
			{include: "#string_multiline"},
			{include: "#heregex"},
			{include: "#regex"},
			{include: "orion.lib#string_singleQuote"},
			{include: "#comment_singleLine"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{include: "#reserved_words"},
			{include: "#keyword_control"},
			{include: "#keyword_operator"},
			{include: "#language_variable"},
			{include: "#constant"},
			{include: "#punctuation"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
		],
		repository: {
			comment_block: {
				begin: {match: "###(?!#)", literal: "###"},
				end: {match: "###", literal: "###"},
				name: "comment.block.coffeescript"
			},
			comment_singleLine: {
				match: {match: "#[^$]*", literal: "#"},
				name: "comment.line.coffeescript"
			},
			constant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.coffeescript"
			},
			embedded_js: {
				begin: "`",
				end: "`",
				patterns: [
					{include: "orion.js"}
				]
			},
			heregex: {
				begin: "\\/{3}",
				end: "\\/{3}(?:[gim]{0,3})",
				name: "string.heregexp.coffeescript",
				patterns: [{
					match: "\\s#[^$]*",
					name: "comment.block.coffeescript"
				}]
			},
			keyword_control: {
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.coffeescript"
			},
			keyword_operator: {
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.coffeescript"
			},
			language_variable: {
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.coffeescript"
			},
			punctuation: {
				/* the following match is based on a regular expression from http://coffeescript.org/ */
				match: "[-=]?>|<|[-+*\\/%<>&|^!?=]?=|>>>=?|([-+*/%:])\\1|([&|<>])\\2=?|\\?\\.?|\\.{2,3}|%|@",
				name: "punctuation.operator.coffeescript"
			},
			regex: {
				match: "/(?![\\s\\*])(?:\\\\.|[^/])+/(?:[gim]{0,3})",
				name: "string.regexp.coffeescript"
			},
			string_singleline: {
				match: '"(?:\\\\.|[^"])*"',
				name: "string.quoted.double.coffeescript",
				patterns: [
					{
						begin: "#{",
						end: "}",
						name: "string.interpolated.coffeescript",
						patterns: [
							{include: "#comment_block"},
							{include: "#embedded_js"},
							{include: "#comment_singleLine"},
							{include: "#heregex"},
							{include: "#regex"},
							{include: "orion.lib#string_singleQuote"},
							{include: "orion.lib#number_decimal"},
							{include: "#reserved_words"},
							{include: "#keyword_control"},
							{include: "#keyword_operator"},
							{include: "#language_variable"},
							{include: "#constant"},
							{include: "#punctuation"},
							{include: "orion.lib#brace_open"},
							{include: "orion.lib#brace_close"},
							{include: "orion.lib#bracket_open"},
							{include: "orion.lib#bracket_close"},
							{include: "orion.lib#parenthesis_open"},
							{include: "orion.lib#parenthesis_close"},
						]
					}
				]
			},
			string_multiline: {
				begin: '("("")?)',
				end: '\\1',
				name: "string.quoted.multiline.coffeescript",
				patterns: [
					{
						begin: "#{",
						end: "}",
						name: "string.interpolated.coffeescript",
						patterns: [
							{include: "#comment_block"},
							{include: "#embedded_js"},
							{include: "#comment_singleLine"},
							{include: "#heregex"},
							{include: "#regex"},
							{include: "orion.lib#string_singleQuote"},
							{include: "orion.lib#number_decimal"},
							{include: "#reserved_words"},
							{include: "#keyword_control"},
							{include: "#keyword_operator"},
							{include: "#language_variable"},
							{include: "#constant"},
							{include: "#punctuation"},
							{include: "orion.lib#brace_open"},
							{include: "orion.lib#brace_close"},
							{include: "orion.lib#bracket_open"},
							{include: "orion.lib#bracket_close"},
							{include: "orion.lib#parenthesis_open"},
							{include: "orion.lib#parenthesis_close"}
						]
					}
				]
			},
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: controlKeywords.concat(constants)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/coffeescript/coffeescriptPlugin',['orion/plugin', 'orion/editor/stylers/application_vnd.coffeescript/syntax'], function(PluginProvider, mCoffeescript) {

	function connect() {
		var headers = {
			name: "Orion CoffeeScript Tool Support",
			version: "1.0",
			description: "This plugin provides CoffeeScript tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/vnd.coffeescript",
					"extends": "text/plain",
					name: "CoffeeScript",
					extension: ["coffee"]
				}
			] 
		});
		mCoffeescript.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-c__src/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"alignas", "alignof", "asm", "and_eq", "and", "auto",
		"bitand", "bitor", "bool", "break",
		"case", "catch", "char16_t", "char32_t", "char", "class",
		"compl", "constexpr", "const_cast", "const", "continue",
		"decltype", "default", "delete", "double", "do", "dynamic_cast",
		"else", "enum", "explicit", "export", "extern",
		"false", "float", "for", "friend",
		"goto",
		"if", "inline", "int",
		"long",
		"mutable",
		"namespace", "new", "noexcept", "not_eq", "not", "nullptr",
		"operator", "or_eq", "or",
		"private", "protected", "public",
		"register", "reinterpret_cast", "return",
		"short", "signed", "sizeof", "static_assert",
		"static_cast", "static", "struct", "switch",
		"template", "this", "thread_local", "throw", "true",
		"try", "typedef", "typeid", "typename",
		"union", "unsigned", "using",
		"virtual", "void", "volatile",
		"wchar_t", "while",
		"xor_eq", "xor",
		"_Bool", "_Complex", "_Imaginary"
	];

	var directives = [
		"define", "elif", "else", "endif", "error", "ifdef",
		"ifndef", "if", "include", "line", "pragma", "undef"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.cpp",
		contentTypes: ["text/x-c++src", "text/x-c++"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "#directive"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.cpp"
			}
		],
		repository: {
			directive: {
				match: "#\\s*(?:" + directives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.cpp"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/cpp/cppPlugin',['orion/plugin', 'orion/editor/stylers/text_x-c__src/syntax'], function(PluginProvider, mCPP) {

	function connect() {
		var headers = {
			name: "Orion C++ Tool Support",
			version: "1.0",
			description: "This plugin provides C++ tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-c++src",
					"extends": "text/plain",
					name: "C++",
					extension: ["cpp", "hpp"]
				}
			] 
		});
		mCPP.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-csharp/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"abstract", "as",
		"base", "bool", "break", "byte", "by",
		"case", "catch", "char", "checked", "class", "const", "continue",
		"decimal", "default", "delegate", "descending", "double", "do",
		"else", "enum", "event", "explicit", "extern",
		"false", "finally", "fixed", "float", "foreach", "for", "from",
		"goto", "group",
		"if", "implicit", "int", "interface", "internal", "into", "in", "is",
		"lock", "long",
		"namespace", "new", "null",
		"object", "operator", "orderby", "out", "override",
		"params", "private", "protected", "public",
		"readonly", "ref", "return",
		"sbyte", "sealed", "select", "short", "sizeof", "stackalloc", "static", "string", "struct", "switch", 
		"this", "throw", "true", "try", "typeof",
		"uint", "ulong", "unchecked", "unsafe", "ushort", "using",
		"var", "virtual", "volatile", "void",
		"while", "where",
		"yield"
	];

	var preprocessorDirectives = [
		"define",
		"elif", "else", "endif", "endregion", "error",
		"if",
		"line",
		"pragma checksum", "pragma warning", "pragma",
		"region",
		"undef",
		"warning"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.csharp",
		contentTypes: ["text/x-csharp"],
		patterns: [
			{
				match: "^\\s*#(?:" + preprocessorDirectives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.csharp"
			},
			{include: "#string_verbatim"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "#doc_line"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.csharp"
			}
		],
		repository: {
			doc_block: {
				begin: "/\\*\\*",
				end: "\\*/",
				name: "comment.block.documentation.csharp",
				patterns: [
					{
						match: "<[^\\s>]*>",
						name: "meta.documentation.tag"
					}, {
						match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.block"}
						}
					}
				]
			},
			doc_line: {
				match: "// /.*",
				name: "comment.line.documentation.csharp",
				patterns: [
					{
						match: "<[^\\s>]*>",
						name: "meta.documentation.tag"
					}, {
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			string_verbatim: {
				begin: '@"',
				end: '^(?:""|[^"])*"(?!")',
				name: "string.quoted.verbatim.csharp",
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
/* eslint-disable missing-nls */
define("orion/editor/stylers/application_xml/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.xml",
		contentTypes: ["application/xml", "application/xhtml+xml"],
		patterns: [
			{include: "#comment"},
			{include: "#doctype"},
			{include: "#xmlDeclaration"},
			{include: "#tag"},
			{include: "#ampersandEscape"}
		],
		repository: {
			ampersandEscape: {
				match: "&lt;|&gt;|&amp;",
				name: "constant.character"
			},
			comment: {
				begin: {match: "<!--", literal: "<!--"},
				end: {match: "-->", literal: "-->"},
				name: "comment.block.xml",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!-->).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.line"}
						}
					}
				]
			},
			doctype: {
				begin: "<!(?:doctype|DOCTYPE)",
				end: ">",
				name: "meta.tag.doctype.xml",
				captures: {
					0: {name: "meta.tag.doctype.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				]
			},
			tag: {
				// https://www.w3.org/TR/2006/REC-xml11-20060816/#sec-common-syn
				begin: "</?[A-Za-z:_][A-Za-z0-9:_\\-.]*",
				end: "/?>",
				captures: {
					0: {name: "meta.tag.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				]	
			},
			xmlDeclaration: {
				begin: "<\\?xml",
				end: "\\?>",
				captures: {
					0: {name: "meta.tag.declaration.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				],
				name: "meta.tag.declaration.xml"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_css/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"alignment-adjust", "alignment-baseline", "animation-delay", "animation-direction", "animation-duration", "animation-iteration-count",
		"animation-name", "animation-play-state", "animation-timing-function", "animation", "appearance",
		"azimuth", "backface-visibility", "background-attachment", "background-clip", "background-color", "background-image",
		"background-origin", "background-position", "background-repeat", "background-size", "background", "baseline-shift",
		"binding", "bleed", "bookmark-label", "bookmark-level", "bookmark-state", "bookmark-target", "border-bottom-color",
		"border-bottom-left-radius", "border-bottom-right-radius", "border-bottom-style", "border-bottom-width", "border-bottom",
		"border-collapse", "border-color", "border-image-outset", "border-image-repeat", "border-image-slice", "border-image-source",
		"border-image-width", "border-image", "border-left-color", "border-left-style", "border-left-width", "border-left",
		"border-radius", "border-right-color", "border-right-style", "border-right-width", "border-right", "border-spacing", "border-style",
		"border-top-color", "border-top-left-radius", "border-top-right-radius", "border-top-style", "border-top-width", "border-top",
		"border-width", "border", "bottom", "box-align", "box-decoration-break", "box-direction", "box-flex-group", "box-flex", "box-lines",
		"box-ordinal-group", "box-orient", "box-pack", "box-shadow", "box-sizing", "break-after", "break-before", "break-inside",
		"caption-side", "clear", "clip", "color-profile", "color", "column-count", "column-fill", "column-gap", "column-rule-color",
		"column-rule-style", "column-rule-width", "column-rule", "column-span", "column-width", "columns", "content", "counter-increment",
		"counter-reset", "crop", "cue-after", "cue-before", "cue", "cursor", "direction", "display", "dominant-baseline",
		"drop-initial-after-adjust", "drop-initial-after-align", "drop-initial-before-adjust", "drop-initial-before-align", "drop-initial-size",
		"drop-initial-value", "elevation", "empty-cells", "fit-position", "fit", "flex-align", "flex-flow", "flex-inline-pack", "flex-order",
		"flex-pack", "float-offset", "float", "font-family", "font-size-adjust", "font-size", "font-stretch", "font-style", "font-variant",
		"font-weight", "font", "grid-columns", "grid-rows", "hanging-punctuation", "height", "hyphenate-after",
		"hyphenate-before", "hyphenate-character", "hyphenate-lines", "hyphenate-resource", "hyphens", "icon", "image-orientation",
		"image-rendering", "image-resolution", "inline-box-align", "left", "letter-spacing", "line-height", "line-stacking-ruby",
		"line-stacking-shift", "line-stacking-strategy", "line-stacking", "list-style-image", "list-style-position", "list-style-type",
		"list-style", "margin-bottom", "margin-left", "margin-right", "margin-top", "margin", "mark-after", "mark-before", "mark",
		"marker-offset", "marks", "marquee-direction", "marquee-loop", "marquee-play-count", "marquee-speed", "marquee-style", "max-height",
		"max-width", "min-height", "min-width", "move-to", "nav-down", "nav-index", "nav-left", "nav-right", "nav-up", "opacity", "orphans",
		"outline-color", "outline-offset", "outline-style", "outline-width", "outline", "overflow-style", "overflow-x", "overflow-y",
		"overflow", "padding-bottom", "padding-left", "padding-right", "padding-top", "padding", "page-break-after", "page-break-before", "page-break-inside",
		"page-policy", "page", "pause-after", "pause-before", "pause", "perspective-origin", "perspective", "phonemes", "pitch-range",
		"pitch", "play-during", "position", "presentation-level", "punctuation-trim", "quotes", "rendering-intent", "resize",
		"rest-after", "rest-before", "rest", "richness", "right", "rotation-point", "rotation", "ruby-align", "ruby-overhang", "ruby-position",
		"ruby-span", "size", "speak-header", "speak-numeral", "speak-punctuation", "speak", "speech-rate", "stress", "string-set", "table-layout",
		"target-name", "target-new", "target-position", "target", "text-align-last", "text-align", "text-decoration", "text-emphasis",
		"text-height", "text-indent", "text-justify", "text-outline", "text-shadow", "text-transform", "text-wrap", "top", "transform-origin",
		"transform-style", "transform", "transition-delay", "transition-duration", "transition-property", "transition-timing-function",
		"transition", "unicode-bidi", "vertical-align", "visibility", "voice-balance", "voice-duration", "voice-family",
		"voice-pitch-range", "voice-pitch", "voice-rate", "voice-stress", "voice-volume", "volume", "white-space-collapse", "white-space",
		"widows", "width", "word-break", "word-spacing", "word-wrap", "z-index"
	];
	var colors = [
		"AliceBlue", "AntiqueWhite", "Aquamarine", "Aqua", "Azure",
		"Beige", "Bisque", "Black", "BlanchedAlmond", "BlueViolet", "Blue", "Brown", "BurlyWood",
		"CadetBlue", "Chartreuse", "Chocolate", "Coral", "CornflowerBlue", "Cornsilk", "Crimson", "Cyan",
		"DarkBlue", "DarkCyan", "DarkGoldenRod", "DarkGray", "DarkGrey", "DarkGreen", "DarkKhaki", "DarkMagenta", "DarkOliveGreen",
		"DarkOrange", "DarkOrchid", "DarkRed", "DarkSalmon", "DarkSeaGreen", "DarkSlateBlue", "DarkSlateGray", "DarkSlateGrey",
		"DarkTurquoise", "DarkViolet", "DeepPink", "DeepSkyBlue", "DimGray", "DimGrey", "DodgerBlue",
		"FireBrick", "FloralWhite", "ForestGreen", "Fuchsia",
		"Gainsboro", "GhostWhite", "Gold", "GoldenRod", "Gray", "Grey", "GreenYellow", "Green",
		"HoneyDew", "HotPink",
		"IndianRed", "Indigo", "Ivory",
		"Khaki",
		"LavenderBlush", "Lavender", "LawnGreen", "LemonChiffon", "LightBlue", "LightCoral", "LightCyan", "LightGoldenRodYellow",
		"LightGray", "LightGrey", "LightGreen", "LightPink", "LightSalmon", "LightSeaGreen", "LightSkyBlue", "LightSlateGray",
		"LightSlateGrey", "LightSteelBlue", "LightYellow", "LimeGreen", "Lime", "Linen",
		"Magenta", "Maroon", "MediumAquaMarine", "MediumBlue", "MediumOrchid", "MediumPurple", "MediumSeaGreen", "MediumSlateBlue",
		"MediumSpringGreen", "MediumTurquoise", "MediumVioletRed", "MidnightBlue", "MintCream", "MistyRose", "Moccasin",
		"NavajoWhite", "Navy",
		"OldLace", "OliveDrab", "Olive", "OrangeRed", "Orange", "Orchid",
		"PaleGoldenRod", "PaleGreen", "PaleTurquoise", "PaleVioletRed", "PapayaWhip", "PeachPuff", "Peru", "Pink", "Plum", "PowderBlue", "Purple",
		"RebeccaPurple", "Red", "RosyBrown", "RoyalBlue",
		"SaddleBrown", "Salmon", "SandyBrown", "SeaGreen", "SeaShell", "Sienna", "Silver", "SkyBlue", "SlateBlue", "SlateGray", "SlateGrey", "Snow", "SpringGreen", "SteelBlue",
		"Tan", "Teal", "Thistle", "Tomato", "Turquoise",
		"Violet",
		"Wheat", "WhiteSmoke", "White",
		"YellowGreen", "Yellow"
	];
	var directives = ["charset", "document", "font-face", "import", "keyframes", "media", "namespace", "page", "supports"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.css",
		contentTypes: ["text/css"],
		patterns: [
			{include: "#string_single_multiline"},
			{include: "#string_double_multiline"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#number_decimal"},
			{include: "#number_hex"},
			{include: "#numeric_value"},
			{include: "#color"},
			{include: "#keyword"},
			{include: "#directive"}
		],
		repository: {
			color: {
				match: "(?i)\\b(?:" + colors.join("|") + ")\\b",
				name: "constant.other.color.css"
			},
			directive: {
				match: "(^|\\s)(@("  + directives.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.directive.css"}
				}
			},
			keyword: {
				match: "(?:-webkit-|-moz-|-ms-|-o-|\\b)(?:" + keywords.join("|") + ")\\b",
				name: "support.type.propertyName.css"
			},
			number_hex: {
				match: "#[0-9A-Fa-f]+\\b",
				name: "constant.numeric.hex.css"
			},
			numeric_value: {
				match: "(?i)\\b-?(?:\\.\\d+|\\d+\\.?\\d*)(?:%|em|ex|ch|rem|vw|vh|vmin|vmax|in|cm|mm|pt|pc|px|deg|grad|rad|turn|s|ms|Hz|kHz|dpi|dpcm|dppx)?\\b",
				name: "constant.numeric.value.css"
			},
			string_double_multiline: {
				begin: '"(?:\\\\.|[^\\\\"])*\\\\$',
				end: '^(?:$|(?:\\\\.|[^\\\\"])*("|[^\\\\]$))',
				name: "string.quoted.double.css"
			},
			string_single_multiline: {
				begin: "'(?:\\\\.|[^\\\\'])*\\\\$",
				end: "^(?:$|(?:\\\\.|[^\\\\'])*('|[^\\\\]$))",
				name: "string.quoted.single.css"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_html/syntax", ["orion/editor/stylers/application_javascript/syntax", "orion/editor/stylers/text_css/syntax", "orion/editor/stylers/application_xml/syntax"],
	/* eslint-disable missing-nls */
	function(mJS, mCSS, mXML) {
	var attributes = [
		"accept-charset", "accept", "accesskey", "action", "align", "alt",  
		"async", "autocomplete", "autoplay", "autosave",
		"bgcolor", "border", "buffered", 
		"challenge", "charset", "checked", "cite", "class", "codebase", "code", "color",
		"colspan", "cols", "contenteditable", "content", "contextmenu", "controls", "coords",
		"data-[A-Za-z_:][\\w.:-]*", "data", "datetime", "default", "defer", "dirname", "dir",
		"disabled", "download", "draggable", "dropzone",
		"enctype",
		"formaction", "form", "for", 
		"headers", "height", "hidden", "high", "hreflang", "href", "http-equiv",
		"icon", "id", "ismap", "itemprop",
		"keytype", "kind", 
		"label", "language", "lang", "list", "loop", "low",  
		"manifest", "maxlength", "max", "media", "method", "min", "multiple",
		"name", "novalidate", 
		"open", "optimum", 
		"pattern", "ping", "placeholder", "poster", "preload", "pubdate",  
		"radiogroup", "readonly", "rel", "required", "reversed", "rowspan", "rows",
		"sandbox", "scoped", "scope", "seamless", "selected", "shape", "sizes", "size", "span", "spellcheck",
		"srcdoc", "srclang","srcset", "src", "start", "step", "style", "summary",
		"tabindex", "target", "title", "type",
		"usemap",
		"value",
		"width", "wrap" 
	];

	var grammars = [];
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push.apply(grammars, mCSS.grammars);
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push({
		id: "orion.html",
		contentTypes: ["text/html"],
		patterns: [
			{
				begin: "(?i)(<style)([^>]*)(>)",
				end: "(?i)(</style>)",
				captures: {
					1: {name: "meta.tag.html"},
					3: {name: "meta.tag.html"}
				},
				contentName: "source.css.embedded.html",
				patterns: [
					{include: "orion.css"}
				]
			}, {
				begin: "(?i)<script\\s*>|<script\\s.*?(?:language\\s*=\\s*(['\"])javascript\\1|type\\s*=\\s*(['\"])(?:text|application)/(?:javascript|ecmascript)\\2).*?>",
				end: "(?i)</script>",
				captures: {
					0: {name: "meta.tag.html"}
				},
				contentName: "source.js.embedded.html",
				patterns: [
					{include: "orion.js"}
				]
			}, {
				begin: "</?[A-Za-z0-9]+",
				end: "/?>",
				captures: {
					0: {name: "meta.tag.html"}
				},
				patterns: [
					{include: "orion.xml#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
					{include: "#attribute"}
				]
			},
			{include: "orion.xml#comment"},
			{include: "orion.xml#doctype"},
			{include: "orion.xml#ampersandEscape"}
		],
		repository: {
			attribute:{
				match: "\\b(?:" + attributes.join("|") + ")\\b",  
				name: "meta.tag.attribute.html"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: [],
		attributes: attributes
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-cshtml/syntax", [
	"orion/editor/stylers/application_xml/syntax",
	"orion/editor/stylers/text_html/syntax",
	"orion/editor/stylers/text_x-csharp/syntax"], function(mXML, mHTML, mCSharp) {

	var grammars = [];
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push.apply(grammars, mHTML.grammars);
	grammars.push.apply(grammars, mCSharp.grammars);
	grammars.push({
		id: "orion.cshtml",
		contentTypes: ["text/x-cshtml"],
		patterns: [
			{include: "#comment"},
			{include: "#codeBlock"},
			{include: "#expression"},
			{include: "#reference"},
			{include: "orion.html"},
		],
		repository: {
			comment: {
				begin: {match: "@\\*", literal: "@*"},
				end: {match: "\\*@", literal: "*@"},
				name: "comment.block.cshtml",
			},
			codeBlock: {
				begin: "(^\\s*)(@)(?=([^{]*){)",
				end: "}",
				captures: {
					2: {name: "entity.name.declaration.csharp"}
				},
				contentName: "source.csharp.embedded.cshtml",
				patterns: [
				    {include: "orion.xml#tag"},
				    {include: "#reference"},
					{include: "orion.csharp"},
				]
			},
			expression: {
				match: "^\\s*@[^{]*$",
				name: "source.csharp.embedded.cshtml",
				patterns: [
				    {include: "#reference"},
					{include: "orion.csharp"},
				]
			},
			reference: {
				match: "@",
				name: "entity.name.declaration.csharp"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/csharp/csharpPlugin',['orion/plugin', 'orion/editor/stylers/text_x-csharp/syntax', 'orion/editor/stylers/text_x-cshtml/syntax'], function(PluginProvider, mCSharp, mCSHtml) {

	function connect() {
		var headers = {
			name: "Orion C# Tool Support",
			version: "1.0",
			description: "This plugin provides C# tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-csharp",
					"extends": "text/plain",
					name: "C#",
					extension: ["cs"]
				}, {id: "text/x-cshtml",
					"extends": "text/plain",
					name: "cshtml",
					extension: ["cshtml"]
				}
			] 
		});
		mCSharp.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		mCSHtml.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-dockerfile/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"add",
		"cmd", "copy",
		"entrypoint", "env", "expose",
		"from",
		"maintainer",
		"onbuild",
		"run",
		"user",
		"volume",
		"workdir"
	];

	var id = "orion.dockerfile";
	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: id,
		contentTypes: ["text/x-dockerfile"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "#numberSignComment"},
			{
				match: "\\b-?[0-9]+(\\.[0-9]+)?\\b",
				name: "constant.numeric.dockerfile"
			}, {
				match: "(?i)^\\s*(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.dockerfile"
			}
		],
		repository: {
			numberSignComment: {
				begin: {match: "^\\s*#", literal: "#"},
				end: {match: "$", literal: ""},
				name: "comment.line.number-sign.dockerfile",
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}
		}
	});

	return {
		id: id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/docker/dockerPlugin',['orion/plugin', 'orion/editor/stylers/text_x-dockerfile/syntax'], function(PluginProvider, mDockerfile) {

	function connect() {
		var headers = {
			name: "Orion Editor Docker Tool Support",
			version: "1.0",
			description: "This plugin provides Docker tools support for the Orion editor."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-dockerfile",
					"extends": "text/plain",
					name: "dockerfile",
					extension: ["dockerfile"]
				}
			] 
		});
		mDockerfile.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-erlang/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"xor",
		"when",
		"try",
		"rem", "receive",
		"query",
		"orelse", "or", "of",
		"not",
		"let",
		"if",
		"fun",
		"end",
		"div",
		"cond", "catch", "case",
		"bxor", "bsr", "bsl", "bor", "bnot", "begin", "band",
		"andalso", "and", "after"
	];

	// For Preprocessors, Records and so on specified with hyphens
	var hyphenKeywords = [
		"vsn",
		"undef",
		"type",
		"spec",
		"record",
		"on_load", "opaque",
		"module",
		"include_lib", "include", "import", "ifndef", "ifdef",
		"file",
		"export_type", "export", "endif", "else",
		"define",
		"callback", "compile",
		"behaviour"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.erlang",
		contentTypes: ["text/x-erlang"],
		patterns: [
			{include: "#comment"},
			{include: "#stringSingleLine"},
			{include: "#stringMultiLine"},
			{include: "#method"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.erlang"
			},
			{
				match: "^\\s*-(?:" + hyphenKeywords.join("|") + ")\\b",
				name: "keyword.operator.erlang"
			}
		],
		repository: {
			comment: {
				match: {match: "%.*", literal: "%"},
				name: "comment.line.erlang",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			method: {
				match: "(^|\\s)[a-zA-Z0-9_.]+(?=\\(|\\s\\()",
			    name: "entity.name.function.erlang"
			},
			stringMultiLine: {
				begin: '"(?:\\\\.|[^\\\\"])*$',
				end: '^(?:\\\\.|[^\\\\"])*"',
				name: "string.quoted.double.erlang"
			},
			stringSingleLine: {
				match: '"(?:\\\\.|[^\\\\"])*"',
				name: "string.quoted.double.erlang"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/erlang/erlangPlugin',['orion/plugin', 'orion/editor/stylers/text_x-erlang/syntax'], function(PluginProvider, mErlang) {

	function connect() {
		var headers = {
			name: "Orion Erlang Tool Support",
			version: "1.0",
			description: "This plugin provides Erlang tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-erlang",
					"extends": "text/plain",
					name: "Erlang",
					extension: ["erl", "hrl"]
				}
			] 
		});
		mErlang.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global escape*/
define('orion/form',[], function() {
	function x_www_form_urlencode(value) {
		return encodeURIComponent(value).replace(/[!'()*]/g, escape).replace('%20', '+'); //$NON-NLS-0$ //$NON-NLS-1$
	}

	/**
	 * @name orion.form
	 * @class Utilities for handling encoding.
	 */
	return /** @lends orion.form */ {
		/**
		 * Encodes an object of form fields and values into an <code>application/x-www-form-urlencoded</code> string.
		 * @static
		 * @param {Object} data The form data to encode.
		 * @returns {String} The <code>x-www-form-urlencoded</code> string.
		 */
		encodeFormData: function(data) {
			data = data || {};
			var paramNames = Object.keys(data);
			var buf = [];
			for (var i=0; i < paramNames.length; i++) {
				var param = paramNames[i], value = data[param];
				buf.push(x_www_form_urlencode(param) + '=' + x_www_form_urlencode(value)); //$NON-NLS-0$
			}
			return buf.join('&'); //$NON-NLS-0$
		},
		/**
		 * Encodes a string into an <a href="http://tools.ietf.org/html/rfc5023#section-9.7.1">RFC 5023</a>-compliant
		 * <tt>Slug</tt> header.
		 * @static
		 * @param {String} s The string to encode.
		 * @returns {String} The encoded <tt>Slug</tt>.
		 */
		encodeSlug: function(s) {
			return s.replace(/([^\u0020-\u007e]|%)+/g, encodeURIComponent);
		}
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10/.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-go/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"break",
		"case", "const", "continue",
		"default", "defer",
		"else",
		"fallthrough", "false", "for", "func",
		"goto", "go",
		"if", "import",
		"nil",
		"package",
		"range", "return",
		"select", "switch",
		"true", "type",
		"var"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.go",
		contentTypes: ["text/x-go"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{
				begin: "`",
				end: "`",
				name: "string.quoted.raw.go",
			},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.go"
			},
			{
				match: "\\b(?:len|cap|new|make|append|close|copy|delete|complex|real|imag|panic|recover)\\b",
				name: "support.function.go"
			},
			{
				match: "\\b(?:bool|chan|uint8|uint16|uint32|uint64|int8|int16|int32|int64|float32|float64|complex64|complex128|byte|map|rune|uint|interface|int|uintptr|string|struct|error)\\b",
				name: "support.function.type"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser,amd*/
define('plugins/languages/go/goPlugin',['orion/plugin', 'orion/form', 'orion/editor/stylers/text_x-go/syntax'], function (PluginProvider, form, mGo) {

	function connect() {
		var headers = {
			name: "Orion Go Language Tool Support",
			version: "1.0",
			description: "This plugin provides Go language tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-go",
					"extends": "text/plain",
					name: "Go",
					extension: ["go"]
				}
			] 
		});
		mGo.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		pluginProvider.registerServiceProvider("orion.edit.contentAssist", {
			computeProposals: function (buffer, offset, context) {
				var NL = context.delimiter;
				var INDENT = context.indentation;
				var TAB = context.tab;

				var constructs = [
					"inner", "if", "if",            "if ${cond} {"                         +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "func", "func",        "func ${name}() (${retval} ${type}) {" +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"inner", "for", "for",          "for ${cond} {"                        +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"inner", "switch", "switch",    "switch {"                             +NL+
					    INDENT+                     "case ${cond}:"                        +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "default:"                             +NL+
					    INDENT+                     "}",

					"inner", "select", "select",    "select {"                             +NL+
					    INDENT+                     "case ${cond}:"                        +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "default:"                             +NL+
					    INDENT+                     "}",

					"outer", "var", "var",          "var ("                                +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "const", "const",      "const ("                              +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "import", "import",    "import ("                             +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "", "method",          "func (this *${type}) ${name}() (${retval} ${type}) {"+NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "", "struct",          "type ${name} struct {"                +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "", "interface",       "type ${name} interface {"             +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"inner", "", "make channel",    "ch := make(chan ${type}, 0)",

					"inner", "", "make array",      "arr := make([]${type}, 1, 1)",

					"outer", "", "main",            "func main() {"                        +NL+
					    INDENT+                      TAB+"${cursor}"                       +NL+
					    INDENT+                     "}"
				];

				var proposals = [];

				if (buffer.length === 0 && offset === 0) {
					proposals.push({
						description: "new file template",
						proposal:	"// COPYRIGHT"          +NL+
									""                      +NL+
									"// GODOC"              +NL+
									"package ${name}"       +NL+
									                         NL+
									"import ("              +NL+
									TAB+"${import}"         +NL+
									")"                     +NL+
									                         NL+
									"func main() {"         +NL+
									TAB                     +NL+
									"}"                     +NL,
						escapePosition: 68 + NL.length * 10,
						positions: [{offset: 28 + NL.length * 3, length: 7}, {offset: 43 +  NL.length * 7, length: 9}]
					});

					proposals.push({
						description: "new test template",
						proposal:	"// COPYRIGHT"               +NL+
									""                           +NL+
									"package main"               +NL+
									                              NL+
									"import ("                   +NL+
									TAB+"testing"                +NL+
									")"                          +NL+
									                              NL+
									"func Test1(t *testing.T) {" +NL+
									TAB                          +NL+
									"}"                          +NL,
						escapePosition: 68 + NL.length * 9,
						positions: []
					});
				}

				for (var i = 0; i < constructs.length; i = i + 4) {
					var type = constructs[i];
					var matcher = constructs[i + 1];
					var keyword = constructs[i + 2];
					var proposal = constructs[i + 3];

					if (matcher.indexOf(context.prefix) !== 0) {
						continue;
					}

					// Check whether this proposal is an "outer" (outside of a var, const, func block)
					// or an "inner"
					if (type === "inner" && INDENT === "") {
						continue;
					}
					if (type === "outer" && INDENT !== "") {
						continue;
					}

					// Ellide the proposal with the existing prefix text where possible
					if (proposal.indexOf(context.prefix) === 0) {
						proposal = proposal.substring(context.prefix.length);
					}

					var propObj = {description: keyword, positions: []};

					// Calculate positions for the variables and the cursor
					var cursorIdx = proposal.indexOf("${cursor}");
					if (cursorIdx !== -1) {
						propObj.escapePosition = cursorIdx + offset;
						proposal = proposal.replace("${cursor}", "");
					}

					propObj.proposal = proposal;

					var idx = 0;
					while (idx !== -1 && idx < proposal.length - 4) {
						idx = proposal.indexOf("${", idx + 1);
						if (idx !== -1) {
							var off = idx + offset;
							var len = proposal.indexOf("}", idx + 1) + 1 - idx;
							if (len > 0) {
								propObj.positions.push({offset: off, length: len});
							}
						}
					}
	
					proposals.push(propObj);
				}

				return proposals;
			}
		}, {
			name: "Go content assist",
			contentType: ["text/x-go"]
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/text_x-git-ignore/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.gitignore",
		contentTypes: ["text/x-git-ignore"],
		patterns: [
			// http://git-scm.com/docs/gitignore
			{
				match: {match: "^\\s*#[^$]*", literal: "#"},
				name: "comment.line.number-sign.ignore"
			}, {
				match: "^\\s*!",
				name: "punctuation.operator.negation.ignore"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/text_x-git-config/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.gitconfig",
		contentTypes: ["text/x-git-config"],
		patterns: [
			/* http://git-scm.com/docs/git-config */
			{
				match: "[;#][^$]*",
				name: "comment.line.gitconfig"
			}, {
				match: "=",
				name: "punctuation.operator.equals.gitconfig"
			}, {
				match: "^\\s*\\[[-.A-Za-z0-9]+\\]\\s*$",
				name: "entity.section.name.gitconfig"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/git/gitFilePlugin',['orion/plugin', 'orion/editor/stylers/text_x-git-ignore/syntax', 'orion/editor/stylers/text_x-git-config/syntax'], function(PluginProvider, mGitIgnore, mGitConfig) {

	function connect() {
		var headers = {
			name: "Orion Git File Support",
			version: "1.0",
			description: "This plugin provides Git file syntax highlighting for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-git-ignore",
					"extends": "text/plain",
					name: "Git Ignore",
					extension: ["gitignore"],
				}, {id: "text/x-git-config",
					"extends": "text/plain",
					name: "Git Config",
					filename: ["config"],
				}, {id: "text/x-cf-ignore",
					"extends": "text/x-git-ignore",
					name: "cf Ignore",
					extension: ["cfignore"],
				}
			] 
		});
		mGitIgnore.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		mGitConfig.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-ruby/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"alias_method", "alias", "attr_accessor", "attr_reader", "attr_writer", "attr",
		"BEGIN",
		"class",
		"defined?", "def",
		"END", "extend",
		"gem",
		"include", "initialize", "in",
		"load",  "lambda",
		"module_function", "module",
		"new", "not",
		"public", "prepend", "private", "protected",
		"require_relative", "require",
		"undef",
		"__ENCODING__", "__END__", "__FILE__", "__LINE__"
	];
	var controlKeywords = [
		"and",
		"begin", "break",
		"case", "catch",
		"do",
		"else", "elsif", "end", "ensure",
		"fail", "for",
		"if",
		"loop",
		"next",
		"or",
		"raise", "redo", "rescue", "retry", "return",
		"then", "throw",
		"unless", "until",
		"when", "while",
		"yield"
	];
	var constants = [
		"false", "nil", "true"
	];
	var languageVariables = ["self", "super"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.ruby",
		contentTypes: ["text/x-ruby"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "#symbol_quoted_single"},
			{include: "#symbol_quoted_double"},
			{include: "#symbol"},
			{include: "#classRef"},
			{
				match: "/(?![\\s])(?:\\\\.|[^/])+/(?:[ioxmuesn]\\b)?",
				name: "string.regexp.ruby"
			}, {
				match: {match: "#.*", literal: "#"},
				name: "comment.line.number-sign.ruby",
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}, {
				begin: {match: "^=begin\\b", literal: "\n=begin "},
				end: {match: "^=end\\b", literal: "\n=end "},
				name: "comment.block.ruby",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.block"}
						}
					}
				]
			},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{include: "#variable"},
			{
				match: "\\b0[bB][01]+\\b",
				name: "constant.numeric.binary.ruby"
			},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.ruby"
			},
			{
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.ruby"
			},
			{
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.ruby"
			},
			{
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.ruby"
			}
		],
		repository: {
			classRef: {
				match: "\\w+::\\w+"
			},
			symbol: {
				match: ":\\w+",
				name: "entity.name.symbol.ruby"
			},
			symbol_quoted_single: {
				match: ":'[^']*'",
				name: "entity.name.symbol.quoted.single.ruby"
			},
			symbol_quoted_double: {
				match: ':"[^"]*"',
				name: "entity.name.symbol.quoted.double.ruby"
			},
			variable: {
				match: "@\\w+",
				name: "entity.name.variable.ruby"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(constants).concat(languageVariables)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-haml/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/text_x-ruby/syntax"],
	function(mLib, mRuby) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mRuby.grammars);
	grammars.push({
		id: "orion.haml",
		contentTypes: ["text/x-haml"],
		patterns: [
			{include: "#inlineRuby"},
			{include: "#interpolatedRuby"},
			{include: "#tagWithRubySymbols"},
			{include: "#tagWithHTMLAttributes"},
			{include: "#doctype"},
			{include: "#tag"},
			{include: "#htmlComment"},
		],
		repository: {
			doctype: {
				match: "^!!!.*$",
				name: "meta.tag.doctype.haml",
			},
			htmlComment: {
				// TODO multi-line comments
				match: {match: "/[^[].*$", literal: "/"},
				name: "comment.line.html.haml"
			},
			inlineRuby: {
				// TODO end match should not eat the last content character, really need a negative look-behind
				begin: "(?:^|[^\\\\])(?:=|-|~|&==?|!==?)",
				end: "(?:^|[^,])$",
				contentName: "source.ruby.embedded.haml",
				patterns: [
					{include: "orion.ruby"},
				]
			},
			interpolatedRuby: {
				begin: "#{",
				end: "}",
				contentName: "source.ruby.embedded.haml",
				patterns: [
					{include: "orion.ruby"},
				]
			},
			tag: {
				match: "^\\s*%[^\\b]+?\\b",
				name: "meta.tag.haml",
			},
			tagWithHTMLAttributes: {
				begin: "(^\\s*)(%[^\\s(]+?)\\(",
				end: "\\)\\s*$",
				beginCaptures: {
					2: {name: "meta.tag.haml"}
				},
				patterns: [
					{
						match: "[^\\s=]+(?==)",
						name: "entity.name.attribute.html.haml"
					},
					{include: "orion.ruby#variable"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			},
			tagWithRubySymbols: {
				begin: "(^\\s*)(%[^\\b]+?)\\b{",
				end: "}\\s*$",
				beginCaptures: {
					2: {name: "meta.tag.haml"}
				},
				patterns: [
					{include: "orion.ruby#symbol"},
					{include: "orion.ruby#variable"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/haml/hamlPlugin',['orion/plugin', 'orion/editor/stylers/text_x-haml/syntax'], function(PluginProvider, mHaml) {

	function connect() {
		var headers = {
			name: "Orion Haml Tool Support",
			version: "1.0",
			description: "This plugin provides Haml tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-haml",
					"extends": "text/plain",
					name: "Haml",
					extension: ["haml"]
				}
			] 
		});
		mHaml.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-jade/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/application_javascript/syntax"], function(mLib, mJS) {
	var keywords = [
		"&attributes",
		"block",
		"case",
		"default", "doctype",
		"each", "else", "extends",
		"for",
		"if", "include",
		"mixin",
		"unless",
		"when", "while"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push({
		id: "orion.jade",
		contentTypes: ["text/x-jade"],
		patterns: [
			{include: "#comment_singleLine"},
			{include: "#code"},
			{include: "#control"},
			{include: "#caseBranch"},
			{include: "#mixinWithParameters"},
			{include: "#mixinRefWithArguments"},
			{include: "#tagWithAttributes"},
			{include: "#interpolatedJS"},
			{include: "#interpolatedTag"},
			{include: "#mixin"},
			{include: "#mixinRef"},
			{include: "#doctype"},
			{include: "#filter"},
			{include: "#case"},
			{include: "#andAttributes"},
			{include: "#otherKeywords"},
			{include: "#tag"},
		],
		repository: {
			andAttributes: {
				match: "&attributes\\b",
				name: "keyword.operator.jade"
			},
			"case": {
				match: "(^\\s*)(case)\\b",
				captures: {
					2: {name: "keyword.control.jade"}
				}				
			},
			caseBranch: {
				begin: "(^\\s*)(when|default)\\s*",
				end: ":|$",
				contentName: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				],
				beginCaptures: {
					2: {name: "keyword.control.jade"}
				},
			},
			code: {
				match: "(^\\s*- |= |!= ).*$",
				name: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				]
			},
			comment_singleLine: {
				match: {match: "^\\s*//.*", literal: "//"},
				name: "comment.line.double-slash.jade",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			control: {
				begin: "(^\\s*)(if|else( if)?|each|for|unless|while)\\b",
				end: "$",
				beginCaptures: {
					2: {name: "keyword.control.jade"}
				},
				contentName: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				]
			},
			doctype: {
				match: "(^\\s*)(doctype.+$)",
				captures: {
					2: {name: "meta.tag.doctype.jade"}
				}
			},
			filter: {
				match: "(^\\s*)(:\\w+)",
				captures: {
					2: {name: "entity.other.filter.jade"}
				}
			},
			interpolatedJS: {
				begin: "(#{)",
				end: "(})",
				captures: {
					1: {name: "string.interpolated.js.jade"}
				},
				contentName: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				]
			},
			interpolatedTag: {
				begin: "(#\\[)",
				end: "(\\])",
				captures: {
					1: {name: "string.interpolated.tag.jade"}
				},
				patterns: [
					{
						begin: "(\\.|\\w+)\\s*\\(",
						end: "(\\))(/)?",
						beginCaptures: {
							1: {name: "meta.tag.jade"}
						},
						endCaptures: {
							2: {name: "meta.tag.jade"}
						},
						contentName: "source.js.embedded.jade",
						patterns: [
							{include: "orion.js"}
						]
					}
				]
			},
			mixin: {
				match: "(^\\s*)(mixin)(\\s+)(\\w+)",
				captures: {
					2: {name: "keyword.operator.jade"},
					4: {name: "entity.name.mixin.jade"}
				}
			},
			mixinRef: {
				match: "(^\\s*)(\\+\\w+)",
				captures: {
					2: {name: "entity.name.mixin.jade"}
				}
			},
			mixinRefWithArguments: {
				begin: "(^\\s*)(\\+\\w+)\\s*\\(",
				end: "\\)|$",
				captures: {
					2: {name: "entity.name.mixin.jade"}
				},
				patterns: [
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
					{include: "orion.lib#number_decimal"}
				]
			},
			mixinWithParameters: {
				begin: "(^\\s*)(mixin)(\\s+)(\\w+)\\s*\\(",
				end: "\\)|$",
				beginCaptures: {
					2: {name: "keyword.operator.jade"},
					4: {name: "entity.name.mixin.jade"}
				},
				patterns: [
					{
						match: "[^\\s,]+",
						name: "variable.parameter.jade"
					}	
				]
			},
			otherKeywords: {
				match: "(^\\s*)(block|extends|include)\\b",
				captures: {
					2: {name: "keyword.operator.jade"}
				}
			},
			tag: {
				match: "(^\\s*)(\\w+|(?=\\.)|(?=#))(#\\w*|\\.\\w*)*(/?)",
				captures: {
					2: {name: "meta.tag.jade"},
					4: {name: "meta.tag.jade"}
				}
			},
			tagWithAttributes: {
				begin: "(^\\s*)(\\w+|(?=\\.)|(?=#))(#\\w*|\\.\\w*)*\\s*\\(",
				end: "(\\))(/)?",
				beginCaptures: {
					2: {name: "meta.tag.jade"}
				},
				endCaptures: {
					2: {name: "meta.tag.jade"}
				},
				contentName: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				]
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/jade/jadePlugin',['orion/plugin', 'orion/editor/stylers/text_x-jade/syntax'], function(PluginProvider, mJade) {

	function connect() {
		var headers = {
			name: "Orion Jade Tool Support",
			version: "1.0",
			description: "This plugin provides Jade tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-jade",
					"extends": "text/plain",
					name: "Jade",
					extension: ["jade"]
				}
			] 
		});
		mJade.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-java-source/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"abstract",
		"boolean", "byte",
		"char", "class",
		"double",
		"extends",
		"final", "float",
		"implements", "import", "instanceof", "int", "interface",
		"long",
		"native", "new",
		"package", "private", "protected", "public",
		"short", "static", "synchronized",
		"throws", "transient",
		"void", "volatile"
	];
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "do",
		"else",
		"finally", "for",
		"if",
		"return",
		"switch",
		"throw", "try",
		"while"
	];
	var constants = [
		"false", "null", "true"
	];
	var languageVariables = ["this", "super"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.java",
		contentTypes: ["text/x-java-source"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.java"
			},
			{
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.java"
			},
			{
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.java"
			},
			{
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.java"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(constants).concat(languageVariables)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/application_x-jsp/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/text_x-java-source/syntax", "orion/editor/stylers/text_html/syntax"],
	function(mLib, mJava, mHTML) {

	var pageAttributeNames = [
		"autoFlush", "buffer", "contentType", "errorPage", "extends",
		"import", "info", "isErrorPage", "isThreadSafe", "language", "pageEncoding", "session"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mJava.grammars);
	grammars.push.apply(grammars, mHTML.grammars);
	grammars.push({
		id: "orion.jsp",
		contentTypes: ["application/x-jsp"],
		patterns: [
			{include: "orion.html"},
			{include: "#jspComment"},
			{include: "#jspJavaFragment"},
			{include: "#jspDirectiveInclude"},
			{include: "#jspDirectivePage"},
			{include: "#jspDirectiveTaglib"}
		],
		repository: {
			jspComment: {
				begin: {match: "<%--", literal: "<%--"},
				end: {match: "--%>", literal: "<%--"},
				name: "comment.block.jsp",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!--%>).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.line"}
						}
					}
				]
			},
			jspDirectiveInclude: {
				begin: "<%@\\s+include(?:\\s|$)",
				end: "%>",
				captures: {
					0: {name: "entity.name.directive.include.jsp"}
				},
				patterns: [
					{
						match: "\\bfile\\b",
						name: "entity.other.attribute-name.jsp"
					},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			},
			jspDirectivePage: {
				begin: "<%@\\s+page(?:\\s|$)",
				end: "%>",
				captures: {
					0: {name: "entity.name.directive.page.jsp"}
				},
				patterns: [
					{
						match: "\\b(?:" + pageAttributeNames.join("|") + ")\\b",
						name: "entity.other.attribute-name.jsp"
					},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			},
			jspDirectiveTaglib: {
				begin: "<%@\\s+taglib(?:\\s|$)",
				end: "%>",
				captures: {
					0: {name: "entity.name.directive.taglib.jsp"}
				},
				patterns: [
					{
						match: "\\b(?:uri|prefix)\\b",
						name: "entity.other.attribute-name.jsp"
					},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			},
			jspJavaFragment: {
				begin: "<%(?:=|!)?(?:\\s|$)",
				end: "%>",
				captures: {
					0: {name: "entity.name.declaration.java"}
				},
				contentName: "source.java.embedded.jsp",
				patterns: [
					{include: "orion.java"}
				]
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/java/javaPlugin',['orion/plugin', 'orion/editor/stylers/text_x-java-source/syntax', 'orion/editor/stylers/application_x-jsp/syntax'], function(PluginProvider, mJava, mJSP) {

	function connect() {
		var headers = {
			name: "Orion Java Tool Support",
			version: "1.0",
			description: "This plugin provides Java tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-java-source",
					"extends": "text/plain",
					name: "Java",
					extension: ["java"]
				}, {id: "application/x-jsp",
					"extends": "text/plain",
					name: "Java Server Page",
					extension: ["jsp"]
				}
			]
		});
		mJava.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		mJSP.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_jsx/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/application_javascript/syntax", "orion/editor/stylers/application_xml/syntax"],
	/* eslint-disable missing-nls */
	function(mLib, mJS, mXML) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push({
		id: "orion.jsx",
		contentTypes: ["text/jsx"],
		patterns: [
			{
				begin: "</?[A-Za-z0-9]+",
				end: "/?>",
				captures: {
					0: {name: "meta.tag.jsx"}
				},
				patterns: [
					{include: "#jsExpression"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
					{include: "orion.c-like#comment_singleLine"},
					{include: "orion.c-like#comment_block"}
				]
			}, {
				include: "orion.xml#ampersandEscape"
			}, {
				include: "orion.js"
			}
		],
		repository: {
			jsExpression: {
				begin: "\\{",
				end: "\\}",
				contentName: "source.js.embedded.jsx",
				patterns: [
					{include: "orion.js"}
				]
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/jsx/jsxPlugin',['orion/plugin', 'orion/editor/stylers/text_jsx/syntax'], function(PluginProvider, mJSX) {

	function connect() {
		var headers = {
			name: "Orion JSX Tool Support",
			version: "1.0",
			description: "This plugin provides JSX tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/jsx",
					"extends": "text/plain",
					name: "JSX",
					extension: ["jsx"]
				}
			] 
		});
		mJSX.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/launch/launchPlugin',['orion/plugin'], function(PluginProvider) {

	function connect() {
		var headers = {
			name: "Orion Launch File Tool Support",
			version: "1.0",
			description: "This plugin provides tools support for Orion Launch files."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-launch",
					"extends": "application/json",
					name: "Launch file",
					extension: ["launch"]
				}
			] 
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-less/syntax", ["orion/editor/stylers/text_css/syntax", "orion/editor/stylers/lib/syntax"], function(mCSS, mLib) {
	var flags = ["important"];
	var constants = ["true"];
	var directives = ["arguments"];
	var keywords = ["all", "and", "not", "when"];
	var importKeywords = [
		"css",
		"inline",
		"less",
		"multiple",
		"once", "optional",
		"reference"
	];

	var grammars = [];
	grammars.push.apply(grammars, mCSS.grammars);
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.less",
		contentTypes: ["text/x-less"],
		patterns: [
			{include: "orion.css#string_single_multiline"},
			{include: "orion.css#string_double_multiline"},
			{include: "orion.c-like#comment_block"},
			{include: "#string_doubleQuote"},
			{include: "#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#importDirective"},
			{include: "orion.css#directive"},
			{include: "#directive"},
			{include: "#variable"},
			{include: "#interpolated"},
			{include: "#constant"},
			{include: "#flag"},
			{include: "#operator"},
			{include: "#keyword"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.css#number_hex"},
			{include: "orion.css#numeric_value"},
			{include: "orion.css#color"},
			{include: "orion.css#keyword"}
		],
		repository: {
			constant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.less"
			},
			directive: {
				match: "(^|\\s)(@("  + directives.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.directive.less"}
				}
			},
			flag: {
				match: "(^|\\s)(!("  + flags.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.flag.less"}
				}
			},
			importDirective: {
				begin: "(@import)\\s*\\(",
				end: "\\)",
				beginCaptures: {
					1: {name: "keyword.other.directive.less"}
				},
				patterns: [{
					match: "\\b(?:" + importKeywords.join("|") + ")\\b",
					name: "keyword.operator.less"
				}]
			},
			interpolated: {
				match: "@\\{[^}]*\\}",
				name: "string.interpolated.less"
			},
			keyword: {
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.less"
			},
			operator: {
				match: "\\+_?|\\*|\\/|=|>=?|<=?|&",
				name: "punctuation.operator.less"
			},
			string_doubleQuote: {
				match: '~?"(?:\\\\.|[^"])*"?',
				name: "string.quoted.double"
			},
			string_singleQuote: {
				match: "~?'(?:\\\\.|[^'])*'?",
				name: "string.quoted.single"
			},
			variable: {
				match: "@[\\w-]+(\\.{3})?",
				name: "variable.other.less"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: directives.concat(flags).concat(importKeywords).concat(keywords)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/less/lessPlugin',['orion/plugin', 'orion/editor/stylers/text_x-less/syntax'], function(PluginProvider, mLess) {

	function connect() {
		var headers = {
			name: "Orion Less Tool Support",
			version: "1.0",
			description: "This plugin provides Less tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-less",
					"extends": "text/plain",
					name: "Less",
					extension: ["less"]
				}
			] 
		});
		mLess.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-lua/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	// from http://www.lua.org/ftp/refman-5.0.pdf
	var keywords = [
		"and",
		"break",
		"do",
		"else", "elseif", "end",
		"false", "for", "function",
		"if", "in",
		"local",
		"nil", "not",
		"or",
		"repeat", "return",
		"then", "true",
		"until",
		"while",
	];

	// base library functions and variables
	// from http://lua-users.org/files/wiki_insecure/users/thomasl/luarefv51.pdf
	var base_functions = [
		"assert", "arg",
		"collectgarbage",
		"dofile",
		"error",
		"getfenv", "getmetatable",
		"ipairs",
		"load", "loadfile", "loadstring",
		"next",
		"pairs", "pcall", "print",
		"rawequal", "rawget", "rawset", "require",
		"select", "setfenv", "setmetatable",
		"tonumber", "tostring", "type",
		"unpack",
		"xpcall"
	];
	var base_variables = [
		"_G",
		"LUA_INIT", "LUA_PATH", "LUA_CPATH",
		"_VERSION"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.lua",
		contentTypes: ["text/x-luasrc", "text/x-lua"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#comment_block_dash_dash"},
			{include: "#comment_singleLine_dash_dash"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{include: "#base_functions"},
			{include: "#base_variables"},
			{include: "#reserved_underscore_capital"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.lua"
			}
		],
		repository: {
			comment_block_dash_dash: {
				begin: {match: "--\\[\\[", literal: "--[["},
				end: {match: "\\]\\]", literal: "]]"},
				name: "comment.block.dash-dash.lua",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!\\]\\]).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.block"}
						}
					}
				]
			},
			comment_singleLine_dash_dash: {
				begin: {match: "--", literal: "--"},
				end: {match: "$", literal: ""},
				name: "comment.line.dash-dash.lua",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			base_functions: {
				match: "\\b(?:" + base_functions.join("|") + ")\\b",
				name: "support.function.lua"
			},
			base_variables: {
				match: "\\b(?:" + base_variables.join("|") + ")\\b",
				name: "support.variable.lua"
			},
			// _ANYCAPITALS is reserved in Lua
			reserved_underscore_capital: {
				match: "\\b_[A-Z]*\\b",
				name: "constant.other.userdefined.lua"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/lua/luaPlugin',['orion/plugin', 'orion/editor/stylers/text_x-lua/syntax'], function(PluginProvider, mLua) {

	function connect() {
		var headers = {
			name: "Orion Lua Tool Support",
			version: "1.0",
			description: "This plugin provides Lua tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-lua",
					"extends": "text/plain",
					name: "Lua",
					extension: ["lua"]
				}
			] 
		});
		mLua.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global requirejs*/
define('orion/i18nUtil',[], function() {
	/**
	 * Performs string substitution. Can be invoked in 2 ways:
	 *
	 * i) vargs giving numbered substition values:
	 *   formatMessage("${0} is ${1}", "foo", "bar")  // "foo is bar"
	 *
	 * ii) a map giving the substitutions:
	 *   formatMessage("${thing} is ${1}", {1: "bar", thing: "foo"})  // "foo is bar"
	 */
	function formatMessage(msg) {
		var pattern = /\$\{([^\}]+)\}/g, args = arguments;
		if (args.length === 2 && args[1] && typeof args[1] === "object") {
			return msg.replace(pattern, function(str, key) {
				return args[1][key];
			});
		}
		return msg.replace(pattern, function(str, index) {
			return args[(index << 0) + 1];
		});
	}
	return {
		formatMessage: formatMessage
	};
});
/**
 * @license RequireJS i18n 2.0.4 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/requirejs/i18n for details
 */
/*jslint regexp: true */
/*global require: false, navigator: false, define: false */

/**
 * This plugin handles i18n! prefixed modules. It does the following:
 *
 * 1) A regular module can have a dependency on an i18n bundle, but the regular
 * module does not want to specify what locale to load. So it just specifies
 * the top-level bundle, like "i18n!nls/colors".
 *
 * This plugin will load the i18n bundle at nls/colors, see that it is a root/master
 * bundle since it does not have a locale in its name. It will then try to find
 * the best match locale available in that master bundle, then request all the
 * locale pieces for that best match locale. For instance, if the locale is "en-us",
 * then the plugin will ask for the "en-us", "en" and "root" bundles to be loaded
 * (but only if they are specified on the master bundle).
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/colors bundle to be that mixed in locale.
 *
 * 2) A regular module specifies a specific locale to load. For instance,
 * i18n!nls/fr-fr/colors. In this case, the plugin needs to load the master bundle
 * first, at nls/colors, then figure out what the best match locale is for fr-fr,
 * since maybe only fr or just root is defined for that locale. Once that best
 * fit is found, all of its locale pieces need to have their bundles loaded.
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/fr-fr/colors bundle to be that mixed in locale.
 */
(function () {
    'use strict';

    //regexp for reconstructing the master bundle name from parts of the regexp match
    //nlsRegExp.exec("foo/bar/baz/nls/en-ca/foo") gives:
    //["foo/bar/baz/nls/en-ca/foo", "foo/bar/baz/nls/", "/", "/", "en-ca", "foo"]
    //nlsRegExp.exec("foo/bar/baz/nls/foo") gives:
    //["foo/bar/baz/nls/foo", "foo/bar/baz/nls/", "/", "/", "foo", ""]
    //so, if match[5] is blank, it means this is the top bundle definition.
    var nlsRegExp = /(^.*(^|\/)nls(\/|$))([^\/]*)\/?([^\/]*)/;

    //Helper function to avoid repeating code. Lots of arguments in the
    //desire to stay functional and support RequireJS contexts without having
    //to know about the RequireJS contexts.
    function addPart(locale, master, needed, toLoad, prefix, suffix) {
        if (master[locale]) {
            needed.push(locale);
            if (master[locale] === true || master[locale] === 1) {
                toLoad.push(prefix + locale + '/' + suffix);
            }
        }
    }

    function addIfExists(req, locale, toLoad, prefix, suffix) {
        var fullName = prefix + locale + '/' + suffix;
        if (require._fileExists(req.toUrl(fullName + '.js'))) {
            toLoad.push(fullName);
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force) {
        var prop;
        for (prop in source) {
            if (source.hasOwnProperty(prop) && (!target.hasOwnProperty(prop) || force)) {
                target[prop] = source[prop];
            } else if (typeof source[prop] === 'object') {
                if (!target[prop] && source[prop]) {
                    target[prop] = {};
                }
                mixin(target[prop], source[prop], force);
            }
        }
    }

    define('i18n',['module'], function (module) {
        var masterConfig = module.config ? module.config() : {};
        masterConfig = masterConfig || {};

        return {
            version: '2.0.4',
            /**
             * Called when a dependency needs to be loaded.
             */
            load: function (name, req, onLoad, config) {
                config = config || {};

                if (config.locale) {
                    masterConfig.locale = config.locale;
                }

                var masterName,
                    match = nlsRegExp.exec(name),
                    prefix = match[1],
                    locale = match[4],
                    suffix = match[5],
                    parts = locale.split("-"),
                    toLoad = [],
                    value = {},
                    i, part, current = "";

                //If match[5] is blank, it means this is the top bundle definition,
                //so it does not have to be handled. Locale-specific requests
                //will have a match[4] value but no match[5]
                if (match[5]) {
                    //locale-specific bundle
                    prefix = match[1];
                    masterName = prefix + suffix;
                } else {
                    //Top-level bundle.
                    masterName = name;
                    suffix = match[4];
                    locale = masterConfig.locale;
                    if (!locale) {
                        locale = masterConfig.locale =
                            typeof navigator === "undefined" ? "root" :
                            (navigator.language ||
                             navigator.userLanguage || "root").toLowerCase();
                    }
                    parts = locale.split("-");
                }

                if (config.isBuild) {
                    //Check for existence of all locale possible files and
                    //require them if exist.
                    toLoad.push(masterName);
                    addIfExists(req, "root", toLoad, prefix, suffix);
                    for (i = 0; i < parts.length; i++) {
                        part = parts[i];
                        current += (current ? "-" : "") + part;
                        addIfExists(req, current, toLoad, prefix, suffix);
                    }
                                        
                    if(config.locales) {
                    	var j, k; 
                    	for (j = 0; j < config.locales.length; j++) {
                    		locale = config.locales[j];
                    		parts = locale.split("-");
                    		current = "";
	                    	for (k = 0; k < parts.length; k++) {
		                        part = parts[k];
		                        current += (current ? "-" : "") + part;
		                        addIfExists(req, current, toLoad, prefix, suffix);
	                    	}
                    	}
                    }

                    req(toLoad, function () {
                        onLoad();
                    });
                } else {
                    //First, fetch the master bundle, it knows what locales are available.
                    req([masterName], function (master) {
                        //Figure out the best fit
                        var needed = [],
                            part;

                        //Always allow for root, then do the rest of the locale parts.
                        addPart("root", master, needed, toLoad, prefix, suffix);
                        for (i = 0; i < parts.length; i++) {
                            part = parts[i];
                            current += (current ? "-" : "") + part;
                            addPart(current, master, needed, toLoad, prefix, suffix);
                        }

                        //Load all the parts missing.
                        req(toLoad, function () {
                            var i, partBundle, part;
                            for (i = needed.length - 1; i > -1 && needed[i]; i--) {
                                part = needed[i];
                                partBundle = master[part];
                                if (partBundle === true || partBundle === 1) {
                                    partBundle = req(prefix + part + '/' + suffix);
                                }
                                mixin(value, partBundle);
                            }

                            //All done, notify the loader.
                            onLoad(value);
                        });
                    });
                }
            }
        };
    });
}());

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/nls/messages',{
	root:true
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/nls/root/messages',{//Default message bundle
	"Navigator": "Navigator",
	"Sites": "Sites",
	"Shell": "Shell",
	"Get Plugins": "Get Plug-ins",
	"Global": "Global",
	"Editor": "Editor",
	"EditorRelatedLink": "Show Current Folder",
	"EditorRelatedLinkParent": "Show Enclosing Folder",
	"EditorLinkWorkspace": "Edit",
	"EditorRelatedLinkProj": "Show Project",
	"navigationBar": "Navigation Bar",
	"Filter bindings": "Filter bindings",
	"BindingPrompt": "Enter the new binding",
	"NoBinding": "---",
	"orionClientLabel": "Orion client repository",
	"Orion Editor": "Orion Editor",
	"Orion Image Viewer": "Orion Image Viewer",
	"Orion Markdown Editor": "Orion Markdown Editor",
	"Orion Markdown Viewer": "Orion Markdown Viewer",
	"Orion JSON Editor": "Orion JSON Editor",
	"View on Site": "View on Site",
	"View this file or folder on a web site hosted by Orion": "View this file or folder on a web site hosted by Orion.",
	"ShowAllKeyBindings": "Show a list of all the keybindings on this page",
	"Show Keys": "Show Keys",
	"HideShowBannerFooter": "Hide or show the page banner",
	"Toggle banner and footer": "Toggle banner",
	"ChooseFileOpenEditor": "Choose a file by name and open an editor on it",
	"FindFile": "Open File...",
	"System Configuration Details": "System Configuration Details",
	"System Config Tooltip": "Go to the System Configuration Details page",
	"Background Operations": "Background Operations",
	"Background Operations Tooltip": "Go to the Background Operations page",
	"Operation status is unknown": "Operation status is unknown",
	"Unknown item": "Unknown item",
	"NoSearchAvailableErr": "Can't search: no search service is available",
	"Related": "Related",
	"Options": "Options",
	"LOG: ": "LOG: ",
	"View": "View",
	"SplashTitle": "Setting up Workspace",
	"SplashTitleSettings": "Loading Settings",
	"SplashTitleGit": "Loading Git Repositories",
	"LoadingPage": "Loading Page",
	"LoadingPlugins": "Loading Plugins",
	"LoadingResources": "Loading Resources",
	"plugin_started": "\"${0}\" started",
	"plugin_lazy activation": "\"${0}\" lazily activated",
	"plugin_starting": "\"${0}\" starting",
	"no parent": "no parent",
	"no tree model": "no tree model",
	"no renderer": "no renderer",
	"could not find table row ": "could not find table row ",
	"Operations": "Operations",
	"Operations running": "Operations running",
	"SomeOpWarning": "Some operations finished with warning",
	"SomeOpErr": "Some operations finished with error",
	"no service registry": "no service registry",
	"Tasks": "Tasks",
	"Close": "Close",
	"Expand all": "Expand all",
	"Collapse all": "Collapse all",
	"Search" : "Search",
	"Advanced search" : "Advanced search",
	"Submit" : "Submit",
	"More" : "More",
	"Recent searches" : "Recent searches",
	"Regular expression" : "Regular expression",
	"Search options" : "Search options",
	"Global search" : "Global search",
	"Orion Home" : "Orion Home",
	"Close notification" : "Close notification",
	"OpPressSpaceMsg" : "Operations - Press spacebar to show current operations",
	"Toggle side panel" : "Toggle side panel",
	"Open or close the side panel": "Open or close the side panel",
	"Projects" : "Projects",
	"Toggle Sidebar" : "Toggle Sidebar",
	"Sample HTML5 Site": "Sample HTML5 Site",
	"Generate an HTML5 'Hello World' website, including JavaScript, HTML, and CSS files.": "Generate an HTML5 'Hello World' website, including JavaScript, HTML, and CSS files.",
	"Sample Orion Plugin": "Sample Orion Plug-in",
	"Generate a sample plugin for integrating with Orion.": "Generate a sample plug-in for integrating with Orion.",
	"Browser": "Web Browser",
	"OutlineProgress": "Getting outline for ${0} from ${1}",
	"outlineTimeout": "Outline service timed out. Try reloading the page and opening the outline again.",
	"UnknownError": "An unknown error occurred.",
	"Filter": "Filter (* = any string, ? = any character)",
	"TemplateExplorerLabel": "Templates",
	"OpenTemplateExplorer": "Open Template Explorer",
	"Edit": "Edit",
	"CentralNavTooltip": "Toggle Navigation Menu",
	"Wrote: ${0}": "Wrote: ${0}",
	"GenerateHTML": "Generate HTML file",
	"GenerateHTMLTooltip": "Write an HTML file generated from the current Markdown editor content",
	"alt text": "alt text",
	"blockquote": "blockquote",
	"code": "code",
	"code (block)": "code (block)",
	"code (span)": "code (span)",
	"emphasis": "emphasis",
	"fenced code (${0})": "fenced code (${0})",
	"header (${0})": "header (${0})",
	"horizontal rule": "horizontal rule",
	"label": "label",
	"link (auto)": "link (auto)",
	"link (image)": "link (image)",
	"link (inline)": "link (inline)",
	"link label": "link label",
	"link label (optional)": "link label (optional)",
	"link (ref)": "link (ref)",
	"list item (bullet)": "list item (bullet)",
	"list item (numbered)": "list item (numbered)",
	"strikethrough (${0})": "strikethrough (${0})",
	"strong": "strong",
	"table (${0})": "table (${0})",
	"text": "text",
	"title (optional)": "title (optional)",
	"url": "url",
	"workedProgress": "${0} (${1}/${2})",
	"TogglePaneOrientationTooltip": "Toggle split pane orientation",
	"WarningDuplicateLinkId": "Duplicate link ID: ${0} (link IDs are not case-sensitive)",
	"WarningHeaderTooDeep": "Header level cannot exceed 6",
	"WarningLinkHasNoText": "Link has no text",
	"WarningLinkHasNoURL": "Link has no URL",
	"WarningOrderedListItem": "Ordered list item within unordered list",
	"WarningOrderedListShouldStartAt1": "The first item in an ordered list should have index 1",
	"WarningUndefinedLinkId": "Undefined link ID: ${0}",
	"WarningUnorderedListItem": "Unordered list item within ordered list",
	"PageTitleFormat": "${0} - ${1}", // ${0} is the file or resource being edited; ${1} is the task (eg. "Editor")
	// Display names for keys:
	"KeyCTRL": "Ctrl",
	"KeySHIFT": "Shift",
	"KeyALT": "Alt",
	"KeyBKSPC": "Backspace",
	"KeyDEL": "Del",
	"KeyEND": "End",
	"KeyENTER": "Enter",
	"KeyESCAPE": "Esc",
	"KeyHOME": "Home",
	"KeyINSERT": "Ins",
	"KeyPAGEDOWN": "Page Down",
	"KeyPAGEUP": "Page Up",
	"KeySPACE": "Space",
	"KeyTAB": "Tab",
	// Display elapsed time:
	"a year": "a year",
	"years": "${0} years",
	"a month": "a month",
	"months": "${0} months",
	"a day": "a day",
	"days": "${0} days",
	"an hour": "an hour",
	"hours": "${0} hours",
	"a minute": "a minute",
	"minutes": "${0} minutes",
	"timeAgo": "${0} ago", //${0} represent the time elapsed
	"justNow": "just now", //Represent that the time elapsed is less than 1 minute
	"fixAll": "Fix all"
});


/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser,amd*/
define('plugins/languages/markdown/markdownPlugin',['orion/plugin', 'orion/i18nUtil', 'i18n!orion/nls/messages'], function (PluginProvider, i18nUtil, messages) { //$NON-NLS-0$

	function connect() {
		var headers = {
			name: "Orion Extended Markdown Language Tool Support",
			version: "1.0",
			description: "This plugin provides extended Markdown language tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-markdown",
					"extends": "text/plain",
					name: "Markdown",
					extension: ["md"]
				}
			] 
		});
		pluginProvider.registerServiceProvider("orion.edit.contentAssist", { //$NON-NLS-0$
			computeProposals: function (buffer, offset, context) {
				var selection = context.selection.start !== context.selection.end ?
					buffer.substring(context.selection.start, context.selection.end) :
					null;
				var NL = context.delimiter;
				var multilineSelection = selection && selection.indexOf(NL) !== -1;
				var onEmptyLine = /^[ ]{0,3}$/.test(context.line);
				var result = [];

				result.push({
					description: messages["emphasis"],
					escapePosition: selection ? null : offset + 1,
					proposal: "*" + (selection ? selection : "") + "*" //$NON-NLS-1$ //$NON-NLS-0$
				});

				result.push({
					description: messages["strong"],
					escapePosition: selection ? null : offset + 2,
					proposal: "**" + (selection ? selection : "") + "**" //$NON-NLS-1$ //$NON-NLS-0$
				});

				if (!multilineSelection) {
					var headerText = onEmptyLine && !selection ? messages["text"] : "";
					result.push({
						description: i18nUtil.formatMessage(messages["header (${0})"], "atx"), //$NON-NLS-0$
						positions: onEmptyLine ? [{offset: offset + 1, length: headerText.length}] : null,
						proposal: (onEmptyLine ? "" : NL) + "#" + headerText + (selection ? selection : "") //$NON-NLS-0$
					});
				}

				if (!multilineSelection) {
					result.push({
						description: messages["link (auto)"],
						escapePosition: selection ? null : offset + 1,
						proposal: "<" + (selection ? selection : "") + ">" //$NON-NLS-1$ //$NON-NLS-0$
					});
				}

				var inlineLinkText = selection || messages["text"];
				var inlineLinkUrl = messages["url"];
				var inlineLinkTitle = messages["title (optional)"];
				result.push({
					description: messages["link (inline)"],
					positions: [
						{offset: offset + 1, length: inlineLinkText.length},
						{offset: offset + 3 + inlineLinkText.length, length: inlineLinkUrl.length},
						{offset: offset + 5 + inlineLinkText.length + inlineLinkUrl.length, length: inlineLinkTitle.length}],
					proposal: "[" + inlineLinkText + "](" + inlineLinkUrl + " \"" + inlineLinkTitle + "\")" //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
				});

				var imageLinkText = selection || messages["alt text"];
				var imageLinkUrl = messages["url"];
				var imageLinkTitle = messages["title (optional)"];
				result.push({
					description: messages["link (image)"],
					positions: [
						{offset: offset + 2, length: imageLinkText.length},
						{offset: offset + 4 + imageLinkText.length, length: imageLinkUrl.length},
						{offset: offset + 6 + imageLinkText.length + imageLinkUrl.length, length: imageLinkTitle.length}],
					proposal: "![" + imageLinkText + "](" + imageLinkUrl + " \"" + imageLinkTitle + "\")" //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
				});
	
				var refLinkText = selection || messages["text"];
				var refLinkLabel = messages["link label (optional)"];
				result.push({
					description: messages["link (ref)"],
					positions: [
						{offset: offset + 1, length: refLinkText.length},
						{offset: offset + 3 + refLinkText.length, length: refLinkLabel.length}],
					proposal: "[" + refLinkText + "][" + refLinkLabel + "]" //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
				});

				if (!multilineSelection) {
					var linkLabelText = selection || messages["label"];
					var linkLabelUrl = messages["url"];
					var linkLabelTitle = messages["title (optional)"];
					var linkLabelNL = onEmptyLine ? "" : NL;
					result.push({
						description: messages["link label"],
						positions: [
							{offset: offset + linkLabelNL.length + 1, length: linkLabelText.length},
							{offset: offset + linkLabelNL.length + 4 + linkLabelText.length, length: linkLabelUrl.length},
							{offset: offset + linkLabelNL.length + 5 + linkLabelText.length + linkLabelUrl.length, length: linkLabelTitle.length}],
						proposal: linkLabelNL + "[" + linkLabelText + "]: " + linkLabelUrl + " " + linkLabelTitle + NL //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
					});
				}

				if (!multilineSelection) {
					var codeBlockText = onEmptyLine && !selection ? messages["code"] : "";
					var indent = onEmptyLine ? "    ".substring(0, 4 - context.indentation.length) : NL + NL + "    "; //$NON-NLS-1$ //$NON-NLS-0$
					result.push({
						description: messages["code (block)"],
						positions: codeBlockText ? [{offset: offset + indent.length, length: codeBlockText.length}] : null,
						proposal: indent + codeBlockText + (selection ? selection : "")
					});
				}

				result.push({
					description: messages["code (span)"],
					escapePosition: selection ? null : offset + 1,
					proposal: "`" + (selection ? selection : "") + "`" //$NON-NLS-1$ //$NON-NLS-0$
				});

				result.push({
					description: messages["horizontal rule"],
					proposal: (onEmptyLine ? "" : NL) + "- - -" + NL + (selection ? selection : "") //$NON-NLS-0$
				});

				if (!multilineSelection) {
					var blockquoteText = onEmptyLine && !selection ? messages["text"] : "";
					result.push({
						description: messages["blockquote"],
						positions: onEmptyLine ? [{offset: offset + 2, length: blockquoteText.length}] : null,
						proposal: (onEmptyLine ? "" : NL) + "> " + blockquoteText + (selection ? selection : "") //$NON-NLS-0$
					});
				}

				var listItemText = onEmptyLine && !selection ? messages["text"] : "";
				result.push({
					description: messages["list item (numbered)"],
					positions: onEmptyLine ? [{offset: offset + 3, length: listItemText.length}] : null,
					proposal: (onEmptyLine ? "" : NL) + "1. " + listItemText + (selection ? selection : "") //$NON-NLS-0$
				});

				result.push({
					description: messages["list item (bullet)"],
					positions: onEmptyLine ? [{offset: offset + 2, length: listItemText.length}] : null,
					proposal: (onEmptyLine ? "" : NL) + "* " + listItemText + (selection ? selection : "") //$NON-NLS-0$
				});

				result.push({ /* gfm items separator */
					style: "hr" //$NON-NLS-0$
				});
	
				result.push({
					description: i18nUtil.formatMessage(messages["strikethrough (${0})"], "gfm"), //$NON-NLS-0$
					escapePosition: selection ? null : offset + 2,
					proposal: "~~" + (selection ? selection : "") + "~~" //$NON-NLS-1$ //$NON-NLS-0$
				});

				var tableNL = onEmptyLine ? "" : NL;
				result.push({
					description: i18nUtil.formatMessage(messages["table (${0})"], "gfm"), //$NON-NLS-0$
					positions: [
						{offset: offset + tableNL.length, length: 5},
						{offset: offset + tableNL.length + 7, length: 9},
						{offset: offset + tableNL.length + 16 + NL.length, length: 6},
						{offset: offset + tableNL.length + 23 + NL.length, length: 9},
						{offset: offset + tableNL.length + 32 + 2 * NL.length, length: 4},
						{offset: offset + tableNL.length + 39 + 2 * NL.length, length: 4}],
					proposal: tableNL + "hLeft |hCentered" + NL + ":-----|:-------:" + NL + "item  |item     " + NL + (selection ? selection : "") //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
				});
	
				result.push({
					description: i18nUtil.formatMessage(messages["fenced code (${0})"], "gfm"), //$NON-NLS-0$
					escapePosition: selection ? null : offset + 3,
					proposal: "```" + (selection ? selection : "") + "```" //$NON-NLS-1$ //$NON-NLS-0$
				});
	
				return result;
			}
		}, {
			name: "Markdown content assist", //$NON-NLS-0$
			contentType: ["text/x-markdown"] //$NON-NLS-0$
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-objective-c/syntax", ["orion/editor/stylers/text_x-csrc/syntax"], function(mC) {
	var keywords = [
		"atomic",
		"BOOL", "bycopy", "byref",
		"Class",
		"id", "IMP", "inout", "in",
		"nil", "nonatomic", "NO", "NULL",
		"oneway", "out",
		"Protocol",
		"retain",
		"SEL", "self", "super",
		"YES"
	];

	var atKeywords = [
		"@catch", "@class",
		"@dynamic",
		"@end",
		"@finally",
		"@implementation", "@interface",
		"@private", "@property", "@protected", "@protocol", "@public",
		"@selector",
		"@synthesize",
		"@throw", "@try"
	];

	var directives = ["import"];

	var grammars = [];
	grammars.push.apply(grammars, mC.grammars);
	grammars.push({
		id: "orion.objectiveC",
		contentTypes: ["text/x-objective-c"],
		patterns: [
			{include: "#objectiveCString"},
			{include: "orion.c"},
			{include: "#objectiveCDirective"},
			{include: "#objectiveCKeyword"}
		],
		repository: {
			objectiveCDirective: {
				match: "#\\s*(?:" + directives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.objective-c"
			},
			objectiveCKeyword: {
				match: "(\\b(?:" + keywords.join("|") + ")|(?:" + atKeywords.join("|") + "))\\b",
				name: "keyword.operator.objective-c"
			},
			objectiveCString: {
				match: '@"(?:\\\\.|[^"])*"?',
				name: "string.quoted.double.objective-c"
			},
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: mC.keywords.concat(keywords).concat(directives).concat(atKeywords)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/objectiveC/objectiveCPlugin',['orion/plugin', 'orion/editor/stylers/text_x-objective-c/syntax'], function(PluginProvider, mObjectiveC) {

	function connect() {
		var headers = {
			name: "Orion Objective-C Tool Support",
			version: "1.0",
			description: "This plugin provides Objective-C tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-objective-c",
					"extends": "text/plain",
					name: "Objective-C",
					extension: ["m", "mm", "h"]
				}
			] 
		});
		mObjectiveC.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-php/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"abstract", "and", "array", "as",
		"callable", "class", "clone", "const",
		"declare",
		"echo", "empty", "eval", "extends",
		"final", "function",
		"global",
		"implements", "include", "include_once", "insteadof", "interface", "instanceof", "isset",
		"list",
		"namespace", "new",
		"or",
		"parent", "print", "private", "protected", "public",
		"require", "require_once",
		"static",
		"trait",
		"unset", "use",
		"var",
		"xor",
		"__halt_compiler", "__CLASS__", "__DIR__", "__FILE__", "__FUNCTION__",
		"__LINE__", "__METHOD__", "__NAMESPACE__", "__TRAIT__"
	];
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "die", "do",
		"else", "elseif", "enddeclare", "endfor", "endforeach", "endif", "endswitch", "endwhile", "exit",
		"finally", "for", "foreach",
		"goto",
		"if",
		"return",
		"switch",
		"throw", "try",
		"while",
		"yield"
	];
	var constants = [
		"false", "FALSE", "null", "NULL", "true", "TRUE"
	];
	var languageVariables = ["self"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.php-core",
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{
				match: {match: "#.*", literal: "#"},
				name: "comment.line.number-sign.php",
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}, {
				begin: "<<<(\\w+)$",
				end: "^\\1;$",
				name: "string.unquoted.heredoc.php"
			}, {
				begin: "<<<'(\\w+)'$",
				end: "^\\1;$",
				name: "string.unquoted.heredoc.nowdoc.php"
			},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{
				match: "\\b0[bB][01]+\\b",
				name: "constant.numeric.binary.php"
			},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.php"
			},
			{
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.php"
			},
			{
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.php"
			},
			{
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.php"
			}
		]
	});
	
	grammars.push({
		id: "orion.php",
		contentTypes: ["text/x-php"],
		patterns: [
			{
				begin: "(?i)<(\\?|%(?!php))(?:=|php)?(?:\\s|$)",
				end: "[\\1]>",
				captures: {
					0: {name: "entity.name.declaration.php"}
				},
				contentName: "source.php.embedded",
				patterns: [
					{include: "orion.php-core"}
				]
			}, {
				include: "orion.html"
			}
		]
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(constants).concat(languageVariables)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/php/phpPlugin',['orion/plugin', 'orion/editor/stylers/text_x-php/syntax'], function(PluginProvider, mPHP) {

	function connect() {
		var headers = {
			name: "Orion PHP Tool Support",
			version: "1.0",
			description: "This plugin provides PHP tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-php",
					"extends": "text/plain",
					name: "PHP",
					extension: ["php", "php3", "php4", "php5", "phpt", "phtml", "aw", "ctp"]
				}
			] 
		});
		mPHP.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-properties/syntax", [], function() {
	var grammars = [];
	grammars.push({
		id: "orion.properties",
		contentTypes: ["text/x-properties"],
		patterns: [
			{
				match: "^\\s*#[^$]*",
				name: "comment.line.properties"
			}, {
				match: "(^[^=]*)(=)([^$]*)",
				captures: {
					1: {name: "entity.other.attribute-name.properties"},
					3: {name: "string.unquoted.properties"}
				}
			}, {
				match: "^[^=]*$",
				name: "string.unquoted.properties"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/properties/propertiesPlugin',['orion/plugin', 'orion/editor/stylers/text_x-properties/syntax'], function(PluginProvider, mProperties) {

	function connect() {
		var headers = {
			name: "Orion Properties File Support",
			version: "1.0",
			description: "This plugin provides Properties file support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-properties",
					"extends": "text/plain",
					name: "Properties",
					extension: ["properties"]
				}
			] 
		});
		mProperties.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-python/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"and", "as", "assert",
		"break",
		"class", "continue",
		"def", "del",
		"exec", "elif", "else", "except", "Ellipsis",
		"False", "finally", "for", "from",
		"global",
		"if", "import", "in", "is",
		"lambda",
		"not", "None", "NotImplemented",
		"or",
		"pass", "print",
		"raise", "return",
		"try", "True",
		"while", "with",
		"yield"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.python",
		contentTypes: ["text/x-python"],
		patterns: [
			{
				begin: "(['\"])\\1\\1",
				end: "\\1\\1\\1",
				name: "string.quoted.triple.python"
			}, 
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{
				begin: {match: "#", literal: "#"},
				end: {match: "$", literal: ""},
				name: "comment.line.number-sign.python",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "#number_decimal"},
			{include: "orion.lib#number_hex"}, 
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.python"
			}
		],
		repository: {
			number_decimal: {
				match: "\\b-?(?:\\.\\d+|\\d+\\.?\\d*)[lL]?\\b",
				name: "constant.numeric.number.python"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/python/pythonPlugin',['orion/plugin', 'orion/editor/stylers/text_x-python/syntax'], function(PluginProvider, mPython) {

	function connect() {
		var headers = {
			name: "Orion Python Tool Support",
			version: "1.0",
			description: "This plugin provides Python tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-python",
					"extends": "text/plain",
					name: "Python",
					extension: ["py", "rpy", "pyw", "cpy", "SConstruct", "Sconstruct", "sconstruct", "SConscript", "gyp", "gypi"]
				}
			] 
		});
		mPython.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/ruby/rubyPlugin',['orion/plugin', 'orion/editor/stylers/text_x-ruby/syntax'], function(PluginProvider, mRuby) {

	function connect() {
		var headers = {
			name: "Orion Ruby Tool Support",
			version: "1.0",
			description: "This plugin provides Ruby tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-ruby",
					"extends": "text/plain",
					name: "Ruby",
					extension: ["rb", "rbx", "rjs", "Rakefile", "rake", "cgi", "fcgi", "gemspec", "irbrc", "capfile", "ru", "prawn", "Gemfile", "Guardfile", "Vagrantfile", "Appraisals", "Rantfile"]
				}
			] 
		});
		mRuby.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-scss/syntax", ["orion/editor/stylers/text_css/syntax", "orion/editor/stylers/lib/syntax"], function(mCSS, mLib) {
	var flags = [
		"default", "global", "optional"
	];
	var constants = [
		"false", "null", "true"
	];
	var directives = [
		"at-root",
		"content",
		"debug",
		"each", "else", "error", "extend",
		"for", "function",
		"if", "include",
		"mixin",
		"return",
		"warn", "while"
	];

	var grammars = [];
	grammars.push.apply(grammars, mCSS.grammars);
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.scss",
		contentTypes: ["text/x-scss"],
		patterns: [
			{include: "orion.css#string_single_multiline"},
			{include: "orion.css#string_double_multiline"},
			{include: "orion.c-like#comment_block"},
			{include: "#eachin"},
			{include: "#forto"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "#variable"},
			{include: "#placeholder"},
			{include: "#flag"},
			{include: "#directive"},
			{include: "orion.css#directive"},
			{include: "#constant"},
			{include: "#interpolated"},
			{include: "#operator"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.css#number_hex"},
			{include: "orion.css#numeric_value"},
			{include: "orion.css#color"},
			{include: "orion.css#keyword"}
		],
		repository: {
			constant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.scss"
			},
			directive: {
				match: "(^|\\s)(@("  + directives.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.directive.scss"}
				}
			},
			eachin: {
				begin: "@each\\s",
				end: "\\sin\\b",
				captures: {
					0: {name: "keyword.other.directive.scss"},
				},
				patterns: [
					{include: "#variable"}
				]
			},
			flag: {
				match: "(^|\\s)(!("  + flags.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.flag.scss"}
				}
			},
			forto: {
				begin: "@for\\s",
				end: "(^|\\s)(t(o|hrough))(\\s|$)",
				beginCaptures: {
					0: {name: "keyword.other.directive.scss"},
				},
				endCaptures: {
					2: {name: "keyword.other.directive.scss"}
				},
				patterns: [
					{include: "#variable"},
					{include: "orion.lib#number_decimal"},
					{
						match: "(^|\\s)(from)(\\s|$)",
						name: "keyword.other.directive.scss"
					}
				]
			},
			interpolated: {
				match: "#\\{[^}]*\\}",
				name: "string.interpolated.scss"
			},
			operator: {
				match: "\\+|\\*|\\/|%|==?|!=|&|<=?|=?>|!",
				name: "punctuation.operator.scss"
			},
			placeholder: {
				match: "%[\\w-]+",
				name: "variable.other.placeholder.sas"
			},
			variable: {
				match: "\\$[\\w-]+(\\.{3})?",
				name: "variable.other.scss"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: directives.concat(flags)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/scss/scssPlugin',['orion/plugin', 'orion/editor/stylers/text_x-scss/syntax'], function(PluginProvider, mScss) {

	function connect() {
		var headers = {
			name: "Orion SCSS Tool Support",
			version: "1.0",
			description: "This plugin provides SCSS tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-scss",
					"extends": "text/plain",
					name: "SCSS",
					extension: ["scss", "sass"]
				}
			] 
		});
		mScss.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/text_x-smarty/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/text_html/syntax", "orion/editor/stylers/text_x-php/syntax"],
	function(mLib, mHTML, mPHP) {

	var constants = [
		"false", "no", "off", "on", "true", "yes"
	];

	/* these can be redefined in the file, this is not currently handled */
	var DELIMITER_OPEN = "{";
	var DELIMITER_CLOSE = "}";
	
	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mHTML.grammars);
	grammars.push.apply(grammars, mPHP.grammars);
	grammars.push({
		id: "orion.smarty",
		contentTypes: ["text/x-smarty"],
		patterns: [
			{include: "orion.html"},
			{include: "#smartyCommentBlock"},
			{include: "#literalBlock"},
			{include: "#phpBlock"},
			{include: "#smartyBlock"}
		],
		repository: {
			literalBlock: {
				begin: "({)literal(})",
				end: "({)/literal(})",
				captures: {
					1: "punctuation.brace.begin.smarty",
					2: "punctuation.brace.end.smarty"
				}
			},
			phpBlock: {
				begin: "({)php(})",
				end: "({)/php(})",
				captures: {
					1: "punctuation.brace.begin.smarty",
					2: "punctuation.brace.end.smarty"
				},
				contentName: "source.php.embedded.smarty",
				patterns: [
					{include: "orion.php-core"}
				]
			},
			smartyBlock: {
				begin: "(" + DELIMITER_OPEN + ")",
				end: "(" + DELIMITER_CLOSE + ")",
				captures: {
					1: "punctuation.brace.begin.smarty",
					2: "punctuation.brace.end.smarty"
				},
				patterns: [
					{include: "orion.lib#string_singleQuote"},
					{include: "#smartyString_doubleQuote"},
					{include: "#smartyVariable"},
					{include: "#smartyConfigVariable"},
					{include: "#smartyConstant"},
					{include: "orion.lib#number_decimal"},
				]
			},
			smartyCommentBlock: {
				begin: {match: DELIMITER_OPEN + "\\*", literal: DELIMITER_OPEN + "*"},
				end: {match: "\\*" + DELIMITER_CLOSE, literal: "*" + DELIMITER_CLOSE},
				name: "comment.block.smarty",
			},
			smartyConfigVariable: {
				match: "#\\w+#",
				name: "variable.other.config.smarty",
			},
			smartyConstant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.smarty"
			},
			smartyEscapedVariable: {
				match: "`\\$[^`]+`",
				name: "variable.other.escaped.smarty",
			},
			smartyString_doubleQuote: {
				begin: '"',
				end: '"',
				name: "string.quoted.double.smarty",
				patterns: [
					{include: "#smartyEscapedVariable"},
					{include: "#smartyVariable"},
					{include: "#smartyConfigVariable"}
				]
			},
			smartyVariable: {
				match: "\\$(?:smarty\\.(?:config|server)\\.)?\\w+",
				name: "variable.other.smarty",
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/smarty/smartyPlugin',['orion/plugin', 'orion/editor/stylers/text_x-smarty/syntax'], function(PluginProvider, mSmarty) {

	function connect() {
		var headers = {
			name: "Orion Smarty Tool Support",
			version: "1.0",
			description: "This plugin provides Smarty tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-smarty",
					"extends": "text/plain",
					name: "Smarty",
					extension: ["tpl"]
				}
			] 
		});
		mSmarty.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/application_sql/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"ADD", "ALTER\\s+TABLE", "AND", "ASC",
		"BETWEEN",
		"COMMIT", "CREATE\\s+(TABLE|DATABASE)", "CREATE\\s+(UNIQUE\\s+)?INDEX",
		"DELETE\\s+FROM", "DESC", "DROP(\\s+(INDEX|TABLE|DATABASE))?",
		"FROM",
		"GROUP\\s+BY",
		"HAVING",
		"INSERT\\s+INTO", "IN",
		"LIKE",
		"MODIFY",
		"ON", "ORDER\\s+BY", "OR",
		"PRIMARY\\s+KEY",
		"ROLLBACK",
		"SELECT(\\s+(COUNT|DISTINCT|SUM))?", "SET",
		"TO", "TRUNCATE\\s+TABLE",
		"UPDATE", "USE",
		"VALUES",
		"WHERE"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.sql",
		contentTypes: ["application/sql"],
		patterns: [
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#number_decimal"},
			{
				match: "(?i)\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.sql"
			}, {
				match: "<>|>=?|<=?|=",
				name: "punctuation.operator.sql"
			}
		],
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});


/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/sql/sqlPlugin',['orion/plugin', 'orion/editor/stylers/application_sql/syntax'], function(PluginProvider, mSql) {

	function connect() {
		var headers = {
			name: "Orion SQL Tool Support",
			version: "1.0",
			description: "This plugin provides SQL tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/sql",
					"extends": "text/plain",
					name: "SQL",
					extension: ["sql"]
				}
			] 
		});
		mSql.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-swift/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"associativity", "as",
		"class", "convenience",
		"deinit", "didSet", "dynamicType", "dynamic",
		"enum", "extension",
		"final", "func",
		"get",
		"import", "infix", "init", "inout", "internal", "in", "is",
		"lazy", "left", "let",
		"mutating",
		"none", "nonmutating",
		"operator", "optional", "override",
		"postfix", "precedence", "prefix", "private", "protocol", "Protocol", "public",
		"required", "right",
		"Self", "set", "static", "struct", "subscript",
		"typealias", "Type",
		"unowned",
		"var",
		"weak", "willSet",
		"@objc"
	];
	var controlKeywords = [
		"break",
		"case", "continue",
		"default", "do",
		"else",
		"fallthrough", "for",
		"if",
		"return",
		"switch",
		"where", "while"
	];
	var constants = [
		"false", "nil", "true"
	];
	var languageVars1 = [
		"self", "super"
	];
	var languageVars2 = [
		"__COLUMN__", "__FILE__", "__FUNCTION__", "__LINE__"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.swift",
		contentTypes: ["text/x-swift"],
		patterns: [
			{include: "#string_doubleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "#number_binary"},
			{include: "#number_hex"},
			{include: "#number_octal"},
			{include: "#number_decimal"},
			{include: "#keywords_operator"},
			{include: "#keywords_control"},
			{include: "#constants"},
			{include: "#languageVars1"},
			{include: "#languageVars2"},
		],
		repository: {
			comment_block: {
				begin: {match: "/\\*", literal: "/*"},
				end: {match: "\\*/", literal: "*/"}, 
				name: "comment.block.swift",
				patterns: [
					{include: "#comment_block"},
					{
						match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task.swift"},
							4: {name: "comment.block.swift"}
						}
					}
				]
			},
			constants: {
				match: "(^|[^\\w`])(" + constants.join("|") + ")\\b",
				captures: {
					2: {name: "constant.language.swift"}
				}
			},
			keywords_operator: {
				match: "(^|[^\\w`])(" + keywords.join("|") + ")\\b",
				captures: {
					2: {name: "keyword.operator.swift"}
				}
			},
			keywords_control: {
				match: "(^|[^\\w`])(" + controlKeywords.join("|") + ")\\b",
				captures: {
					2: {name: "keyword.control.swift"}
				}
			},
			languageVars1: {
				match: "(^|[^\\w`])(" + languageVars1.join("|") + ")\\b",
				captures: {
					2: {name: "variable.language.swift"}
				}
			},
			languageVars2: {
				match: "(^|[^\\w`])(" + languageVars2.join("|") + ")(?:$|[^\\w])",
				captures: {
					2: {name: "variable.language.swift"}
				}
			},
			number_binary: {
				match: "\\b0b[01]+\\b",
				name: "constant.numeric.binary.swift"
			},
			number_decimal: {
				match: "\\b-?(?:\\.\\d[\\d_]*|\\d[\\d_]*\\.?[\\d_]*)(?:[eE][+-]?\\d[\\d_]*)?\\b",
				name: "constant.numeric.decimal.swift"
			},
			number_hex: {
				match: "\\b0[xX](?:\\.[0-9A-Fa-f][0-9A-Fa-f_]*|[0-9A-Fa-f][0-9A-Fa-f_]*\\.?[0-9A-Fa-f_]*)(?:[pP][+-]?\\d[\\d_]*)?\\b",
				name: "constant.numeric.hex.swift"
			},
			number_octal: {
				match: "\\b0o[01234567][01234567_]*\\b",
				name: "constant.numeric.octal.swift"
			},
			segment: {
				begin: "\\(",
				end: "\\)",
				patterns: [
					{include: "#segment"},
					{include: "#comment_block"},
					{include: "#number_binary"},
					{include: "#number_hex"},
					{include: "#number_octal"},
					{include: "#number_decimal"},
					{include: "#keywords_operator"},
					{include: "#keywords_control"},
					{include: "#constants"},
					{include: "#languageVars1"},
					{include: "#languageVars2"}
				]
			},
			string_doubleQuote: {
				match: '"(?:\\\\.|[^"])*"?',
				name: "string.quoted.double.swift",
				patterns: [
					{
						begin: "\\\\\\(",
						end: "\\)",
						name: "string.interpolated.swift",
						patterns: [
							{include: "#segment"},
							{include: "#comment_block"},
							{include: "#number_binary"},
							{include: "#number_hex"},
							{include: "#number_octal"},
							{include: "#number_decimal"},
							{include: "#keywords_operator"},
							{include: "#keywords_control"},
							{include: "#constants"},
							{include: "#languageVars1"},
							{include: "#languageVars2"}
						]
					}
				]
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(constants).concat(languageVars1).concat(languageVars2)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/swift/swiftPlugin',['orion/plugin', 'orion/editor/stylers/text_x-swift/syntax'], function(PluginProvider, mSwift) {

	function connect() {
		var headers = {
			name: "Orion Swift Tool Support",
			version: "1.0",
			description: "This plugin provides Swift tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-swift",
					"extends": "text/plain",
					name: "Swift",
					extension: ["swift"]
				}
			] 
		});
		mSwift.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-typescript/syntax", ["orion/editor/stylers/application_javascript/syntax"], function(mJS) {
	var keywords = ["constructor", "declare", "module"];
	var types = ["any", "boolean", "number", "string"];

	var grammars = [];
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push({
		id: "orion.typescript",
		contentTypes: ["text/x-typescript"],
		patterns: [
			{include: "orion.js"},
			{include: "#keywords"},
			{include: "#types"},
		],
		repository: {
			keywords: {
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.typescript"
			},
			types: {
				match: "\\b(?:" + types.join("|") + ")\\b",
				name: "storage.type.typescript"
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(types)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/typescript/typescriptPlugin',['orion/plugin', 'orion/editor/stylers/text_x-typescript/syntax'], function(PluginProvider, mTypescript) {

	function connect() {
		var headers = {
			name: "Orion Typescript Tool Support",
			version: "1.0",
			description: "This plugin provides Typescript tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-typescript",
					"extends": "text/plain",
					name: "Typescript",
					extension: ["typescript"]
				}
			] 
		});
		mTypescript.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-vb/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
	    "AddHandler", "AddressOf", "Aggregate", "Alias", "AndAlso",
	    "And", "Ansi", "Assembly", "Async", "As", "Auto", "Await",
	    "Binary", "Boolean", "ByRef", "Byte", "ByVal",
	    "Call", "Case", "Catch", "CBool", "CByte", "CChar", "CDate",
	    "CDbl", "CDec", "Char", "CInt", "Class", "CLng", "CObj", "Compare",
	    "Const", "CShort", "CSng", "CStr", "CType", "Custom",
	    "Date", "Decimal", "Declare", "Default", "Delegate", "Dim",
	    "DirectCast", "Distinct", "Double", "Do",
	    "Each", "ElseIf", "Else", "EndIf", "End", "Enum", "Equals",
	    "Erase", "Error", "Event", "Exit", "Explicit",
	    "False", "Finally", "For", "Friend", "From", "Function",
	    "GetType", "Get", "GoSub", "GoTo", "Group By", "Group Join",
	    "Handles",
	    "If", "Implements", "Imports", "Inherits", "Integer", "Interface",
	    "Into", "In", "IsFalse", "IsTrue", "Is", "Iterator",
	    "Join",
	    "Key",
	    "Let", "Lib", "Like", "Long", "Loop",
	    "Me", "Mid", "Module", "Mod", "MustInherit", "MustOverride",
	    "MyBase", "MyClass",
	    "Namespace", "New", "Next", "Nothing", "NotInheritable",
	    "NotOverridable", "Not",
	    "Object", "Off", "On", "Optional", "Option", "Order By", "OrElse",
	    "Or", "Overloads", "Overridable", "Overrides",
	    "ParamArray", "Preserve", "Private", "Property", "Protected", "Public",
	    "RaiseEvent", "ReadOnly", "ReDim", "REM", "RemoveHandler", "Resume", "Return",
	    "Select", "Set", "Shadows", "Shared", "Short", "Single", "Skip While", "Skip",
	    "Static", "Step", "Stop", "Strict", "String", "Structure", "Sub", "SyncLock",
	    "Take While", "Take", "Text", "Then", "Throw", "To", "True", "Try", "TypeOf",
	    "Unicode", "Until",
	    "Variant",
	    "Wend", "When", "Where", "While", "WithEvents", "With", "WriteOnly",
	    "Xor",
	    "Yield"
	];

	var preprocessorDirectives = ["Const", "ElseIf", "Else", "End", "ExternalSource", "If", "Region"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.vb",
		contentTypes: ["text/x-vb"],
		patterns: [
			{
				match: "^\\s*#(?:" + preprocessorDirectives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.vb"
			},
			{include: "orion.lib#string_doubleQuote"},
			{include: "#doc"},
			{include: "#comment"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.vb"
			}
		],
		repository: {
			comment: {
				begin: {match: "'", literal: "'"},
				end: {match: "$", literal: ""},
				name: "comment.line.vb",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			doc: {
				match: {match: "'''.*", literal: "'''"},
				name: "comment.line.documentation.vb",
				patterns: [
					{
						match: "<[^\\s>]*>",
						name: "meta.documentation.tag"
					}, {
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			number_hex: {
				match: "&[hH][0-9A-Fa-f]+\\b",
				name: "constant.numeric.hex.vb"
			},
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-vbhtml/syntax", [
	"orion/editor/stylers/application_xml/syntax",
	"orion/editor/stylers/text_html/syntax",
	"orion/editor/stylers/text_x-vb/syntax"], function(mXML, mHTML, mVB) {

	var grammars = [];
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push.apply(grammars, mHTML.grammars);
	grammars.push.apply(grammars, mVB.grammars);
	grammars.push({
		id: "orion.vbhtml",
		contentTypes: ["text/x-vbhtml"],
		patterns: [
			{include: "#vbhtmlComment"},
			{include: "#codeBlock"},
			{include: "#expression"},
			{include: "#reference"},
			{include: "orion.html"},
		],
		repository: {
			vbhtmlComment: {
				begin: {match: "@\\*", literal: "@*"},
				end: {match: "\\*@", literal: "*@"},
				name: "comment.block.vbhtml",
			},
			codeBlock: {
				begin: "(?i)^\\s*@code",
				end: "(?i)end code",
				captures: {
					0: {name: "entity.name.declaration.vb"}
				},
				contentName: "source.vb.embedded.vbhtml",
				patterns: [
				    {include: "orion.xml#tag"},
				    {include: "#reference"},
					{include: "orion.vb"},
				]
			},
			expression: {
				match: "(?i)^\\s*@(?!code)[^$]*",
				name: "source.vb.embedded.vbhtml",
				patterns: [
				    {include: "#reference"},
					{include: "orion.vb"},
				]
			},
			reference: {
				match: "@",
				name: "entity.name.declaration.vb"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/vb/vbPlugin',['orion/plugin', 'orion/editor/stylers/text_x-vb/syntax', 'orion/editor/stylers/text_x-vbhtml/syntax'], function(PluginProvider, mVB, mVBHtml) {

	function connect() {
		var headers = {
			name: "Orion VB.NET Tool Support",
			version: "1.0",
			description: "This plugin provides VB.NET tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-vb",
					"extends": "text/plain",
					name: "VB.NET",
					extension: ["vb"]
				}, {id: "text/x-vbhtml",
					"extends": "text/plain",
					name: "vbhtml",
					extension: ["vbhtml"]
				}
			]
		});
		mVB.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		mVBHtml.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/xml/xmlPlugin',['orion/plugin', 'orion/editor/stylers/application_xml/syntax'], function(PluginProvider, mXML) {

	function connect() {
		var headers = {
			name: "Orion XML Tool Support",
			version: "1.0",
			description: "This plugin provides XML tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/xml",
					"extends": "text/plain",
					name: "XML",
					extension: ["xml", "xib"],
					imageClass: "file-sprite-xml"
				}, {id: "application/xhtml+xml",
					"extends": "text/plain",
					name: "XHTML",
					extension: ["xhtml", "xht"],
					imageClass: "file-sprite-xml"
				}
			] 
		});
		mXML.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
 
/*eslint-env browser, amd*/
define("orion/editor/stylers/application_xquery/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"zero-digit",
		"xquery",
		"window", "where", "when",
		"version", "variable", "validate",
		"unordered", "union",
		"typeswitch", "type", "tumbling", "try",
		"treat", "to", "then", "text",
		"switch","strip", "strict", "start", "stable", "some",
		"sliding", "self", "schema-element", "schema-attribute", "schema", "satisfies",
		"return",
		"processing-instruction", "previous", "preserve", "preceding-sibling",
		"preceding", "percent", "per-mille", "pattern-separator", "parent",
		"ordering", "order", "or", "option", "only", "of",
		"node", "no-preserve", "no-inherit", "next",
		"ne", "NaN", "namespace-node", "namespace",
		"module", "mod", "minus-sign",
		"lt", "let", "least", "le", "lax",
		"item", "is", "intersect", "instance", "inherit",
		"infinity", "in", "import", "if", "idiv",
		"gt", "grouping-separator", "group", "greatest", "ge",
		"function", "for", "following-sibling", "following",
		"external", "except", "every", "eq", "end",
		"encoding", "empty-sequence", "empty", "else", "element",
		"document-node", "div", "digit", "descending", "descendant-or-self",
		"descendant", "default", "declare", "decimal-separator", "decimal-format",
		"count", "copy-namespaces", "context", "construction", "comment",
		"collation", "child", "catch", "castable", "cast", "case",
		"by", "boundary-space", "base-uri",
		"attribute", "at", "ascending", "as",
		"and", "ancestor-or-self", "ancestor", "allowing"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.xquery",
		contentTypes: ["application/xquery"],
		patterns: [
			{include: "#comment"},
			{include: "#variable"},
			{include: "#multiLineDoubleQuote"}, 
			{include: "#multiLineSingleQuote"}, 
			{include: "orion.xml#tag"}, 
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.js"
			}
		],
		repository: {
			comment: {
				begin: {match: "\\(:", literal: "(:"},
				end: {match: ":\\)", literal: ":)"}, 
				name: "comment.block.xquery",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!:\\)).)*)", // match: "(\\b)(TODO)(\\b)(((?!-->).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.block"}
						}
					}
				]
			},
			variable: {
				match: "\\$[a-zA-z0-9_]+",
				name: "variable.other.xquery" 
			},
			multiLineDoubleQuote: {
				begin: "\"",
				end: "\"",
				name: "string.quoted.double"
			},
			multiLineSingleQuote: {
				begin: "\'",
				end: "\'",
				name: "string.quoted.single"
			}
		}

	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/xquery/xqueryPlugin',['orion/plugin', 'orion/editor/stylers/application_xquery/syntax'], function(PluginProvider, mXQuery) {
	
	function connect() {
		var headers = {
			name: "Orion XQuery Tool Support",
			version: "1.0",
			description: "This plugin provides XQuery tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/xquery",
					"extends": "text/plain",
					name: "XQuery",
					extension: ["xq", "xqy", "xquery"]
				}
			] 
		});
		mXQuery.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-yaml/syntax", [], function() {
	var keywords = ["false", "null", "true"];
	var casts = ["!!bool", "!!float", "!!int", "!!map", "!!null", "!!omap", "!!seq", "!!str"];

	var id = "orion.yaml";
	var grammar = {
		id: id,
		contentTypes: ["text/x-yaml"],
		patterns: [
			{include: "#numberSignComment"},
			{
				match: "^%(?:YAML|TAG)\\s.*",
				name: "meta.directive.yaml"
			}, {
				begin: "^.*?:(?:[\\t ]|$)",
				end: "$",
				contentName: "string.unquoted.yaml",
				beginCaptures: {
					0: {
						name: "entity.name.key.yaml"
					}
				},
				patterns: [
					{include: "#numberSignComment"},
					{
						match: "^\\s*[&*]\\s*$",
						name: "entity.name.tag.yaml"
					}, {
						match: "(?i)^\\s*(?:" + keywords.join("|") + ")\\s*$",
						name: "keyword.operator.yaml"
					}, {
						match: "(?i)^\\s*(?:" + casts.join("|") + ")\\b",
						name: "keyword.operator.yaml"
					}, {
						match: "(?i)^\\s*(?:-?[0-9]*(?:\\.[0-9]+)?(?:e[-+][1-9][0-9]*)?)\\s*$",
						name: "constant.numeric.yaml"
					}, {
						match: "(?i)^\\s*(?:-?[1-9][0-9]*|0|-?\\.inf|\\.nan)\\s*$",
						name: "constant.numeric.yaml"
					}
				]
			}, {
				match: "---|\\.\\.\\.",
				name: "meta.separator.yaml"
			}
		],
		repository: {
			numberSignComment: {
				begin: {match: "(?:^|\\s)#", literal: "#"},
				end: {match: "$", literal: ""},
				name: "comment.line.number-sign.yaml",
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}
		}
	};

	return {
		id: id,
		grammars: [grammar],
		keywords: casts.concat(keywords)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/yaml/yamlPlugin',['orion/plugin', 'orion/editor/stylers/text_x-yaml/syntax'], function(PluginProvider, mYAML) {

	function connect() {
		var headers = {
			name: "Orion YAML Tool Support",
			version: "1.0",
			description: "This plugin provides YAML tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-yaml",
					"extends": "text/plain",
					name: "YAML",
					extension: ["yaml", "yml"]
				}
			] 
		});
		mYAML.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languageToolsPlugin',[
	'orion/plugin',
	'plugins/languages/arduino/arduinoPlugin',
	'plugins/languages/c/cPlugin',
	'plugins/languages/coffeescript/coffeescriptPlugin',
	'plugins/languages/cpp/cppPlugin',
	'plugins/languages/csharp/csharpPlugin',
	'plugins/languages/docker/dockerPlugin',
	'plugins/languages/erlang/erlangPlugin',
	'plugins/languages/go/goPlugin',
	'plugins/languages/git/gitFilePlugin',
	'plugins/languages/haml/hamlPlugin',
	'plugins/languages/jade/jadePlugin',
	'plugins/languages/java/javaPlugin',
	'plugins/languages/jsx/jsxPlugin',
	'plugins/languages/launch/launchPlugin',
	'plugins/languages/less/lessPlugin',
	'plugins/languages/lua/luaPlugin',
	'plugins/languages/markdown/markdownPlugin',
	'plugins/languages/objectiveC/objectiveCPlugin',
	'plugins/languages/php/phpPlugin',
	'plugins/languages/properties/propertiesPlugin',
	'plugins/languages/python/pythonPlugin',
	'plugins/languages/ruby/rubyPlugin',
	'plugins/languages/scss/scssPlugin',
	'plugins/languages/smarty/smartyPlugin',
	'plugins/languages/sql/sqlPlugin',
	'plugins/languages/swift/swiftPlugin',
	'plugins/languages/typescript/typescriptPlugin',
	'plugins/languages/vb/vbPlugin',
	'plugins/languages/xml/xmlPlugin',
	'plugins/languages/xquery/xqueryPlugin',
	'plugins/languages/yaml/yamlPlugin'
], function(PluginProvider) {

	var plugins = Array.prototype.slice.call(arguments);
	plugins.shift();

	function connect() {
		var headers = {
			name: "Orion Languages Tool Support",
			version: "1.0",
			description: "This plugin provides tooling for languages that are not included in other core Orion plugins."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(provider) {
		plugins.forEach(function(plugin) {
			plugin.registerServiceProviders(provider);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});


//# sourceMappingURL=languageToolsPlugin.js.map