/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 *   
 * $Id: OWLXMLSaver.java,v 1.3 2007/03/18 10:24:39 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource;

import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.rdfweb.NamespaceDefinition;
import org.eclipse.eodm.rdf.resource.parser.impl.RDFNamespaceMap;
import org.eclipse.eodm.vocabulary.OWL;
import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;
import org.eclipse.eodm.vocabulary.XSD;

/**
 * Save an EODM document into a file or output stream in RDF/XML syntax.
 * 
 */
public class OWLXMLSaver {

    /**
     * Save an EODM document into a file in RDF/XML syntax using the system
     * default encoding.
     * 
     * @param document
     *            The EODM-based RDF ontology to be saved.
     * @param fileName
     *            The name of the file to save to.
     * @throws IOException
     * @throws UnsupportedViewTypeException 
     */
    public static void saveToFile(Document document, String fileName)
            throws IOException, UnsupportedViewTypeException {
        saveToStream(document, new FileOutputStream(fileName));
    }

    /**
     * Save an EODM document to a file in RDF/XML syntax using a specified
     * encoding/charset.
     * 
     * @param document
     *            The EODM document
     * @param fileName
     *            The name of the file to save to
     * @param charsetName
     *            The name of the charset that will be used for saving.
     * @throws IOException
     * @throws UnsupportedViewTypeException 
     */
    public static void saveToFile(Document document, String fileName,
            String charsetName) throws IOException, UnsupportedViewTypeException {
        saveToStream(document, new FileOutputStream(fileName), charsetName);
    }

    /**
     * Save an EODM document into an output steam in RDF/XML syntax using the
     * system default encoding.
     * 
     * @param ontology
     *            The EODM document
     * @param os
     *            The output steam to save to
     * @throws IOException
     * @throws UnsupportedViewTypeException 
     */
    public static void saveToStream(Document document, OutputStream os)
            throws IOException, UnsupportedViewTypeException {
        OutputStreamWriter osWriter = new OutputStreamWriter(os);
        saveToStream(document, new BufferedWriter(osWriter), osWriter
                .getEncoding());
    }

    /**
     * Save an EODM document into an output steam in RDF/XML syntax using a
     * specified encoding/charset.
     * 
     * @param ontology
     *            The EODM document
     * @param os
     *            The output stream to write to
     * @param charsetName
     *            The charset that will be used to write to the stream
     * @throws IOException
     * @throws UnsupportedViewTypeException 
     */
    public static void saveToStream(Document document, OutputStream os,
            String charsetName) throws IOException, UnsupportedViewTypeException {
        saveToStream(document, new BufferedWriter(new OutputStreamWriter(os,
                charsetName)), charsetName);
    }

    /**
     * Save an EODM document into a output stream writer in RDF/XML syntax.
     * 
     * @param document
     *            The EODM document to save
     * @param writer
     *            The writer for saving
     * @param encoding
     *            The encoding used for saving, it should be the same as the
     *            encoding used by the writer.
     * @throws IOException
     * @throws UnsupportedViewTypeException 
     */
    protected static void saveToStream(Document document, Writer writer,
            String encoding) throws IOException, UnsupportedViewTypeException {

        writeRDFXMLHeader(document, writer, encoding);

        OWLXMLSaverImpl saverImpl = new OWLXMLSaverImpl(writer, document);
        saverImpl.save();

        writeRDFXMLEnding(writer);

        writer.flush();
    }

    /**
     * Write the ending of a RDF/XML document.
     * 
     * @param writer
     * @throws IOException
     */
    protected static void writeRDFXMLEnding(Writer writer) throws IOException {
        writer.write("</rdf:RDF>\n");
    }

    /**
     * Write the head part of a RDF/XML document, including the XML head and all
     * the namespace definitions.
     * 
     * @param namespaceList
     *            the list of namespaces to write
     * @param writer
     *            the writer to use
     * @param encoding
     *            the encoding to use
     * @throws IOException
     *             If I/O error occurs
     */
    protected static void writeRDFXMLHeader(Document document, Writer writer,
            String encoding) throws IOException {

        // Firstly, re-name the abbreviation of namespaces to aviod conflic
        EList list = document.getNamespaceDefinition();
        RDFNamespaceMap nsMap = new RDFNamespaceMap();
        for (int i = 0; i < list.size() - 1; i++) {
            for (int j = i + 1; j < list.size(); j++) {
                if (((NamespaceDefinition) (list.get(i))).getNamespacePrefix()
                        .equals(
                                ((NamespaceDefinition) (list.get(j)))
                                        .getNamespacePrefix())) {
                    ((NamespaceDefinition) (list.get(j)))
                            .setNamespacePrefix(nsMap.getKey());
                }
            }
        }

        boolean foundRDFNamespace = false;
        boolean foundRDFSNamespace = false;
        boolean foundOWLNamespace = false;
        boolean foundXSDNamespace = false;

        writer.write("<?xml version=\"1.0\" encoding=\"" + encoding + "\"?>\n");
        writer.write("<!DOCTYPE rdf:RDF [\n");

        for (Iterator iterator = document.getNamespaceDefinition().iterator(); iterator
                .hasNext();) {
            NamespaceDefinition ns = (NamespaceDefinition) iterator.next();
            String nsURI = ns.getNamespace().getNamespaceURIRef().getURIString() ;
            if (!nsURI.equals(
                    RDFS.BNODE_NAMESPACE)) {
                writer.write("\t<!ENTITY "
                             + ns.getNamespacePrefix() + " \""
                             + OWLXMLSaverImpl.replaceKeywords( nsURI )
                             + "\">\n");
                if (nsURI.equals(RDF.NAMESPACE))
                    foundRDFNamespace = true;
                if (nsURI.equals(RDFS.NAMESPACE))
                    foundRDFSNamespace = true;
                if (nsURI.equals(OWL.NAMESPACE_URI))
                    foundOWLNamespace = true;
                if (nsURI.equals(XSD.NAMESPACE))
                    foundXSDNamespace = true;
            }
        }

        if (!foundRDFNamespace) {
            writer.write("\t<!ENTITY rdf \"" + RDF.NAMESPACE + "\">\n");
        }

        if (!foundRDFSNamespace) 
            writer.write("\t<!ENTITY rdfs \"" + RDFS.NAMESPACE + "\">\n");

        if (!foundOWLNamespace) 
            writer.write("\t<!ENTITY owl \"" + OWL.NAMESPACE_URI + "\">\n");
        
        if (!foundXSDNamespace) 
            writer.write("\t<!ENTITY xsd \"" + XSD.NAMESPACE + "\">\n");

        writer.write("]>\n");

        writer.write("<rdf:RDF\n");

        for (Iterator iterator = document.getNamespaceDefinition().iterator(); iterator
                .hasNext();) {
            NamespaceDefinition ns = (NamespaceDefinition) iterator.next();
            if (!ns.getNamespace().getNamespaceURIRef().getURIString().equals(
                    RDFS.BNODE_NAMESPACE)) {
                writer.write("         xmlns:"
                             + ns.getNamespacePrefix() + "=\"&"
                             + ns.getNamespacePrefix() + ";\"\n");
            }
        }

        if (!foundRDFNamespace) {
            writer.write("         xmlns:rdf=\"&rdf;\"\n");
        }

        if (!foundRDFSNamespace) {
            writer.write("         xmlns:rdfs=\"&rdfs;\"\n");
        }
        if (!foundOWLNamespace) {
            writer.write("         xmlns:owl=\"&owl;\"\n");
        }
        if (!foundXSDNamespace) {
            writer.write("         xmlns:xsd=\"&xsd;\"\n");
        }

        writer.write(">\n");
    }

}