/*******************************************************************************
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Mark Czotter - initial API and implementation
 *******************************************************************************/
package org.eclipse.incquery.runtime.api;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.runtime.api.impl.BasePatternGroup;

/**
 * Generic implementation of {@link IPatternGroup}, covering an arbitrarily chosen set of patterns. Use the public
 * constructor or static GenericPatternGroup.of(...) methods to instantiate.
 * 
 * @author Mark Czotter
 * 
 */
public class GenericPatternGroup extends BasePatternGroup {

    private final Set<Pattern> patterns;

    /**
     * Creates a GenericPatternGroup object with a set of patterns.
     * 
     * @param patterns
     */
    public GenericPatternGroup(Set<Pattern> patterns) {
        this.patterns = patterns;
    }

    @Override
    public Set<Pattern> getPatterns() {
        return patterns;
    }

    /**
     * Creates a generic {@link IPatternGroup} instance from {@link IQuerySpecification} objects.
     * 
     * @param querySpecifications
     * @return
     */
    public static IPatternGroup of(Set<IQuerySpecification<?>> querySpecifications) {
        return new GenericPatternGroup(patterns(querySpecifications));
    }

    /**
     * Creates a generic {@link IPatternGroup} instance from {@link Pattern} objects.
     * 
     * @param querySpecifications
     * @return
     */
    public static IPatternGroup of(Pattern... patterns) {
        return new GenericPatternGroup(new HashSet<Pattern>(Arrays.asList(patterns)));
    }

    /**
     * Creates a generic {@link IPatternGroup} instance from {@link IQuerySpecification} objects.
     * 
     * @param querySpecifications
     * @return
     */
    public static IPatternGroup of(IQuerySpecification<?>... querySpecifications) {
        return of(new HashSet<IQuerySpecification<?>>(Arrays.asList(querySpecifications)));
    }

    /**
     * Creates a generic {@link IPatternGroup} instance from other {@link IPatternGroup} objects (subgroups).
     * 
     * @param querySpecifications
     * @return
     */
    public static IPatternGroup of(IPatternGroup... subGroups) {
        Set<Pattern> patterns = new HashSet<Pattern>();
        for (IPatternGroup group : subGroups) {
            patterns.addAll(group.getPatterns());
        }
        return new GenericPatternGroup(patterns);
    }
}
