/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.eclipse.aether.internal.impl.synccontext.named;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.metadata.DefaultMetadata;
import org.eclipse.aether.metadata.Metadata;
import org.eclipse.aether.named.NamedLockKey;
import org.eclipse.aether.spi.locking.LockingInhibitor;
import org.junit.jupiter.api.Test;

import static java.util.Collections.singletonList;
import static org.junit.jupiter.api.Assertions.assertEquals;

public class InhibitingNameMapperTest extends NameMapperTestSupport {
    NameMapper mapper = new InhibitingNameMapper(
            NameMappers.gavNameMapper(true),
            Arrays.asList(
                    new LockingInhibitor() {
                        @Override
                        public boolean preventArtifactLocking(Artifact artifact) {
                            return "no.lock".equals(artifact.getGroupId());
                        }
                    },
                    new LockingInhibitor() {
                        @Override
                        public boolean preventMetadataLocking(Metadata metadata) {
                            return "no.lock".equals(metadata.getGroupId());
                        }
                    }));

    @Test
    void singleArtifactWithLocking() {
        DefaultArtifact artifact = new DefaultArtifact("group:artifact:1.0");
        Collection<NamedLockKey> names = mapper.nameLocks(session, singletonList(artifact), null);
        assertEquals(1, names.size());
        assertEquals("artifact~group~artifact~1.0.lock", names.iterator().next().name());
    }

    @Test
    void singleArtifactWithoutLocking() {
        DefaultArtifact artifact = new DefaultArtifact("no.lock:artifact:1.0");
        Collection<NamedLockKey> names = mapper.nameLocks(session, singletonList(artifact), null);
        assertEquals(0, names.size());
    }

    @Test
    void combined() {
        Collection<NamedLockKey> names = mapper.nameLocks(
                session,
                Arrays.asList(new DefaultArtifact("group:artifact:1.0"), new DefaultArtifact("no.lock:artifact:1.0")),
                Arrays.asList(
                        new DefaultMetadata(
                                "group", "artifact", "maven-metadata.xml", Metadata.Nature.RELEASE_OR_SNAPSHOT),
                        new DefaultMetadata(
                                "no.lock", "artifact", "maven-metadata.xml", Metadata.Nature.RELEASE_OR_SNAPSHOT)));
        assertEquals(2, names.size());
        ArrayList<NamedLockKey> keys = new ArrayList<>(names);
        assertEquals("artifact~group~artifact~1.0.lock", keys.get(0).name());
        assertEquals("metadata~group~artifact.lock", keys.get(1).name());
    }
}
