/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.imap.processor;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import jakarta.inject.Inject;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.james.core.Username;
import org.apache.james.imap.api.display.HumanReadableText;
import org.apache.james.imap.api.message.Capability;
import org.apache.james.imap.api.message.request.ImapRequest;
import org.apache.james.imap.api.message.response.StatusResponseFactory;
import org.apache.james.imap.api.process.ImapProcessor;
import org.apache.james.imap.api.process.ImapSession;
import org.apache.james.imap.main.PathConverter;
import org.apache.james.imap.message.request.AuthenticateRequest;
import org.apache.james.imap.message.request.IRAuthenticateRequest;
import org.apache.james.imap.message.response.AuthenticateResponse;
import org.apache.james.imap.processor.AbstractAuthProcessor;
import org.apache.james.imap.processor.CapabilityImplementingProcessor;
import org.apache.james.jwt.OidcJwtTokenVerifier;
import org.apache.james.jwt.introspection.IntrospectionEndpoint;
import org.apache.james.mailbox.MailboxManager;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.protocols.api.OIDCSASLParser;
import org.apache.james.protocols.api.OidcSASLConfiguration;
import org.apache.james.util.MDCBuilder;
import org.apache.james.util.ReactorUtils;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class AuthenticateProcessor
extends AbstractAuthProcessor<AuthenticateRequest>
implements CapabilityImplementingProcessor {
    public static final String AUTH_PLAIN = "AUTH=PLAIN";
    public static final Capability AUTH_PLAIN_CAPABILITY = Capability.of("AUTH=PLAIN");
    private static final Logger LOGGER = LoggerFactory.getLogger(AuthenticateProcessor.class);
    private static final String AUTH_TYPE_PLAIN = "PLAIN";
    private static final String AUTH_TYPE_OAUTHBEARER = "OAUTHBEARER";
    private static final String AUTH_TYPE_XOAUTH2 = "XOAUTH2";
    private static final List<Capability> OAUTH_CAPABILITIES = ImmutableList.of((Object)Capability.of("AUTH=OAUTHBEARER"), (Object)Capability.of("AUTH=XOAUTH2"));
    public static final Capability SASL_CAPABILITY = Capability.of("SASL-IR");

    @Inject
    public AuthenticateProcessor(MailboxManager mailboxManager, StatusResponseFactory factory, MetricFactory metricFactory, PathConverter.Factory pathConverterFactory) {
        super(AuthenticateRequest.class, mailboxManager, factory, metricFactory, pathConverterFactory);
    }

    @Override
    public List<Class<? extends AuthenticateRequest>> acceptableClasses() {
        return ImmutableList.of(AuthenticateRequest.class, IRAuthenticateRequest.class);
    }

    @Override
    protected void processRequest(AuthenticateRequest request, ImapSession session, ImapProcessor.Responder responder) {
        String authType = request.getAuthType();
        if (authType.equalsIgnoreCase(AUTH_TYPE_PLAIN)) {
            if (session.isPlainAuthDisallowed()) {
                LOGGER.warn("Login attempt over clear channel rejected");
                this.no(request, responder, HumanReadableText.DISABLED_LOGIN);
            } else if (request instanceof IRAuthenticateRequest) {
                IRAuthenticateRequest irRequest = (IRAuthenticateRequest)request;
                this.doPlainAuth(irRequest.getInitialClientResponse(), session, request, responder);
            } else {
                session.executeSafely(() -> {
                    responder.respond(new AuthenticateResponse());
                    responder.flush();
                    session.pushLineHandler((requestSession, data) -> Mono.fromRunnable(() -> {
                        this.doPlainAuth(AuthenticateProcessor.extractInitialClientResponse(data), requestSession, request, responder);
                        requestSession.popLineHandler();
                        responder.flush();
                    }).subscribeOn(ReactorUtils.BLOCKING_CALL_WRAPPER).then());
                });
            }
        } else if (authType.equalsIgnoreCase(AUTH_TYPE_OAUTHBEARER) || authType.equalsIgnoreCase(AUTH_TYPE_XOAUTH2)) {
            if (request instanceof IRAuthenticateRequest) {
                IRAuthenticateRequest irRequest = (IRAuthenticateRequest)request;
                this.doOAuth(irRequest.getInitialClientResponse(), session, request, responder);
            } else {
                session.executeSafely(() -> {
                    responder.respond(new AuthenticateResponse());
                    responder.flush();
                    session.pushLineHandler((requestSession, data) -> Mono.fromRunnable(() -> {
                        this.doOAuth(AuthenticateProcessor.extractInitialClientResponse(data), requestSession, request, responder);
                        requestSession.popLineHandler();
                        responder.flush();
                    }).subscribeOn(ReactorUtils.BLOCKING_CALL_WRAPPER).then());
                });
            }
        } else {
            LOGGER.debug("Unsupported authentication mechanism '{}'", (Object)authType);
            this.no(request, responder, HumanReadableText.UNSUPPORTED_AUTHENTICATION_MECHANISM);
        }
    }

    protected void doPlainAuth(String initialClientResponse, ImapSession session, ImapRequest request, ImapProcessor.Responder responder) {
        AbstractAuthProcessor.AuthenticationAttempt authenticationAttempt = this.parseDelegationAttempt(initialClientResponse);
        if (authenticationAttempt.isDelegation()) {
            this.doAuthWithDelegation(authenticationAttempt, session, request, responder);
        } else {
            this.doAuth(authenticationAttempt, session, request, responder, HumanReadableText.AUTHENTICATION_FAILED);
        }
        session.stopDetectingCommandInjection();
    }

    private AbstractAuthProcessor.AuthenticationAttempt parseDelegationAttempt(String initialClientResponse) {
        try {
            String userpass = new String(Base64.getDecoder().decode(initialClientResponse));
            List tokens = Arrays.stream(userpass.split("\u0000")).filter(token -> !token.isBlank()).collect(Collectors.toList());
            Preconditions.checkArgument((tokens.size() == 2 || tokens.size() == 3 ? 1 : 0) != 0);
            if (tokens.size() == 2) {
                return AuthenticateProcessor.noDelegation(Username.of((String)((String)tokens.get(0))), (String)tokens.get(1));
            }
            return AuthenticateProcessor.delegation(Username.of((String)((String)tokens.get(0))), Username.of((String)((String)tokens.get(1))), (String)tokens.get(2));
        }
        catch (Exception e) {
            LOGGER.info("Invalid syntax in AUTHENTICATE initial client response", (Throwable)e);
            return AuthenticateProcessor.noDelegation(null, null);
        }
    }

    @Override
    public List<Capability> getImplementedCapabilities(ImapSession session) {
        ArrayList<Capability> caps = new ArrayList<Capability>();
        if (!session.isPlainAuthDisallowed()) {
            caps.add(AUTH_PLAIN_CAPABILITY);
        }
        caps.add(SASL_CAPABILITY);
        if (session.supportsOAuth()) {
            caps.addAll(OAUTH_CAPABILITIES);
        }
        return ImmutableList.copyOf(caps);
    }

    @Override
    protected MDCBuilder mdc(AuthenticateRequest request) {
        return MDCBuilder.create().addToContext("action", "AUTHENTICATE").addToContext("authType", request.getAuthType());
    }

    protected void doOAuth(String initialResponse, ImapSession session, ImapRequest request, ImapProcessor.Responder responder) {
        if (!session.supportsOAuth()) {
            this.no(request, responder, HumanReadableText.UNSUPPORTED_AUTHENTICATION_MECHANISM);
        } else {
            OIDCSASLParser.parse((String)initialResponse).flatMap(oidcInitialResponseValue -> session.oidcSaslConfiguration().map(configure -> Pair.of((Object)oidcInitialResponseValue, (Object)configure))).ifPresentOrElse(pair -> this.doOAuth((OIDCSASLParser.OIDCInitialResponse)pair.getLeft(), (OidcSASLConfiguration)pair.getRight(), session, request, responder), () -> this.manageFailureCount(session, request, responder));
        }
        session.stopDetectingCommandInjection();
    }

    private void doOAuth(OIDCSASLParser.OIDCInitialResponse oidcInitialResponse, OidcSASLConfiguration oidcSASLConfiguration, ImapSession session, ImapRequest request, ImapProcessor.Responder responder) {
        this.validateToken(oidcSASLConfiguration, oidcInitialResponse.getToken()).ifPresentOrElse(authenticatedUser -> {
            Username associatedUser = Username.of((String)oidcInitialResponse.getAssociatedUser());
            if (!associatedUser.equals(authenticatedUser)) {
                this.doAuthWithDelegation(() -> this.getMailboxManager().authenticate(authenticatedUser).as(associatedUser), session, request, responder, (Username)authenticatedUser, associatedUser);
            } else {
                this.authSuccess((Username)authenticatedUser, session, request, responder);
            }
        }, () -> this.manageFailureCount(session, request, responder));
    }

    private Optional<Username> validateToken(OidcSASLConfiguration oidcSASLConfiguration, String token) {
        if (oidcSASLConfiguration.isCheckTokenByIntrospectionEndpoint()) {
            return AuthenticateProcessor.validTokenWithIntrospection(oidcSASLConfiguration, token);
        }
        if (oidcSASLConfiguration.isCheckTokenByUserinfoEndpoint()) {
            return AuthenticateProcessor.validTokenWithUserInfo(oidcSASLConfiguration, token);
        }
        return OidcJwtTokenVerifier.verifySignatureAndExtractClaim((String)token, (URL)oidcSASLConfiguration.getJwksURL(), (String)oidcSASLConfiguration.getClaim()).map(Username::of);
    }

    private static Optional<Username> validTokenWithUserInfo(OidcSASLConfiguration oidcSASLConfiguration, String token) {
        return Mono.from((Publisher)OidcJwtTokenVerifier.verifyWithUserinfo((String)token, (URL)oidcSASLConfiguration.getJwksURL(), (String)oidcSASLConfiguration.getClaim(), (URL)((URL)oidcSASLConfiguration.getUserInfoEndpoint().orElseThrow()))).blockOptional().map(Username::of);
    }

    private static Optional<Username> validTokenWithIntrospection(OidcSASLConfiguration oidcSASLConfiguration, String token) {
        return Mono.from((Publisher)OidcJwtTokenVerifier.verifyWithIntrospection((String)token, (URL)oidcSASLConfiguration.getJwksURL(), (String)oidcSASLConfiguration.getClaim(), (IntrospectionEndpoint)oidcSASLConfiguration.getIntrospectionEndpoint().map(endpoint -> new IntrospectionEndpoint(endpoint, oidcSASLConfiguration.getIntrospectionEndpointAuthorization())).orElseThrow())).blockOptional().map(Username::of);
    }

    private static String extractInitialClientResponse(byte[] data) {
        return new String(data, 0, data.length - 2, StandardCharsets.US_ASCII);
    }
}

