/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.quotas;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClusterMetrics;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.RegionStatesCount;
import org.apache.hadoop.hbase.quotas.NoopQuotaLimiter;
import org.apache.hadoop.hbase.quotas.QuotaLimiter;
import org.apache.hadoop.hbase.quotas.QuotaState;
import org.apache.hadoop.hbase.quotas.QuotaUtil;
import org.apache.hadoop.hbase.quotas.UserQuotaState;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.regionserver.RegionServerServices;
import org.apache.hadoop.hbase.util.ConcurrentMapUtils;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class QuotaCache
implements Stoppable {
    private static final Logger LOG = LoggerFactory.getLogger(QuotaCache.class);
    public static final String REFRESH_CONF_KEY = "hbase.quota.refresh.period";
    private static final int REFRESH_DEFAULT_PERIOD = 300000;
    private static final int EVICT_PERIOD_FACTOR = 5;
    static boolean TEST_FORCE_REFRESH = false;
    private final ConcurrentMap<String, QuotaState> namespaceQuotaCache = new ConcurrentHashMap<String, QuotaState>();
    private final ConcurrentMap<TableName, QuotaState> tableQuotaCache = new ConcurrentHashMap<TableName, QuotaState>();
    private final ConcurrentMap<String, UserQuotaState> userQuotaCache = new ConcurrentHashMap<String, UserQuotaState>();
    private final ConcurrentMap<String, QuotaState> regionServerQuotaCache = new ConcurrentHashMap<String, QuotaState>();
    private volatile boolean exceedThrottleQuotaEnabled = false;
    private volatile double machineQuotaFactor = 1.0;
    private final ConcurrentHashMap<TableName, Double> tableMachineQuotaFactors = new ConcurrentHashMap();
    private final RegionServerServices rsServices;
    private QuotaRefresherChore refreshChore;
    private boolean stopped = true;

    public QuotaCache(RegionServerServices rsServices) {
        this.rsServices = rsServices;
    }

    public void start() throws IOException {
        this.stopped = false;
        Configuration conf = this.rsServices.getConfiguration();
        int period = conf.getInt(REFRESH_CONF_KEY, 300000);
        this.refreshChore = new QuotaRefresherChore(period, this);
        this.rsServices.getChoreService().scheduleChore(this.refreshChore);
    }

    @Override
    public void stop(String why) {
        if (this.refreshChore != null) {
            LOG.debug("Stopping QuotaRefresherChore chore.");
            this.refreshChore.shutdown(true);
        }
        this.stopped = true;
    }

    @Override
    public boolean isStopped() {
        return this.stopped;
    }

    public QuotaLimiter getUserLimiter(UserGroupInformation ugi, TableName table) {
        if (table.isSystemTable()) {
            return NoopQuotaLimiter.get();
        }
        return this.getUserQuotaState(ugi).getTableLimiter(table);
    }

    public UserQuotaState getUserQuotaState(UserGroupInformation ugi) {
        return ConcurrentMapUtils.computeIfAbsent(this.userQuotaCache, ugi.getShortUserName(), UserQuotaState::new, this::triggerCacheRefresh);
    }

    public QuotaLimiter getTableLimiter(TableName table) {
        return this.getQuotaState(this.tableQuotaCache, table).getGlobalLimiter();
    }

    public QuotaLimiter getNamespaceLimiter(String namespace) {
        return this.getQuotaState(this.namespaceQuotaCache, namespace).getGlobalLimiter();
    }

    public QuotaLimiter getRegionServerQuotaLimiter(String regionServer) {
        return this.getQuotaState(this.regionServerQuotaCache, regionServer).getGlobalLimiter();
    }

    protected boolean isExceedThrottleQuotaEnabled() {
        return this.exceedThrottleQuotaEnabled;
    }

    private <K> QuotaState getQuotaState(ConcurrentMap<K, QuotaState> quotasMap, K key) {
        return ConcurrentMapUtils.computeIfAbsent(quotasMap, key, QuotaState::new, this::triggerCacheRefresh);
    }

    void triggerCacheRefresh() {
        this.refreshChore.triggerNow();
    }

    long getLastUpdate() {
        return this.refreshChore.lastUpdate;
    }

    Map<String, QuotaState> getNamespaceQuotaCache() {
        return this.namespaceQuotaCache;
    }

    Map<String, QuotaState> getRegionServerQuotaCache() {
        return this.regionServerQuotaCache;
    }

    Map<TableName, QuotaState> getTableQuotaCache() {
        return this.tableQuotaCache;
    }

    Map<String, UserQuotaState> getUserQuotaCache() {
        return this.userQuotaCache;
    }

    static interface Fetcher<Key, Value> {
        public Get makeGet(Map.Entry<Key, Value> var1);

        public Map<Key, Value> fetchEntries(List<Get> var1) throws IOException;
    }

    private class QuotaRefresherChore
    extends ScheduledChore {
        private long lastUpdate;

        public QuotaRefresherChore(int period, Stoppable stoppable) {
            super("QuotaRefresherChore", stoppable, period);
            this.lastUpdate = 0L;
        }

        @Override
        @SuppressWarnings(value={"GC_UNRELATED_TYPES"}, justification="I do not understand why the complaints, it looks good to me -- FIX")
        protected void chore() {
            for (TableName table : ((HRegionServer)QuotaCache.this.rsServices).getOnlineTables()) {
                if (table.isSystemTable()) continue;
                QuotaCache.this.tableQuotaCache.computeIfAbsent(table, key -> new QuotaState());
                String ns = table.getNamespaceAsString();
                QuotaCache.this.namespaceQuotaCache.computeIfAbsent(ns, key -> new QuotaState());
            }
            QuotaCache.this.regionServerQuotaCache.computeIfAbsent("all", key -> new QuotaState());
            this.updateQuotaFactors();
            this.fetchNamespaceQuotaState();
            this.fetchTableQuotaState();
            this.fetchUserQuotaState();
            this.fetchRegionServerQuotaState();
            this.fetchExceedThrottleQuota();
            this.lastUpdate = EnvironmentEdgeManager.currentTime();
        }

        private void fetchNamespaceQuotaState() {
            this.fetch("namespace", QuotaCache.this.namespaceQuotaCache, new Fetcher<String, QuotaState>(){

                @Override
                public Get makeGet(Map.Entry<String, QuotaState> entry) {
                    return QuotaUtil.makeGetForNamespaceQuotas(entry.getKey());
                }

                @Override
                public Map<String, QuotaState> fetchEntries(List<Get> gets) throws IOException {
                    return QuotaUtil.fetchNamespaceQuotas(QuotaCache.this.rsServices.getConnection(), gets, QuotaCache.this.machineQuotaFactor);
                }
            });
        }

        private void fetchTableQuotaState() {
            this.fetch("table", QuotaCache.this.tableQuotaCache, new Fetcher<TableName, QuotaState>(){

                @Override
                public Get makeGet(Map.Entry<TableName, QuotaState> entry) {
                    return QuotaUtil.makeGetForTableQuotas(entry.getKey());
                }

                @Override
                public Map<TableName, QuotaState> fetchEntries(List<Get> gets) throws IOException {
                    return QuotaUtil.fetchTableQuotas(QuotaCache.this.rsServices.getConnection(), gets, QuotaCache.this.tableMachineQuotaFactors);
                }
            });
        }

        private void fetchUserQuotaState() {
            final Set namespaces = QuotaCache.this.namespaceQuotaCache.keySet();
            final Set tables = QuotaCache.this.tableQuotaCache.keySet();
            this.fetch("user", QuotaCache.this.userQuotaCache, new Fetcher<String, UserQuotaState>(){

                @Override
                public Get makeGet(Map.Entry<String, UserQuotaState> entry) {
                    return QuotaUtil.makeGetForUserQuotas(entry.getKey(), tables, namespaces);
                }

                @Override
                public Map<String, UserQuotaState> fetchEntries(List<Get> gets) throws IOException {
                    return QuotaUtil.fetchUserQuotas(QuotaCache.this.rsServices.getConnection(), gets, QuotaCache.this.tableMachineQuotaFactors, QuotaCache.this.machineQuotaFactor);
                }
            });
        }

        private void fetchRegionServerQuotaState() {
            this.fetch("regionServer", QuotaCache.this.regionServerQuotaCache, new Fetcher<String, QuotaState>(){

                @Override
                public Get makeGet(Map.Entry<String, QuotaState> entry) {
                    return QuotaUtil.makeGetForRegionServerQuotas(entry.getKey());
                }

                @Override
                public Map<String, QuotaState> fetchEntries(List<Get> gets) throws IOException {
                    return QuotaUtil.fetchRegionServerQuotas(QuotaCache.this.rsServices.getConnection(), gets);
                }
            });
        }

        private void fetchExceedThrottleQuota() {
            try {
                QuotaCache.this.exceedThrottleQuotaEnabled = QuotaUtil.isExceedThrottleQuotaEnabled(QuotaCache.this.rsServices.getConnection());
            }
            catch (IOException e) {
                LOG.warn("Unable to read if exceed throttle quota enabled from quota table", (Throwable)e);
            }
        }

        private <K, V extends QuotaState> void fetch(String type, ConcurrentMap<K, V> quotasMap, Fetcher<K, V> fetcher) {
            long now = EnvironmentEdgeManager.currentTime();
            long refreshPeriod = this.getPeriod();
            long evictPeriod = refreshPeriod * 5L;
            ArrayList<Get> gets = new ArrayList<Get>();
            ArrayList toRemove = new ArrayList();
            for (Map.Entry entry : quotasMap.entrySet()) {
                long lastUpdate = ((QuotaState)entry.getValue()).getLastUpdate();
                long lastQuery = ((QuotaState)entry.getValue()).getLastQuery();
                if (lastQuery > 0L && now - lastQuery >= evictPeriod) {
                    toRemove.add(entry.getKey());
                    continue;
                }
                if (!TEST_FORCE_REFRESH && now - lastUpdate < refreshPeriod) continue;
                gets.add(fetcher.makeGet(entry));
            }
            for (Map.Entry key : toRemove) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("evict " + type + " key=" + key);
                }
                quotasMap.remove(key);
            }
            if (!gets.isEmpty()) {
                try {
                    for (Map.Entry entry : fetcher.fetchEntries(gets).entrySet()) {
                        QuotaState quotaInfo = (QuotaState)quotasMap.putIfAbsent(entry.getKey(), entry.getValue());
                        if (quotaInfo != null) {
                            quotaInfo.update((QuotaState)entry.getValue());
                        }
                        if (!LOG.isTraceEnabled()) continue;
                        LOG.trace("refresh " + type + " key=" + entry.getKey() + " quotas=" + quotaInfo);
                    }
                }
                catch (IOException e) {
                    LOG.warn("Unable to read " + type + " from quota table", (Throwable)e);
                }
            }
        }

        private void updateQuotaFactors() {
            ClusterMetrics clusterMetrics;
            try {
                clusterMetrics = QuotaCache.this.rsServices.getConnection().getAdmin().getClusterMetrics(EnumSet.of(ClusterMetrics.Option.SERVERS_NAME, ClusterMetrics.Option.TABLE_TO_REGIONS_COUNT));
            }
            catch (IOException e) {
                LOG.warn("Failed to get cluster metrics needed for updating quotas", (Throwable)e);
                return;
            }
            int rsSize = clusterMetrics.getServersName().size();
            if (rsSize != 0) {
                QuotaCache.this.machineQuotaFactor = 1.0 / (double)rsSize;
            }
            Map<TableName, RegionStatesCount> tableRegionStatesCount = clusterMetrics.getTableRegionStatesCount();
            for (TableName tableName : QuotaCache.this.tableQuotaCache.keySet()) {
                if (tableRegionStatesCount.containsKey(tableName)) {
                    double factor = 1.0;
                    try {
                        long regionSize = tableRegionStatesCount.get(tableName).getOpenRegions();
                        if (regionSize == 0L) {
                            factor = 0.0;
                        } else {
                            int localRegionSize = QuotaCache.this.rsServices.getRegions(tableName).size();
                            factor = 1.0 * (double)localRegionSize / (double)regionSize;
                        }
                    }
                    catch (IOException e) {
                        LOG.warn("Get table regions failed: {}", (Object)tableName, (Object)e);
                    }
                    QuotaCache.this.tableMachineQuotaFactors.put(tableName, factor);
                    continue;
                }
                QuotaCache.this.tableMachineQuotaFactors.remove(tableName);
            }
        }
    }
}

