// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.AlwaysNotNullable;
import org.apache.doris.nereids.trees.expressions.functions.ExplicitlyCastableSignature;
import org.apache.doris.nereids.trees.expressions.functions.ExpressionTrait;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.ArrayType;
import org.apache.doris.nereids.types.DataType;
import org.apache.doris.nereids.types.coercion.FollowToArgumentType;
import org.apache.doris.nereids.util.TypeCoercionUtils;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * ScalarFunction 'array'. This class is generated by GenerateFunction.
 */
public class Array extends ScalarFunction
        implements ExplicitlyCastableSignature, AlwaysNotNullable {

    public static final List<FunctionSignature> SIGNATURES = ImmutableList.of(
            FunctionSignature.ret(ArrayType.SYSTEM_DEFAULT).args()
    );

    /**
     * constructor with 0 or more arguments.
     */
    public Array(Expression... varArgs) {
        super("array", varArgs);
    }

    /**
     * withChildren.
     */
    @Override
    public Array withChildren(List<Expression> children) {
        return new Array(children.toArray(new Expression[0]));
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitArray(this, context);
    }

    @Override
    public List<FunctionSignature> getSignatures() {
        if (arity() == 0) {
            return SIGNATURES;
        }
        Optional<DataType> commonDataType = TypeCoercionUtils.findWiderCommonTypeForCaseWhen(
                children.stream().map(ExpressionTrait::getDataType).collect(Collectors.toList()));
        if (commonDataType.isPresent()) {
            return ImmutableList.of(
                    FunctionSignature.ret(ArrayType.of(commonDataType.get())).varArgs(commonDataType.get()));
        }
        Map<Boolean, List<DataType>> partitioned = children.stream()
                .map(ExpressionTrait::getDataType)
                .collect(Collectors.partitioningBy(TypeCoercionUtils::hasCharacterType));
        List<DataType> needTypeCoercion = Lists.newArrayList(Sets.newHashSet(partitioned.get(true)));
        if (needTypeCoercion.size() > 1 || !partitioned.get(false).isEmpty()) {
            needTypeCoercion = needTypeCoercion.stream()
                    .map(TypeCoercionUtils::replaceCharacterToString)
                    .collect(Collectors.toList());
        }
        partitioned = partitioned.get(false).stream()
                .collect(Collectors.partitioningBy(TypeCoercionUtils::hasDecimalV2Type));
        if (!partitioned.get(true).isEmpty()) {
            needTypeCoercion.addAll(partitioned.get(true).stream()
                    .map(TypeCoercionUtils::replaceDecimalV2WithDefault).collect(Collectors.toList()));
        }
        partitioned = partitioned.get(false).stream()
                .collect(Collectors.partitioningBy(TypeCoercionUtils::hasDecimalV3Type));
        if (!partitioned.get(true).isEmpty()) {
            needTypeCoercion.addAll(partitioned.get(true).stream()
                    .map(TypeCoercionUtils::replaceDecimalV3WithWildcard).collect(Collectors.toList()));
        }
        partitioned = partitioned.get(false).stream()
                .collect(Collectors.partitioningBy(TypeCoercionUtils::hasDateTimeV2Type));
        if (!partitioned.get(true).isEmpty()) {
            needTypeCoercion.addAll(partitioned.get(true).stream()
                    .map(TypeCoercionUtils::replaceDateTimeV2WithMax).collect(Collectors.toList()));
        }
        needTypeCoercion.addAll(partitioned.get(false));
        return needTypeCoercion.stream()
                .map(dataType -> FunctionSignature.ret(ArrayType.of(new FollowToArgumentType(0))).varArgs(dataType))
                .collect(ImmutableList.toImmutableList());
    }
}
