/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.google.sheets;

import java.security.SecureRandom;
import java.util.Collections;
import java.util.Map;
import java.util.UUID;

import com.google.api.services.sheets.v4.model.Sheet;
import com.google.api.services.sheets.v4.model.SheetProperties;
import com.google.api.services.sheets.v4.model.Spreadsheet;
import com.google.api.services.sheets.v4.model.SpreadsheetProperties;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelExecutionException;
import org.apache.camel.Component;
import org.apache.camel.test.junit5.CamelTestSupport;
import org.junit.jupiter.api.TestInstance;

/**
 * Abstract base class for GoogleSheets Integration tests generated by Camel API component maven plugin.
 */
@TestInstance(TestInstance.Lifecycle.PER_METHOD)
public abstract class AbstractGoogleSheetsTestSupport extends CamelTestSupport {

    protected static final String TEST_SHEET = "TestData";
    protected static final String TEST_RANGE = TEST_SHEET + "!A1:B2";

    private Spreadsheet spreadsheet;

    /**
     * Create test spreadsheet that is used throughout all tests.
     */
    private void createTestSpreadsheet() {
        Spreadsheet spreadsheet = new Spreadsheet();
        spreadsheet.setFactory(MockGoogleSheetsClientFactory.JSON_FACTORY);
        spreadsheet.setSpreadsheetId(UUID.randomUUID().toString());

        SpreadsheetProperties spreadsheetProperties = new SpreadsheetProperties();
        spreadsheetProperties.setTitle("camel-sheets-" + new SecureRandom().nextInt(Integer.MAX_VALUE));
        spreadsheet.setProperties(spreadsheetProperties);

        Sheet sheet = new Sheet();
        SheetProperties sheetProperties = new SheetProperties();
        sheetProperties.setTitle(TEST_SHEET);
        sheet.setProperties(sheetProperties);

        spreadsheet.setSheets(Collections.singletonList(sheet));

        this.spreadsheet = spreadsheet;
    }

    @Override
    protected CamelContext createCamelContext() throws Exception {

        final CamelContext context = super.createCamelContext();

        Component component = getComponent(context);
        context.addComponent(getComponentName(), component);

        return context;
    }

    protected Component getComponent(CamelContext context) throws Exception {
        GoogleSheetsConfiguration configuration = new GoogleSheetsConfiguration();
        configuration.setServiceAccountKey("mock");

        GoogleSheetsComponent component = new GoogleSheetsComponent(context);
        component.setConfiguration(configuration);
        component.setClientFactory(getClientFactory());
        return component;
    }

    protected String getComponentName() {
        return "google-sheets";
    }

    protected abstract GoogleSheetsClientFactory getClientFactory() throws Exception;

    @SuppressWarnings("unchecked")
    protected <T> T requestBodyAndHeaders(String endpointUri, Object body, Map<String, Object> headers)
            throws CamelExecutionException {
        return (T) template().requestBodyAndHeaders(endpointUri, body, headers);
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBody(String endpoint, Object body) throws CamelExecutionException {
        return (T) template().requestBody(endpoint, body);
    }

    public Spreadsheet getSpreadsheet() {
        if (spreadsheet == null) {
            createTestSpreadsheet();
        }
        return spreadsheet;
    }

}
