unit Conv;

{ Standard definitions for file-to-file conversions.
                JCC  2 Jan 90

   2 Jan 90: SwOrigin[] in ParsedCmdLine
   6 Aug 89: Allow '=' as well as ':' for separator of switch and tail.

  For TURBO PASCAL version >= 5.0.
  Some of the routines here are machine dependent, and are intended to
  hide machine dependence -- notably ParseCmd and SetFiles.
  }

interface

uses Utils, DOS;

const
     MAXARG = 15;
     SwitchChars: Set of Char = ['/', '-'];
                                  { Allow both UNIX and MS-DOS switches. }
type
    ArgList    = array[1..MAXARG] of FileName;
    ParsedCmdLine = record
           NumFiles, NumSw : integer;
           Files, Switch, SwOrigin, Tail, TailNotUC: ArgList;
        end;

procedure ParseSwitch (var arg, switch, tail: filename);
procedure ParseCmd (var CmdLine: ParsedCmdLine);
procedure SetFiles(SignOn, SourcePrompt, destPrompt: ScreenLine;
  var CmdLine: ParsedCmdLine;
  var SourceName, DestName : FileName;
  var error: boolean);

implementation
{$V-}

procedure ParseSwitch (var arg, switch, tail: filename);
{ Assume arg is of form /<switch>:<tail> or /<switch>=<tail> or /<switch>.
  ('/' is not checked.
  Convert switch, but not tail to upper case.
}
var icol: integer;
BEGIN
   icol := pos(':', arg);
   IF (icol = 0) THEN
      icol := pos('=', arg);
   IF (icol = 0) THEN BEGIN
      switch := copy (arg, 2, length(arg) -1);
      tail := '';
   END
   ELSE BEGIN
      switch := copy (arg, 2, icol - 2);
      tail := copy (arg, icol + 1, length(arg) -icol);
   END;
   UC (switch);
END;


procedure ParseCmd (var CmdLine: ParsedCmdLine);
{ Separate command line arguments into file names -- not beginning
  with '/' -- and switches -- they have the form /<switch>:<tail>
  or /<switch>.
  The '/' may be any of the characters in SwitchChars.
  All parts are converted to upper case.  But a copy of the tails without
  the uppercase conversion is left in CmdLine.TailNoUC.
}
var i: integer;
    ThisArg: filename;
BEGIN
with CmdLine do
   BEGIN { with }
      NumFiles := 0;
      NumSw := 0;
      FOR i := 1 to ParamCount do
         BEGIN { i'th parameter }
            ThisArg := ParamStr(i);
            IF (ThisArg[1] in SwitchChars) and (NumSw < MAXARG) THEN
               BEGIN { Switch }
                  NumSw := NumSw + 1;
                  SwOrigin[NumSw] := ThisArg;
                  ParseSwitch (ThisArg, Switch[NumSw], TailNotUC[NumSw]);
                  Tail[NumSw] := TailNotUC[NumSw];
                  UC (Tail[NumSw]);
               END { Switch }
            ELSE IF (not (ThisArg[1] in SwitchChars))
                    and (NumFiles < MAXARG) THEN
               BEGIN { File }
                  NumFiles := NumFiles + 1;
                  Files[NumFiles] := ThisArg;
                  UC (Files[NumFiles]);
               END { File }
            ELSE
               writeln ('I have run out of space to parse the command line');
         END { i'th parameter };
   END {with};
END {ParseCmd};

procedure SetFiles(SignOn, SourcePrompt, destPrompt: ScreenLine;
  var CmdLine: ParsedCmdLine;
  var SourceName, DestName : FileName;
  var error: boolean);
{ Given command line data, get filename info from command line or console. }
{ Test for existence of the files but do not leave open.                   }
{ If destination file is same as source, then error.                       }
{ Use switches /Q /NOQ -Q -NOQ to control query or no query if destination }
{    already exists.  (Default is /Q).                                     }
{ But if user is prompted for destination, then always query if destination}
{ exists.  Never query if destination is one of the MSDOS standard devices:}
{ CON, PRN, AUX or NUL.                                                    }

var done, query : boolean;
    i : integer;
function StdDev: boolean;
begin
   StdDev := (DestName = 'CON') or (DestName = 'PRN') or (DestName = 'AUX')
             or (DestName = 'NUL');
end;

begin
 writeln(SignOn);
 SourceName := '';  {For safety, blank out file names.}
 DestName := '';
 error := false;
 query := false;

 with CmdLine do for i := 1 to NumSw do
    BEGIN {Obey switches}
      if (Switch[i] = 'Q') then query := true
      else if (Switch[i] = 'NOQ') then query := false;
    END {Obey switches};

 if (CmdLine.NumFiles > 0) then
    BEGIN {Source on CmdLine}
       SourceName := CmdLine.Files[1];
       rtb (SourceName);
       UC  (SourceName);
       error := not exist (SourceName);
       if error then writeln('File ', SourceName, ' does not exist.');
    END {Source on CmdLine}
 else
    BEGIN {Source from console}
       done := false;
       repeat
          inquire (SourcePrompt, SourceName);
          rtb (SourceName);
          UC  (SourceName);
          if SourceName = '' then
             begin
                writeln('NO FILE SPECIFIED.  I WILL STOP');
                error := true;
             end
          else if not exist(SourceName) then
             begin
                writeln ('File ', SourceName, ' does not exist. C/R to abort.');
             end
          else
             done := true;
       until done or error;
    END {Source from console};
 if error then exit;

 if (CmdLine.NumFiles > 1) then
    BEGIN {Dest from cmdline}
       DestName := CmdLine.Files[2];
       rtb (DestName);
       UC  (DestName);
       if (DestName = SourceName) and not StdDev then
          begin
             writeln ('Source and destination files must not be same.', bell);
             error := true;
             DestName := '';
          end
       else if query and exist (DestName) and not StdDev then
          error := not ask ('Do you want to overwrite ' + DestName + bell);
    END {Dest from cmdline}
 else
    BEGIN {Dest from console}
       done := false;
       repeat
          inquire (DestPrompt, DestName);
          rtb (DestName);
          UC  (DestName);
          if DestName = '' then
             begin
                writeln('NO FILE SPECIFIED.  I WILL STOP');
                error := true;
             end
          else if (DestName = SourceName) and not StdDev then
             begin
                writeln ('Source and destination files must not be same.', bell);
                done := false;
             end
          else if exist (DestName) and not StdDev then
             done := ask ('Do you want to overwrite ' + DestName)
          else
             done := true;
       until done or error;
    END {Dest from console};

END {SetFiles};

end.
