% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gt_cluster_pca_best_k.R
\name{gt_cluster_pca_best_k}
\alias{gt_cluster_pca_best_k}
\title{Find the best number of clusters based on principal components}
\usage{
gt_cluster_pca_best_k(
  x,
  stat = c("BIC", "AIC", "WSS"),
  criterion = c("diffNgroup", "min", "goesup", "smoothNgoesup", "goodfit"),
  quiet = FALSE
)
}
\arguments{
\item{x}{a \code{gt_cluster_pca} object obtained with \code{\link[=gt_cluster_pca]{gt_cluster_pca()}}}

\item{stat}{a statistics, one of "BIC", "AIC" or "WSS"}

\item{criterion}{one of "diffNgroup", "min", "goesup", "smoothNgoesup",
"goodfit", see details for a discussion of each approach.}

\item{quiet}{boolean on whether to silence outputting information to the
screen (defaults to FALSE)}
}
\value{
a 'gt_cluster_pca' object with an added element 'best_k'
}
\description{
This function selects the best k value based on a chosen metric and
criterion. It is equivalent to plotting the metric against the \emph{k} values,
and selecting the \emph{k} that fulfills a given criterion (see details for an
explanation of each criterion). This function simply adds an element 'best_k'
to the \code{gt_cluster_pca} returned by \code{\link[=gt_cluster_pca]{gt_cluster_pca()}}. The choice can be
over-ridden simply by assigning a different value to that element (e.g. for
an object x and a desired \emph{k} of 8, simply use \code{x$best_k <- 8})
}
\details{
The analysis of data simulated under various population genetics models (see
reference) suggested an ad-hoc rule for the selection of the optimal number
of clusters. First important result is that BIC seems more efficient than AIC
and WSS to select the appropriate number of clusters (see example). The rule
of thumb consists in increasing K until it no longer leads to an appreciable
improvement of fit (i.e., to a decrease of BIC). In the most simple models
(island models), BIC decreases until it reaches the optimal K, and then
increases. In these cases, the best rule amounts to choosing the lowest K. In
other models such as stepping stones, the decrease of BIC often continues
after the optimal K, but is much less steep, so a change in slope can be
taken as an indication of where the best \emph{k} lies.

This function provides a programmatic way to select \emph{k}. Note that it is
highly recommended to look at the graph of BIC versus the numbers of
clusters, to understand and validate the programmatic selection. The criteria
available in this function are:
\itemize{
\item "diffNgroup": differences between successive values of the summary
statistics (by default, BIC) are split into two groups using a Ward's
clustering method (see ?hclust), to differentiate sharp decrease from mild
decreases or increases. The retained K is the one before the first group
switch. This criterion appears to work well for island/hierarchical models,
and decently for isolation by distance models, albeit with some instability.
It can be confounded by an initial, very sharp decrease of the test
statistics. IF UNSURE ABOUT THE CRITERION TO USE, USE THIS ONE.
\item "min": the model with the minimum summary statistics (as specified by
stat argument, BIC by default) is retained. Is likely to work for simple
island model, using BIC. It is likely to fail in models relating to stepping
stones, where the BIC always decreases (albeit by a small amount) as K
increases. In general, this approach tends to over-estimate the number of
clusters.
\item "goesup": the selected model is the K after which increasing the number
of clusters leads to increasing the summary statistics. Suffers from
inaccuracy, since i) a steep decrease might follow a small 'bump' of increase
of the statistics, and ii) increase might never happen, or happen after
negligible decreases. Is likely to work only for clear-cut island models.
\item "smoothNgoesup": a variant of "goesup", in which the summary statistics
is first smoothed using a lowess approach. Is meant to be more accurate than
"goesup" as it is less prone to stopping to small 'bumps' in the decrease of
the statistics.
\item "goodfit": another criterion seeking a good fit with a minimum number
of clusters. This approach does not rely on differences between successive
statistics, but on absolute fit. It selects the model with the smallest K so
that the overall fit is above a given threshold.
}
}
\examples{
# Create a gen_tibble of lobster genotypes
bed_file <-
  system.file("extdata", "lobster", "lobster.bed", package = "tidypopgen")
lobsters <- gen_tibble(bed_file,
  backingfile = tempfile("lobsters"),
  quiet = TRUE
)

# Remove monomorphic loci and impute
lobsters <- lobsters \%>\% select_loci_if(loci_maf(genotypes) > 0)
lobsters <- gt_impute_simple(lobsters, method = "mode")

# Create PCA object
pca <- gt_pca_partialSVD(lobsters)

# Run clustering on the first 10 PCs
cluster_pca <- gt_cluster_pca(
  x = pca,
  n_pca = 10,
  k_clusters = c(1, 5),
  method = "kmeans",
  n_iter = 1e5,
  n_start = 10,
  quiet = FALSE
)

# Find best K through minimum BIC
cluster_pca <- gt_cluster_pca_best_k(cluster_pca,
  stat = "BIC",
  criterion = "min",
  quiet = FALSE
)
# Best K is stored in the object
cluster_pca$best_k
}
\references{
Jombart T, Devillard S and Balloux F (2010) Discriminant analysis
of principal components: a new method for the analysis of genetically
structured populations. BMC Genetics 11:94. doi:10.1186/1471-2156-11-94
}
