% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lppinv.R
\name{lppinv}
\alias{lppinv}
\title{Solve a linear program via Convex Least Squares Programming (CLSP).}
\usage{
lppinv(
  c = NULL,
  A_ub = NULL,
  b_ub = NULL,
  A_eq = NULL,
  b_eq = NULL,
  non_negative = TRUE,
  bounds = NULL,
  replace_value = NA_real_,
  tolerance = sqrt(.Machine$double.eps),
  final = TRUE,
  alpha = NULL,
  ...
)
}
\arguments{
\item{c}{numeric vector of length \eqn{p}, optional.
Objective-function coefficients. Included for API parity with
Python's \code{pylppinv}; not used by CLSP.}

\item{A_ub}{numeric matrix of size \eqn{i \times p}, optional.
Matrix of inequality constraints
\eqn{\mathbf{A}_{ub} \mathbf{x} \le \mathbf{b}_{ub}}.}

\item{b_ub}{numeric vector of length \eqn{i}, optional.
Right-hand side for the inequality constraints.}

\item{A_eq}{numeric matrix of size \eqn{j \times p}, optional.
Matrix of equality constraints
\eqn{\mathbf{A}_{eq} \mathbf{x} = \mathbf{b}_{eq}}.}

\item{b_eq}{numeric vector of length \eqn{j}, optional.
Right-hand side for the equality constraints.}

\item{non_negative}{logical scalar, default = \code{TRUE}.
If \code{FALSE}, no default nonnegativity bound is applied.}

\item{bounds}{NULL, \code{numeric(2)}, or list of \code{numeric(2)}.
Bounds on variables. If a single pair \code{c(low, high)} is
given, it is applied to all variables. If \code{NULL},
defaults to \code{c(0, NA)} for each variable (non-negativity).}

\item{replace_value}{numeric scalar or \code{NA}, default = \code{NA}.
Final replacement value for any variable that violates the bounds by
more than the given tolerance.}

\item{tolerance}{numeric scalar, default = \code{sqrt(.Machine$double.eps)}.
Convergence tolerance for bounds.}

\item{final}{logical scalar, default = \code{TRUE}
If \code{FALSE}, only the first step of the CLSP estimator is performed.}

\item{alpha}{numeric scalar, numeric vector, or \code{NULL},
Regularization parameter for the second step of the CLSP estimator.}

\item{...}{Additional arguments passed to the \pkg{rclsp} solver.}
}
\value{
An object of class \code{"clsp"} containing the fitted CLSP model.
}
\description{
Solve a linear program via Convex Least Squares Programming (CLSP).
}
\examples{
\donttest{
  ## Linear Programming via Regularized Least Squares (LPPinv)
  ## Underdetermined and potentially infeasible LP system

  set.seed(123456789)

  # sample (dataset)
  A_ub <- matrix(rnorm(50 * 500), nrow = 50, ncol = 500)
  A_eq <- matrix(rnorm(25 * 500), nrow = 25, ncol = 500)
  b_ub <- matrix(rnorm(50),       ncol = 1)
  b_eq <- matrix(rnorm(25),       ncol = 1)

  # model (no default non-negativity, unique MNBLUE solution)
  model <- lppinv(
      A_ub = A_ub,
      A_eq = A_eq,
      b_ub = b_ub,
      b_eq = b_eq,
      non_negative = FALSE,
      final = TRUE,
      alpha = 1.0                                   # unique MNBLUE estimator
  )

  # coefficients
  print("x hat (x_M hat):")
  print(round(model$x, 4))

  # numerical stability (if available)
  if (!is.null(model$kappaC)) {
      cat("\nNumerical stability:\n")
      cat("  kappaC :", round(model$kappaC, 4), "\n")
  }
  if (!is.null(model$kappaB)) {
      cat("  kappaB :", round(model$kappaB, 4), "\n")
  }
  if (!is.null(model$kappaA)) {
      cat("  kappaA :", round(model$kappaA, 4), "\n")
  }

  # goodness-of-fit diagnostics (if available)
  if (!is.null(model$nrmse)) {
      cat("\nGoodness-of-fit:\n")
      cat("  NRMSE :", round(model$nrmse, 6), "\n")
  }
  if (!is.null(model$x_lower)) {
      cat("  Diagnostic band (min):", round(min(model$x_lower), 4), "\n")
  }
  if (!is.null(model$x_upper)) {
      cat("  Diagnostic band (max):", round(max(model$x_upper), 4), "\n")
  }

  # bootstrap NRMSE t-test (if supported by rclsp)
  if ("ttest" \%in\% names(model)) {
      cat("\nBootstrap t-test:\n")
      tt <- model$ttest(sample_size = 30,
                        seed = 123456789,
                        distribution = "normal")
      for (nm in names(tt)) {
          cat("   ", nm, ": ", round(tt[[nm]], 6), "\n", sep = "")
      }
  }
}

}
\seealso{
\link[rclsp]{clsp}

\link[CVXR]{CVXR-package}
}
