#' checkDymamicChips
#'
#' Generate a dataframe of summary infformation for each chip defined using makeDynamicChipsSF().
#'
#' Generate set of summary information about each chip defined by makeDynamicChipsSF() including
#' all the information in the sf object generated by makeDynamicChipsSF() and the number of columns
#' of cells in the predictor variable raster and mask ("cCntImg" and "cCntMsk"), number of rows
#' of cells in the predictor variable raster and mask ("rCntImg" and "rCntMsk"), number of input
#' predictor variables ("bCntImg"), and count of NA pixels in the predictor variable raster and mask
#' ("naCntImg" and "naCntMsk"). This function is used primarily to check the chips prior to attempting
#' to use them in a training or validation process.
#'
#' @param chipsSF sf object created by makeDynamicChipsSF().
#' @param chipSize Size of desired image chips. Default is 512 (512x512 cells)
#' @param cellSize Cells size of input data. Default is 1 m.
#' @param doJitter Whether or not to add random noise to chip center coordinates. Default is FALSE.
#' @param jitterSD If doJitter is TRUE, standard deviation of random positional noise to add in
#' both the x and y directions. Default is 15 (15 meters).
#' @param useSeed Whether or not to use a random seed for added jitter noise. Default is FALSE.
#' @param seed Random seed value.
#' @return Dataframe of summary information about image chips.
#' @export
checkDynamicChips <- function(chipsSF,
                              chipSize=512,
                              cellSize=1,
                              doJitter=FALSE,
                              jitterSD=15,
                              useSeed=TRUE,
                              seed=42){

  checkDF <- data.frame()

  for(i in 1:nrow(chipsSF)){
    c1 <- makeDynamicChip(chipIn=chipsSF[i,],
                          chipSize=chipSize,
                          cellSize=cellSize,
                          doJitter=doJitter,
                          jitterSD=jitterSD,
                          useSeed=useSeed,
                          seed=seed)

    c1DF <- data.frame(cCntImg = terra::ncol(c1$image),
                       rCntImg = terra::nrow(c1$image),
                       bCntImg = terra::nlyr(c1$image),
                       naCntImg = terra::global(is.na(c1$image), fun = "sum")[1,1],
                       cCntMsk = terra::ncol(c1$mask),
                       rCntMsk = terra::nrow(c1$mask),
                       naCntMsk = terra::global(is.na(c1$mask), fun = "sum")[1,1])

    checkDF <- dplyr::bind_rows(checkDF, c1DF)

  }

  chipsSF <- dplyr::bind_cols(chipsSF, checkDF)

  return(chipsSF)
}
