#################################################################################################
## Functions for Scale Mixture Skew Normal (SMSN) distribution
##
## Description: This script contains functions for probability density function (pdf),
##              cumulative distribution function (cdf), moments and
##              other related operations for the Scale Mixture Skew Normal (SMSN) distribution.
##
#################################################################################################

skewness_simp  <- function(x, na.rm = FALSE) {
  if (na.rm) x <- x[!is.na(x)]
  m <- mean(x)
  n <- length(x)
  num <- sum((x - m)^3) / n
  den <- (sum((x - m)^2) / n)^(3/2)
  num / den
}


##################################################
######### E_phi e E_Phi - Simetricas #############
##################################################


E_Phi <- function(r,a,nu,delta,type=type)
{
  n <- length(a)
  if(type=="N")
  {
    resp <- pnorm(a)
  }
  if(type=="T")
  {
    Aux0 <- gamma(0.5*(nu+(2*r)))
    Aux1 <- gamma(nu/2)
    Aux2 <- Aux0/Aux1
    Aux3 <- (0.5*nu)^(-r)
    Aux4 <- AcumPearsonVII(a,0,1,nu+(2*r),nu)
    resp <- Aux2*Aux3*Aux4
  }
  if(type=="CN")
  {
    Aux0 <- (nu[2]^r)*AcumNormalC(a,0,1,nu)
    Aux1 <- (1-nu[2]^r)*(1-nu[1])*pnorm(a)
    resp <- Aux0 + Aux1
  }
  return(resp)
}

################################################################################
################################################################################

E_phi <- function(r,a,nu,delta,type=type)
{
  n <- length(a)
  b <- rep(Inf,n)
  b1<- rep(-Inf,n)

  if(setequal(a,b)== TRUE | setequal(a,b1)== TRUE)
  {
    resp <- rep(0,n)
  }
  else
  {
    if(type=="N")
    {
      resp <- dnorm(a)
    }
    if(type=="T")
    {
      Aux0 <- gamma(0.5*(nu+2*r))
      Aux1 <- gamma(nu/2)*sqrt(2*pi)
      Aux2 <- Aux0/Aux1
      Aux3 <- (0.5*nu)^(nu/2)
      Aux4 <- (0.5*(a^2+nu))^(-0.5*(nu+2*r))
      resp <- Aux2*Aux3*Aux4
    }
    if(type=="CN")
    {
      Aux0 <- nu[1]*(nu[2]^r)*dnorm(a*sqrt(nu[2]))
      Aux1 <- (1-nu[1])*dnorm(a)
      resp <- Aux0 + Aux1
    }
  }
  return(resp)
}


##################################################
######### E_phi e E_Phi - Assimetricas ###########
##################################################


E_phiSNI <- function(r,a,nu,delta,lambda,type=type)
{
  n <- length(a)
  b <- rep(Inf,n)
  b1<- rep(-Inf,n)
  if(setequal(a,b)== TRUE | setequal(a,b1)== TRUE)
  {
    resp <- rep(0,n)
  }
  else
  {
    if(type=="SN")
    {
      resp <- 2*dnorm(a)*pnorm(lambda*a)
    }
    if(type=="ST")
    {
      Aux0 <- (2^(r+1))*(nu^(nu/2))*gamma(0.5*nu+r)
      Aux1 <- gamma(nu/2)*sqrt(2*pi)
      Aux2 <- (a^2+nu)^(0.5*nu+r)
      Aux3 <- Aux0/(Aux1*Aux2)
      Aux4 <- sqrt((2*r+nu)/(a^2+nu))*lambda*a
      Aux5 <- pt(Aux4,df=(2*r+nu))
      resp <- Aux3*Aux5
    }
    if(type=="SCN")
    {
      Aux0 <- nu[1]*(nu[2]^r)*(2*dnorm(a*sqrt(nu[2]))*pnorm(lambda*a*sqrt(nu[2])))
      Aux1 <- (1-nu[1])*(2*dnorm(a)*pnorm(lambda*a))
      resp <- Aux0 + Aux1
    }
  }
  if(length(which(resp == 0)) > 0) resp[which(resp == 0)] <- .Machine$double.xmin
  return(resp)
}

E_PhiSNI <- function(r,a,nu,delta,lambda,type=type)
{
  deltinha <- lambda/sqrt(1+lambda^2)
  n <- length(a)
  if(type=="SN")
  {
    resp <- cdfSNI(a,mu=0,sigma2=1,lambda,nu=NULL,type=type)
  }
  if(type=="ST")
  {
    Aux0 <- 2^(r+1)
    Aux1 <- gamma(0.5*nu+r)
    Aux2 <- gamma(nu/2)
    Aux3 <- nu^(r)
    Aux4 <- (Aux0*Aux1)/(Aux2*Aux3)
    Aux5 <- sqrt((2*r+nu)/nu)
    valor <- Aux5*c(a,0)
    mean1 <- c(0,0)
    Sigma1 <- matrix(c(1,-deltinha,-deltinha,1), 2, 2)
    GB = mvtnorm::GenzBretz(maxpts = 5e4, abseps = 1e-9, releps = 0)
    Aux6 <- mvtnorm::pmvt(lower = -Inf, upper = valor-mean1, sigma = Sigma1,  df=round(2*r+nu), algorithm = GB)[1]
    resp <- Aux4*Aux6
  }
  if(type=="SCN")
  {
    Aux0 <- (nu[2]^r)*cdfSNI(a,mu=0,sigma2=1,lambda,nu,type="SCN")
    Aux1 <- 2*(1-nu[2]^r)*(1-nu[1])
    valor <- c(a,0)
    mean1 <- c(0,0)
    Sigma1 <- matrix(c(1,-deltinha,-deltinha,1), 2, 2)
    if(a==Inf)
    {
      valor=c(500,0)
    }
    Aux3 <- mnormt::pmnorm(valor, mu=c(0,0), varcov=Sigma1)
    resp <- Aux0 + (Aux1*Aux3)
  }
  if(length(which(resp == 0)) > 0) resp[which(resp == 0)] <- .Machine$double.xmin
  return(resp)
}


################################################################
##########              Densities                   ############
################################################################

## SN density with scale location ##
dSN <- function(y, mu = 0, sigma2 = 1, shape=1){
  dens <- 2*dnorm(y, mu, sqrt(sigma2))*pnorm(shape*((y - mu)/sqrt(sigma2)))
  return(dens)
}


## ST density with scale location ##
dt.ls <- function(x, loc = 0, sigma2 = 1,shape=1, nu = 4){
  d <- (x - loc)/sqrt(sigma2)
  dens <- 2*dt(d, df = nu)*pt(sqrt((1+nu)/(d^2+nu))*d*shape,1+nu)/sqrt(sigma2)
  return(dens)
}


## SCN density with scale location ##
DenSCN <-  function(y, mu, sigma2, shape, nu){
  dens <- 2*(nu[1]*dnorm(y, mu, sqrt(sigma2/nu[2]))*pnorm(sqrt(nu[2])*shape*sigma2^(-1/2)*(y-mu)) + (1 - nu[1])*dnorm(y, mu, sqrt(sigma2))*pnorm(shape*sigma2^(-1/2)*(y-mu)))
  return(dens)
}


## Auxiliary function for likelihood of T ##
dTMod <- function(cc, y, mu, sigma2 = 1, nu=4, cens="Left", LS=NULL){
  densN<- vector(mode = "numeric", length = length(y))
  aux<-(y-mu)/sqrt(sigma2)
  densN[cc==0] <- dt(aux[cc==0],nu)/sqrt(sigma2)
   if (cens=="Left"){
   densN[cc==1]<- pt(aux[cc==1],nu)
       }
   if (cens=="Right"){
   densN[cc==1]<- 1-pt(aux[cc==1],nu)
       }
  if (cens=="Interv"){
  aux1<-(LS-mu)/sqrt(sigma2)
   densN[cc==1]<- pt(aux1[cc==1],nu)-pt(aux[cc==1],nu)
       }
  if(length(which(densN == 0)) > 0) densN[which(densN == 0)] <- .Machine$double.xmin
  return(densN)
}


## Auxiliary function for likelihood of ST ##
dSTMod <- function(cc, y, mu, sigma2 = 1, shape, nu=4, cens="Left", LS=NULL){
  densN<- vector(mode = "numeric", length = length(y))
  densN[cc==0] <- dt.ls(y[cc==0], mu[cc==0], sigma2, shape, nu )
   if (cens=="Left"){
   densN[cc==1]<- cdfSNI(y[cc==1], mu[cc==1], sigma2, shape, nu, type = "ST")
       }
   if (cens=="Right"){
   densN[cc==1]<- 1-cdfSNI(y[cc==1], mu[cc==1], sigma2, shape, nu, type = "ST")
       }
  if (cens=="Interv"){
    if (length(y[cc == 1]) > 0) {
      densN[cc == 1] <- pSNI2(y[cc == 1], LS[cc == 1], mu[cc == 1], sigma2, shape, nu, type = "ST")
    }
       }
  if(length(which(densN == 0)) > 0) densN[which(densN == 0)] <- .Machine$double.xmin
  return(densN)
}

## Auxiliary function for likelihood of SN ##
dSNMod <- function(y, mu = 0, sigma2 = 1, shape=1){
  densN <- 2*dnorm(y, mu, sqrt(sigma2))*pnorm(shape*((y - mu)/sqrt(sigma2)))
  if(length(which(densN == 0)) > 0) densN[which(densN == 0)] <- .Machine$double.xmin
  return(densN)
}


## Auxiliary function for likelihood of SCN ##
dSCNMod <- function(cc, y, mu, sigma2 = 1, shape, nu, cens="Left", LS=NULL){
  densN<- vector(mode = "numeric", length = length(y))
  densN[cc==0] <- DenSCN(y[cc==0], mu[cc==0], sigma2, shape, nu )
  if (cens=="Left"){
    densN[cc==1]<- AcumSCN(y[cc==1], mu[cc==1], sigma2, shape, nu)
  }
  if (cens=="Right"){
    densN[cc==1]<- 1-AcumSCN(y[cc==1], mu[cc==1], sigma2, shape, nu)
  }
  if (cens=="Interv"){
    if (length(y[cc == 1]) > 0) {
      densN[cc==1]<-pSNI_NC(y[cc==1], LS[cc==1], mu[cc==1], sigma2, shape, nu, type = "SCN")
    }
  }
  if(length(which(densN == 0)) > 0) densN[which(densN == 0)] <- .Machine$double.xmin
  return(densN)
}


################################################################
##########    Cumulative distribution functions     ############
################################################################

### N-SN-T-ST CDF
cdfSNI<- function(x, mu, sigma2, lambda, nu, type = "SN")
{
  n <- length(x)
  resp <- matrix(0, n, 1)
  if (type == "N") {
    resp <- pnorm((x - mu)/sqrt(sigma2))
    return(resp)
  }

  if (type == "T") {
    resp <- pt((x - mu)/sqrt(sigma2), df = nu)
    return(resp)
  }

  if (type == "SN") {
    delta <- lambda/sqrt(1 + lambda^2)
    SIGMA <- matrix(c(sigma2, -delta * sqrt(sigma2), -delta * sqrt(sigma2),
                      1), byrow = TRUE, ncol = 2, nrow = 2)
    if (length(mu) == 1) {
      MU <- cbind(rep(mu, n), 0)
    }
    if (length(mu) == n) {
      MU <- cbind(mu, 0)
    }
    Y <- cbind(x, 0)
    for (i in 1:n) {
      resp[i] <- 2 * mnormt::pmnorm(x = Y[i, ], mean = MU[i, ], varcov = SIGMA)
    }
    return(resp)
  }

  if (type == "ST") {
    if (lambda == 0) {
      resp <- pt((x - mu) / sqrt(sigma2), df = nu)
      return(resp)
    }
    delta <- lambda/sqrt(1 + lambda^2)
    SIGMA <- matrix(c(sigma2, -delta * sqrt(sigma2), -delta * sqrt(sigma2),
                      1), byrow = TRUE, ncol = 2, nrow = 2)
    if (length(mu) == 1) {
      MU <- cbind(rep(mu, n), 0)
    }
    if (length(mu) == n) {
      MU <- cbind(mu, 0)
    }
    Y <- cbind(x, 0)
    Y[x==-Inf,]<-c(-Inf,-Inf)
    if(nu%%1 == 0){
      #nu integer
      for (i in 1:n){
        resp[i] <- 2 * mnormt::pmt(x = Y[i, ], mean = MU[i, ], S = SIGMA, df = nu)
      }
    }else{
      #nu decimal
      for (i in 1:n) {
        if (Y[i, 1] > MU[i, 1] + 15 * sqrt(sigma2) && Y[i, 1] != Inf) {
            resp[i] <- 1
        }
        else {
          resp[i] <- 2 * acumt2(a = NULL, b = Y[i, ], mu = MU[i, ], Sigma = SIGMA, nu = nu)
        }
      }
    }
    return(resp)
  }
}

### SCN CDF
AcumSCN <- function(y,mu,sigma2,shape,nu){
  n      <- length(y)
  resp   <- matrix(0, n, 1)
  delta  <- shape/sqrt(1 + shape^2)
  SIGMA1 <- matrix(c(sigma2/(nu[2]), -delta * sqrt(sigma2/nu[2]),
                     -delta * sqrt(sigma2/nu[2]), 1), byrow = TRUE, ncol = 2, nrow = 2)
  SIGMA2 <- matrix(c(sigma2, -delta * sqrt(sigma2),
                     -delta * sqrt(sigma2), 1), byrow = TRUE, ncol = 2, nrow = 2)
  if (length(mu) == 1) {
    MU <- cbind(rep(mu, n), 0)
  }
  if (length(mu) == n) {
    MU <- cbind(mu, 0)
  }
  Y <- cbind(y, 0)

  for (i in 1:n) {
    resp[i] <- 2 * ( nu[1]*mnormt::pmnorm(x = Y[i, ], mean = MU[i, ], varcov = SIGMA1)
                     + (1-nu[1]) * mnormt::pmnorm(x = Y[i, ], mean = MU[i, ], varcov = SIGMA2))
  }
  return(resp)
}


###  CDF in intervals
pSNI2<- function(LI, LS, mu, sigma2, lambda, nu, type = "SN"){
  n <- length(LI)
  resp <- matrix(0, n, 1)
  if (type == "N") {
    resp <- pnorm((LS - mu)/sqrt(sigma2))-pnorm((LI - mu)/sqrt(sigma2))
    if(length(which(resp <= 0)) > 0) resp[which(resp <= 0)] <- .Machine$double.xmin
    return(resp)
  }

  if (type == "T") {
    resp <- pt((LS - mu)/sqrt(sigma2), df = nu)-pt((LI - mu)/sqrt(sigma2), df = nu)
    if(length(which(resp <= 0)) > 0) resp[which(resp <= 0)] <- .Machine$double.xmin
    return(resp)
  }

  if (type == "SN") {
    delta <- lambda/sqrt(1 + lambda^2)
    SIGMA <- matrix(c(sigma2, -delta * sqrt(sigma2), -delta * sqrt(sigma2),
                      1), byrow = TRUE, ncol = 2, nrow = 2)
    if (length(mu) == 1) {
      MU <- cbind(rep(mu, n), 0)
    }
    if (length(mu) == n) {
      MU <- cbind(mu, 0)
    }
    YI <- cbind(LI, 0)
    YS <- cbind(LS, 0)
    for (i in 1:n) {
      resp[i] <- 2 * (mnormt::pmnorm(x = YS[i, ], mean = MU[i, ], varcov = SIGMA)-mnormt::pmnorm(x = YI[i, ], mean = MU[i, ], varcov = SIGMA))
    }
    if(length(which(resp <= 0)) > 0) resp[which(resp <= 0)] <- .Machine$double.xmin
    return(resp)
  }
  if (type == "ST") {
    if (lambda == 0) {
      resp <- pt((LS - mu) / sqrt(sigma2), df = nu) - pt((LI - mu) / sqrt(sigma2), df = nu)
      resp[resp <= 0] <- .Machine$double.xmin
      return(resp)
    }
    delta <- lambda/sqrt(1 + lambda^2)
    SIGMA <- matrix(c(sigma2, -delta * sqrt(sigma2), -delta * sqrt(sigma2),
                      1), byrow = TRUE, ncol = 2, nrow = 2)
    if (length(mu) == 1) {
      MU <- cbind(rep(mu, n), 0)
    }
    if (length(mu) == n) {
      MU <- cbind(mu, 0)
    }
    YI <- cbind(LI, 0)
    YI[LI==-Inf,]<-c(-Inf,-Inf)
    YS <- cbind(LS, 0)
    if(nu%%1 == 0){
      #nu integer
      for (i in 1:n){
        resp[i] <- 2 * (mnormt::pmt(x=YS[i, ], mean = MU[i, ], S = SIGMA, df = nu)-mnormt::pmt(x=YI[i, ], mean = MU[i, ], S = SIGMA, df = nu))
      }
    }else{
      #nu decimal
      for (i in 1:n) {
        if (all(YS[i, ] == YI[i, ])) {
          resp[i] <- 0
        }
        else if ((YS[i, 1] > MU[i, 1] + 4 * sqrt(sigma2) || YI[i, 1] > MU[i, 1] + 4 * sqrt(sigma2)) && YS[i, 1] != Inf) {
          resp[i] <- tryCatch(
            {
              integrate(
                f = function(x) dt.ls(x, loc = MU[i, 1], sigma2 = sigma2, shape = lambda, nu = nu),
                lower = YI[i, 1],
                upper = YS[i, 1],
                subdivisions = 1000,
                rel.tol = 1e-8
              )$value
            },
            error = function(e) {
              cubature::cubintegrate(
                f = function(x) dt.ls(x, loc = MU[i, 1], sigma2 = sigma2, shape = lambda, nu = nu),
                lower = YI[i, 1],
                upper = YS[i, 1],
                method = "pcubature",
              )$integral
            }
          )
        }
        else {
          resp[i] <- 2 * (
            acumt2(a = NULL, b = c(YS[i, ] - MU[i, ]), mu = c(0, 0), Sigma = SIGMA, nu = nu) -
              acumt2(a = NULL, b = c(YI[i, ] - MU[i, ]), mu = c(0, 0), Sigma = SIGMA, nu = nu)
          )
        }
      }
    }
    if(length(which(resp <= 0)) > 0) resp[which(resp <= 0)] <- .Machine$double.xmin
    return(resp)
  }
}


###  CDF in intervals SCN ##
pSNI_NC<- function(LI, LS, mu, sigma2, shape, nu, type = "SCN"){
  n      <- length(LI)
  resp   <- matrix(0, n, 1)
  delta  <- shape/sqrt(1 + shape^2)
  SIGMA1 <- matrix(c(sigma2/(nu[2]), -delta * sqrt(sigma2/nu[2]),
                     -delta * sqrt(sigma2/nu[2]), 1), byrow = TRUE, ncol = 2, nrow = 2)
  SIGMA2 <- matrix(c(sigma2, -delta * sqrt(sigma2),
                     -delta * sqrt(sigma2), 1), byrow = TRUE, ncol = 2, nrow = 2)
  if (length(mu) == 1) {
    MU <- cbind(rep(mu, n), 0)
  }
  if (length(mu) == n) {
    MU <- cbind(mu, 0)
  }
  YI <- cbind(LI, 0)
  YI[LI==-Inf,]<-c(-Inf,-Inf)
  YS <- cbind(LS, 0)
  for (i in 1:n) {
    resp[i] <- 2 * (( nu[1]*mnormt::pmnorm(x = YS[i, ], mean = MU[i, ], varcov = SIGMA1)
                      + (1-nu[1]) * mnormt::pmnorm(x = YS[i, ], mean = MU[i, ], varcov = SIGMA2)) -
                      ( nu[1]*mnormt::pmnorm(x = YI[i, ], mean = MU[i, ], varcov = SIGMA1)
                        + (1-nu[1]) * mnormt::pmnorm(x = YI[i, ], mean = MU[i, ], varcov = SIGMA2)))
  }
  if(length(which(resp <= 0)) > 0) resp[which(resp <= 0)] <- .Machine$double.xmin
  return(resp)
}


##################################################################
####   Cumulative distribution functions (NI)   ##################
##################################################################

AcumPearsonVII <- function(y,mu,sigma2,nu,delta)
{
  Acum <- z <- vector(mode = "numeric", length = length(y))
  sigma2a <- sigma2*(delta/nu)
  for (i in 1:length(y))
  {
    z[i] <- (y[i]-mu)/sqrt(sigma2a)
    Acum[i] <- pt(z[i],df=nu)
  }
  return(Acum)
}

P <- function(y,mu,sigma2,nu,delta)
{
  A <- z <- vector(mode = "numeric", length = length(y))
  sigma2a <- sigma2*(delta/nu)
  n <- length(y)
  i <- 0
  while (i<n)
  {
    i <- i +1
    z[i] <- (y[i]-mu)/sqrt(sigma2a)
    A[i] <- pt(z[i],df=nu)
  }
  return(A)
}

AcumSlash <- function(y,mu,sigma2,nu)
{
  Acum <- z <- vector(mode = "numeric", length = length(y))
  for (i in 1:length(y))
  {
    z[i] <- (y[i]-mu)/sqrt(sigma2)
    f1 <- function(u) nu*u^(nu-1)*pnorm(z[i]*sqrt(u))
    Acum[i]<- integrate(f1,0,1)$value
  }
  return(Acum)
}

AcumNormalC <- function(y,mu,sigma2,nu)
{
  Acum <- vector(mode = "numeric", length = length(y))
  for (i in 1:length(y))
  {
    eta  <- nu[1]
    gama <- nu[2]
    Acum[i] <- eta*pnorm(y[i],mu,sqrt(sigma2/gama)) + (1-eta)*pnorm(y[i],mu,sqrt(sigma2))
  }
  return(Acum)
}



##################################################
########  TRUNCATED - SNI MOMENTS -     ##########
##################################################

MomenSNI2023 <- function(mu,sigma2,lambda,nu,delta,Lim1,Lim2,type)
{
  Lim11 <- (Lim1-mu)/sqrt(sigma2)
  Lim21 <- (Lim2-mu)/sqrt(sigma2)
  b <- sqrt(2/pi)
  n <- length(Lim11)
  lambda1 <- sqrt(1+lambda^2)
  Slambda1 <- b*lambda/(lambda1)^2
  if(type=="SN")
  {
    type1 <- "N"
    EU <-  1
  }
  if(type=="ST")
  {
    type1 <- "T"
  }
  if(type=="SCN")
  {
    type1 <- "CN"
  }
  if(Lim11==-Inf)
  {
    FNIa <- 0
    S4 <- (Lim21)*E_phiSNI(-0.5,Lim21,nu,delta=NULL,lambda,type=type)
  }
  else
  {
    FNIa <- cdfSNI(Lim11,mu=0,sigma2=1,lambda,nu=nu,type=type)
  }
  if(Lim21==Inf)
  {
    FNIb <- 1
    S4 <- - (Lim11)*E_phiSNI(-0.5,Lim11,nu,delta=NULL,lambda,type=type)
  }
  else
  {
    FNIb <- cdfSNI(Lim21,mu=0,sigma2=1,lambda,nu,type=type)
  }

  if(Lim11==-Inf & Lim21==Inf)
  {
    FNIb <- 1
    FNIa <- 0
    S4   <- 0
  }

  den <- FNIb-FNIa
  if (den == 0) {
    den <- 1-0.9999
  }

  K <- 1/(den)

  if (K == Inf) {
    K <- 1/.Machine$double.xmin
  }
  S0 <- (b*lambda/lambda1)*(E_Phi(-0.5,Lim21*lambda1, nu,delta,type=type1)- E_Phi(-0.5,Lim11*lambda1, nu,delta,type=type1))
  S1 <- E_phiSNI(-0.5,Lim21,nu,delta=NULL,lambda,type=type)- E_phiSNI(-0.5,Lim11,nu,delta=NULL,lambda,type=type)
  S2 <- E_PhiSNI(-1,Lim21,nu,delta,lambda,type=type)- E_PhiSNI(-1,Lim11,nu,delta,lambda,type=type)
  S3 <- Slambda1*(E_phi(-1,Lim21*lambda1,nu,delta,type=type1)-E_phi(-1,Lim11*lambda1,nu,delta,type=type1))
  if(setequal(Lim11,-Inf)== FALSE & setequal(Lim21,Inf)== FALSE)
  {
    S4 <- (Lim21)*E_phiSNI(-0.5,Lim21,nu,delta=NULL,lambda,type=type)- (Lim11)*E_phiSNI(-0.5,Lim11,nu,delta=NULL,lambda,type=type)
  }
  EUX1 <- K*(S0-S1)
  EUX2 <- K*(S2-S3-S4)
  sigma <- sqrt(sigma2)
  EUY1 <- mu + sigma*EUX1
  EUY2 <- mu^2 + 2*mu*sigma*EUX1 + (sigma^2)*EUX2
  return(list(EUY1=EUY1,EUY2=EUY2))
}

