/*  JavaScript Library - SQL String Functions

   DESCRIPTION:  This library of functions is designed to be used to manipulate
	strings for the purposes of forming valid SQL statements.  SQL statements
	do not allow special characters such as carriage returns in the data
	strings.  
		Suppose, however, that you wish to store data entered by
	a user from a multi-line text box in an HTML form.  The user may very
	well enter several carriage returns into the resulting string that comes from
	the text-box data.  The functions below are designed to help with this
	problem.  
		The two major functions "escString()" and "unEscString()" can
	be used to convert a string to all-valid characters, and decode a converted
	string again, respectively.  The other functions listed below are support
	functions for escString() and unEscString, however they may be used
	on their own for other purposes as well.
	
   SPECIAL NOTES:  Note that these functions may be used for any purpose,
	not just SQL string manipulation.  Each function below is described
	in detail, with it's input parameters and return value described as well.
	The functions may be used on the server-side in a LiveWire application,
	or they may be copied and pasted into a client-side HTML page.

	 These functions were tested and verified for client-side use on March 19, 1996

   KNOWN PROBLEMS:  The "unEscString()" function does not handle the case
	where it is given an escape string such as:
		%xxx;
	where "xxx" is a number greater than the maximum ASCII character
	value 255.

*/

/* ============================================================

FUNCTION:  isHexNum( num )

INPUT:  	num	- a character which will be tested to see if it is a digit from 0-9.

RETURNS:  	true if num is a digit from 0-9, false otherwise

DESCRIPTION:  This function determines if a character is a digit from 0-9 or not.

============================================================ */

function isHexNum ( num )
{
	num = num.toUpperCase();
	if ( ((num >= "0") && (num <= "9")) || ((num >= "A") && (num <= "F")) ) {
		returnval = true
	} else {
		returnval = false
	}

	return returnval

}

/* =========================================================

   FUNCTION:  	isSQLok( testchar )

   INPUT:	testchar - the character to be tested

   RETURNS:    	true if the character is valid in an SQL statement
 	    	false if the character is invalid in SQL statements

   DESCRIPTION:

   This function tests a character to see if it is a printable standard ASCII character.
   SQL statements will not accept non-printable characters, so this function tests
   to see if the character must be escaped before being included in an SQL query
   string.

   The % character must also cause a return value of false because it is the prelude
   character for "escaped characters"

   The ' character (single quote, NOT a backquote) must be escaped so that any
   SQL statements constructed in code that construct query strings are not broken
   up by a single quote in the string.  For example:

	insert into mytable values ('532.6',' this is a 'quote' of mine')

    As you can see, the single quote characters around the word quote above
    will disrupt the string in the 2-argument value list in the SQL insert statement,
    so it must be escaped to look like this:

	insert into mytable values ('532.6','this is a %36;quote%36; of mine')

    This way, the single quote delimiters are not disrupted.
   ========================================================== */

function isSQLok(testchar)
{
	var	result = true

	// is the character printable?  If not, return false.
	if ( (testchar < " ") || (testchar > "~") ) {
		result = false
	}
	
	// is the character a special case?  If so, return false.
	if ( (testchar == "%") || (testchar == "'") ) {
		result = false
	}

	return result

}  // END function


/* ==============================================================

   FUNCTION:  escString( str )

   INPUT:  str - the string of data to be searched for characters that need to be escaped

   RETURNS:  a the processed string of data, where all characters that need to be are
		escaped in "%37;" format, where "37" is the ASCII value of the escaped
		character and "%" and ";" are the delimiters.  (See also the
		JavaScript documentation for a description of the "escape()" function).

   DESCRIPTION:  This function processes the input string of data and converts
		all non-printable characters, extended-ASCII characters and
		the special case characters:  the single quote " ' "  and the
		"%" character (since it is the delimiter for escaped characters)

============================================================== */

function  escString( str )
{

	var	resultstr = ""
	
	// Search through the entire input string for characters
	// that need to be escaped.

	for(i = 0; i < str.length; i++) {
		if ( isSQLok(str.charAt(i)) ) {
			resultstr += str.charAt(i)
		}  else if (escape(str.charAt(i)) == "%0A")
			resultstr += "";  // eliminate linefeed characters completely
		   else {
			resultstr += escape(str.charAt(i)) + ";"
		}
	} // end for loop

	return resultstr
} // END FUNCTION


/* ============================================================

   FUNCTION:  	unEscString ( str )

   INPUT:	str - the string to which will have all escaped characters converted
		 	back to their original forms.

   RETURNS:	a string of characters all in their original, unescapes forms.

   DESCRIPTION:  This function simply undoes what escString() does to a string
		by unescaping all of the escape sequences in it.  For example, suppose
		"37" is the ASCII value of the escaped character and "%" and ";"
		are the delimiting characters, so that the escape sequence is "%37;".  This
		function will translate it back into it's original character form.  (See
		also the JavaScript documentation on the "unescape()" function)

   ============================================================ */


function unEscString (str)
{

	var resultstr = ""
	var substr = ""
	var validFlag = true

	// begin scanning entire input string
	for (i = 0; i < str.length; i++) {

		if (str.charAt(i) == "%") {
			// start building our possible escape-string with the percent symbol
			// (required for the "unescape" function)
			substr += str.charAt(i)
			i++
			count = 1

			/* Scan for the ending delimiter, a semi-colon.  Build the substring
			 with the characters occuring before the semi-colon.
			 The loop only iterates 2 times because there are only 255 characters
			 possible requiring only 2 hex digits at most after the % sign for escaped
			 ASCII representation.  Therefore, if we go longer than 2 numbers after
			the % sign, we know it cannot be an escape representation, but is just
			a % sign with some numbers following it.
			*/
			while( (str.charAt(i) != ";") && (count <= 2) && (validFlag) ) {

				 if ( !isHexNum(str.charAt(i)) ) {
					validFlag = false
				}
				
				substr += str.charAt(i)
				i++
				count++
			} // end while

			/* At end of loop if we have a valid escape string, "i" should point to the 	
			";"  If it doesn't, we have either an un-terminated escape string (case 2) 	
			or an escape string with a number that is not a valid escape string (case
			4).  In either case, we simply treat it as a string of characters which is
			not an escape string, and just tack it on to the result string.
			*/

			if (str.charAt(i) != ";") {		// Fixes cases 2 & 4 (described above)
				validFlag = false
			}
			
			if (validFlag) {
				chr = unescape(substr)
				resultstr += chr
			} else {
				substr += str.charAt(i)	// Tack on the semi-colon to the
 							// so it doesn't get skipped when the
 							// for loop iterates
				resultstr += substr
			}

			substr = ""	// reset substr since we are done with it in this iteration
		} else {
			resultstr += str.charAt(i)	// if the character wasn't a "%" we simply
							// copy it to the resultstr string.
		} // end outermost if
				
	} // end for loop

	return resultstr
	
} // END FUNCTION



