/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import io.fabric8.kubernetes.api.model.NamedContext;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.http.HttpClient;
import io.fabric8.kubernetes.client.http.HttpRequest;
import io.fabric8.kubernetes.client.http.HttpResponse;
import io.fabric8.kubernetes.client.internal.KubeConfigUtils;
import io.fabric8.kubernetes.client.internal.SSLUtils;
import io.fabric8.kubernetes.client.utils.Serialization;
import io.fabric8.kubernetes.client.utils.URLUtils;
import io.fabric8.kubernetes.client.utils.Utils;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import javax.net.ssl.KeyManager;
import javax.net.ssl.TrustManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpenIDConnectionUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(OpenIDConnectionUtils.class);
    public static final String EMPTY = "";
    public static final String ID_TOKEN_KUBECONFIG = "id-token";
    public static final String ISSUER_KUBECONFIG = "idp-issuer-url";
    public static final String REFRESH_TOKEN_KUBECONFIG = "refresh-token";
    public static final String REFRESH_TOKEN_PARAM = "refresh_token";
    public static final String GRANT_TYPE_PARAM = "grant_type";
    public static final String CLIENT_ID_PARAM = "client_id";
    public static final String CLIENT_SECRET_PARAM = "client_secret";
    public static final String ID_TOKEN_PARAM = "id_token";
    public static final String ACCESS_TOKEN_PARAM = "access_token";
    public static final String CLIENT_ID_KUBECONFIG = "client-id";
    public static final String CLIENT_SECRET_KUBECONFIG = "client-secret";
    public static final String IDP_CERT_DATA = "idp-certificate-authority-data";
    public static final String TOKEN_ENDPOINT_PARAM = "token_endpoint";
    public static final String WELL_KNOWN_OPENID_CONFIGURATION = ".well-known/openid-configuration";
    public static final String GRANT_TYPE_REFRESH_TOKEN = "refresh_token";

    private OpenIDConnectionUtils() {
    }

    public static CompletableFuture<String> resolveOIDCTokenFromAuthConfig(Map<String, String> currentAuthProviderConfig, HttpClient.Builder clientBuilder) {
        String accessToken = currentAuthProviderConfig.get(ID_TOKEN_KUBECONFIG);
        String issuer = currentAuthProviderConfig.get(ISSUER_KUBECONFIG);
        String clientId = currentAuthProviderConfig.get(CLIENT_ID_KUBECONFIG);
        String refreshToken = currentAuthProviderConfig.get(REFRESH_TOKEN_KUBECONFIG);
        String clientSecret = currentAuthProviderConfig.getOrDefault(CLIENT_SECRET_KUBECONFIG, EMPTY);
        String idpCert = currentAuthProviderConfig.get(IDP_CERT_DATA);
        if (OpenIDConnectionUtils.isTokenRefreshSupported(currentAuthProviderConfig)) {
            return OpenIDConnectionUtils.getOIDCProviderTokenEndpointAndRefreshToken(issuer, clientId, refreshToken, clientSecret, accessToken, idpCert, clientBuilder);
        }
        return CompletableFuture.completedFuture(accessToken);
    }

    static CompletableFuture<String> getOIDCProviderTokenEndpointAndRefreshToken(HttpClient client, Map<String, Object> wellKnownOpenIdConfiguration, String clientId, String refreshToken, String clientSecret, String accessToken, boolean shouldPersistUpdatedTokenInKubeConfig) {
        String oidcTokenEndpoint = OpenIDConnectionUtils.getParametersFromDiscoveryResponse(wellKnownOpenIdConfiguration, TOKEN_ENDPOINT_PARAM);
        CompletableFuture<String> freshAccessToken = OpenIDConnectionUtils.refreshToken(client, oidcTokenEndpoint, clientId, refreshToken, clientSecret, shouldPersistUpdatedTokenInKubeConfig);
        return freshAccessToken.thenApply(s -> Utils.getNonNullOrElse(s, accessToken));
    }

    static boolean isTokenRefreshSupported(Map<String, String> currentAuthProviderConfig) {
        return Utils.isNotNull(currentAuthProviderConfig.get(REFRESH_TOKEN_KUBECONFIG));
    }

    static CompletableFuture<String> refreshToken(HttpClient client, String oidcTokenEndpoint, String clientId, String refreshToken, String clientSecret, boolean shouldPersistUpdatedTokenInKubeConfig) {
        CompletableFuture<Map<String, Object>> response = OpenIDConnectionUtils.refreshOidcToken(client, clientId, refreshToken, clientSecret, oidcTokenEndpoint);
        return response.thenApply(map -> {
            if (!map.containsKey(ID_TOKEN_PARAM)) {
                LOGGER.warn("token response did not contain an id_token, either the scope \\\"openid\\\" wasn't requested upon login, or the provider doesn't support id_tokens as part of the refresh response.");
                return null;
            }
            try {
                if (shouldPersistUpdatedTokenInKubeConfig && !OpenIDConnectionUtils.persistKubeConfigWithUpdatedToken(map)) {
                    LOGGER.warn("oidc: failure while persisting new tokens into KUBECONFIG");
                }
            }
            catch (IOException e) {
                LOGGER.warn("Failure in fetching refresh token: ", e);
            }
            return String.valueOf(map.get(ID_TOKEN_PARAM));
        });
    }

    static CompletableFuture<Map<String, Object>> refreshOidcToken(HttpClient client, String clientId, String refreshToken, String clientSecret, String tokenURL) {
        HttpRequest request = OpenIDConnectionUtils.getTokenRefreshHttpRequest(client, tokenURL, clientId, refreshToken, clientSecret);
        CompletableFuture<HttpResponse<String>> response = client.sendAsync(request, String.class);
        return response.thenApply(r -> {
            String body = (String)r.body();
            if (body != null) {
                if (r.isSuccessful()) {
                    try {
                        return OpenIDConnectionUtils.convertJsonStringToMap(body);
                    }
                    catch (JsonProcessingException e) {
                        LOGGER.warn("Failure in fetching refresh token: ", e);
                    }
                } else {
                    LOGGER.warn("Response: {}", (Object)body);
                }
            }
            return Collections.emptyMap();
        });
    }

    static CompletableFuture<Map<String, Object>> getOIDCDiscoveryDocumentAsMap(HttpClient client, String issuer) {
        HttpRequest request = client.newHttpRequestBuilder().uri(OpenIDConnectionUtils.getWellKnownUrlForOpenIDIssuer(issuer)).build();
        return client.sendAsync(request, String.class).thenApply(response -> {
            try {
                if (response.isSuccessful() && response.body() != null) {
                    return OpenIDConnectionUtils.convertJsonStringToMap((String)response.body());
                }
                String responseBody = (String)response.body();
                LOGGER.warn("oidc: failed to query metadata endpoint: {} {}", (Object)response.code(), (Object)responseBody);
            }
            catch (IOException e) {
                LOGGER.warn("Could not refresh OIDC token, failure in getting refresh URL", e);
            }
            return Collections.emptyMap();
        });
    }

    static String getWellKnownUrlForOpenIDIssuer(String issuer) {
        return URLUtils.join(issuer, "/", WELL_KNOWN_OPENID_CONFIGURATION);
    }

    static String getParametersFromDiscoveryResponse(Map<String, Object> responseAsJson, String key) {
        if (responseAsJson.containsKey(key)) {
            return String.valueOf(responseAsJson.get(key));
        }
        LOGGER.warn("oidc: oidc: discovery object doesn't contain a {}", (Object)key);
        return EMPTY;
    }

    static boolean persistKubeConfigWithUpdatedToken(String kubeConfigPath, Map<String, Object> updatedAuthProviderConfig) throws IOException {
        io.fabric8.kubernetes.api.model.Config config = KubeConfigUtils.parseConfig(new File(kubeConfigPath));
        NamedContext currentNamedContext = KubeConfigUtils.getCurrentContext(config);
        if (currentNamedContext != null) {
            int currentUserIndex = KubeConfigUtils.getNamedUserIndexFromConfig(config, currentNamedContext.getContext().getUser());
            Map<String, String> authProviderConfig = config.getUsers().get(currentUserIndex).getUser().getAuthProvider().getConfig();
            authProviderConfig.put(ID_TOKEN_KUBECONFIG, String.valueOf(updatedAuthProviderConfig.get(ID_TOKEN_PARAM)));
            authProviderConfig.put(REFRESH_TOKEN_KUBECONFIG, String.valueOf(updatedAuthProviderConfig.get("refresh_token")));
            config.getUsers().get(currentUserIndex).getUser().getAuthProvider().setConfig(authProviderConfig);
            try {
                KubeConfigUtils.persistKubeConfigIntoFile(config, kubeConfigPath);
                return true;
            }
            catch (IOException exception) {
                LOGGER.warn("failed to write file {}", (Object)kubeConfigPath, (Object)exception);
            }
        }
        return false;
    }

    private static Map<String, Object> convertJsonStringToMap(String jsonString) throws JsonProcessingException {
        return Serialization.jsonMapper().readValue(jsonString, Map.class);
    }

    private static HttpClient getDefaultHttpClientWithPemCert(String idpCert, HttpClient.Builder clientBuilder) {
        TrustManager[] trustManagers = null;
        KeyManager[] keyManagers = null;
        String pemCert = new String(Base64.getDecoder().decode(idpCert));
        try {
            trustManagers = SSLUtils.trustManagers(pemCert, null, false, null, null);
            keyManagers = SSLUtils.keyManagers(pemCert, null, null, null, null, null, null, null);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException | InvalidKeySpecException e) {
            throw new RuntimeException("Could not import idp certificate", e);
        }
        clientBuilder.sslContext(keyManagers, trustManagers);
        return clientBuilder.build();
    }

    private static HttpRequest getTokenRefreshHttpRequest(HttpClient client, String tokenEndpointUrl, String clientId, String refreshToken, String clientSecret) {
        HttpRequest.Builder httpRequestBuilder = client.newHttpRequestBuilder().uri(tokenEndpointUrl);
        Map<String, String> requestBody = OpenIDConnectionUtils.getRequestBodyContentForRefresh(clientId, refreshToken, clientSecret);
        String credentials = Base64.getEncoder().encodeToString((clientId + ':' + clientSecret).getBytes(StandardCharsets.UTF_8));
        httpRequestBuilder.header("Authorization", "Basic " + credentials);
        httpRequestBuilder.post(requestBody);
        return httpRequestBuilder.build();
    }

    private static Map<String, String> getRequestBodyContentForRefresh(String clientId, String refreshToken, String clientSecret) {
        LinkedHashMap<String, String> result = new LinkedHashMap<String, String>();
        result.put("refresh_token", refreshToken);
        result.put(GRANT_TYPE_PARAM, "refresh_token");
        result.put(CLIENT_ID_PARAM, clientId);
        result.put(CLIENT_SECRET_PARAM, clientSecret);
        return result;
    }

    private static CompletableFuture<String> getOIDCProviderTokenEndpointAndRefreshToken(String issuer, String clientId, String refreshToken, String clientSecret, String accessToken, String idpCert, HttpClient.Builder clientBuilder) {
        HttpClient newClient = OpenIDConnectionUtils.getDefaultHttpClientWithPemCert(idpCert, clientBuilder);
        CompletableFuture<Map<String, Object>> wellKnownOpenIdConfiguration = OpenIDConnectionUtils.getOIDCDiscoveryDocumentAsMap(newClient, issuer);
        CompletionStage result = wellKnownOpenIdConfiguration.thenCompose(config -> OpenIDConnectionUtils.getOIDCProviderTokenEndpointAndRefreshToken(newClient, config, clientId, refreshToken, clientSecret, accessToken, true));
        ((CompletableFuture)result).whenComplete((s, t) -> newClient.close());
        return result;
    }

    private static boolean persistKubeConfigWithUpdatedToken(Map<String, Object> updatedAuthProviderConfig) throws IOException {
        return OpenIDConnectionUtils.persistKubeConfigWithUpdatedToken(Config.getKubeconfigFilename(), updatedAuthProviderConfig);
    }
}

