/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.timeline;

import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.WritableComparator;
import org.apache.hadoop.service.AbstractService;
import org.apache.hadoop.shaded.org.apache.commons.collections.map.LRUMap;
import org.apache.hadoop.shaded.org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.shaded.org.fusesource.leveldbjni.JniDBFactory;
import org.apache.hadoop.shaded.org.iq80.leveldb.DB;
import org.apache.hadoop.shaded.org.iq80.leveldb.DBException;
import org.apache.hadoop.shaded.org.iq80.leveldb.DBIterator;
import org.apache.hadoop.shaded.org.iq80.leveldb.Options;
import org.apache.hadoop.shaded.org.iq80.leveldb.ReadOptions;
import org.apache.hadoop.shaded.org.iq80.leveldb.WriteBatch;
import org.apache.hadoop.shaded.org.nustaq.serialization.FSTClazzNameRegistry;
import org.apache.hadoop.shaded.org.nustaq.serialization.FSTConfiguration;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.yarn.api.records.timeline.TimelineDomain;
import org.apache.hadoop.yarn.api.records.timeline.TimelineDomains;
import org.apache.hadoop.yarn.api.records.timeline.TimelineEntities;
import org.apache.hadoop.yarn.api.records.timeline.TimelineEntity;
import org.apache.hadoop.yarn.api.records.timeline.TimelineEvent;
import org.apache.hadoop.yarn.api.records.timeline.TimelineEvents;
import org.apache.hadoop.yarn.api.records.timeline.TimelinePutResponse;
import org.apache.hadoop.yarn.proto.YarnServerCommonProtos;
import org.apache.hadoop.yarn.server.records.Version;
import org.apache.hadoop.yarn.server.records.impl.pb.VersionPBImpl;
import org.apache.hadoop.yarn.server.timeline.EntityIdentifier;
import org.apache.hadoop.yarn.server.timeline.GenericObjectMapper;
import org.apache.hadoop.yarn.server.timeline.NameValuePair;
import org.apache.hadoop.yarn.server.timeline.RollingLevelDB;
import org.apache.hadoop.yarn.server.timeline.TimelineDataManager;
import org.apache.hadoop.yarn.server.timeline.TimelineReader;
import org.apache.hadoop.yarn.server.timeline.TimelineStore;
import org.apache.hadoop.yarn.server.timeline.util.LeveldbUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class RollingLevelDBTimelineStore
extends AbstractService
implements TimelineStore {
    private static final Logger LOG = LoggerFactory.getLogger(RollingLevelDBTimelineStore.class);
    private static FSTConfiguration fstConf = FSTConfiguration.createDefaultConfiguration();
    private static FSTConfiguration fstConf224 = FSTConfiguration.createDefaultConfiguration();
    private static final int LINKED_HASH_MAP_224_CODE = 83;
    @InterfaceAudience.Private
    @VisibleForTesting
    static final String FILENAME = "leveldb-timeline-store";
    static final String DOMAIN = "domain-ldb";
    static final String ENTITY = "entity-ldb";
    static final String INDEX = "indexes-ldb";
    static final String STARTTIME = "starttime-ldb";
    static final String OWNER = "owner-ldb";
    @VisibleForTesting
    static final String BACKUP_EXT = ".backup-";
    private static final byte[] DOMAIN_ID_COLUMN;
    private static final byte[] EVENTS_COLUMN;
    private static final byte[] PRIMARY_FILTERS_COLUMN;
    private static final byte[] OTHER_INFO_COLUMN;
    private static final byte[] RELATED_ENTITIES_COLUMN;
    private static final byte[] DESCRIPTION_COLUMN;
    private static final byte[] OWNER_COLUMN;
    private static final byte[] READER_COLUMN;
    private static final byte[] WRITER_COLUMN;
    private static final byte[] TIMESTAMP_COLUMN;
    private static final byte[] EMPTY_BYTES;
    private static final String TIMELINE_STORE_VERSION_KEY = "timeline-store-version";
    private static final Version CURRENT_VERSION_INFO;
    private static long writeBatchSize;
    @InterfaceAudience.Private
    @VisibleForTesting
    static final FsPermission LEVELDB_DIR_UMASK;
    private Map<EntityIdentifier, Long> startTimeWriteCache;
    private Map<EntityIdentifier, Long> startTimeReadCache;
    private DB domaindb;
    private RollingLevelDB entitydb;
    private RollingLevelDB indexdb;
    private DB starttimedb;
    private DB ownerdb;
    private Thread deletionThread;
    private JniDBFactory factory;

    public RollingLevelDBTimelineStore() {
        super(RollingLevelDBTimelineStore.class.getName());
    }

    @VisibleForTesting
    void setFactory(JniDBFactory fact) {
        this.factory = fact;
    }

    protected void serviceInit(Configuration conf) throws Exception {
        Preconditions.checkArgument((conf.getLong("yarn.timeline-service.ttl-ms", 604800000L) > 0L ? 1 : 0) != 0, (String)"%s property value should be greater than zero", (Object)"yarn.timeline-service.ttl-ms");
        Preconditions.checkArgument((conf.getLong("yarn.timeline-service.leveldb-timeline-store.ttl-interval-ms", 300000L) > 0L ? 1 : 0) != 0, (String)"%s property value should be greater than zero", (Object)"yarn.timeline-service.leveldb-timeline-store.ttl-interval-ms");
        Preconditions.checkArgument((conf.getLong("yarn.timeline-service.leveldb-timeline-store.read-cache-size", 0x6400000L) >= 0L ? 1 : 0) != 0, (String)"%s property value should be greater than or equal to zero", (Object)"yarn.timeline-service.leveldb-timeline-store.read-cache-size");
        Preconditions.checkArgument((conf.getLong("yarn.timeline-service.leveldb-timeline-store.start-time-read-cache-size", 10000L) > 0L ? 1 : 0) != 0, (String)" %s property value should be greater than zero", (Object)"yarn.timeline-service.leveldb-timeline-store.start-time-read-cache-size");
        Preconditions.checkArgument((conf.getLong("yarn.timeline-service.leveldb-timeline-store.start-time-write-cache-size", 10000L) > 0L ? 1 : 0) != 0, (String)"%s property value should be greater than zero", (Object)"yarn.timeline-service.leveldb-timeline-store.start-time-write-cache-size");
        Preconditions.checkArgument((conf.getLong("yarn.timeline-service.leveldb-timeline-store.max-open-files", 1000L) > 0L ? 1 : 0) != 0, (String)"%s property value should be greater than zero", (Object)"yarn.timeline-service.leveldb-timeline-store.max-open-files");
        Preconditions.checkArgument((conf.getLong("yarn.timeline-service.leveldb-timeline-store.write-buffer-size", 0x1000000L) > 0L ? 1 : 0) != 0, (String)"%s property value should be greater than zero", (Object)"yarn.timeline-service.leveldb-timeline-store.write-buffer-size");
        Options options = new Options();
        options.createIfMissing(true);
        options.cacheSize(conf.getLong("yarn.timeline-service.leveldb-timeline-store.read-cache-size", 0x6400000L));
        if (this.factory == null) {
            this.factory = new JniDBFactory();
        }
        Path dbPath = new Path(conf.get("yarn.timeline-service.leveldb-timeline-store.path"), FILENAME);
        Path domainDBPath = new Path(dbPath, DOMAIN);
        Path starttimeDBPath = new Path(dbPath, STARTTIME);
        Path ownerDBPath = new Path(dbPath, OWNER);
        try (LocalFileSystem localFS = FileSystem.getLocal((Configuration)conf);){
            if (!localFS.exists(dbPath)) {
                if (!localFS.mkdirs(dbPath)) {
                    throw new IOException("Couldn't create directory for leveldb timeline store " + dbPath);
                }
                localFS.setPermission(dbPath, LEVELDB_DIR_UMASK);
            }
            if (!localFS.exists(domainDBPath)) {
                if (!localFS.mkdirs(domainDBPath)) {
                    throw new IOException("Couldn't create directory for leveldb timeline store " + domainDBPath);
                }
                localFS.setPermission(domainDBPath, LEVELDB_DIR_UMASK);
            }
            if (!localFS.exists(starttimeDBPath)) {
                if (!localFS.mkdirs(starttimeDBPath)) {
                    throw new IOException("Couldn't create directory for leveldb timeline store " + starttimeDBPath);
                }
                localFS.setPermission(starttimeDBPath, LEVELDB_DIR_UMASK);
            }
            if (!localFS.exists(ownerDBPath)) {
                if (!localFS.mkdirs(ownerDBPath)) {
                    throw new IOException("Couldn't create directory for leveldb timeline store " + ownerDBPath);
                }
                localFS.setPermission(ownerDBPath, LEVELDB_DIR_UMASK);
            }
        }
        options.maxOpenFiles(conf.getInt("yarn.timeline-service.leveldb-timeline-store.max-open-files", 1000));
        options.writeBufferSize(conf.getInt("yarn.timeline-service.leveldb-timeline-store.write-buffer-size", 0x1000000));
        LOG.info("Using leveldb path " + dbPath);
        this.domaindb = LeveldbUtils.loadOrRepairLevelDb(this.factory, domainDBPath, options);
        this.entitydb = new RollingLevelDB(ENTITY);
        this.entitydb.init(conf);
        this.indexdb = new RollingLevelDB(INDEX);
        this.indexdb.init(conf);
        this.starttimedb = LeveldbUtils.loadOrRepairLevelDb(this.factory, starttimeDBPath, options);
        this.ownerdb = LeveldbUtils.loadOrRepairLevelDb(this.factory, ownerDBPath, options);
        this.checkVersion();
        this.startTimeWriteCache = Collections.synchronizedMap(new LRUMap(RollingLevelDBTimelineStore.getStartTimeWriteCacheSize(conf)));
        this.startTimeReadCache = Collections.synchronizedMap(new LRUMap(RollingLevelDBTimelineStore.getStartTimeReadCacheSize(conf)));
        writeBatchSize = conf.getInt("yarn.timeline-service.leveldb-timeline-store.write-batch-size", 10000);
        super.serviceInit(conf);
    }

    protected void serviceStart() throws Exception {
        if (this.getConfig().getBoolean("yarn.timeline-service.ttl-enable", true)) {
            this.deletionThread = new EntityDeletionThread(this.getConfig());
            this.deletionThread.start();
        }
        super.serviceStart();
    }

    protected void serviceStop() throws Exception {
        if (this.deletionThread != null) {
            this.deletionThread.interrupt();
            LOG.info("Waiting for deletion thread to complete its current action");
            try {
                this.deletionThread.join();
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while waiting for deletion thread to complete, closing db now", (Throwable)e);
            }
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.domaindb});
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.starttimedb});
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.ownerdb});
        this.entitydb.stop();
        this.indexdb.stop();
        super.serviceStop();
    }

    @Override
    public TimelineEntity getEntity(String entityId, String entityType, EnumSet<TimelineReader.Field> fields) throws IOException {
        Long revStartTime = this.getStartTimeLong(entityId, entityType);
        if (revStartTime == null) {
            LOG.debug("Could not find start time for {} {} ", (Object)entityType, (Object)entityId);
            return null;
        }
        byte[] prefix = LeveldbUtils.KeyBuilder.newInstance().add(entityType).add(GenericObjectMapper.writeReverseOrderedLong(revStartTime)).add(entityId).getBytesForLookup();
        DB db = this.entitydb.getDBForStartTime(revStartTime);
        if (db == null) {
            LOG.debug("Could not find db for {} {} ", (Object)entityType, (Object)entityId);
            return null;
        }
        try (DBIterator iterator = db.iterator();){
            iterator.seek(prefix);
            TimelineEntity timelineEntity = RollingLevelDBTimelineStore.getEntity(entityId, entityType, revStartTime, fields, iterator, prefix, prefix.length);
            return timelineEntity;
        }
    }

    private static TimelineEntity getEntity(String entityId, String entityType, Long startTime, EnumSet<TimelineReader.Field> fields, DBIterator iterator, byte[] prefix, int prefixlen) throws IOException {
        byte[] key;
        if (fields == null) {
            fields = EnumSet.allOf(TimelineReader.Field.class);
        }
        TimelineEntity entity = new TimelineEntity();
        boolean events = false;
        boolean lastEvent = false;
        if (fields.contains((Object)TimelineReader.Field.EVENTS)) {
            events = true;
        } else if (fields.contains((Object)TimelineReader.Field.LAST_EVENT_ONLY)) {
            lastEvent = true;
        } else {
            entity.setEvents(null);
        }
        boolean relatedEntities = false;
        if (fields.contains((Object)TimelineReader.Field.RELATED_ENTITIES)) {
            relatedEntities = true;
        } else {
            entity.setRelatedEntities(null);
        }
        boolean primaryFilters = false;
        if (fields.contains((Object)TimelineReader.Field.PRIMARY_FILTERS)) {
            primaryFilters = true;
        } else {
            entity.setPrimaryFilters(null);
        }
        boolean otherInfo = false;
        if (fields.contains((Object)TimelineReader.Field.OTHER_INFO)) {
            otherInfo = true;
        } else {
            entity.setOtherInfo(null);
        }
        while (iterator.hasNext() && LeveldbUtils.prefixMatches(prefix, prefixlen, key = (byte[])iterator.peekNext().getKey())) {
            if (key.length != prefixlen) {
                if (key[prefixlen] == PRIMARY_FILTERS_COLUMN[0]) {
                    if (primaryFilters) {
                        RollingLevelDBTimelineStore.addPrimaryFilter(entity, key, prefixlen + PRIMARY_FILTERS_COLUMN.length);
                    }
                } else if (key[prefixlen] == OTHER_INFO_COLUMN[0]) {
                    if (otherInfo) {
                        Object o = null;
                        String keyStr = RollingLevelDBTimelineStore.parseRemainingKey(key, prefixlen + OTHER_INFO_COLUMN.length);
                        try {
                            o = fstConf.asObject((byte[])iterator.peekNext().getValue());
                            entity.addOtherInfo(keyStr, o);
                        }
                        catch (Exception ignore) {
                            try {
                                o = fstConf224.asObject((byte[])iterator.peekNext().getValue());
                                entity.addOtherInfo(keyStr, o);
                            }
                            catch (Exception e) {
                                LOG.warn("Error while decoding " + entityId + ":otherInfo:" + keyStr, (Throwable)e);
                            }
                        }
                    }
                } else if (key[prefixlen] == RELATED_ENTITIES_COLUMN[0]) {
                    if (relatedEntities) {
                        RollingLevelDBTimelineStore.addRelatedEntity(entity, key, prefixlen + RELATED_ENTITIES_COLUMN.length);
                    }
                } else if (key[prefixlen] == EVENTS_COLUMN[0]) {
                    TimelineEvent event;
                    if ((events || lastEvent && entity.getEvents().size() == 0) && (event = RollingLevelDBTimelineStore.getEntityEvent(null, key, prefixlen + EVENTS_COLUMN.length, (byte[])iterator.peekNext().getValue())) != null) {
                        entity.addEvent(event);
                    }
                } else if (key[prefixlen] == DOMAIN_ID_COLUMN[0]) {
                    byte[] v = (byte[])iterator.peekNext().getValue();
                    String domainId = new String(v, StandardCharsets.UTF_8);
                    entity.setDomainId(domainId);
                } else {
                    LOG.warn(String.format("Found unexpected column for entity %s of type %s (0x%02x)", entityId, entityType, key[prefixlen]));
                }
            }
            iterator.next();
        }
        entity.setEntityId(entityId);
        entity.setEntityType(entityType);
        entity.setStartTime(startTime);
        return entity;
    }

    @Override
    public TimelineEvents getEntityTimelines(String entityType, SortedSet<String> entityIds, Long limit, Long windowStart, Long windowEnd, Set<String> eventType) throws IOException {
        TimelineEvents events = new TimelineEvents();
        if (entityIds == null || entityIds.isEmpty()) {
            return events;
        }
        TreeMap<byte[], ArrayList<EntityIdentifier>> startTimeMap = new TreeMap<byte[], ArrayList<EntityIdentifier>>(new Comparator<byte[]>(){

            @Override
            public int compare(byte[] o1, byte[] o2) {
                return WritableComparator.compareBytes((byte[])o1, (int)0, (int)o1.length, (byte[])o2, (int)0, (int)o2.length);
            }
        });
        for (String string : entityIds) {
            byte[] startTime = this.getStartTime(string, entityType);
            if (startTime == null) continue;
            ArrayList<EntityIdentifier> entities = (ArrayList<EntityIdentifier>)startTimeMap.get(startTime);
            if (entities == null) {
                entities = new ArrayList<EntityIdentifier>();
                startTimeMap.put(startTime, entities);
            }
            entities.add(new EntityIdentifier(string, entityType));
        }
        for (Map.Entry entry : startTimeMap.entrySet()) {
            byte[] revStartTime = (byte[])entry.getKey();
            for (EntityIdentifier entityIdentifier : (List)entry.getValue()) {
                DB db;
                TimelineEvents.EventsOfOneEntity entity = new TimelineEvents.EventsOfOneEntity();
                entity.setEntityId(entityIdentifier.getId());
                entity.setEntityType(entityType);
                events.addEvent(entity);
                LeveldbUtils.KeyBuilder kb = LeveldbUtils.KeyBuilder.newInstance().add(entityType).add(revStartTime).add(entityIdentifier.getId()).add(EVENTS_COLUMN);
                byte[] prefix = kb.getBytesForLookup();
                if (windowEnd == null) {
                    windowEnd = Long.MAX_VALUE;
                }
                byte[] revts = GenericObjectMapper.writeReverseOrderedLong(windowEnd);
                kb.add(revts);
                byte[] first = kb.getBytesForLookup();
                byte[] last = null;
                if (windowStart != null) {
                    last = LeveldbUtils.KeyBuilder.newInstance().add(prefix).add(GenericObjectMapper.writeReverseOrderedLong(windowStart)).getBytesForLookup();
                }
                if (limit == null) {
                    limit = 100L;
                }
                if ((db = this.entitydb.getDBForStartTime(GenericObjectMapper.readReverseOrderedLong(revStartTime, 0))) == null) continue;
                DBIterator iterator = db.iterator();
                Throwable throwable = null;
                try {
                    byte[] key;
                    iterator.seek(first);
                    while ((long)entity.getEvents().size() < limit && iterator.hasNext() && LeveldbUtils.prefixMatches(prefix, prefix.length, key = (byte[])iterator.peekNext().getKey()) && (last == null || WritableComparator.compareBytes((byte[])key, (int)0, (int)key.length, (byte[])last, (int)0, (int)last.length) <= 0)) {
                        TimelineEvent event = RollingLevelDBTimelineStore.getEntityEvent(eventType, key, prefix.length, (byte[])iterator.peekNext().getValue());
                        if (event != null) {
                            entity.addEvent(event);
                        }
                        iterator.next();
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (iterator == null) continue;
                    if (throwable != null) {
                        try {
                            iterator.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    iterator.close();
                }
            }
        }
        return events;
    }

    @Override
    public TimelineEntities getEntities(String entityType, Long limit, Long windowStart, Long windowEnd, String fromId, Long fromTs, NameValuePair primaryFilter, Collection<NameValuePair> secondaryFilters, EnumSet<TimelineReader.Field> fields, TimelineDataManager.CheckAcl checkAcl) throws IOException {
        if (primaryFilter == null) {
            return this.getEntityByTime(EMPTY_BYTES, entityType, limit, windowStart, windowEnd, fromId, fromTs, secondaryFilters, fields, checkAcl, false);
        }
        byte[] base = LeveldbUtils.KeyBuilder.newInstance().add(primaryFilter.getName()).add(fstConf.asByteArray(primaryFilter.getValue()), true).getBytesForLookup();
        return this.getEntityByTime(base, entityType, limit, windowStart, windowEnd, fromId, fromTs, secondaryFilters, fields, checkAcl, true);
    }

    private TimelineEntities getEntityByTime(byte[] base, String entityType, Long limit, Long starttime, Long endtime, String fromId, Long fromTs, Collection<NameValuePair> secondaryFilters, EnumSet<TimelineReader.Field> fields, TimelineDataManager.CheckAcl checkAcl, boolean usingPrimaryFilter) throws IOException {
        LeveldbUtils.KeyBuilder kb = LeveldbUtils.KeyBuilder.newInstance().add(base).add(entityType);
        byte[] prefix = kb.getBytesForLookup();
        if (endtime == null) {
            endtime = Long.MAX_VALUE;
        }
        if (fields == null) {
            fields = EnumSet.allOf(TimelineReader.Field.class);
        }
        long firstStartTime = Long.MAX_VALUE;
        byte[] first = null;
        if (fromId != null) {
            Long fromIdStartTime = this.getStartTimeLong(fromId, entityType);
            if (fromIdStartTime == null) {
                return new TimelineEntities();
            }
            if (fromIdStartTime <= endtime) {
                firstStartTime = fromIdStartTime;
                first = kb.add(GenericObjectMapper.writeReverseOrderedLong(fromIdStartTime)).add(fromId).getBytesForLookup();
            }
        }
        if (first == null) {
            firstStartTime = endtime;
            first = kb.add(GenericObjectMapper.writeReverseOrderedLong(endtime)).getBytesForLookup();
        }
        byte[] last = null;
        if (starttime != null) {
            last = LeveldbUtils.KeyBuilder.newInstance().add(base).add(entityType).add(GenericObjectMapper.writeReverseOrderedLong(starttime)).getBytesForLookup();
        }
        if (limit == null) {
            limit = 100L;
        }
        TimelineEntities entities = new TimelineEntities();
        RollingLevelDB rollingdb = null;
        rollingdb = usingPrimaryFilter ? this.indexdb : this.entitydb;
        DB db = rollingdb.getDBForStartTime(firstStartTime);
        while ((long)entities.getEntities().size() < limit && db != null) {
            DBIterator iterator = db.iterator();
            Throwable throwable = null;
            try {
                byte[] key;
                iterator.seek(first);
                block10: while ((long)entities.getEntities().size() < limit && iterator.hasNext() && LeveldbUtils.prefixMatches(prefix, prefix.length, key = (byte[])iterator.peekNext().getKey()) && (last == null || WritableComparator.compareBytes((byte[])key, (int)0, (int)key.length, (byte[])last, (int)0, (int)last.length) <= 0)) {
                    long insertTime;
                    LeveldbUtils.KeyParser kp = new LeveldbUtils.KeyParser(key, prefix.length);
                    Long startTime = kp.getNextLong();
                    String entityId = kp.getNextString();
                    if (fromTs != null && (insertTime = GenericObjectMapper.readReverseOrderedLong((byte[])iterator.peekNext().getValue(), 0)) > fromTs) {
                        byte[] firstKey = key;
                        while (iterator.hasNext()) {
                            key = (byte[])iterator.peekNext().getKey();
                            iterator.next();
                            if (LeveldbUtils.prefixMatches(firstKey, kp.getOffset(), key)) continue;
                            continue block10;
                        }
                        continue;
                    }
                    EnumSet<TimelineReader.Field> queryFields = EnumSet.copyOf(fields);
                    boolean addPrimaryFilters = false;
                    boolean addOtherInfo = false;
                    if (secondaryFilters != null && secondaryFilters.size() > 0) {
                        if (!queryFields.contains((Object)TimelineReader.Field.PRIMARY_FILTERS)) {
                            queryFields.add(TimelineReader.Field.PRIMARY_FILTERS);
                            addPrimaryFilters = true;
                        }
                        if (!queryFields.contains((Object)TimelineReader.Field.OTHER_INFO)) {
                            queryFields.add(TimelineReader.Field.OTHER_INFO);
                            addOtherInfo = true;
                        }
                    }
                    TimelineEntity entity = null;
                    if (usingPrimaryFilter) {
                        entity = this.getEntity(entityId, entityType, queryFields);
                        iterator.next();
                    } else {
                        entity = RollingLevelDBTimelineStore.getEntity(entityId, entityType, startTime, queryFields, iterator, key, kp.getOffset());
                    }
                    if (entity == null) continue;
                    boolean filterPassed = true;
                    if (secondaryFilters != null) {
                        for (NameValuePair filter : secondaryFilters) {
                            Object v = entity.getOtherInfo().get(filter.getName());
                            if (v == null) {
                                Set vs = (Set)entity.getPrimaryFilters().get(filter.getName());
                                if (vs != null && vs.contains(filter.getValue())) continue;
                                filterPassed = false;
                                break;
                            }
                            if (v.equals(filter.getValue())) continue;
                            filterPassed = false;
                            break;
                        }
                    }
                    if (!filterPassed) continue;
                    if (entity.getDomainId() == null) {
                        entity.setDomainId("DEFAULT");
                    }
                    if (checkAcl != null && !checkAcl.check(entity)) continue;
                    if (addPrimaryFilters) {
                        entity.setPrimaryFilters(null);
                    }
                    if (addOtherInfo) {
                        entity.setOtherInfo(null);
                    }
                    entities.addEntity(entity);
                }
                db = rollingdb.getPreviousDB(db);
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (iterator == null) continue;
                if (throwable != null) {
                    try {
                        iterator.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                iterator.close();
            }
        }
        return entities;
    }

    /*
     * WARNING - void declaration
     */
    private long putEntities(TreeMap<Long, RollingLevelDB.RollingWriteBatch> entityUpdates, TreeMap<Long, RollingLevelDB.RollingWriteBatch> indexUpdates, TimelineEntity entity, TimelinePutResponse response) {
        long putCount = 0L;
        ArrayList<EntityIdentifier> relatedEntitiesWithoutStartTimes = new ArrayList<EntityIdentifier>();
        byte[] revStartTime = null;
        Map primaryFilters = null;
        try {
            void var23_41;
            DB dB;
            RollingLevelDB.RollingWriteBatch rollingWriteBatch;
            Map map;
            Map otherInfo;
            byte[] value;
            DB db2;
            List events = entity.getEvents();
            Long startTime = this.getAndSetStartTime(entity.getEntityId(), entity.getEntityType(), entity.getStartTime(), events);
            if (startTime == null) {
                TimelinePutResponse.TimelinePutError error = new TimelinePutResponse.TimelinePutError();
                error.setEntityId(entity.getEntityId());
                error.setEntityType(entity.getEntityType());
                error.setErrorCode(1);
                response.addError(error);
                return putCount;
            }
            if (StringUtils.isEmpty((CharSequence)entity.getDomainId())) {
                TimelinePutResponse.TimelinePutError error = new TimelinePutResponse.TimelinePutError();
                error.setEntityId(entity.getEntityId());
                error.setEntityType(entity.getEntityType());
                error.setErrorCode(5);
                response.addError(error);
                return putCount;
            }
            revStartTime = GenericObjectMapper.writeReverseOrderedLong(startTime);
            long roundedStartTime = this.entitydb.computeCurrentCheckMillis(startTime);
            RollingLevelDB.RollingWriteBatch rollingWriteBatch2 = entityUpdates.get(roundedStartTime);
            if (rollingWriteBatch2 == null && (db2 = this.entitydb.getDBForStartTime(startTime)) != null) {
                WriteBatch writeBatch = db2.createWriteBatch();
                rollingWriteBatch2 = new RollingLevelDB.RollingWriteBatch(db2, writeBatch);
                entityUpdates.put(roundedStartTime, rollingWriteBatch2);
            }
            if (rollingWriteBatch2 == null) {
                TimelinePutResponse.TimelinePutError error = new TimelinePutResponse.TimelinePutError();
                error.setEntityId(entity.getEntityId());
                error.setEntityType(entity.getEntityType());
                error.setErrorCode(7);
                response.addError(error);
                return putCount;
            }
            WriteBatch writeBatch = rollingWriteBatch2.getWriteBatch();
            byte[] entityIdBytes = entity.getEntityId().getBytes(StandardCharsets.UTF_8);
            byte[] entityTypeBytes = entity.getEntityType().getBytes(StandardCharsets.UTF_8);
            byte[] domainIdBytes = entity.getDomainId().getBytes(StandardCharsets.UTF_8);
            byte[] markerKey = LeveldbUtils.KeyBuilder.newInstance(3).add(entityTypeBytes, true).add(revStartTime).add(entityIdBytes, true).getBytesForLookup();
            writeBatch.put(markerKey, EMPTY_BYTES);
            ++putCount;
            byte[] domainkey = LeveldbUtils.KeyBuilder.newInstance(4).add(entityTypeBytes, true).add(revStartTime).add(entityIdBytes, true).add(DOMAIN_ID_COLUMN).getBytes();
            writeBatch.put(domainkey, domainIdBytes);
            ++putCount;
            if (events != null) {
                for (TimelineEvent timelineEvent : events) {
                    byte[] byArray = GenericObjectMapper.writeReverseOrderedLong(timelineEvent.getTimestamp());
                    byte[] byArray2 = LeveldbUtils.KeyBuilder.newInstance().add(entityTypeBytes, true).add(revStartTime).add(entityIdBytes, true).add(EVENTS_COLUMN).add(byArray).add(timelineEvent.getEventType().getBytes(StandardCharsets.UTF_8)).getBytes();
                    value = fstConf.asByteArray(timelineEvent.getEventInfo());
                    writeBatch.put(byArray2, value);
                    ++putCount;
                }
            }
            if ((primaryFilters = entity.getPrimaryFilters()) != null) {
                for (Map.Entry entry : primaryFilters.entrySet()) {
                    for (Object e : (Set)entry.getValue()) {
                        byte[] key2 = LeveldbUtils.KeyBuilder.newInstance(6).add(entityTypeBytes, true).add(revStartTime).add(entityIdBytes, true).add(PRIMARY_FILTERS_COLUMN).add((String)entry.getKey()).add(fstConf.asByteArray(e)).getBytes();
                        writeBatch.put(key2, EMPTY_BYTES);
                        ++putCount;
                    }
                }
            }
            if ((otherInfo = entity.getOtherInfo()) != null) {
                for (Map.Entry entry : otherInfo.entrySet()) {
                    byte[] byArray = LeveldbUtils.KeyBuilder.newInstance(5).add(entityTypeBytes, true).add(revStartTime).add(entityIdBytes, true).add(OTHER_INFO_COLUMN).add((String)entry.getKey()).getBytes();
                    value = fstConf.asByteArray(entry.getValue());
                    writeBatch.put(byArray, value);
                    ++putCount;
                }
            }
            if ((map = entity.getRelatedEntities()) != null) {
                for (Map.Entry entry : map.entrySet()) {
                    String relatedEntityType = (String)entry.getKey();
                    for (String relatedEntityId : (Set)entry.getValue()) {
                        DB db3;
                        Long relatedStartTimeLong = this.getStartTimeLong(relatedEntityId, relatedEntityType);
                        if (relatedStartTimeLong == null) {
                            relatedEntitiesWithoutStartTimes.add(new EntityIdentifier(relatedEntityId, relatedEntityType));
                            continue;
                        }
                        byte[] relatedEntityStartTime = GenericObjectMapper.writeReverseOrderedLong(relatedStartTimeLong);
                        long relatedRoundedStartTime = this.entitydb.computeCurrentCheckMillis(relatedStartTimeLong);
                        RollingLevelDB.RollingWriteBatch relatedRollingWriteBatch = entityUpdates.get(relatedRoundedStartTime);
                        if (relatedRollingWriteBatch == null && (db3 = this.entitydb.getDBForStartTime(relatedStartTimeLong)) != null) {
                            WriteBatch relatedWriteBatch = db3.createWriteBatch();
                            relatedRollingWriteBatch = new RollingLevelDB.RollingWriteBatch(db3, relatedWriteBatch);
                            entityUpdates.put(relatedRoundedStartTime, relatedRollingWriteBatch);
                        }
                        if (relatedRollingWriteBatch == null) {
                            TimelinePutResponse.TimelinePutError error = new TimelinePutResponse.TimelinePutError();
                            error.setEntityId(entity.getEntityId());
                            error.setEntityType(entity.getEntityType());
                            error.setErrorCode(7);
                            response.addError(error);
                            continue;
                        }
                        byte[] relatedDomainIdBytes = relatedRollingWriteBatch.getDB().get(RollingLevelDBTimelineStore.createDomainIdKey(relatedEntityId, relatedEntityType, relatedEntityStartTime));
                        String domainId = null;
                        domainId = relatedDomainIdBytes == null ? "DEFAULT" : new String(relatedDomainIdBytes, StandardCharsets.UTF_8);
                        if (!domainId.equals(entity.getDomainId())) {
                            TimelinePutResponse.TimelinePutError error = new TimelinePutResponse.TimelinePutError();
                            error.setEntityId(entity.getEntityId());
                            error.setEntityType(entity.getEntityType());
                            error.setErrorCode(6);
                            response.addError(error);
                            continue;
                        }
                        byte[] key3 = RollingLevelDBTimelineStore.createRelatedEntityKey(relatedEntityId, relatedEntityType, relatedEntityStartTime, entity.getEntityId(), entity.getEntityType());
                        WriteBatch relatedWriteBatch = relatedRollingWriteBatch.getWriteBatch();
                        relatedWriteBatch.put(key3, EMPTY_BYTES);
                        ++putCount;
                    }
                }
            }
            if ((rollingWriteBatch = indexUpdates.get(roundedStartTime)) == null && (dB = this.indexdb.getDBForStartTime(startTime)) != null) {
                WriteBatch indexWriteBatch = dB.createWriteBatch();
                RollingLevelDB.RollingWriteBatch rollingWriteBatch3 = new RollingLevelDB.RollingWriteBatch(dB, indexWriteBatch);
                indexUpdates.put(roundedStartTime, rollingWriteBatch3);
            }
            if (var23_41 == null) {
                TimelinePutResponse.TimelinePutError timelinePutError = new TimelinePutResponse.TimelinePutError();
                timelinePutError.setEntityId(entity.getEntityId());
                timelinePutError.setEntityType(entity.getEntityType());
                timelinePutError.setErrorCode(7);
                response.addError(timelinePutError);
                return putCount;
            }
            WriteBatch writeBatch2 = var23_41.getWriteBatch();
            putCount += RollingLevelDBTimelineStore.writePrimaryFilterEntries(writeBatch2, primaryFilters, markerKey, EMPTY_BYTES);
        }
        catch (IOException e) {
            LOG.error("Error putting entity " + entity.getEntityId() + " of type " + entity.getEntityType(), (Throwable)e);
            TimelinePutResponse.TimelinePutError error = new TimelinePutResponse.TimelinePutError();
            error.setEntityId(entity.getEntityId());
            error.setEntityType(entity.getEntityType());
            error.setErrorCode(2);
            response.addError(error);
        }
        for (EntityIdentifier relatedEntity : relatedEntitiesWithoutStartTimes) {
            try {
                DB db;
                Long relatedEntityStartAndInsertTime = this.getAndSetStartTime(relatedEntity.getId(), relatedEntity.getType(), GenericObjectMapper.readReverseOrderedLong(revStartTime, 0), null);
                if (relatedEntityStartAndInsertTime == null) {
                    throw new IOException("Error setting start time for related entity");
                }
                long relatedStartTimeLong = relatedEntityStartAndInsertTime;
                long relatedRoundedStartTime = this.entitydb.computeCurrentCheckMillis(relatedStartTimeLong);
                RollingLevelDB.RollingWriteBatch relatedRollingWriteBatch = entityUpdates.get(relatedRoundedStartTime);
                if (relatedRollingWriteBatch == null && (db = this.entitydb.getDBForStartTime(relatedStartTimeLong)) != null) {
                    WriteBatch relatedWriteBatch = db.createWriteBatch();
                    relatedRollingWriteBatch = new RollingLevelDB.RollingWriteBatch(db, relatedWriteBatch);
                    entityUpdates.put(relatedRoundedStartTime, relatedRollingWriteBatch);
                }
                if (relatedRollingWriteBatch == null) {
                    TimelinePutResponse.TimelinePutError error = new TimelinePutResponse.TimelinePutError();
                    error.setEntityId(entity.getEntityId());
                    error.setEntityType(entity.getEntityType());
                    error.setErrorCode(7);
                    response.addError(error);
                    continue;
                }
                WriteBatch relatedWriteBatch = relatedRollingWriteBatch.getWriteBatch();
                byte[] relatedEntityStartTime = GenericObjectMapper.writeReverseOrderedLong(relatedEntityStartAndInsertTime);
                byte[] key = RollingLevelDBTimelineStore.createDomainIdKey(relatedEntity.getId(), relatedEntity.getType(), relatedEntityStartTime);
                relatedWriteBatch.put(key, entity.getDomainId().getBytes(StandardCharsets.UTF_8));
                ++putCount;
                relatedWriteBatch.put(RollingLevelDBTimelineStore.createRelatedEntityKey(relatedEntity.getId(), relatedEntity.getType(), relatedEntityStartTime, entity.getEntityId(), entity.getEntityType()), EMPTY_BYTES);
                ++putCount;
                relatedWriteBatch.put(RollingLevelDBTimelineStore.createEntityMarkerKey(relatedEntity.getId(), relatedEntity.getType(), relatedEntityStartTime), EMPTY_BYTES);
                ++putCount;
            }
            catch (IOException e) {
                LOG.error("Error putting related entity " + relatedEntity.getId() + " of type " + relatedEntity.getType() + " for entity " + entity.getEntityId() + " of type " + entity.getEntityType(), (Throwable)e);
                TimelinePutResponse.TimelinePutError error = new TimelinePutResponse.TimelinePutError();
                error.setEntityId(entity.getEntityId());
                error.setEntityType(entity.getEntityType());
                error.setErrorCode(2);
                response.addError(error);
            }
        }
        return putCount;
    }

    private static long writePrimaryFilterEntries(WriteBatch writeBatch, Map<String, Set<Object>> primaryFilters, byte[] key, byte[] value) throws IOException {
        long putCount = 0L;
        if (primaryFilters != null) {
            for (Map.Entry<String, Set<Object>> pf : primaryFilters.entrySet()) {
                for (Object pfval : pf.getValue()) {
                    writeBatch.put(RollingLevelDBTimelineStore.addPrimaryFilterToKey(pf.getKey(), pfval, key), value);
                    ++putCount;
                }
            }
        }
        return putCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public TimelinePutResponse put(TimelineEntities entities) {
        LOG.debug("Starting put");
        TimelinePutResponse response = new TimelinePutResponse();
        TreeMap<Long, RollingLevelDB.RollingWriteBatch> entityUpdates = new TreeMap<Long, RollingLevelDB.RollingWriteBatch>();
        TreeMap<Long, RollingLevelDB.RollingWriteBatch> indexUpdates = new TreeMap<Long, RollingLevelDB.RollingWriteBatch>();
        long entityCount = 0L;
        long indexCount = 0L;
        try {
            for (TimelineEntity entity : entities.getEntities()) {
                entityCount += this.putEntities(entityUpdates, indexUpdates, entity, response);
            }
            for (RollingLevelDB.RollingWriteBatch entityUpdate : entityUpdates.values()) {
                entityUpdate.write();
            }
            for (RollingLevelDB.RollingWriteBatch indexUpdate : indexUpdates.values()) {
                indexUpdate.write();
            }
        }
        finally {
            for (RollingLevelDB.RollingWriteBatch entityRollingWriteBatch : entityUpdates.values()) {
                entityRollingWriteBatch.close();
            }
            for (RollingLevelDB.RollingWriteBatch indexRollingWriteBatch : indexUpdates.values()) {
                indexRollingWriteBatch.close();
            }
        }
        LOG.debug("Put {} new leveldb entity entries and {} new leveldb index entries from {} timeline entities", new Object[]{entityCount, indexCount, entities.getEntities().size()});
        return response;
    }

    private byte[] getStartTime(String entityId, String entityType) throws IOException {
        Long l = this.getStartTimeLong(entityId, entityType);
        return l == null ? null : GenericObjectMapper.writeReverseOrderedLong(l);
    }

    private Long getStartTimeLong(String entityId, String entityType) throws IOException {
        EntityIdentifier entity = new EntityIdentifier(entityId, entityType);
        if (this.startTimeReadCache.containsKey(entity)) {
            return this.startTimeReadCache.get(entity);
        }
        byte[] b = RollingLevelDBTimelineStore.createStartTimeLookupKey(entity.getId(), entity.getType());
        byte[] v = this.starttimedb.get(b);
        if (v == null) {
            return null;
        }
        Long l = GenericObjectMapper.readReverseOrderedLong(v, 0);
        this.startTimeReadCache.put(entity, l);
        return l;
    }

    private Long getAndSetStartTime(String entityId, String entityType, Long startTime, List<TimelineEvent> events) throws IOException {
        EntityIdentifier entity = new EntityIdentifier(entityId, entityType);
        Long time = this.startTimeWriteCache.get(entity);
        if (time != null) {
            return time;
        }
        if (startTime == null && events != null) {
            startTime = Long.MAX_VALUE;
            for (TimelineEvent e : events) {
                if (e.getTimestamp() >= startTime) continue;
                startTime = e.getTimestamp();
            }
        }
        return this.checkStartTimeInDb(entity, startTime);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Long checkStartTimeInDb(EntityIdentifier entity, Long suggestedStartTime) throws IOException {
        Long startAndInsertTime = null;
        byte[] b = RollingLevelDBTimelineStore.createStartTimeLookupKey(entity.getId(), entity.getType());
        RollingLevelDBTimelineStore rollingLevelDBTimelineStore = this;
        synchronized (rollingLevelDBTimelineStore) {
            byte[] v = this.starttimedb.get(b);
            if (v == null) {
                if (suggestedStartTime == null) {
                    return null;
                }
                startAndInsertTime = suggestedStartTime;
                this.starttimedb.put(b, GenericObjectMapper.writeReverseOrderedLong(suggestedStartTime));
            } else {
                startAndInsertTime = GenericObjectMapper.readReverseOrderedLong(v, 0);
            }
        }
        this.startTimeWriteCache.put(entity, startAndInsertTime);
        this.startTimeReadCache.put(entity, startAndInsertTime);
        return startAndInsertTime;
    }

    private static byte[] createStartTimeLookupKey(String entityId, String entityType) throws IOException {
        return LeveldbUtils.KeyBuilder.newInstance().add(entityType).add(entityId).getBytes();
    }

    private static byte[] createEntityMarkerKey(String entityId, String entityType, byte[] revStartTime) throws IOException {
        return LeveldbUtils.KeyBuilder.newInstance().add(entityType).add(revStartTime).add(entityId).getBytesForLookup();
    }

    private static byte[] addPrimaryFilterToKey(String primaryFilterName, Object primaryFilterValue, byte[] key) throws IOException {
        return LeveldbUtils.KeyBuilder.newInstance().add(primaryFilterName).add(fstConf.asByteArray(primaryFilterValue), true).add(key).getBytes();
    }

    private static TimelineEvent getEntityEvent(Set<String> eventTypes, byte[] key, int offset, byte[] value) throws IOException {
        LeveldbUtils.KeyParser kp = new LeveldbUtils.KeyParser(key, offset);
        long ts = kp.getNextLong();
        String tstype = kp.getNextString();
        if (eventTypes == null || eventTypes.contains(tstype)) {
            TimelineEvent event = new TimelineEvent();
            event.setTimestamp(ts);
            event.setEventType(tstype);
            Object o = null;
            try {
                o = fstConf.asObject(value);
            }
            catch (Exception ignore) {
                try {
                    o = fstConf224.asObject(value);
                }
                catch (Exception e) {
                    LOG.warn("Error while decoding " + tstype, (Throwable)e);
                }
            }
            if (o == null) {
                event.setEventInfo(null);
            } else if (o instanceof Map) {
                Map m = (Map)o;
                event.setEventInfo(m);
            } else {
                throw new IOException("Couldn't deserialize event info map");
            }
            return event;
        }
        return null;
    }

    private static void addPrimaryFilter(TimelineEntity entity, byte[] key, int offset) throws IOException {
        LeveldbUtils.KeyParser kp = new LeveldbUtils.KeyParser(key, offset);
        String name = kp.getNextString();
        byte[] bytes = kp.getRemainingBytes();
        Object value = null;
        try {
            value = fstConf.asObject(bytes);
            entity.addPrimaryFilter(name, value);
        }
        catch (Exception ignore) {
            try {
                value = fstConf224.asObject(bytes);
                entity.addPrimaryFilter(name, value);
            }
            catch (Exception e) {
                LOG.warn("Error while decoding " + name, (Throwable)e);
            }
        }
    }

    private static String parseRemainingKey(byte[] b, int offset) {
        return new String(b, offset, b.length - offset, StandardCharsets.UTF_8);
    }

    private static byte[] createRelatedEntityKey(String entityId, String entityType, byte[] revStartTime, String relatedEntityId, String relatedEntityType) throws IOException {
        return LeveldbUtils.KeyBuilder.newInstance().add(entityType).add(revStartTime).add(entityId).add(RELATED_ENTITIES_COLUMN).add(relatedEntityType).add(relatedEntityId).getBytes();
    }

    private static void addRelatedEntity(TimelineEntity entity, byte[] key, int offset) throws IOException {
        LeveldbUtils.KeyParser kp = new LeveldbUtils.KeyParser(key, offset);
        String type = kp.getNextString();
        String id = kp.getNextString();
        entity.addRelatedEntity(type, id);
    }

    private static byte[] createDomainIdKey(String entityId, String entityType, byte[] revStartTime) throws IOException {
        return LeveldbUtils.KeyBuilder.newInstance().add(entityType).add(revStartTime).add(entityId).add(DOMAIN_ID_COLUMN).getBytes();
    }

    @VisibleForTesting
    void clearStartTimeCache() {
        this.startTimeWriteCache.clear();
        this.startTimeReadCache.clear();
    }

    @VisibleForTesting
    static int getStartTimeReadCacheSize(Configuration conf) {
        return conf.getInt("yarn.timeline-service.leveldb-timeline-store.start-time-read-cache-size", 10000);
    }

    @VisibleForTesting
    static int getStartTimeWriteCacheSize(Configuration conf) {
        return conf.getInt("yarn.timeline-service.leveldb-timeline-store.start-time-write-cache-size", 10000);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    long evictOldStartTimes(long minStartTime) throws IOException {
        LOG.info("Searching for start times to evict earlier than " + minStartTime);
        long batchSize = 0L;
        long totalCount = 0L;
        long startTimesCount = 0L;
        WriteBatch writeBatch = null;
        ReadOptions readOptions = new ReadOptions();
        readOptions.fillCache(false);
        try (DBIterator iterator = this.starttimedb.iterator(readOptions);){
            iterator.seekToFirst();
            writeBatch = this.starttimedb.createWriteBatch();
            while (iterator.hasNext()) {
                Map.Entry current = (Map.Entry)iterator.next();
                byte[] entityKey = (byte[])current.getKey();
                byte[] entityValue = (byte[])current.getValue();
                long startTime = GenericObjectMapper.readReverseOrderedLong(entityValue, 0);
                if (startTime < minStartTime) {
                    ++startTimesCount;
                    writeBatch.delete(entityKey);
                    if (++batchSize >= writeBatchSize) {
                        LOG.debug("Preparing to delete a batch of {} old start times", (Object)batchSize);
                        this.starttimedb.write(writeBatch);
                        LOG.debug("Deleted batch of {}. Total start times deleted so far this cycle: {}", (Object)batchSize, (Object)startTimesCount);
                        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{writeBatch});
                        writeBatch = this.starttimedb.createWriteBatch();
                        batchSize = 0L;
                    }
                }
                ++totalCount;
            }
            LOG.debug("Preparing to delete a batch of {} old start times", (Object)batchSize);
            this.starttimedb.write(writeBatch);
            LOG.debug("Deleted batch of {}. Total start times deleted so far this cycle: {}", (Object)batchSize, (Object)startTimesCount);
            LOG.info("Deleted " + startTimesCount + "/" + totalCount + " start time entities earlier than " + minStartTime);
        }
        catch (Throwable throwable) {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{writeBatch});
            throw throwable;
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{writeBatch});
        return startTimesCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void discardOldEntities(long timestamp) throws IOException, InterruptedException {
        long totalCount = 0L;
        long t1 = System.currentTimeMillis();
        try {
            totalCount += this.evictOldStartTimes(timestamp);
            this.indexdb.evictOldDBs();
            this.entitydb.evictOldDBs();
        }
        finally {
            long t2 = System.currentTimeMillis();
            LOG.info("Discarded " + totalCount + " entities for timestamp " + timestamp + " and earlier in " + (double)(t2 - t1) / 1000.0 + " seconds");
        }
    }

    Version loadVersion() throws IOException {
        byte[] data = this.starttimedb.get(JniDBFactory.bytes((String)TIMELINE_STORE_VERSION_KEY));
        if (data == null || data.length == 0) {
            return Version.newInstance(1, 0);
        }
        VersionPBImpl version = new VersionPBImpl(YarnServerCommonProtos.VersionProto.parseFrom(data));
        return version;
    }

    @VisibleForTesting
    void storeVersion(Version state) throws IOException {
        this.dbStoreVersion(state);
    }

    private void dbStoreVersion(Version state) throws IOException {
        String key = TIMELINE_STORE_VERSION_KEY;
        byte[] data = ((VersionPBImpl)state).getProto().toByteArray();
        try {
            this.starttimedb.put(JniDBFactory.bytes((String)key), data);
        }
        catch (DBException e) {
            throw new IOException(e);
        }
    }

    Version getCurrentVersion() {
        return CURRENT_VERSION_INFO;
    }

    private void checkVersion() throws IOException {
        Version loadedVersion = this.loadVersion();
        LOG.info("Loaded timeline store version info " + loadedVersion);
        if (loadedVersion.equals(this.getCurrentVersion())) {
            return;
        }
        if (!loadedVersion.isCompatibleTo(this.getCurrentVersion())) {
            String incompatibleMessage = "Incompatible version for timeline store: expecting version " + this.getCurrentVersion() + ", but loading version " + loadedVersion;
            LOG.error(incompatibleMessage);
            throw new IOException(incompatibleMessage);
        }
        LOG.info("Storing timeline store version info " + this.getCurrentVersion());
        this.dbStoreVersion(CURRENT_VERSION_INFO);
    }

    @Override
    public void put(TimelineDomain domain) throws IOException {
        try (WriteBatch domainWriteBatch = this.domaindb.createWriteBatch();
             WriteBatch ownerWriteBatch = this.ownerdb.createWriteBatch();){
            if (domain.getId() == null || domain.getId().length() == 0) {
                throw new IllegalArgumentException("Domain doesn't have an ID");
            }
            if (domain.getOwner() == null || domain.getOwner().length() == 0) {
                throw new IllegalArgumentException("Domain doesn't have an owner.");
            }
            byte[] domainEntryKey = RollingLevelDBTimelineStore.createDomainEntryKey(domain.getId(), DESCRIPTION_COLUMN);
            byte[] ownerLookupEntryKey = RollingLevelDBTimelineStore.createOwnerLookupKey(domain.getOwner(), domain.getId(), DESCRIPTION_COLUMN);
            if (domain.getDescription() != null) {
                domainWriteBatch.put(domainEntryKey, domain.getDescription().getBytes(StandardCharsets.UTF_8));
                ownerWriteBatch.put(ownerLookupEntryKey, domain.getDescription().getBytes(StandardCharsets.UTF_8));
            } else {
                domainWriteBatch.put(domainEntryKey, EMPTY_BYTES);
                ownerWriteBatch.put(ownerLookupEntryKey, EMPTY_BYTES);
            }
            domainEntryKey = RollingLevelDBTimelineStore.createDomainEntryKey(domain.getId(), OWNER_COLUMN);
            ownerLookupEntryKey = RollingLevelDBTimelineStore.createOwnerLookupKey(domain.getOwner(), domain.getId(), OWNER_COLUMN);
            domainWriteBatch.put(domainEntryKey, domain.getOwner().getBytes(StandardCharsets.UTF_8));
            ownerWriteBatch.put(ownerLookupEntryKey, domain.getOwner().getBytes(StandardCharsets.UTF_8));
            domainEntryKey = RollingLevelDBTimelineStore.createDomainEntryKey(domain.getId(), READER_COLUMN);
            ownerLookupEntryKey = RollingLevelDBTimelineStore.createOwnerLookupKey(domain.getOwner(), domain.getId(), READER_COLUMN);
            if (domain.getReaders() != null && domain.getReaders().length() > 0) {
                domainWriteBatch.put(domainEntryKey, domain.getReaders().getBytes(StandardCharsets.UTF_8));
                ownerWriteBatch.put(ownerLookupEntryKey, domain.getReaders().getBytes(StandardCharsets.UTF_8));
            } else {
                domainWriteBatch.put(domainEntryKey, EMPTY_BYTES);
                ownerWriteBatch.put(ownerLookupEntryKey, EMPTY_BYTES);
            }
            domainEntryKey = RollingLevelDBTimelineStore.createDomainEntryKey(domain.getId(), WRITER_COLUMN);
            ownerLookupEntryKey = RollingLevelDBTimelineStore.createOwnerLookupKey(domain.getOwner(), domain.getId(), WRITER_COLUMN);
            if (domain.getWriters() != null && domain.getWriters().length() > 0) {
                domainWriteBatch.put(domainEntryKey, domain.getWriters().getBytes(StandardCharsets.UTF_8));
                ownerWriteBatch.put(ownerLookupEntryKey, domain.getWriters().getBytes(StandardCharsets.UTF_8));
            } else {
                domainWriteBatch.put(domainEntryKey, EMPTY_BYTES);
                ownerWriteBatch.put(ownerLookupEntryKey, EMPTY_BYTES);
            }
            domainEntryKey = RollingLevelDBTimelineStore.createDomainEntryKey(domain.getId(), TIMESTAMP_COLUMN);
            ownerLookupEntryKey = RollingLevelDBTimelineStore.createOwnerLookupKey(domain.getOwner(), domain.getId(), TIMESTAMP_COLUMN);
            long currentTimestamp = System.currentTimeMillis();
            byte[] timestamps = this.domaindb.get(domainEntryKey);
            if (timestamps == null) {
                timestamps = new byte[16];
                GenericObjectMapper.writeReverseOrderedLong(currentTimestamp, timestamps, 0);
                GenericObjectMapper.writeReverseOrderedLong(currentTimestamp, timestamps, 8);
            } else {
                GenericObjectMapper.writeReverseOrderedLong(currentTimestamp, timestamps, 8);
            }
            domainWriteBatch.put(domainEntryKey, timestamps);
            ownerWriteBatch.put(ownerLookupEntryKey, timestamps);
            this.domaindb.write(domainWriteBatch);
            this.ownerdb.write(ownerWriteBatch);
        }
    }

    private static byte[] createDomainEntryKey(String domainId, byte[] columnName) throws IOException {
        return LeveldbUtils.KeyBuilder.newInstance().add(domainId).add(columnName).getBytes();
    }

    private static byte[] createOwnerLookupKey(String owner, String domainId, byte[] columnName) throws IOException {
        return LeveldbUtils.KeyBuilder.newInstance().add(owner).add(domainId).add(columnName).getBytes();
    }

    @Override
    public TimelineDomain getDomain(String domainId) throws IOException {
        try (DBIterator iterator = this.domaindb.iterator();){
            byte[] prefix = LeveldbUtils.KeyBuilder.newInstance().add(domainId).getBytesForLookup();
            iterator.seek(prefix);
            TimelineDomain timelineDomain = RollingLevelDBTimelineStore.getTimelineDomain(iterator, domainId, prefix);
            return timelineDomain;
        }
    }

    @Override
    public TimelineDomains getDomains(String owner) throws IOException {
        try (DBIterator iterator = this.ownerdb.iterator();){
            byte[] key;
            byte[] prefix = LeveldbUtils.KeyBuilder.newInstance().add(owner).getBytesForLookup();
            iterator.seek(prefix);
            ArrayList<TimelineDomain> domains = new ArrayList<TimelineDomain>();
            while (iterator.hasNext() && LeveldbUtils.prefixMatches(prefix, prefix.length, key = (byte[])iterator.peekNext().getKey())) {
                byte[] prefixExt;
                LeveldbUtils.KeyParser kp = new LeveldbUtils.KeyParser(key, prefix.length);
                String domainId = kp.getNextString();
                TimelineDomain domainToReturn = RollingLevelDBTimelineStore.getTimelineDomain(iterator, domainId, prefixExt = LeveldbUtils.KeyBuilder.newInstance().add(owner).add(domainId).getBytesForLookup());
                if (domainToReturn == null) continue;
                domains.add(domainToReturn);
            }
            Collections.sort(domains, new Comparator<TimelineDomain>(){

                @Override
                public int compare(TimelineDomain domain1, TimelineDomain domain2) {
                    int result = domain2.getCreatedTime().compareTo(domain1.getCreatedTime());
                    if (result == 0) {
                        return domain2.getModifiedTime().compareTo(domain1.getModifiedTime());
                    }
                    return result;
                }
            });
            TimelineDomains domainsToReturn = new TimelineDomains();
            domainsToReturn.addDomains(domains);
            TimelineDomains timelineDomains = domainsToReturn;
            return timelineDomains;
        }
    }

    private static TimelineDomain getTimelineDomain(DBIterator iterator, String domainId, byte[] prefix) throws IOException {
        byte[] key;
        TimelineDomain domain = new TimelineDomain();
        domain.setId(domainId);
        boolean noRows = true;
        while (iterator.hasNext() && LeveldbUtils.prefixMatches(prefix, prefix.length, key = (byte[])iterator.peekNext().getKey())) {
            byte[] value;
            if (noRows) {
                noRows = false;
            }
            if ((value = (byte[])iterator.peekNext().getValue()) != null && value.length > 0) {
                if (key[prefix.length] == DESCRIPTION_COLUMN[0]) {
                    domain.setDescription(new String(value, StandardCharsets.UTF_8));
                } else if (key[prefix.length] == OWNER_COLUMN[0]) {
                    domain.setOwner(new String(value, StandardCharsets.UTF_8));
                } else if (key[prefix.length] == READER_COLUMN[0]) {
                    domain.setReaders(new String(value, StandardCharsets.UTF_8));
                } else if (key[prefix.length] == WRITER_COLUMN[0]) {
                    domain.setWriters(new String(value, StandardCharsets.UTF_8));
                } else if (key[prefix.length] == TIMESTAMP_COLUMN[0]) {
                    domain.setCreatedTime(Long.valueOf(GenericObjectMapper.readReverseOrderedLong(value, 0)));
                    domain.setModifiedTime(Long.valueOf(GenericObjectMapper.readReverseOrderedLong(value, 8)));
                } else {
                    LOG.error("Unrecognized domain column: " + key[prefix.length]);
                }
            }
            iterator.next();
        }
        if (noRows) {
            return null;
        }
        return domain;
    }

    static {
        fstConf.setShareReferences(false);
        fstConf224.setShareReferences(false);
        FSTClazzNameRegistry registry = fstConf224.getClassRegistry();
        registry.registerClass(LinkedHashMap.class, 83, fstConf224);
        DOMAIN_ID_COLUMN = "d".getBytes(StandardCharsets.UTF_8);
        EVENTS_COLUMN = "e".getBytes(StandardCharsets.UTF_8);
        PRIMARY_FILTERS_COLUMN = "f".getBytes(StandardCharsets.UTF_8);
        OTHER_INFO_COLUMN = "i".getBytes(StandardCharsets.UTF_8);
        RELATED_ENTITIES_COLUMN = "r".getBytes(StandardCharsets.UTF_8);
        DESCRIPTION_COLUMN = "d".getBytes(StandardCharsets.UTF_8);
        OWNER_COLUMN = "o".getBytes(StandardCharsets.UTF_8);
        READER_COLUMN = "r".getBytes(StandardCharsets.UTF_8);
        WRITER_COLUMN = "w".getBytes(StandardCharsets.UTF_8);
        TIMESTAMP_COLUMN = "t".getBytes(StandardCharsets.UTF_8);
        EMPTY_BYTES = new byte[0];
        CURRENT_VERSION_INFO = Version.newInstance(1, 0);
        writeBatchSize = 10000L;
        LEVELDB_DIR_UMASK = FsPermission.createImmutable((short)448);
    }

    private class EntityDeletionThread
    extends Thread {
        private final long ttl;
        private final long ttlInterval;

        EntityDeletionThread(Configuration conf) {
            this.ttl = conf.getLong("yarn.timeline-service.ttl-ms", 604800000L);
            this.ttlInterval = conf.getLong("yarn.timeline-service.leveldb-timeline-store.ttl-interval-ms", 300000L);
            LOG.info("Starting deletion thread with ttl " + this.ttl + " and cycle interval " + this.ttlInterval);
        }

        @Override
        public void run() {
            Thread.currentThread().setName("Leveldb Timeline Store Retention");
            while (true) {
                long timestamp = System.currentTimeMillis() - this.ttl;
                try {
                    RollingLevelDBTimelineStore.this.discardOldEntities(timestamp);
                    Thread.sleep(this.ttlInterval);
                    continue;
                }
                catch (IOException e) {
                    LOG.error(e.toString());
                    continue;
                }
                catch (InterruptedException e) {
                    LOG.info("Deletion thread received interrupt, exiting");
                    return;
                }
                break;
            }
        }
    }
}

