/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.nfs;

import com.google.common.base.Preconditions;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import org.apache.commons.net.util.SubnetUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.nfs.AccessPrivilege;
import org.apache.hadoop.util.LightWeightCache;
import org.apache.hadoop.util.LightWeightGSet;
import org.apache.hadoop.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NfsExports {
    private static NfsExports exports = null;
    public static final Logger LOG = LoggerFactory.getLogger(NfsExports.class);
    private static final String IP_ADDRESS = "(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})";
    private static final String SLASH_FORMAT_SHORT = "(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})/(\\d{1,3})";
    private static final String SLASH_FORMAT_LONG = "(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})/(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})";
    private static final Pattern CIDR_FORMAT_SHORT = Pattern.compile("(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})/(\\d{1,3})");
    private static final Pattern CIDR_FORMAT_LONG = Pattern.compile("(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})/(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})");
    private static final String LABEL_FORMAT = "[a-zA-Z0-9]([a-zA-Z0-9\\-]{0,61}[a-zA-Z0-9])?";
    private static final Pattern HOSTNAME_FORMAT = Pattern.compile("^([a-zA-Z0-9]([a-zA-Z0-9\\-]{0,61}[a-zA-Z0-9])?\\.)*[a-zA-Z0-9]([a-zA-Z0-9\\-]{0,61}[a-zA-Z0-9])?$");
    private final List<Match> mMatches;
    private final LightWeightCache<AccessCacheEntry, AccessCacheEntry> accessCache;
    private final long cacheExpirationPeriod;

    public static synchronized NfsExports getInstance(Configuration conf) {
        if (exports == null) {
            String matchHosts = conf.get("nfs.exports.allowed.hosts", "* rw");
            int cacheSize = conf.getInt("nfs.exports.cache.size", 512);
            long expirationPeriodNano = conf.getLong("nfs.exports.cache.expirytime.millis", 900000L) * 1000L * 1000L;
            try {
                exports = new NfsExports(cacheSize, expirationPeriodNano, matchHosts);
            }
            catch (IllegalArgumentException e) {
                LOG.error("Invalid NFS Exports provided: ", (Throwable)e);
                return exports;
            }
        }
        return exports;
    }

    NfsExports(int cacheSize, long expirationPeriodNano, String matchHosts) {
        this.cacheExpirationPeriod = expirationPeriodNano;
        this.accessCache = new LightWeightCache(cacheSize, cacheSize, expirationPeriodNano, 0L);
        String[] matchStrings = matchHosts.split(";");
        this.mMatches = new ArrayList<Match>(matchStrings.length);
        for (String mStr : matchStrings) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Processing match string '" + mStr + "'");
            }
            if ((mStr = mStr.trim()).isEmpty()) continue;
            this.mMatches.add(NfsExports.getMatch(mStr));
        }
    }

    public String[] getHostGroupList() {
        int listSize = this.mMatches.size();
        String[] hostGroups = new String[listSize];
        for (int i = 0; i < this.mMatches.size(); ++i) {
            hostGroups[i] = this.mMatches.get(i).getHostGroup();
        }
        return hostGroups;
    }

    public AccessPrivilege getAccessPrivilege(InetAddress addr) {
        return this.getAccessPrivilege(addr.getHostAddress(), addr.getCanonicalHostName());
    }

    AccessPrivilege getAccessPrivilege(String address, String hostname) {
        long now = System.nanoTime();
        AccessCacheEntry newEntry = new AccessCacheEntry(address, AccessPrivilege.NONE, now + this.cacheExpirationPeriod);
        AccessCacheEntry cachedEntry = (AccessCacheEntry)this.accessCache.get((Object)newEntry);
        if (cachedEntry != null && now < cachedEntry.expirationTime) {
            return cachedEntry.access;
        }
        for (Match match : this.mMatches) {
            if (!match.isIncluded(address, hostname)) continue;
            if (match.accessPrivilege == AccessPrivilege.READ_ONLY) {
                newEntry.access = AccessPrivilege.READ_ONLY;
                break;
            }
            if (match.accessPrivilege != AccessPrivilege.READ_WRITE) continue;
            newEntry.access = AccessPrivilege.READ_WRITE;
        }
        this.accessCache.put((Object)newEntry);
        return newEntry.access;
    }

    private static Match getMatch(String line) {
        String host;
        String[] parts = line.split("\\s+");
        AccessPrivilege privilege = AccessPrivilege.READ_ONLY;
        switch (parts.length) {
            case 1: {
                host = StringUtils.toLowerCase((String)parts[0]).trim();
                break;
            }
            case 2: {
                host = StringUtils.toLowerCase((String)parts[0]).trim();
                String option = parts[1].trim();
                if (!"rw".equalsIgnoreCase(option)) break;
                privilege = AccessPrivilege.READ_WRITE;
                break;
            }
            default: {
                throw new IllegalArgumentException("Incorrectly formatted line '" + line + "'");
            }
        }
        if (host.equals("*")) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using match all for '" + host + "' and " + (Object)((Object)privilege));
            }
            return new AnonymousMatch(privilege);
        }
        if (CIDR_FORMAT_SHORT.matcher(host).matches()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using CIDR match for '" + host + "' and " + (Object)((Object)privilege));
            }
            return new CIDRMatch(privilege, new SubnetUtils(host).getInfo());
        }
        if (CIDR_FORMAT_LONG.matcher(host).matches()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using CIDR match for '" + host + "' and " + (Object)((Object)privilege));
            }
            String[] pair = host.split("/");
            return new CIDRMatch(privilege, new SubnetUtils(pair[0], pair[1]).getInfo());
        }
        if (host.contains("*") || host.contains("?") || host.contains("[") || host.contains("]") || host.contains("(") || host.contains(")")) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using Regex match for '" + host + "' and " + (Object)((Object)privilege));
            }
            return new RegexMatch(privilege, host);
        }
        if (HOSTNAME_FORMAT.matcher(host).matches()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using exact match for '" + host + "' and " + (Object)((Object)privilege));
            }
            return new ExactMatch(privilege, host);
        }
        throw new IllegalArgumentException("Invalid hostname provided '" + host + "'");
    }

    private static class RegexMatch
    extends Match {
        private final Pattern pattern;

        private RegexMatch(AccessPrivilege accessPrivilege, String wildcard) {
            super(accessPrivilege);
            this.pattern = Pattern.compile(wildcard, 2);
        }

        @Override
        public boolean isIncluded(String address, String hostname) {
            if (this.pattern.matcher(address).matches() || this.pattern.matcher(hostname).matches()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RegexMatcher '" + this.pattern.pattern() + "', allowing client '" + address + "', '" + hostname + "'");
                }
                return true;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("RegexMatcher '" + this.pattern.pattern() + "', denying client '" + address + "', '" + hostname + "'");
            }
            return false;
        }

        @Override
        public String getHostGroup() {
            return this.pattern.toString();
        }
    }

    private static class ExactMatch
    extends Match {
        private final String ipOrHost;

        private ExactMatch(AccessPrivilege accessPrivilege, String ipOrHost) {
            super(accessPrivilege);
            this.ipOrHost = ipOrHost;
        }

        @Override
        public boolean isIncluded(String address, String hostname) {
            if (this.ipOrHost.equalsIgnoreCase(address) || this.ipOrHost.equalsIgnoreCase(hostname)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("ExactMatcher '" + this.ipOrHost + "', allowing client " + "'" + address + "', '" + hostname + "'");
                }
                return true;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("ExactMatcher '" + this.ipOrHost + "', denying client " + "'" + address + "', '" + hostname + "'");
            }
            return false;
        }

        @Override
        public String getHostGroup() {
            return this.ipOrHost;
        }
    }

    private static class CIDRMatch
    extends Match {
        private final SubnetUtils.SubnetInfo subnetInfo;

        private CIDRMatch(AccessPrivilege accessPrivilege, SubnetUtils.SubnetInfo subnetInfo) {
            super(accessPrivilege);
            this.subnetInfo = subnetInfo;
        }

        @Override
        public boolean isIncluded(String address, String hostname) {
            if (this.subnetInfo.isInRange(address)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("CIDRNMatcher low = " + this.subnetInfo.getLowAddress() + ", high = " + this.subnetInfo.getHighAddress() + ", allowing client '" + address + "', '" + hostname + "'");
                }
                return true;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("CIDRNMatcher low = " + this.subnetInfo.getLowAddress() + ", high = " + this.subnetInfo.getHighAddress() + ", denying client '" + address + "', '" + hostname + "'");
            }
            return false;
        }

        @Override
        public String getHostGroup() {
            return this.subnetInfo.getAddress() + "/" + this.subnetInfo.getNetmask();
        }
    }

    private static class AnonymousMatch
    extends Match {
        private AnonymousMatch(AccessPrivilege accessPrivilege) {
            super(accessPrivilege);
        }

        @Override
        public boolean isIncluded(String address, String hostname) {
            return true;
        }

        @Override
        public String getHostGroup() {
            return "*";
        }
    }

    private static abstract class Match {
        private final AccessPrivilege accessPrivilege;

        private Match(AccessPrivilege accessPrivilege) {
            this.accessPrivilege = accessPrivilege;
        }

        public abstract boolean isIncluded(String var1, String var2);

        public abstract String getHostGroup();
    }

    static class AccessCacheEntry
    implements LightWeightCache.Entry {
        private final String hostAddr;
        private AccessPrivilege access;
        private final long expirationTime;
        private LightWeightGSet.LinkedElement next;

        AccessCacheEntry(String hostAddr, AccessPrivilege access, long expirationTime) {
            Preconditions.checkArgument((hostAddr != null ? 1 : 0) != 0);
            this.hostAddr = hostAddr;
            this.access = access;
            this.expirationTime = expirationTime;
        }

        public int hashCode() {
            return this.hostAddr.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj instanceof AccessCacheEntry) {
                AccessCacheEntry entry = (AccessCacheEntry)obj;
                return this.hostAddr.equals(entry.hostAddr);
            }
            return false;
        }

        public void setNext(LightWeightGSet.LinkedElement next) {
            this.next = next;
        }

        public LightWeightGSet.LinkedElement getNext() {
            return this.next;
        }

        public void setExpirationTime(long timeNano) {
        }

        public long getExpirationTime() {
            return this.expirationTime;
        }
    }
}

