% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/highest_concentration_terra.R
\name{find_highest_concentration}
\alias{find_highest_concentration}
\title{Find highest concentration}
\usage{
find_highest_concentration(
  df,
  value,
  top_n = 1,
  radius = 200,
  cell_size = 100,
  grid_precision = 1,
  lon = "lon",
  lat = "lat",
  crs_metric = 3035,
  print_progress = TRUE
)
}
\arguments{
\item{df}{A data.frame containing demand points. Must include at least
columns for longitude, latitude, and the value of interest.}

\item{value}{Column name in \code{df} with the value of interest to
summarize.}

\item{top_n}{Positive integer greater or equal to 1 (default is 1).
Specifies how many highest concentration circles are returned.
If \code{top_n > 1}, then after each iteration the points belonging to
the highest concentration are removed from \code{df}. This prevents the
subsequent concentrations from being located in the same area, which
would otherwise repeatedly select overlapping points with the largest
values.}

\item{radius}{Numeric. Radius of the circle in meters (default = 200).}

\item{cell_size}{Numeric. Size of the grid cell in meters (default is 100).
Defines the resolution of the initial raster grid. The choice of cell
size depends on the size of the study area. For example, for a country
the size of the Netherlands, cells of 100 × 100 meters are typically
sufficient. For larger areas such as Germany, a cell size of 200 × 200
meters may be more appropriate. The choice of \code{cell_size} does not
affect the final result, only the computational speed.}

\item{grid_precision}{Numeric. Precision of the search grid in meters
(default is 1). Determines the spacing of sub-points within each raster
cell. For example, with \code{cell_size = 100} and
\code{grid_precision = 1}, 10,000 sub-points (100 × 100) are evaluated
per cell. Larger values reduce the number of sub-points (and runtime),
but also reduce spatial accuracy.}

\item{lon}{Column name in \code{df} for longitude (default = \code{"lon"}).
Must be in EPSG:4326.}

\item{lat}{Column name in \code{df} for latitude (default = \code{"lat"}).
Must be in EPSG:4326.}

\item{crs_metric}{Numeric. Metric Coordinate Reference System (CRS) used in
background calculations. For Europe use
\href{https://epsg.io/3035}{EPSG:3035} (default).
For the United States use \href{https://epsg.io/6317}{EPSG:6317}.
For Asia-Pacific use \href{https://epsg.io/8859}{EPSG:8859}.}

\item{print_progress}{Logical. Whether to print progress messages
(\code{TRUE}/\code{FALSE}).}
}
\value{
A list with two elements:
\enumerate{
\item A data.frame with the \code{top_n} highest concentrations.
\item A data.frame with the subset of \code{df} corresponding to those
concentrations.
}
}
\description{
Identifies the central coordinates of a circle with a fixed
radius that maximizes the coverage of demand points.
}
\details{
A recent regulation by the European Commission mandates insurers to
report the maximum insured value of fire risk policies for all buildings
partly or fully within a circle of radius 200 meters (see Article 132 - fire
risk sub-module - of the Delegated Regulation). This captures the risk of
catastrophic fire or explosion, including terrorist attacks.

The problem resembles a Maximal Covering Location Problem (MCLP) with a
fixed radius, a classic facility location problem. The goal is to select the
best locations to maximize coverage of demand points, ensuring each demand
point lies within the radius of at least one selected facility.
}
\examples{
# Find single highest concentration
x <- find_highest_concentration(Groningen, value = "amount")
plot(x)

# Find top 2 concentrations with smaller grid cells
y <- find_highest_concentration(Groningen, "amount",
                                top_n = 2, cell_size = 50)
plot(y)

}
\references{
Commission Delegated Regulation (EU) (2015). Solvency II
Delegated Act 2015/35. Official Journal of the European Union, 58:124.
}
\author{
Martin Haringa
}
