% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findtarget.R
\name{FindTarget}
\alias{FindTarget}
\title{Find requirements for target power (or other objective)}
\usage{
FindTarget(
  x,
  par_to_search = "n",
  find_lowest = TRUE,
  target_value = 0.9,
  target_at_least = TRUE,
  method = "step"
)
}
\arguments{
\item{x}{An array, most commonly of class \code{power_array}, possibly the result
of taking a slice of an object of class \code{power_array} using
\code{\link{ArraySlicer}} or the power_array []-indexing method.}

\item{par_to_search}{Which parameter should be searched to achieve the
required target value. In the typical power analysis case, this is n.}

\item{find_lowest}{If TRUE, the lowest value of par_to_search is found that
yields a value that meets the target. This is typical for n in a sample
size estimation, where one searches the lowest n to achieve a certain
power. For, e.g. the variance, one would however search for the maximum
where the target power can still be achieved.}

\item{target_value}{The required value in x (e.g., .9, if the values
represent power)}

\item{target_at_least}{Is the \code{target_value} a minimum (e.g., the power) or
a maximum (e.g., the size of a confidence interval)}

\item{method}{How is the required \code{par_to_search} to achieve
\code{target_value} found. Either \code{'step'}: walking in steps along
\code{par_to_search} or \code{'lm'}: Interpolating assuming a linear
relation between \code{par_to_search} and \code{(qnorm(x) + qnorm(1 -
  0.05)) ^ 2}. Setting 'lm' is inspired on the implementation in the sse
package by Thomas Fabbro.}
}
\value{
Returns an array or vector: containing the value that is found for
the par_to_search (say, n) meeting the target following above criteria
(say, the lowest n for which the power is larger than .9), for each
crossing of the levels of the other dimensions (say, delta, SD).
}
\description{
For most use cases of powergrid, you will not need this function, but rather
use more convenient functions, most notable \code{\link{Example}}. \code{Example}
shows you the smallest sample size to still find enough power, or the
largest standard deviation at which your CI95 does not get too large. More
insight about the relation between parameters and the resulting power may be
gained with \code{\link{PowerPlot}} or \code{\link{GridPlot}}.

Only if you need to work with, say, the required n for a range of
assumptions over and above PowerPlot and GridPlot, you will need to use
FindTarget.

FindTarget takes as input an array (typically of class \code{power_array}).
FindTarget then searches (up or down) along one chosen dimension for a value
that meets a set target value (at least or at most). It does so for each
combination of the remaining dimensions. Concretely, this may mean: The
array contains the calculated power for each combination of dimensions n,
effect size, and SD. The function may then find, for each combination of
effect size and SD, the lowest n for which power of at least, say, .8 is
achieved. The result would be an array of effect size by SD, containing the
n's yielding acceptable power.
}
\details{
By default FindTarget searches along the dimension called \code{n}
(\code{par_to_search}), searching for the lowest value (\code{find_lowest}
= TRUE) where the array contains a value of at least
(\code{target_at_least} = TRUE) .9 (the \code{target_value}), thus finding the
minimal sample size required to achieve a power of 90\%. These arguments
may seem a bit confusing at first, but they allow for three additional
purposes:

First, the implementation also allows to search for a value that is \emph{at
most} the \code{target_value}, by setting \code{target_at_least} to FALSE. This may
be used, for example, when the aim is to find a sample size yielding a
confidence interval that is not bigger than some maximum width.

Second, the implementation allows to search along another \emph{named} dimension
of x than n.

Third, the implementation allows to search for a certain target value to be
achieved by maximizing (find_lowest = FALSE) the parameter on the searched
dimension. This may be used, for example, when the aim is to find the
maximum standard deviation at which a study's power is still acceptable.

\code{FindTarget} is most often called as the workhorse of
\code{\link{Example}}, \code{\link{PowerPlot}} or \code{\link{GridPlot}}.
}
\examples{
## ============================================
## A basic power analysis example:
## ============================================
sse_pars = list(
  n = seq(from = 10, to = 60, by = 2),
  sig_level = seq(.01, .1, .01),
  delta = seq(from = 0.5, to = 1.5, by = 0.2), ## effect size
  sd = seq(.1, .9, .2)) ## Standard deviation
PowFun <- function(n, sig_level, delta, sd){
  ptt = power.t.test(n = n/2, delta = delta, sd = sd,
                     sig.level = sig_level)
  return(ptt$power)
}
power_array = PowerGrid(pars = sse_pars, fun = PowFun, n_iter = NA)
summary(power_array) # four dimensions

## We can use Example so find the required sample size, but only for one example:
Example(power_array,
        example = list(delta = .7, sd = .7, sig_level = .05),
        target_value = .9)

## If we want to see the required sample size for all delta's, we can use
## FindTarget. Get the minimal n needed for achieving a value of 0.9, at sd =
## .3:
n_by_delta_sd_03 = FindTarget(power_array[, sig_level = '0.05', , sd = '0.3'],
                              par_to_search = 'n',
                              target_value = .9)

n_by_delta_sd_03
## just as an illustration, a figure (that can be much more aestetically made
## using PowerPlot)
plot(as.numeric(names(n_by_delta_sd_03)),
     n_by_delta_sd_03, type = 'l')

## =================================
## Higher dimensionality
## =================================

## The function works also for higher dimensionality:
n_by_delta_sd = FindTarget(power_array,
                           par_to_search = 'n',
                           target_value = .85)
## what is the minimum n to achieve .85 for different values of delta, sd,
## when  sig_level = 0.05:
n_by_delta_sd[5, , ] # note that for some combinations of delta and sd, there is
                     # no n yielding the required power at this significance
                     # level (NAs).
}
\seealso{
\code{\link{PowerGrid}}, \code{\link{Example}},
\code{\link{PowerPlot}}
}
\author{
Gilles Dutilh
}
