% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apply_citation_matching.R
\name{applyCitationMatching}
\alias{applyCitationMatching}
\title{Apply citation normalization to bibliometrix data frame}
\usage{
applyCitationMatching(M, threshold = 0.9, method = "jw", min_chars = 20)
}
\arguments{
\item{M}{A bibliometrix data frame, typically created by \code{\link{convert2df}}.
Must contain the columns:
\itemize{
  \item \code{SR}: Short reference identifier for each document
  \item \code{CR}: Cited references field (citations separated by semicolons)
  \item \code{DB}: (Optional) Database source identifier for format detection
}}

\item{threshold}{Numeric value between 0 and 1 indicating the similarity threshold
for matching citations. Default is 0.85. See \code{\link{normalize_citations}}
for details on selecting appropriate thresholds.}

\item{method}{String distance method to use for fuzzy matching. Options include:
\itemize{
  \item "jw" (default): Jaro-Winkler distance, optimized for bibliographic strings
  \item "lv": Levenshtein distance
  \item Other methods supported by \code{\link[stringdist]{stringdistmatrix}}
}}

\item{min_chars}{Minimum characters for valid citations (default: 20)}
}
\value{
A list with four elements:
\describe{
  \item{full_data}{A data frame with columns:
    \itemize{
      \item \code{SR}: Source document identifier
      \item \code{CR}: Original citation string
      \item \code{CR_canonical}: Canonical (normalized) citation
      \item \code{cluster_id}: Unique cluster identifier
      \item \code{n_cluster}: Size of the citation cluster
      \item \code{first_author}, \code{year}, \code{journal}, \code{volume}: Extracted metadata
    }
  }
  \item{summary}{A data frame summarizing citation frequencies with columns:
    \itemize{
      \item \code{CR_canonical}: The canonical citation for each cluster
      \item \code{n}: Total number of times this work was cited
      \item \code{n_variants}: Number of different formatting variants found
      \item \code{variants_example}: Sample of variant formats (up to 3 examples)
    }
    Sorted by citation frequency (n) in descending order.
  }
  \item{matched_citations}{Complete output from \code{\link{normalize_citations}},
    useful for detailed analysis of the matching process.}
  \item{CR_normalized}{A data frame with columns:
    \itemize{
      \item \code{SR}: Source document identifier
      \item \code{CR}: Reconstructed CR field with normalized citations (semicolon-separated)
      \item \code{n_references}: Number of unique references after normalization
    }
    This can be merged back with M to replace the original CR field.
  }
}
}
\description{
This is a convenience wrapper function that applies \code{\link{normalize_citations}}
to a bibliometrix data frame (typically loaded with \code{\link{convert2df}}). It
extracts citations from the CR field, performs normalization and matching, and
returns comprehensive results including per-paper citation lists and summary statistics.
}
\details{
The function automatically handles the new Scopus citation format (where the year
appears at the end in parentheses) by converting it to the classic format before
processing.


The function performs the following steps:
\enumerate{
  \item Splits the CR field by semicolons to extract individual citations
  \item Detects and converts new Scopus format citations to classic format
  \item Trims whitespace from each citation
  \item Applies \code{\link{normalize_citations}} to identify duplicate citations
  \item Links normalized citations back to source documents (SR)
  \item Generates summary statistics and reconstructs normalized CR fields
}

The normalized CR field can be used to replace the original CR field in subsequent
bibliometric analyses, ensuring that citation counts and network analyses are not
inflated by duplicate citations with minor formatting differences.
}
\examples{
\dontrun{
# Load bibliometric data
file <- "https://www.bibliometrix.org/datasets/savedrecs.txt"
M <- convert2df(file, dbsource = "wos", format = "plaintext")

# Apply citation normalization
results <- applyCitationMatching(M, threshold = 0.85)

# View top cited works (after normalization)
head(results$summary, 20)

# See how many variants were found for the top citation
top_citation <- results$summary$CR_canonical[1]
variants <- subset(results$full_data, CR_canonical == top_citation)
unique(variants$CR)

# Replace original CR with normalized CR in the data frame
M_normalized <- M \%>\%
  rename(CR_orig = CR) \%>\%
  left_join(results$CR_normalized, by = "SR")

# Compare citation counts before and after normalization
original_citations <- strsplit(M$CR, ";") \%>\%
  unlist() \%>\%
  trimws() \%>\%
  table() \%>\%
  length()

normalized_citations <- nrow(results$summary)

cat("Original unique citations:", original_citations, "\n")
cat("After normalization:", normalized_citations, "\n")
cat("Duplicates found:", original_citations - normalized_citations, "\n")

# Use normalized data for further analysis
CR_analysis <- citations(M_normalized, field = "article", sep = ";")
}

}
\references{
Aria, M. & Cuccurullo, C. (2017). bibliometrix: An R-tool for comprehensive
science mapping analysis. Journal of Informetrics, 11(4), 959-975.
}
\seealso{
\code{\link{normalize_citations}} for the underlying normalization algorithm
\code{\link{citations}} for citation analysis
\code{\link{localCitations}} for local citation analysis
}
