\name{predQuant}
\alias{predQuant}
\title{Predictive quantile based on the generalized Pareto model}
\usage{
predQuant(
  qlev,
  postsamp,
  threshold,
  lb,
  ub,
  type = c("continuous", "discrete"),
  extrapolation = FALSE,
  p,
  k,
  n
)
}
\arguments{
\item{qlev}{double, quantile level}

\item{postsamp}{a \code{m} by \code{2} matrix with columns containing the posterior samples of scale and shape parameters of the generalized Pareto distribution, respectively}

\item{threshold}{threshold for the generalized Pareto model, corresponding to the \eqn{n-k}th order statistic of the sample}

\item{lb}{double, the lower bound of the admissible region for the quantile value}

\item{ub}{double, the upper bound of the admissible region for the quantile value}

\item{type}{data type, either \code{"continuous"} or \code{"discrete"} for count data.}

\item{extrapolation}{logical; if \code{TRUE}, extrapolate using the threshold stability property}

\item{p}{scalar tail probability for the extrapolation. Must be smaller than \eqn{k/n} (only used if \code{extrapolation=TRUE})}

\item{k}{integer, number of exceedances for the generalized Pareto (only used if \code{extrapolation=TRUE})}

\item{n}{integer, number of observations in the full sample. Must be greater than \code{k}  (only used if \code{extrapolation=TRUE})}
}
\value{
a double indicating the value of the quantile
}
\description{
Bayesian Generalize Pareto-based predictive quantile for continuous and discrete predictive distribution conditioned on intermediate and extreme levels.
}
\examples{
\dontrun{
# generate data
set.seed(1234)
n <- 500
samp <- evd::rfrechet(n,0,3,4)
# set effective sample size and threshold
k <- 50
threshold <- sort(samp,decreasing = TRUE)[k+1]
# preliminary mle estimates of scale and shape parameters
mlest <- evd::fpot(samp, threshold)
# empirical bayes procedure
proc <- estPOT(
  samp,
  k = k,
  pn = c(0.01, 0.005),
  type = "continuous",
  method = "bayesian",
  prior = "empirical",
  start = as.list(mlest$estimate),
  sig0 = 0.1)
# predictive density estimation
yg <- seq(0, 50, by = 2)
nyg <- length(yg)
predDens_int <- predDens(
  yg,
  proc$post_sample,proc$t,
  "continuous",
  extrapolation=FALSE)
predQuant_int <- predQuant(
  0.5,
  proc$post_sample,
  proc$t,
  proc$t + 0.01,
  50,
  "continuous",
  extrapolation = FALSE)
predDens_ext <- predDens(
  yg,
  proc$post_sample,
  proc$t,
  "continuous",
  extrapolation = TRUE,
  p = 0.001,
  k = k,
  n = n)
predQuant_ext <- predQuant(
  0.5,
  proc$post_sample,
  proc$t,
  proc$t + 0.01,
  100,
  "continuous",
  extrapolation = TRUE,
  p = 0.005,
  k = k,
  n = n)
  }
}
