/*
TOUCH.C

Examples:
    touch <files>
    touch -t yymmddhhmmss <files>
    touch -f filename <files>
    touch -c <files>

The first form sets the date and time of all the files named
in <files> to be the current date and time, as of when the
program starts.  The second form sets the date and time of
all the files named in <files> to be the specified date and
time.  Years from 00-79 are taken to be the years 2000
through 2079.  The third form sets the date and time of all
the files named in <files> equal to the date and time of
filename. 

If the target file does not exist, touch will create it,
giving it the appropriate date/time, according to the
options specified.  If -c is specified, missing files will
not be created. 

This program, it's source code, and it's documentation is
copyright, 1995, by the author, Fred C.  Smith,
fredex@fcshome.stoneham.ma.us.  While the author retains all
rights to the program, it is permissible to distribute the
program freely, as long as this copyright notice remains in
the package, the ownership is not misrepresented, and it is
not sold for profit. 

Further, the author will not be held liable for any
consequences that may arise from the use or misuse of this
program, no matter how dire said consequences may be. 
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <io.h>
#include <sys/types.h>
#include <sys/stat.h>   /* for S_IREAD, etc. */
#include <dos.h>

union REGS regs;

typedef struct 
    {
    unsigned int seconds :5;
    unsigned int minutes :6;
    unsigned int hours :5;
    }TIM;
union utime
    {
    TIM tim;
    int dostime;
    };

typedef struct 
    {
    unsigned int day :5;
    unsigned int month :4;
    unsigned int year :7;
    } DTE;
union udate
    {
    DTE dte;
    int dosdate;
    };

int touch_file (int dostime, int dosdate,
                char *filename, int creatit);
int get_current_time (TIM *ptim, DTE *pdte);
int time2struct (TIM *tim, DTE *dte, char *deftime);
int file_time2struct (int *dostime, int *dosdate,
                char *filename);
void usage(void);
int proc_args (int *pargc, char **argv,
                union utime *tim, union udate *dte);


/* global for command option processing */
int cflag;

void main (int argc, char **argv)
    {
    int pos;
    union utime tim;
    union udate dte;

    proc_args (&argc, argv, &tim, &dte);
    if (argc < 2)
        usage ();
    pos = 1;
    while (pos < argc)
        {
        if (touch_file (tim.dostime, dte.dosdate,
                        argv[pos], !cflag) < 0)
            fprintf (stderr,
                     "Cannot reset date/time of file %s\n",
                     argv[pos]);
        pos++;
        }
    }

/*
 * -------------------- touch_file() -----------------------
 * The actual workhorse. Given the time, date, filname, and
 * creation flag. Returns 0 on success, -1 on error. Uses
 * INT 21h, function 0x57 to do the dirty work.
 */

int touch_file (int dostime, int dosdate,
                char *filename, int creatit) 
    {
    int fd;
    if (!creatit && access (filename, 0) != 0)
        return (0);
    else if (creatit && access (filename, 0) != 0)
        fd = open(filename, O_RDWR | O_CREAT,
                  S_IREAD | S_IWRITE);
    else
        fd = open (filename, O_RDWR);
    if (fd < 0)
        return (-1);
    regs.h.ah = 0x57;
    regs.h.al = 1;
    regs.x.bx = fd;
    regs.x.cx = dostime;
    regs.x.dx = dosdate;
    int86 (0x21, &regs, &regs);

    close (fd);
    return (0);
    }

/*
 * ------------------- get_current_time() ------------------
 * Gets the current system date and time from MS-DOS, stuffs
 * the results into structures of type DTE and TIM.
 */

int get_current_time (TIM *ptim, DTE *pdte)
    {
    
    regs.h.ah = 0x2c;
    int86 (0x21, &regs, &regs);

    ptim->hours = regs.h.ch;
    ptim->minutes = regs.h.cl;
    ptim->seconds = regs.h.dh / 2;

    regs.h.ah = 0x2a;
    int86 (0x21, &regs, &regs);

    pdte->year = regs.x.cx - 1980;
    pdte->month = regs.h.dh;
    pdte->day = regs.h.dl;
    
    return (0);
    }

/*
 * ---------------------- time2struct() --------------------
 * Given the time specified on the command-line with the -t
 * option,this function turns it into structures of type DTE
 * and TIM. Some minimal error checking.
 */

int time2struct (TIM *tim, DTE *dte, char *deftime)
    {
    char buf[3];

    strncpy (buf, deftime + 2, 2);
    buf[2] = '\0';
    dte->month = atoi (buf);
    if (dte->month > 12 || dte->month < 1)
        return (-1);
    
    strncpy (buf, deftime + 4, 2);
    buf[2] = '\0';
    dte->day = atoi (buf);
    if (dte->day > 31 || dte->day < 1)
        return (-1);
    strncpy (buf, deftime, 2);
    buf[2] = '\0';
    if (atoi (buf) < 80)
        dte->year = atoi (buf) + 20;
    else
        dte->year = atoi (buf) - 80;
    strncpy (buf, deftime + 6, 2);
    buf[2] = '\0';
    tim->hours = atoi (buf);
    if (tim->hours > 24)
        return (-1);
    strncpy (buf, deftime + 8, 2);
    buf[2] = '\0';
    tim->minutes = atoi (buf);
    if (tim->minutes > 59)
        return (-1);
    strncpy (buf, deftime + 10, 2);
    buf[2] = '\0';
    if (atoi(buf) > 59)
        return (-1);
    tim->seconds = atoi (buf) / 2;
    return (0);
    }

/*
 * ------------------- file_time2struct() ------------------
 * Given a filename, gets the date/time for that file from
 * MS-DOS, and stuffs the result into a pair of integers
 * whose addresses were passed as parameters.
 */

int file_time2struct (int *dostime,
                      int *dosdate, char *filename) 
    {
    int fd;

    fd = open (filename, O_RDWR);
    if (fd < 0)
        return (-1);
    
    regs.h.ah = 0x57;
    regs.h.al = 0;
    regs.x.bx = fd;
    int86 (0x21, &regs, &regs);
    *dostime = regs.x.cx;
    *dosdate = regs.x.dx;

    close (fd);
    return (0);
    }

/* usage() - Show'em how to use it. */
void usage()
    {
    fprintf (stderr,
       "Usage: touch [-t date] [-f filename] [-c] <files>\n");
    fprintf (stderr, "  Examples:\n");
    fprintf (stderr, "       touch <files>\n");
    fprintf (stderr, "       touch -t yymmddhhmmss <files>\n");
    fprintf (stderr, "       touch -f <filename> <files>\n");
    fprintf (stderr,
             "  The first form sets the timestamp of files "
             "to the then-current time.\n"
             "  The second form sets the timestamp of files "
             "to the specified time.\n"
             "  The third form sets the timestamp of files "
             "to the time of filename.\n"
             "  Touch will create files that do not exist "
             "unless -c is used.\n");
    exit (0);
    }

/* proc_args()
 *
 * Process command-line arguments. Sets a flag for each
 * option argument found, then removes it from argv and
 * decrements argc, so that when it is done only non-option
 * arguments remain.
 */
int proc_args (int *pargc, char **argv,
              union utime *tim, union udate *dte) 
    {
    int tflag, fflag;
    int j, k, m;

    tflag = fflag = cflag = 0;

    if (get_current_time (&tim->tim, &dte->dte) < 0)
        {
        fprintf (stderr,
                "Error getting current system time\n");
        exit (1);
        }
    for ( j = 0 ; j < *pargc ; j++ )
        {
        if (argv[j][0] == '-')
            {
            switch (argv[j][1])
                {
                case 't':
                    if (fflag == 1)
                        usage();
                    if (time2struct (&tim->tim,
                                &dte->dte, argv[j + 1]) < 0)
                        usage();
                    tflag = 1;
                    m = 2;
                    break;
                case 'f':
                    if (tflag == 1)
                        usage();
                    if (file_time2struct ((int *)&tim->tim,
                          (int*)&dte->dte, argv[j + 1]) < 0)
                        usage();
                    fflag = 1;
                    m = 2;
                    break;
                case 'c':
                    cflag = 1;
                    m = 1;
                    break;
                default:
                    usage();
                }
            for ( k = j ; k < *pargc ; k++ )
                if (k + m < *pargc)
                    argv[k] = argv[k + m];
            (*pargc) -= m;
            j--;
            }
        else if (argv[j][0] == '\0')
            return (-1);
        }
    return (1);
    }

