// loaddrv.c - Dynamic driver install/start/stop/remove

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "loaddrv.h"

// globals
HINSTANCE hInst = NULL;
SC_HANDLE hSCMan = NULL;

// private prototypes
LRESULT CALLBACK DlgProc(HWND, UINT, WPARAM, LPARAM);
DWORD DriverInstall(LPSTR, LPSTR);
DWORD DriverStart(LPSTR);
DWORD DriverStop(LPSTR);
DWORD DriverRemove(LPSTR);
void GetServiceName(HWND, LPSTR, LPSTR);
void DisplayStatus(HWND, DWORD);

/**-----------------------------------------------------**/
int APIENTRY WinMain(HINSTANCE hInstance, 
   HINSTANCE hPrevInst, LPSTR lpCmdLine, int nShow)
{
   hInst = hInstance;
   return DialogBox(hInst, "IDD_DRIVER", NULL, DlgProc);
} // WinMain

/**-----------------------------------------------------**/
LRESULT CALLBACK DlgProc(HWND hDlg, UINT uMsg, 
   WPARAM wParam, LPARAM lParam)
{
   DWORD dwStatus = OKAY;
   char szPath[MAX_PATH], szDriver[MAX_PATH];

   switch (uMsg) 
   {
      case WM_INITDIALOG:
         SetDlgItemText(hDlg, IDC_DRVNAME, 
           "c:\\windows\\system32\\drivers\\");

         // connect to local service control manager
         if ((hSCMan = OpenSCManager(NULL, NULL, 
            SC_MANAGER_ALL_ACCESS)) == NULL) 
         {
            MessageBox(hDlg, 
               "Can't connect to service control manager",
               "LoadDrv", MB_OK);
            EndDialog(hDlg, FALSE);
         }
         return TRUE;

      case WM_COMMAND:
         switch(LOWORD(wParam)) 
         {
            case IDC_INSTALL:
               GetServiceName(hDlg, szPath, szDriver);
               dwStatus = DriverInstall(szPath, szDriver);
               DisplayStatus(hDlg, dwStatus);
               break;

            case IDC_START:
               GetServiceName(hDlg, szPath, szDriver);
               dwStatus = DriverStart(szDriver);
               DisplayStatus(hDlg, dwStatus);
               break;

            case IDC_STOP:
               GetServiceName(hDlg, szPath, szDriver);
               dwStatus = DriverStop(szDriver);
               DisplayStatus(hDlg, dwStatus);
               break;

            case IDC_REMOVE:
               GetServiceName(hDlg, szPath, szDriver);
               dwStatus = DriverRemove(szDriver);
               DisplayStatus(hDlg, dwStatus);
               break;

            case IDOK:
               if (hSCMan != NULL) CloseServiceHandle(hSCMan);
               EndDialog(hDlg, TRUE);
               return TRUE;
      } 
   } 
   return FALSE;
} // DlgProc

/**-----------------------------------------------------**/
DWORD DriverInstall(LPSTR lpPath, LPSTR lpDriver)
{
   BOOL dwStatus = OKAY;
   SC_HANDLE hService = NULL;

   // add to service control manager's database
   if ((hService = CreateService(hSCMan, lpDriver, 
      lpDriver, SERVICE_ALL_ACCESS, SERVICE_KERNEL_DRIVER,
      SERVICE_DEMAND_START, SERVICE_ERROR_NORMAL, lpPath, 
      NULL, NULL, NULL, NULL, NULL)) == NULL)
         dwStatus = GetLastError();
   else CloseServiceHandle(hService);

   return dwStatus;
} // DriverInstall

/**-----------------------------------------------------**/
DWORD DriverStart(LPSTR lpDriver)
{
   BOOL dwStatus = OKAY;
   SC_HANDLE hService = NULL;

   // get a handle to the service
   if ((hService = OpenService(hSCMan, lpDriver, 
      SERVICE_ALL_ACCESS)) != NULL) 
   {
      // start the driver
      if (!StartService(hService, 0, NULL))
         dwStatus = GetLastError();
   } else dwStatus = GetLastError();

   if (hService != NULL) CloseServiceHandle(hService);
   return dwStatus;
} // DriverStart

/**-----------------------------------------------------**/
DWORD DriverStop(LPSTR lpDriver)
{
   BOOL dwStatus = OKAY;
   SC_HANDLE hService = NULL;
   SERVICE_STATUS serviceStatus;

   // get a handle to the service
   if ((hService = OpenService(hSCMan, lpDriver, 
      SERVICE_ALL_ACCESS)) != NULL) 
   {
      // stop the driver
      if (!ControlService(hService, SERVICE_CONTROL_STOP,
         &serviceStatus))
            dwStatus = GetLastError();
   } else dwStatus = GetLastError();

   if (hService != NULL) CloseServiceHandle(hService);
   return dwStatus;
} // DriverStop

/**-----------------------------------------------------**/
DWORD DriverRemove(LPSTR lpDriver)
{
   BOOL dwStatus = OKAY;
   SC_HANDLE hService = NULL;

   // get a handle to the service
   if ((hService = OpenService(hSCMan, lpDriver, 
      SERVICE_ALL_ACCESS)) != NULL) 
   {
      // remove the driver
      if (!DeleteService(hService))
         dwStatus = GetLastError();
   } else dwStatus = GetLastError();

   if (hService != NULL) CloseServiceHandle(hService);
   return dwStatus;
} // DriverRemove

/**-----------------------------------------------------**/
void GetServiceName(HWND hDlg, LPSTR lpPath, LPSTR lpDriver)
{
   GetDlgItemText(hDlg, IDC_DRVNAME, lpPath, MAX_PATH);
   _splitpath(lpPath, NULL, NULL, lpDriver, NULL);
} // GetServiceName

/**-----------------------------------------------------**/
void DisplayStatus(HWND hDlg, DWORD dwStatus)
{
   char szStatus[128];

   if (!LoadString(hInst, dwStatus, szStatus, 128))
     LoadString(hInst, UNEXPECTED_ERROR, szStatus, 128);
   SetDlgItemText(hDlg, IDC_STATUS, szStatus);
} // DisplayStatus
