
/*=========================================================
 *
 * INT8.C
 * This program has an interrupt routine that is used to 
 * hook onto the timer interrupt (8h).  The interrupt is a 
 * protected-mode routine.
 *
 * A timer interrupt occurs 18.2 times per second 
 *********************************************************/

#include <string.h>
#include <dos.h>

/* redefine REGS, int86, int86x more Microsoft-compatible */
struct myDWORDREGS {
	unsigned int ax;
	unsigned int bx;
	unsigned int cx;
	unsigned int dx;
	unsigned int si;
	unsigned int di;
	unsigned int cflag;
};
union myREGS {
	struct myDWORDREGS x;
	struct BYTEREGS  h;
};
#define int86(__a,__b,__c) \
	int386(__a,(union REGS *)__b,(union REGS *)__c)
#define int86x(__a,__b,__c,__d) \
	int386x(__a,(union REGS *)__b,(union REGS *)__c,__d)

#define TSVXD_ID 0x4444

extern void __interrupt int8asm(void);
extern void _far *prev_int_8;
extern unsigned long tsactive,tsticks,jdata;
extern void _jmpadr(void);
extern long (_far *vxd)(long/*EAX*/, long/*EDX*/, 
	long/*EBX*/); /*register calling*/
extern long gotwindows;


static int dpmi_lockregion (void *addr, long size)
/**********************************************************
*
* NAME:         dpmi_lockregion
*
* DESCRIPTION:  DPMI function needed by interrupt handlers 
*               to prevent swapping. 
*
***********************************************************/
{
union myREGS ioregs; 

	ioregs.x.ax = 0x0600;
	ioregs.x.bx = *((unsigned short *)&addr+1);
	ioregs.x.cx = (unsigned short)addr;
	ioregs.x.si = *((unsigned short *)&size+1);
	ioregs.x.di = (unsigned short)size;
	int86(0x31, &ioregs, &ioregs);
	return ioregs.x.cflag;
}/*end of dpmi_lockregion*/


static int dpmi_unlockregion (void *addr, long size)
/**********************************************************
*
* NAME:         dpmi_ulockregion
*
* DESCRIPTION:  DPMI function needed by interrupt handlers 
*               to prevent swapping. 
*
**********************************************************/
{
union myREGS ioregs;

	ioregs.x.ax = 0x0601;
	ioregs.x.bx = *((unsigned short *)&addr+1);
	ioregs.x.cx = (unsigned short)addr;
	ioregs.x.si = *((unsigned short *)&size+1);
	ioregs.x.di = (unsigned short)size;
	int86(0x31, &ioregs, &ioregs);
	return ioregs.x.cflag;
}/*end of dpmi_unlockregion*/


long _cdecl switchit()
/**********************************************************
*
* NAME:         switchit
*
* DESCRIPTION:  Called by timer interrupt handler, after 
*               determining that segment registers are OK.
*               _cdecl means ax,bx,cx,dx can be clobbered.
*
* OUTPUTS:      returns with non-zero if we should perfrom 
*               a task switch now.
*
**********************************************************/
{
  if (tsactive && *(unsigned long *)0x46C - tsticks >= 4) {
	tsactive = 0;
	return(1); /* more than 1/4 second, so switch */
  }
  return(0); /* don't switch */
}/*end of switchit*/


void init_int8()
/**********************************************************
*
* NAME:         init_int8
*
* DESCRIPTION:  Initialize interrupt 8 handler for 
*               timeslicing.
*
**********************************************************/
{
union myREGS ioregs;
struct SREGS segregs;

   tsactive = 0;
   
   /* look for Windows VxD */
   
   ioregs.x.ax = 0x1600;
   int86(0x2F, &ioregs, &ioregs); /* Windows function */
   if (ioregs.h.al == 3 && ioregs.h.ah >= 10) {
	/* in Windows 3.1 enhanced mode DOS box */
	if (gotwindows < 0)
		return; /* already tried VxD and failed */

	/* call windows func to see if my VxD is present */
	memset(&segregs, 0, sizeof(segregs));
	ioregs.x.di = 0;
	ioregs.x.ax = 0x1684;
	ioregs.x.bx = TSVXD_ID;
	int86x(0x2F, &ioregs, &ioregs, &segregs);
	if (segregs.es == 0 && ioregs.x.di == 0) {
		gotwindows = -1;
		return; /* forget it */
	}
	
	vxd = MK_FP(segregs.es, ioregs.x.di);

	/* initialize VxD */

	if ((*vxd)(0x100,(long)&jdata,(long)&_jmpadr) < 0){
		gotwindows = -1;
		return; /* VxD initialization error */
	}

	gotwindows = 1;
   }

   dpmi_lockregion((void *)&int8asm, 200); /* approx size */
   dpmi_lockregion((void *)&_jmpadr, 10); /* approx size */
   dpmi_lockregion((void *)&switchit, 200); /* approx size*/
   dpmi_lockregion((void *)&tsticks, 60); /* lock asm vars*/

   ioregs.x.ax = 0x204;
   ioregs.h.bl = 8;
   int86(0x31, &ioregs, &ioregs); /* DPMI get prev int8 */
   prev_int_8 = MK_FP(ioregs.x.cx, ioregs.x.dx);       

   ioregs.x.ax = 0x205;
   ioregs.h.bl = 8;
   segread(&segregs);
   ioregs.x.cx = segregs.cs;
   ioregs.x.dx = (unsigned)&int8asm;
   int86(0x31, &ioregs, &ioregs); /* DPMI install int8 */
   
   tsticks = *(unsigned long *)0x46C; /* read bios ticks */
   tsactive = 1;
}/*end of init_int8*/


void deinit_int8()
/**********************************************************
*
* NAME:         deinit_int8
*
* DESCRIPTION:  Un-initialize interrupt 8 handler.
*
**********************************************************/
{
union myREGS ioregs;

   tsactive = 0;
   
   /* call DPMI func to re-install original int8 handler */

   ioregs.x.ax = 0x205;
   ioregs.h.bl = 8;
   ioregs.x.cx = FP_SEG(prev_int_8);
   ioregs.x.dx = FP_OFF(prev_int_8);
   int86(0x31, &ioregs, &ioregs); 
   
   if (gotwindows > 0) {
	(*vxd)(0x200, 0, 0); /* de-initialize vxd */
	gotwindows = 0;
   }
   
   dpmi_unlockregion((void *)&int8asm, 200);
   dpmi_unlockregion((void *)&_jmpadr, 10); /* approx size*/
   dpmi_unlockregion((void *)&switchit, 200);
   dpmi_unlockregion((void *)&tsticks, 60); /*unlock vars*/
}/*end of deinit_int8*/
