/* listing 2 - hotkeyhk.c
   DLL Module for hooking hot keys within Windows
   copyright 1994 Robert Mashlan
*/

#define STRICT
#include <windows.h>
#include <alloc.h>
#include "hotkeyhk.h"

// structure used to track each hot key
typedef struct {
   WORD wHotKey;   // hot key descriptor - virtual key code
                   //   in the low byte ORed and modifier
                   //   key flags in the high byte.
   HWND hwnd;      // window which gets notification
   LPARAM lParam;  // data passed back in lParam
   DWORD  hHotKey; // handle returned from
                   //   SetVirtualHotKey();
} HOTKEYSTRUCT, FAR *LPHOTKEYSTRUCT;

HINSTANCE hInst;    // hInstance of this module
HHOOK hKeyHook;     // handle for keyboard hook
HHOOK hShellHook;   // handle for shell hook
UINT wm_hotkeypressed; // registered windows message for
                       //  hotkey notification.
FARPROC lpfnVHKD;   // api entry point for VHKD.386

#define VHKDID 0x331b  // VxD id for VHKD

WORD nHotKeys;         // number of hot keys in list
LPHOTKEYSTRUCT lpHK;   // list of hot keys

// a handy macro to get what we need from keyboard
//   message lParam
#define KeyDown(lParam) ((lParam&(1L<<31))?FALSE:TRUE)

LRESULT CALLBACK _export HotKeyHook( int nCode,
   WPARAM wKey, LPARAM lParam )
// Hook procedure - called when a windows application
// is about to retrieve a WM_KEYDOWN, WM_KEYUP
//  WM_SYSKEYDOWN. WM_SYSKEYUP message.
{
   static WORD ModKeys;  // has bits set for ALT/SHIFT/CTRL

   if( nCode >= 0 && nCode != HC_NOREMOVE ) {
      int i;
      // update ModKeys if needed
      if( wKey == VK_SHIFT ) {
         if( KeyDown(lParam) )
            ModKeys |= SHIFT_BIT;
         else
            ModKeys &= ~SHIFT_BIT;
      } else if( wKey == VK_CONTROL ) {
         if( KeyDown(lParam) )
            ModKeys |= CONTROL_BIT;
         else
            ModKeys &= ~CONTROL_BIT;
      } else if( wKey == VK_MENU ) {
         if( KeyDown(lParam) )
            ModKeys |= ALT_BIT;
         else
            ModKeys &= ~ALT_BIT;
      } else {
         // search through list
         for(i=0;i<nHotKeys;i++) {
            if( lpHK[i].wHotKey == wKey+ModKeys ) {
                // found a match
                if( KeyDown(lParam) ) {
                   // if this key is a keydown, post a
                   //   notification message to the
                   //   client window.
                   if( IsWindow(lpHK[i].hwnd) )
                      PostMessage(lpHK[i].hwnd,
                         wm_hotkeypressed,wKey,
                         lpHK[i].lParam);
                }
                return 1;  // discard key message
            }
         }
      }
   }
   // give other hooks a crack at the message
   return CallNextHookEx(hKeyHook,nCode,wKey,lParam);
}

DWORD AddVirtualHotKey( WORD wHotKey )
// define a hot key with VHKD.386
// entry: wHotKey = hot key descriptor
// returns: 32 bit handle
{
   WORD  wScan;
   WORD  wState = 0;  // shift key state
   if( !lpfnVHKD )  // no API, nothing to do
      return 0;

   // convert virtual key to scan code
   wScan = MapVirtualKey(LOBYTE(wHotKey),0);
   // convert Modifier key bit mappings
   if( wHotKey & SHIFT_BIT )
      wState |= 1;
   if( wHotKey & ALT_BIT )
      wState |= 0x100;
   if( wHotKey & CONTROL_BIT )
      wState |= 0x80;
   _asm mov ax,1           // VHKD API function 1
   _asm mov bx,[wScan];    // pass scan code
   _asm mov cx,[wState];   // pass state
   _asm call [lpfnVHKD];   // call API
   // result in dx:ax
#ifdef __BORLANDC__
   // to remove warning
   return ((DWORD)_DX<<16)|_AX;
#endif
}

void  RemoveVirtualHotKey( DWORD hHotKey )
// undefines a hot key with VHKD.386
// entry: hHotKey is 32 bit handle returned
//        by AddVirtualHotkey()
{
   if( !lpfnVHKD ) return; // no API

   _asm mov ax,2;          // VHKD API function 2
   _asm mov bx,word ptr [hHotKey];
   _asm mov cx,word ptr [hHotKey+2];
   _asm call [lpfnVHKD];   // call VHKD API
}

void RemoveKeyHook( void )
// removes Windows keyboard hook and
//   undefines hot keys with VHKD.386
{
   int i;
   // remove virtual hot keys
   for(i=0;i<nHotKeys;i++)
      if( lpHK[i].hHotKey )
         RemoveVirtualHotKey(lpHK[i].hHotKey);
   // remove hook
   if(hKeyHook)
      UnhookWindowsHookEx(hKeyHook);
   hKeyHook = NULL;
}

BOOL InstallKeyHook( void )
// installs Windows keyboard hook and defines
//  hot keys with VHKD.386
{
   int i;
   if( hKeyHook )
     RemoveKeyHook();
   if( !nHotKeys) return FALSE;  // no need to install hook

   // install virtual hot keys
   for(i=0;i<nHotKeys;i++)
      lpHK[i].hHotKey = AddVirtualHotKey(lpHK[i].wHotKey);
   // install hook
   hKeyHook =
      SetWindowsHookEx(WH_KEYBOARD,HotKeyHook,hInst,NULL);
   return (hKeyHook!=NULL);
}


int WINAPI _export AddHotKey( HWND hwnd, WORD wHotKey,
    LPARAM lParam )
// adds a hot key to the list
// entry: hwnd = windows to post notfication
//        wHotKey = hot key descriptor
//                  low byte = virtual key code
//                  high byte = modifier flags
{
   int result = 0;
   int i;

   if( !IsWindow(hwnd) ) return 1;  // invalid hwnd

   RemoveKeyHook();  // remove the hook

   // allocate memory for the new item
   if( nHotKeys == 0 )
      lpHK = farmalloc(sizeof(HOTKEYSTRUCT));
   else {
      LPHOTKEYSTRUCT lpHKnew =
        farrealloc(lpHK,sizeof(HOTKEYSTRUCT)*(nHotKeys+1));
      if( lpHKnew == NULL )
         result = -1; // out of memory!
      else
         lpHK = lpHKnew;
   }
   if( result == 0 ) {
      // save the hot key data
      lpHK[nHotKeys].hwnd = hwnd;
      lpHK[nHotKeys].wHotKey = wHotKey;
      lpHK[nHotKeys].lParam = lParam;
      nHotKeys++;
      // check for a duplicate key
      for(i=0;i<nHotKeys-1;i++) {
         if( lpHK[i].wHotKey == wHotKey ) {
            result = 2;
            break;
         }
      }
      result = 1;
   }
   InstallKeyHook();  // install the hook
   return result;
}

int WINAPI _export RemoveHotKey( HWND hwnd, WORD wHotKey )
// remove a hot key from the list
// entry: hwnd = owner of hot key
//        wHotKey = hot key descriptor or 0 for all
//                    hot keys added by window
{
   int i, j;
   int result = 0;
   RemoveKeyHook();
   for( i=0;i<nHotKeys;i++) {
       if( lpHK[i].hwnd == hwnd &&
         ( lpHK[i].wHotKey == wHotKey || wHotKey == 0 ) )
          // remove entry from list
          for( j = i+1; j<nHotKeys; j++ )
             lpHK[j-1] = lpHK[j];
          nHotKeys--;
          result = 1;
          // resize the memory
          if(nHotKeys)
            farrealloc(lpHK,sizeof(HOTKEYSTRUCT)*nHotKeys);
          else
            farfree(lpHK);
   }
   InstallKeyHook();
   return result;
}


LRESULT CALLBACK _export ShellHook( int nCode,
   WPARAM wParam, LPARAM lParam )
// shell hook to put our keyboard hook first
//  after another app starts.
{
   if( nCode == HSHELL_WINDOWCREATED ) {
      RemoveKeyHook();  // reinstall keyboard hook to put
      InstallKeyHook(); //  our hook at the top of the list
   }
   return CallNextHookEx(hShellHook,nCode,
      wParam,lParam);
}

BOOL GetVHKDEntryPoint( void )
// get the API entry point VHKD.386 if installed
{
   // test for enhanced mode windows
   if( (GetWinFlags() & WF_ENHANCED) == 0 )
      return FALSE;
   // get entry point for VHDK API
   _asm mov ax, 1684h;
   _asm mov bx, VHKDID
   _asm int 2fh
   _asm mov word ptr [lpfnVHKD], di;
   _asm mov word ptr [lpfnVHKD+2], es;
   return lpfnVHKD != NULL;
}


#pragma argsused
int FAR PASCAL LibMain( HINSTANCE hInstance, WORD wDataSeg,
   WORD cbHeapSize, LPSTR lpCmdLine)
{
   hInst = hInstance;

   // get entry point for VHKD
   GetVHKDEntryPoint();
   // register notification message
   wm_hotkeypressed = RegisterWindowMessage(HOTKEYPRESSED);
   // install shell hook
   hShellHook =
      SetWindowsHookEx(WH_SHELL,ShellHook,hInst,NULL);
   return 1;
}

#pragma argsused
int CALLBACK _export WEP( int n )
{
   // remove shell hook
   if(hShellHook)
      UnhookWindowsHookEx(hShellHook);
   RemoveKeyHook();  // remove hook if needed
   return 1;
}



