;*****************************************************;
; WddjTee.asm                                         ;
; -- VxD traps writes to stdout and stderr and "tees" ;
;    them through a PM callback.                      ;
; -- PRB: Modified Sep. 11 '94 to use the VM control  ;
;    block to store the per-VM queue.                 ;
;*****************************************************;
.386p

;*****************************************************;
; Header files.                                       ;
;*****************************************************;
include vmm.inc

;*****************************************************;
; Constants.                                          ;
;*****************************************************;
wDeviceId     equ 31b3h ; Device id from Microsoft.
bVersionMajor equ 01    ; Major byte of version number.
bVersionMinor equ 01    ; Minor byte of version number.
wVersion      equ (bVersionMajor * 256 + bVersionMinor)
cbQueue       equ 10000h ; Size of a circular queue.
nfcVMCreate   equ 0     ; New VM created.
nfcVMDestroy  equ 1     ; VM destroyed.
nfcDataReady  equ 2     ; Data for VM is available.

;*****************************************************;
; Data structures.                                    ;
;*****************************************************;
CQS struc ; Circular Queue Structure.
    hvm       dd (?) ; VM handle.
    hsem      dd (0) ; Overflow prevention semaphore.
    ibHead    dd (?) ; Read end.
    ibTail    dd (?) ; Write end.
    fFull     dd (?) ; If queue is full.
    fInNotify dd (?) ; Client notified of data ready.
    ; Array of cbQueue bytes here.
CQS ends

;*****************************************************;
; Globals.                                            ;
;*****************************************************;
VxD_LOCKED_DATA_SEG
bcqs       dd (0) ; Offset of CQS in Control Block.
VxD_LOCKED_DATA_ENDS

VxD_DATA_SEG
lpfnNotify dd (0) ; Client callback.

; Protect mode API dispatch table.
rgpfn label DWORD
    dd offset32 WGetVersion
    dd offset32 CchRead
    dd offset32 RegisterLpfn
cpfn = ($ - rgpfn) / 4 - 1
VxD_DATA_ENDS

;*****************************************************;
; Device header.                                      ;
;*****************************************************;
Declare_Virtual_Device  wddjtee, bVersionMajor, \
    bVersionMinor, DispatchControl, wDeviceId, \
    Undefined_Init_Order,, PMApi

VxD_Locked_Code_Seg
;*****************************************************;
; Control procedures.                                 ;
;*****************************************************;

BeginProc DispatchControl
;*****************************************************;
; -- Control dispatcher.                              ;
;*****************************************************;
    Begin_Control_Dispatch
    Control_Dispatch Device_Init,  VxDInit
    Control_Dispatch VM_Init,      VMInit
    Control_Dispatch VM_Terminate, VMEnd
    End_Control_Dispatch
    clc
    ret
EndProc DispatchControl

BeginProc VxDInit
;*****************************************************;
; -- Install the V86 Int 21h hook.                    ;
;*****************************************************;
    ; Reserve space in future VM control blocks for
    ; queue structure.
    VMMcall   _Allocate_Device_CB_Area, \
              <<cbQueue + size CQS>, 0>
    or        eax, eax
    jz short  VxDInitFail        ; Did'na work capt'n!
    mov       [bcqs], eax        ; Rememeber offset.

    ; Install the V86 int 21h hook procedure.
    push      esi
    mov       eax, LF_Use_Heap OR LF_Alloc_Error
    mov       ecx, cbQueue + size CQS
    mov       eax, 21h          ; Set ISR.
    mov       esi, OFFSET32 Int21Hook
    VMMCall   Hook_V86_Int_Chain
    pop       esi
    ret

VxDInitFail:
    stc
    ret
EndProc   VxDInit

BeginProc VMInit
;*****************************************************;
; -- Initialize the circular queue for this VM.       ;
;*****************************************************;
    VMMCall   Test_Sys_VM_Handle ; DOS box?
    jz short  VMInitDone         ; No, so skip.

    ; Initialize queue.
    mov       eax, ebx
    call      PcqsFindHvm
    mov       [ebx].hvm, eax

    ; Create overflow prevention semaphore.
    mov       ecx, 0
    VMMCall   Create_Semaphore
    jc short  VMInitDone
    mov       [ebx].hsem, eax
    mov       eax, nfcVMCreate
    call      NotifyClient

VMInitDone:
    clc
    ret
EndProc   VMInit

BeginProc VMEnd
;*****************************************************;
; -- Free the circular queue for this VM.             ;
;*****************************************************;
    pushad
    call      PcqsFindHvm

    ; Destroy semaphore.
    mov       eax, [ebx].hsem
    or        eax, eax
    jz short  VMEndDoNotify
    VMMCall   Destroy_Semaphore

VMEndDoNotify:
    ; Notify client of VM's doom.
    mov       eax, nfcVMDestroy
    call      NotifyClient
    popad
    clc
    ret
EndProc   VMEnd

BeginProc PcqsFindHvm
;*****************************************************;
; -- Return the circular queue for the given VM.      ;
; -- EBX : VM handle on input, queue on output.       ;
;*****************************************************;
    add       ebx, [bcqs]
    ret
EndProc   PcqsFindHvm

VxD_Locked_Code_Ends

VxD_Code_Seg
;*****************************************************;
; Procedures.                                         ;
;*****************************************************;

BeginProc Int21Hook
;*****************************************************;
; -- Virtual 86 mode int 21h hook function.           ;
;*****************************************************;
    pushad                         ; Save regs.

    cmp       [lpfnNotify], 0      ; Is anyone
    je        Int21HookNextHandler ; interested?

    mov       ax, [ebp].Client_AX  ; Function number.
    cmp       ah, 40h              ; Write?
    jne       Int21HookNextHandler ; No, so skip.
    VMMCall   Test_Sys_VM_Handle   ; DOS box?
    jz        Int21HookNextHandler ; No, so skip.
    cmp       [ebp].Client_BX, 1   ; < stdout?
    jl        Int21HookNextHandler ; Yes, so skip.
    cmp       [ebp].Client_BX, 2   ; > stderr?
    jg        Int21HookNextHandler ; Yes, so skip.
    cmp       [ebp].Client_CX, 0
    jz        Int21HookNextHandler

    ; Find queue.
    call      PcqsFindHvm
    mov       edi, ebx
    add       edi, size CQS        ; pcqs->rgb.
    add       edi, [ebx].ibTail

    ; Get linear address of buffer.
    movzx     esi, [ebp].Client_DS ; Src segment.
    shl       esi, 4               ; Linearize
    movzx     eax, [ebp].Client_DX ; plus offset.
    add       esi, eax

    ; Get length of copy.
    movzx     ecx, [ebp].Client_CX ; # bytes out.
    call      CbCqs                ; # bytes queued.
    neg       eax
    add       eax, cbQueue - 1     ; # bytes free.
    cmp       ecx, eax             ; > queue size?
    jl short  Int21HookOverflowChecked
    mov       eax, [ebx].hsem
    push      ecx
    mov       ecx, Block_Svc_Ints
    mov       [ebx].fFull, 1       ; Remember fact.
    VMMCall   Wait_Semaphore
    pop       ecx

Int21HookOverflowChecked:
    mov       eax, [ebx].ibTail    ; Current write pos.
    add       eax, ecx             ; Calculate new
    call      WMod                 ; write position and
    mov       [ebx].ibTail, eax    ; record it.

Int21HookGotRead:
    cld                            ; Prepare for loop.
    mov       eax, ebx
    add       eax, cbQueue + size CQS

Int21HookCopyLoop:
    cmp       edi, eax             ; Wrapped?
    jl short  Int21HookNoWrap      ; No, we're ok.
    mov       edi, ebx             ; Yes, restart.
    add       edi, size CQS

Int21HookNoWrap:
    movsb                          ; Copy a byte.
    loop      Int21HookCopyLoop    ; Until no more.

    ; Notify client of new data.
    cmp       [ebx].fInNotify, 0   ; Is one pending?
    jnz short Int21HookNextHandler
    mov       [ebx].fInNotify, 1
    mov       eax, nfcDataReady
    call      NotifyClient

Int21HookNextHandler:
    popad                          ; Restore regs.
    stc                            ; Chain to next.
    ret
EndProc Int21Hook

BeginProc CbCqs
;*****************************************************;
; -- Return the number of bytes in the queue.         ;
; -- On input  : EBX : pointer to CQS.                ;
; -- On output : EAX : number of bytes.               ;
;*****************************************************;
    mov       eax, [ebx].ibTail
    sub       eax, [ebx].ibHead
    add       eax, cbQueue
    call      WMod
    ret
EndProc   CbCqs

BeginProc WMod
;*****************************************************;
; -- Return the mod of the input argument (eax) with  ;
;    the queue size (cbQueue) in eax.                 ;
;*****************************************************;
    push      edx          ; Save these guys.
    push      ecx
    xor       edx, edx     ; Clear hi 32 bits.
    mov       ecx, cbQueue ; Can't use immediate
    idiv      ecx          ; operand in division.
    mov       eax, edx     ; Get remainder.
    pop       ecx          ; Restore these guys.
    pop       edx
    ret
EndProc   WMod

BeginProc PMApi, Public
;*****************************************************;
; -- Main dispatch point for protected mode API.      ;
; -- On input:                                        ;
;    -- Client_AX : API id to execute.                ;
; -- Jump to corresponding API routine.               ;
;*****************************************************;
    movzx     eax, [ebp].Client_AX ; Function number.
    cmp       ax, cpfn             ; Within range?
    ja short  PMApiDone            ; No, so exit.
    jmp       rgpfn[eax * 4]       ; Yes, call handler.
PMApiDone:
    ret
EndProc   PMApi

;*****************************************************;
; -- Return the version number.                       ;
;*****************************************************;
BeginProc WGetVersion, Public
    mov       [ebp].Client_AX, wVersion
    ret
EndProc   WGetVersion

BeginProc RegisterLpfn, Public
;*****************************************************;
; -- Register the client's callback function.         ;
; -- Client_ES:Client_DI : Pointer to callback.       ;
;*****************************************************;
    mov         ax, [ebp].Client_ES     ; Selector.
    shl         eax, 16
    mov         ax, [ebp].Client_DI     ; Offset.
    mov         [lpfnNotify], eax
    ret
EndProc   RegisterLpfn

BeginProc CchRead, Public
;*****************************************************;
; -- Read the bytes tee'd from the given VM.          ;
; -- On Entry:                                        ;
;    -- Client_DX:Client_BX : VM handle.              ;
; -- On Exit:                                         ;
;    -- Client_DS:Client_SI : Output buffer.          ;
;    -- Client_AX           : # bytes copied.         ;
;*****************************************************;
    push      es                  ; Save stuff
    pushad
    mov       [ebp].Client_AX, 0  ; Assume failure.

    mov       bx, [ebp].Client_DX ; Get VM handle.
    shl       ebx, 16
    mov       bx, [ebp].Client_BX
    VMMCall   Validate_VM_Handle  ; Is it valid?
    jc short  CchReadDone         ; No, so exit.

    call      PcqsFindHvm         ; Get queue ptr.
    call      CbCqs               ; # bytes to copy.
    mov       [ebp].Client_AX, ax ; Return # bytes.
    or        eax, eax            ; Are there any?
    jz short  CchReadDone         ; No, so exit.
    mov       ecx, eax            ; Loop counter.

    mov       es, [ebp].Client_DS ; Get dest buffer.
    movzx     edi, [ebp].Client_SI
    mov       esi, ebx            ; Get head address.
    add       esi, size CQS       ; pcqs->rgb
    mov       eax, [ebx].ibHead
    add       esi, eax            ; + pcqs->ibHead.

    add       eax, ecx            ; Get new head.
    call      WMod
    mov       [ebx].ibHead, eax   ; Record it.

    mov       eax, ebx            ; Get source limit
    add       eax, cbQueue + size CQS ; address.
    cld                           ; Prepare to loop.

CchReadDoLoop:
    cmp       esi, eax            ; Wrapped?
    jl short  CchReadNoWrap       ; No, we're ok.
    mov       esi, ebx            ; Yes, start over.
    add       esi, size CQS
CchReadNoWrap:
    movsb                         ; Copy a byte.
    loop      CchReadDoLoop       ; Until no more.

CchReadDone:    ; If VM is blocked, wake it up.
    cmp       [ebx].fFull, 0
    je short  CchReadOverflowChecked
    mov       eax, [ebx].hsem
    VMMCall   Signal_Semaphore
    mov       [ebx].fFull, 0

CchReadOverflowChecked:
    popad                         ; Restore stuff.
    pop       es
    ret
EndProc   CchRead

BeginProc NotifyClient, Public
;*****************************************************;
; -- Schedule call back to system VM to call client's ;
;    notification callback.                           ;
; -- EAX : Notifcation code.                          ;
; -- EBX : Queue.                                     ;
;*****************************************************;
    pushad
    cmp        [lpfnNotify], 0
    jz short   NotifyClientDone

    mov        edx, [ebx].hvm ; AX <-- code, hi 16 bits
    mov        edx, [edx].CB_VMID
    shl        edx, 16        ; gets VM id.
    or         edx, eax

    VMMCall    Get_Sys_VM_Handle ; Call sys VM.
    mov        esi, offset32 DoNotify
    VMMCall    Call_VM_Event

NotifyClientDone:
    popad
    ret
EndProc   NotifyClient

BeginProc DoNotify, Public
;*****************************************************;
; -- Post notifcation to client via callback.         ;
;*****************************************************;
    pushad
    Push_Client_State
    VMMCall   Begin_Nest_Exec
    mov       eax, edx        ; Notification code.
    VMMCall   Simulate_Push
    mov       ebx, edx        ; Get VM id.
    shr       ebx, 16
    call      PcqsFindVmi     ; Get queue.
    cmp       dx, nfcDataReady
    jnz short DoNotifyPushVmh
    mov       [ebx].fInNotify, 0

DoNotifyPushVmh:
    mov       eax, [ebx].hvm  ; VM handle.
    shr       eax, 16
    VMMCall   Simulate_Push   ; Hi 16 bits.
    mov       eax, [ebx].hvm  ; VM handle.
    VMMCall   Simulate_Push   ; Lo 16 bits.
    cmp       dx, nfcVMDestroy
    jnz short DoNotifyCall

DoNotifyCall:
    mov       cx, word ptr [lpfnNotify + 2]
    movzx     edx, word ptr [lpfnNotify]
    VMMCall   Simulate_Far_Call
    VMMCall   Resume_Exec
    VMMCall   End_Nest_Exec
    Pop_Client_State
    popad
    ret
EndProc   DoNotify

BeginProc PcqsFindVmi
;*****************************************************;
; -- Find the circular queue for the given VM ID.     ;
; -- EBX : VM ID on input, queue on output.           ;
; -- If found carry is clear, else carry is set.      ;
;*****************************************************;
    mov       eax, ebx
    VMMCall   Get_Sys_VM_Handle   ; Start here.

PcqsFindVmiGetNext:
    VMMCall   Get_Next_VM_Handle
    VMMCall   Test_Sys_VM_Handle  ; Back to first?
    jz short  PcqsFindVmiNotFound
    cmp       [ebx].CB_VMID, eax  ; Is this it?
    jnz short PcqsFindVmiGetNext  ; Nope, try again.
    call      PcqsFindHvm         ; Yup, we're done.
    clc
    ret

PcqsFindVmiNotFound:
    stc                           ; Not found.
    ret
EndProc   PcqsFindVmi
VxD_Code_Ends
End
